<?php

namespace App\Http\Controllers\Backend;

use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Repositories\Backend\Auth\UserRepository;
use App\Models\Auth\CountryStudy;
use App\Models\Auth\User;
use App\Models\Auth\AgentCommission;
use App\Models\Auth\AgentDetails;
use Illuminate\Support\Facades\Http;
use App\Models\Auth\UniversityApplication;
use Yajra\DataTables\DataTables;
use App\Http\Requests\Backend\Auth\User\ManageUserRequest;
use App\Events\Backend\Auth\User\UserDeleted;
use App\Models\Auth\Country;
use App\Models\StaffBranch;
use App\Models\UserDocuEsign;
use App\Models\Auth\UserDetails;
use App\Models\MailTemplates;
use App\Models\SMSTemplates;
use App\Notifications\Backend\UserAlert;
use App\Models\UserActivity;
use Log;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;

class AgentManagementController extends Controller
{

  public function __construct()
  {
    $this->middleware(function ($request, $next) {
      $user = \Auth::user();
      if (!(\Auth::user()->isAdmin() || \Auth::user()->isAgent() || \Auth::user()->can('agent management'))) {
        return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
      }

      // if(!env('AGENTS_ENABLE')){
      //   return redirect()->route('admin.dashboard.message')->withFlashDanger('Agent Management is disabled.');
      // }
      return $next($request);
    });
  }

  public function getAgentCount(): int
  {
    return  User::with('roles', 'permissions', 'providers')
      ->whereHas('roles', function ($q) {
        $q->whereIn('name', ['agent']);
      })
      ->count();
  }

  public function index(Request $request)
  {
    if (!(\Auth::user()->isAdmin() || \Auth::user()->can('agent management'))) {
      return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
    }
    $branch = get_staff_branch(\Auth::user()->id);
    $current_branch = $request->input('branch');
    return view('backend.agents.agents_index', compact('branch', 'current_branch'));
  }

	public function agents_list(Request $request){
		// Enable query logging
    	DB::enableQueryLog();

		$columns = array(
			0 => 'first_name',
			2 => 'email',
      		3 => 'mobile',
			4 => 'status',
			5 => 'created_at',
		);

		$totalData = $this->getAgentCount();
		$totalFiltered = $totalData;
		$limit = $request->input('length');
		$start = $request->input('start');
		$order = $columns[$request->input('order.0.column')];
		$dir = $request->input('order.0.dir');
		$get_search = $request->input('search.value');

		$branch = "";
		if ($request->input('branch') != "all") {
			$branch = $request->input('branch');
		}

		$totalData = User::with('roles')
			->select(DB::raw('users.*'))
			->whereHas('roles', function ($q) {
				$q->whereIn('name', ['agent']);
			})
			->when(!empty($get_search), function ($query) use ($get_search) {
				return $query->where('users.first_name', 'like', "%$get_search%")
				->orWhere('users.last_name', 'like', "%$get_search%")
				->orWhere('users.email', 'like', "%$get_search%")
				->orWhere('users.mobile', 'like', "%$get_search%");
			})
			->when(!empty($branch), function ($query) use ($branch) {
				return $query->where('branch', $branch);
			})
			->count();

		$totalFiltered = User::with('roles')
			->leftJoin('agent_details', 'users.id', '=', 'agent_details.agent_id')
			->leftjoin('countries', 'users.country', '=', 'countries.id')
            ->leftjoin('states', 'users.state', '=', 'states.id')
			->leftjoin('branch', 'users.branch', '=', 'branch.id')
			->leftJoin('users as point_of_contact', 'agent_details.point_of_contact', '=', 'point_of_contact.id')
			->select(DB::raw('point_of_contact.first_name as point_cont_first_name, point_of_contact.last_name as point_cont_last_name, branch.branch_name as branch_name, countries.title_en as country_name, states.name as state_name, users.*,agent_details.business_name,agent_details.agent_category,agent_details.agent_id,agent_details.agent_id_proof,agent_details.buis_incorpo_certi_proof,agent_details.logo_approved,agent_details.agent_panel_logo,agent_details.point_of_contact'))
			->whereHas('roles', function ($q) {
				$q->whereIn('name', ['agent']);
			})
			->when(!empty($get_search), function ($query) use ($get_search) {
				return $query->where('users.first_name', 'like', "%$get_search%")
					->orWhere('users.last_name', 'like', "%$get_search%")
					->orWhere('users.email', 'like', "%$get_search%")
					->orWhere('users.mobile', 'like', "%$get_search%")
					->orWhereRaw("CONCAT(`users`.`first_name`, ' ', `users`.`last_name`) LIKE ?", ['%'.$get_search.'%']);
			})
			->when(!empty($branch), function ($query) use ($branch) {
				return $query->where('users.branch', $branch);
			})
			->offset($start)
			->limit($limit)
			->orderBy("users.created_at", "desc")
			->count();

    	$totalFiltered = $totalData;

		$agents = User::with('roles')
			// ->select(DB::raw('users.*'))
			->leftJoin('agent_details', 'users.id', '=', 'agent_details.agent_id')
			->leftjoin('countries', 'users.country', '=', 'countries.id')
            ->leftjoin('states', 'users.state', '=', 'states.id')
			->leftjoin('branch', 'users.branch', '=', 'branch.id')
			->leftJoin('users as point_of_contact', 'agent_details.point_of_contact', '=', 'point_of_contact.id')
			->select(DB::raw('point_of_contact.first_name as point_cont_first_name, point_of_contact.last_name as point_cont_last_name, branch.branch_name as branch_name, countries.title_en as country_name, states.name as state_name, users.*,agent_details.business_name,agent_details.agent_category,agent_details.agent_id,agent_details.agent_id_proof,agent_details.buis_incorpo_certi_proof,agent_details.logo_approved,agent_details.agent_panel_logo,agent_details.point_of_contact'))
			->whereHas('roles', function ($q) {
				$q->whereIn('name', ['agent']);
			})
			->when(!empty($get_search), function ($query) use ($get_search) {
				return $query->where('users.first_name', 'like', "%$get_search%")
					->orWhere('users.last_name', 'like', "%$get_search%")
					->orWhere('users.email', 'like', "%$get_search%")
					->orWhere('users.mobile', 'like', "%$get_search%")
					->orWhereRaw("CONCAT(`users`.`first_name`, ' ', `users`.`last_name`) LIKE ?", ['%'.$get_search.'%']);
			})
			->when(!empty($branch), function ($query) use ($branch) {
				return $query->where('users.branch', $branch);
			})
			->offset($start)
			->limit($limit)
			->orderBy("users.created_at", "desc")
			->get();

		$data = array();
		if (!empty($agents)) {
			foreach ($agents as $page) {
				$customResult['full_name'] = "<strong>
          <a href='".route('admin.agent.details', $page->id)."'><i class='fas fa-user-tie text-sub'></i> " . $page->full_name . "</a></strong>";

				if ($page->agent_panel_logo) {
					$agent_panel_logo = "storage/" . $page->agent_panel_logo;
					$customResult['full_name'] .= '<br><img src="' . url($agent_panel_logo) . '" alt="Current logo" style="height:70px;padding:8px 0px 10px">';
					if (!$page->logo_approved) {
						$customResult['full_name'] .= '<br><a href="' . route('admin.agent.agent-details.approved-disapproved', $page->id) . '" name="confirm_item" class="btn btn-sm text-success btn-link approvedOrDisapproved" data-id=' . $page->id . '> Approve Logo</a>';
					} else {
						$customResult['full_name'] .= '<br><a href="' . route('admin.agent.agent-details.approved-disapproved', $page->id) . '" name="confirm_item" class="btn btn-sm text-danger btn-link approvedOrDisapproved" data-id="' . $page->id . '"> Disapprove Logo</a>';
					}
				}
				if ($page->business_name) {
					$customResult['full_name'] .= "<br><i class='fas fa-briefcase text-sub'></i> " . $page->business_name;
				}

				if ($page->country || $page->city || $page->state)
					$customResult['full_name'] .= "<br /><span style='white-space:nowrap;'><i class='fas fa-globe text-sub'></i> <span class='text-sub'>";
					if ($page->address_1) {
						$customResult['full_name'] .= $page->address_1 . ", ";
					}
					if ($page->city) {
						$customResult['full_name'] .= $page->city . ", ";
					}
					if ($page->state) {
						if ($page->country) {
							$customResult['full_name'] .= $page->state_name . ", <br>";
						} else {
							$customResult['full_name'] .= $page->state_name;
						}
					}
					if ($page->country) {
						$customResult['full_name'] .= $page->country_name . " ";
					}
					$customResult['full_name'] .= "</span></span>";

					// $customResult['full_name'] .= "<br>".$page->address_1;
					// if($page->city){
					//   $customResult['full_name'] .= ", ".$page->city;
					// }
					// if($page->state){
					//   $customResult['full_name'] .= ", ".get_state_name($page->state);
					// }
					// $customResult['full_name'] .= " ".get_country_name($page->country);
					if ($page->branch) {
						$customResult['full_name'] .= "<br><span class='tagPink mb-1'>" . $page->branch_name . "</span>";
					}
					if ($page->point_of_contact) {
						$customResult['full_name'] .= "<br>Point of Contact: " . $page->point_cont_first_name . " " . $page->point_cont_last_name;
					}
					$customResult['agent_cat'] = "<strong>" . $page->agent_category . "</strong>";
					if ($page->active == 1) {
						$customResult['status'] = "<strong><span class='badge badge-success'>Enabled</span></strong>";
					} else {
						$customResult['status'] = "<strong><span class='badge badge-danger'>Disabled</span></strong>";
					}
					$customResult['signed_up'] = $page->created_at ?  date('d M Y', strtotime($page->created_at)) : "--";

					$customResult['email'] = $page->email;
					$customResult['email'] .= $page->confirmed ? " <i title='Confirmed' class=\"fas fa-check text-success\"></i>" : " <i title='Not Confirmed' class=\"fas fa-ban text-muted\"></i>";

					$mobile = "";
					if ($page->dialcode) {
						$mobile = "<i class='fas fa-phone-alt text-sub'></i> " . "+" . $page->dialcode . "-" . $page->mobile;
						$customResult['mobile'] = $mobile;
					} else {
						$mobile = $page->mobile;
						$customResult['mobile'] = "<br><i class='fas fa-phone-alt text-sub'></i> " . $mobile;
					}

					$customResult['options'] = '<div class="" aria-label="">
						<a style="padding: 4px 8px; font-size: 15px;" href="' . route('admin.agent.details', $page->id) . '" data-toggle="tooltip" data-placement="top"   class="btn btn-outline-success""><i class="fa fa-eye"></i> </a>

					<a target="_blank" style="padding: 4px 8px; font-size: 15px;" href="' . route('admin.communicate.mail', $page->id) . '" data-toggle="tooltip" data-placement="top" title="Email" class="btn btn-outline-success"><i class="fa fa-envelope"></i></a>
					<a target="_blank" style="font-size: 15px;padding: 4px 8px;" href="' . route('admin.communicate.sms', $page->id) . '" data-toggle="tooltip" data-placement="top" title="SMS" class="btn btn-outline-primary"><i class="fa fa-comment"></i></a>
					<div class="btn-group btn-group-sm" role="group">
						<button style="padding: 4px 8px !important;" id="userActions" type="button" class="btn btn-outline-danger dropdown-toggle" data-bs-toggle="dropdown" aria-expanded="true"><i class="fas fa-ellipsis-h"></i></button>
						<div class="dropdown-menu dropdown-menu-end dropdown-menu-arrow" data-popper-placement="bottom-end">';
					if (!$page->confirmed) {
						$customResult['options'] .= '<a href="' . route('admin.auth.user.account.confirm.resend', $page->id) . '" class="dropdown-item py-1"><i class="fas fa-share-square text-sub"></i> &nbsp;&nbsp;&nbsp;' . \Lang::get('buttons.backend.access.users.resend_email') . '</a>
								<a href="' . route('admin.auth.agent.confirm', $page->id) . '" class="dropdown-item py-1"><i class="fas fa-check text-sub"></i> &nbsp;&nbsp;&nbsp;Confirm Email</a>';
					}
					$customResult['options'] .= '<a href="' . route('admin.agents.edit', $page->id) . '" class="dropdown-item py-1"><i class="fa fa-edit text-sub"></i> &nbsp;&nbsp;&nbsp;Edit</a>';
					if ($page->active == 1) {
						$customResult['options'] .= '<a href="' . route('admin.auth.agent.mark', [$page->id, 0]) . '" class="dropdown-item py-1 text-danger"><i class="fas fa-eye-slash"></i> &nbsp;&nbsp;&nbsp;Deactivate</a>';
					} else {
						$customResult['options'] .= '<a href="' . route('admin.auth.agent.mark', [$page->id, 1]) . '" class="dropdown-item py-1 text-success"><i class="fas fa-eye"></i> &nbsp;&nbsp;&nbsp;Activate</a>';
					}
					if (!session()->has('admin_user_id') || !session()->has('temp_user_id')) {
						if ($page->id != auth()->id()) {
							$customResult['options'] .= '<a href="' . route('admin.auth.agent.login-as', $page->id) . '" class="dropdown-item py-1"><i class="fa fa-user text-sub"></i> &nbsp;&nbsp;&nbsp;Login as ' . $page->full_name . '</a>';
						}
					}
					$customResult['options'] .= '<a href="' . route('admin.auth.agent.change-password', $page->id) . '" class="dropdown-item py-1"><i class="fas fa-key text-sub"></i> &nbsp;&nbsp;&nbsp;' . \Lang::get('buttons.backend.access.users.change_password') . '</a>';
					if ($page->id != auth()->id() && $page->id != 1) {
						$customResult['options'] .= '<a href="' . route('admin.auth.agent.destroy', $page->id) . '" data-method="delete" data-trans-button-cancel="' . \Lang::get('buttons.general.cancel') . '" data-trans-button-confirm="' . \Lang::get('buttons.general.crud.delete') . '" data-trans-title="' . \Lang::get('strings.backend.general.are_you_sure') . '" class="dropdown-item py-1 text-danger"><i class="fas fa-trash-alt text-danger"></i> &nbsp;&nbsp;&nbsp;' . \Lang::get('buttons.general.crud.delete') . '</a>';
					}
					$customResult['options'] .= '</div>
					</div>
				</div>';
				$data[] = $customResult;
			}

			// Get the logged queries
			$queries = DB::getQueryLog();

			// Count the queries
			$queryCount = count($queries);
			\Log::debug(end($queries));
			\Log::debug("AGENT MANAGEMENT DATABASE QUERY COUNT: " . $queryCount);

			$json_data = array(
				"draw"            => intval($request->input('draw')),
				"recordsTotal"    => intval($totalData),
				"recordsFiltered" => intval($totalFiltered),
				"data"            => $data
			);

			return json_encode($json_data);
		}

	}

	public function create(){
		if (!(\Auth::user()->isAdmin() || \Auth::user()->can('agent management'))) {
			return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
		}
		// $dialcodes = Country::groupBy('tel')->orderBy('tel', 'asc')->get(['tel as code', 'title_en as name']);
		$dialcodes = Country::orderBy('country_code', 'asc')->get(['tel as code', 'code as country_code', 'title_en as name']);
		$country_list = Country::pluck('title_en', 'id');
		$branch = get_staff_branches(\Auth::user()->id);
		$userRepo = new UserRepository();
		$staff = $userRepo->getNonStudentsTrainers();

		return view('backend.agents.add_agent', ['staff' => $staff, 'branch' => $branch, 'dialcodes' => $dialcodes, 'country_list' => $country_list]);
	}

	public function store(Request $request){
		$request->validate([
			'business_name' => ['nullable', 'max:191', 'regex:/^[\pL\s]+$/u'],
			'first_name' => ['required', 'max:50', 'regex:/^[\pL\s]+$/u'],
			'last_name'  => ['required', 'max:50', 'regex:/^[\pL\s]+$/u'],
			'email' => ['required', 'email', 'max:100', Rule::unique('users')],
			'dialcode' => 'required',
			'mobile' => ['required', 'numeric', 'digits_between:' . config('app.USER_MOBILE_MIN_LENGTH') . ',' . config('app.USER_MOBILE_MAX_LENGTH')],
			'country' => 'required',
			'address' => 'nullable|max:191',
			'password' => 'required|min:6|confirmed',
			'branch' => 'required',
      // 'commission_percent' => 'required',
		]);

    // if (!isset($request->dialcode) || $request->dialcode == "") {
    //     $dialcode = env('USER_DEFAULT_DIALCODE');
    // }
    if ($request->file('agent_id_proof')) {
      $request->validate([
        'agent_id_proof' => 'mimes:jpeg,jpg,pdf'
      ], [
        'agent_id_proof.*.mimes' => 'Only pdf, jpeg, jpg files are allowed'
      ]);
    }
    if ($request->file('buis_incor_certi_proof')) {
      $request->validate([
        'buis_incor_certi_proof' => 'mimes:jpeg,jpg,pdf'
      ], [
        'buis_incor_certi_proof.*.mimes' => 'Only pdf, jpeg, jpg files are allowed'
      ]);
    }
    if ($request->file('agent_panel_logo')) {
      $request->validate([
        'agent_panel_logo' => 'image|mimes:jpeg,jpg,gif,png'
      ], [
        'agent_panel_logo.*.mimes' => 'Only pdf, doc, docx files are allowed'
      ]);
    }
    $agent_id_proof = "";
    $buis_incor_certi_proof = "";
    $agent_panel_logo = "";

    if ($request->file('agent_id_proof')) {
      $file = $request->agent_id_proof->getClientOriginalName();
      $filename = pathinfo($file, PATHINFO_FILENAME);
      if (strlen($filename) > 125)
        $filename = substr($filename, 0, 125);
      $filename = $filename . Str::random(20) . "." . $request->agent_id_proof->getClientOriginalExtension();
      $agent_id_proof = $request->file('agent_id_proof')->storeAs('/agent_doc', $filename, 'public');
    }
    if ($request->file('buis_incor_certi_proof')) {
      $file = $request->buis_incor_certi_proof->getClientOriginalName();
      $filename = pathinfo($file, PATHINFO_FILENAME);
      if (strlen($filename) > 125)
        $filename = substr($filename, 0, 125);

      $filename = $filename . Str::random(20) . "." . $request->buis_incor_certi_proof->getClientOriginalExtension();
      $buis_incor_certi_proof = $request->file('buis_incor_certi_proof')->storeAs('/agent_doc', $filename, 'public');
    }
    if ($request->file('agent_panel_logo')) {
      $file = $request->agent_panel_logo->getClientOriginalName();
      $filename = pathinfo($file, PATHINFO_FILENAME);
      if (strlen($filename) > 125)
        $filename = substr($filename, 0, 125);

      $filename = $filename . Str::random(20) . "." . $request->agent_panel_logo->getClientOriginalExtension();
      $agent_panel_logo = $request->file('agent_panel_logo')->storeAs('/agent_doc', $filename, 'public');
    }

    if (!isset($request->country) || $request->country == "") {
      $country = env('USER_DEFAULT_COUNTRY');
    } else {
      $country = $request->get('country');
    }

    $user = User::create([
      'first_name'     => $request->input('first_name'),
      'last_name'  => $request->input('last_name'),
      'is_agent' => 1,
      'email'    => $request->input('email'),
      'dialcode' => $request->get('dialcode'),
      'mobile' => $request->input('mobile'),
      'country' => $country,
      'state' => $request->get('state'),
      'city' => $request->get('city'),
      'address_1' => $request->input('address'),
      'password' => $request->input('password'),
      'branch' => $request->get('branch'),
      'active' => 1,
      'confirmed' => 1,
    ]);

    $agent_details = AgentDetails::create([
      'agent_id' => $user->id,
      'point_of_contact' => $request->input('point_of_contact'),
      'business_name'     => $request->input('business_name'),
      'agent_category'     => $request->input('agent_category'),
      'logo_approved'     => 1,
      // 'agent_website_url' => $request->input('agent_website_url'),
      'agent_id_proof' => $agent_id_proof,
      'buis_incorpo_certi_proof' => $buis_incor_certi_proof,
      'agent_panel_logo' => $agent_panel_logo,
      'commission_percent' => $request->input('commission_percent'),
      // 'facebook' => $request->get('facebook'),
      // 'instagram' => $request->get('instagram'),
      // 'twitter' => $request->get('twitter'),
      // 'others' => $request->get('others'),
    ]);

    if ($user) {
      $user->assignRole('agent');
      StaffBranch::create(["staff_id" => $user->id, "branch_id" => $request->get('branch')]);
    }

    $msg_details = "New Agent Added.<br /><br />Agent Name: " . get_user_full_name_by_id($user->agent_id) . "<br />email: " . $user->email . "<br />Mobile: " . $user->dialcode . ' ' . $user->mobile . "<br />Gender: " . $user->gender;
    $notify_emails = get_site_config('event_notification_email');
    if ($notify_emails) {
      send_event_notification_email(explode(",", $notify_emails), "New Agent Added", $msg_details);
    }

    //Email to Agent
    \Log::debug("Agent sending welcome email.....");
    $mail_tpl = MailTemplates::find("AGENT_CREATE");

    if ($mail_tpl && $mail_tpl->status) {
      \Log::debug("Agent sending welcome email status is active.....");
      $email_message = $mail_tpl->message;
      $full_name = $request->get('first_name') . ' ' . $request->get('last_name');
      $email_message = str_ireplace("#sitename#", app_name(), $email_message);
      $email_message = str_ireplace("#siteurl#", url(''), $email_message);
      $email_message = str_ireplace("#first_name#", $request->get('first_name'), $email_message);
      $email_message = str_ireplace("#full_name#", $full_name, $email_message);
      $email_message = str_ireplace("#email#", $request->get('email'), $email_message);
      $email_message = str_ireplace("#mobile#", $request->get('dialcode') . ' ' . $request->get('mobile'), $email_message);
      $email_message = str_ireplace("#password#", $request->get('password'), $email_message);
      try {
        //$user->notify(new UserAlert($mail_tpl->subject, $email_message));
        user_notify($user, $mail_tpl->subject, $email_message);
        \Log::debug("Agent: mail sent to user- " . $email_message);
        UserActivity::create(['user_id' => $user->id, 'type' => 'email_sent', 'staff' => \Auth::user()->id, 'detail' => "Email Sent: Subject:" . $mail_tpl->subject . "<br>Message:" . $email_message]);
      } catch (\Exception $e) {
        $message = $e->getMessage();
        \Log::debug("Mail Error: (NEW AGENT) " . $message);
      }
    }

    //Whatsapp
    if (get_site_config('whatsapp_enabled')) {
        $numbers = [$user->whatsapp_dialcode.$user->whatsapp]; 

        $hook = 'agent_welcome';

        send_whatsapp_message($numbers, $hook, $user->id);
    }

    $notify_sms = SMSTemplates::find("AGENT_CREATE");
    if ($notify_sms && $notify_sms->status) {
      \Log::debug("Agent sending welcome SMS Sending, status is active.....");
      $notify_sms->message = str_replace("#sitename#", app_name(), $notify_sms->message);
      $notify_sms->message = str_replace("#siteurl#", url('/'), $notify_sms->message);
      $notify_sms->message = str_replace("#first_name#", $request->get('first_name'), $notify_sms->message);
      $notify_sms->message = str_replace("#full_name#", $full_name, $notify_sms->message);
      $notify_sms->message = str_replace("#password#", $request->get('password'), $notify_sms->message);

      \Log::debug("SMS Sending: (AGENT CREATE)" . $notify_sms->message);
      $numbers = [$request->get('dialcode') . $request->get('mobile')];
      send_sms($numbers, $notify_sms->message, $notify_sms->template_id);
    }

    return redirect()->route('admin.agents.index')->withFlashSuccess("Agent created successfully");
  }

  public function approvedOrDisapproved($id)
  {
    $agent_details = AgentDetails::where('agent_id', $id)->first();
    if ($agent_details->logo_approved) {
      $agent_details->logo_approved = 0;
    } else {
      $agent_details->logo_approved = 1;
    }
    $agent_details->save();

    return redirect()->route('admin.agents.index')->withFlashSuccess("Business Logo Status Updated.");
  }

  public function edit($user_id)
  {
    if (!(\Auth::user()->isAdmin() || \Auth::user()->can('agent management'))) {
      return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
    }
    $user = User::find($user_id);
    $agent_details = AgentDetails::where('agent_id', $user_id)->first();
    if (!$agent_details) {
      $agent_details = AgentDetails::create([
        'agent_id' => $user->id,
      ]);
      $agent_details->save();
    }
    $country_list = Country::pluck('title_en', 'id');
		// $dialcodes = Country::groupBy('tel')->orderBy('tel', 'asc')->get(['tel as code', 'title_en as name']);
    $dialcodes = Country::orderBy('country_code', 'asc')->get(['tel as code', 'code as country_code', 'title_en as name']);
    $branch = get_staff_branches(\Auth::user()->id);
    $userRepo = new UserRepository();
    $staff = $userRepo->getNonStudentsTrainers();
    return view('backend.agents.edit_agent', compact('staff', 'user', 'branch', 'dialcodes', 'country_list', 'agent_details'));
  }

  public function update(Request $request, $user_id)
  {
    // dd($request->all());
    $request->validate([
      'business_name' => ['nullable', 'max:191', 'regex:/^[\pL\s]+$/u'],
      'first_name'     => 'required|max:191',
      'last_name'  => 'required|max:191',
      'email'    => 'required|email:filter,filter_unicode|max:191|unique:users,email,' . $user_id,
      'mobile' => 'required',
      'country' => 'required',
      'address' => 'nullable|max:191',
      'branch' => 'required',
      // 'commission_percent' => 'required',
    ]);

    $agent_details = AgentDetails::where('agent_id', $user_id)->first();

    if (!$agent_details) {
      $agent_details = AgentDetails::create([
        'agent_id' => $user_id,
      ]);
    }

    if ($request->file('agent_id_proof')) {
      $file = $request->agent_id_proof->getClientOriginalName();
      $filename = pathinfo($file, PATHINFO_FILENAME);
      if (strlen($filename) > 125)
        $filename = substr($filename, 0, 125);

      $filename = $filename . Str::random(20) . "." . $request->agent_id_proof->getClientOriginalExtension();
      $agent_details->agent_id_proof = $request->file('agent_id_proof')->storeAs('/agent_doc', $filename, 'public');
    }
    if ($request->file('buis_incor_certi_proof')) {
      $file = $request->buis_incor_certi_proof->getClientOriginalName();
      $filename = pathinfo($file, PATHINFO_FILENAME);
      if (strlen($filename) > 125)
        $filename = substr($filename, 0, 125);

      $filename = $filename . Str::random(20) . "." . $request->buis_incor_certi_proof->getClientOriginalExtension();
      $agent_details->buis_incorpo_certi_proof = $request->file('buis_incor_certi_proof')->storeAs('/agent_doc', $filename, 'public');
    }
    if ($request->file('agent_panel_logo')) {
      $file = $request->agent_panel_logo->getClientOriginalName();
      $filename = pathinfo($file, PATHINFO_FILENAME);
      if (strlen($filename) > 125)
        $filename = substr($filename, 0, 125);

      $filename = $filename . Str::random(20) . "." . $request->agent_panel_logo->getClientOriginalExtension();
      $agent_details->agent_panel_logo = $request->file('agent_panel_logo')->storeAs('/agent_doc', $filename, 'public');
    }

    $user = User::find($user_id);
    $user->first_name = $request->input('first_name');
    $user->last_name = $request->input('last_name');
    $user->is_agent = 1;
    $user->email = $request->input('email');
    $user->dialcode = $request->input('dialcode');
    $user->mobile = $request->input('mobile');
    $user->country = $request->get('country');
    $user->state = $request->get('state');
    $user->city = $request->get('city');
    $user->address_1 = $request->input('address');
    $user->branch = $request->input('branch');
    $user->save();

    $agent_details->point_of_contact = $request->input('point_of_contact');
    $agent_details->business_name     = $request->input('business_name');
    $agent_details->agent_category     = $request->input('agent_category');
    if($agent_details->commission_percent != $request->input('commission_percent')){
      $agentStudents = DB::table('users')->join('university_application','university_application.userid','=','users.id')->where('agent_id',$user_id)->get(['university_application.*','users.id as user_ids']);
      if($agentStudents){
        foreach($agentStudents as $student){
          //$userApplication = UniversityApplication::where('userid',$student->id)->first();
          if($student->tuition_fees && $student->my_commission_percent){
              $comm = $student->tuition_fees * $student->my_commission_percent/100;
              $agent_commission_percent = $request->input('commission_percent');
              $agent_commission = $comm * $request->input('commission_percent') / 100;
              DB::table('university_application')->where('id', $student->id)->update(['agent_commission' => $agent_commission,'agent_commission_percent' => $agent_commission_percent]);
            //$student->save();
             //echo $student->user_ids."<br>";
          }
        }
      }
    }
    //dd("done");
    $agent_details->commission_percent = $request->input('commission_percent');
    $agent_details->save();

    StaffBranch::where('staff_id', $user->id)->delete();
    StaffBranch::create(["staff_id" => $user->id, "branch_id" => $request->get('branch')]);

    return redirect()->route('admin.agents.index')->withFlashSuccess("Agent updated successfully");
  }

  public function commission()
  {
    if (!(\Auth::user()->isAdmin() || \Auth::user()->can('commission'))) {
      return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
    }
    $search = "";
    $filters = array("search" => $search, "agent" => "");
    $userRepo = new UserRepository();
    $agents = $userRepo->getAgents();
    \Log::info("agenddts data in agement ,maangendt conrtroller",[$agents]);
    return view('backend.agents.commission', compact('filters', 'agents'));
  }

  public function commission_data(Request $request)
  {
    $columns = array(
      0 => 'created_at',
      1 => 'agent_id',
      2 => 'appl_id',
      4 => 'amount',
      5 => 'deduction',
      6 => 'net_pay',
      7 => 'notes',
      8 => 'paid_date',
      // 8 => 'actions'
    );

    $totalData = AgentCommission::count();

    $totalFiltered = $totalData;

    $agent = $request->input('agent');
    $limit = $request->input('length');
    $start = $request->input('start');
    $order = $columns[$request->input('order.0.column')];
    $status = $request->input('status');
    $dir = $request->input('order.0.dir');

    if (empty($request->input('search.value')) && empty($agent) && empty($status)) {
      $users = AgentCommission::join('users', 'agent_commission.agent_id', '=', 'users.id')
        ->select('users.first_name', 'users.last_name', 'users.email', 'agent_commission.*')
        ->offset($start)
        ->limit($limit)
        ->orderBy($order, $dir)
        ->get();
    } else {
      $get_search = $request->input('search.value');
      $users = AgentCommission::join('users', 'agent_commission.agent_id', '=', 'users.id')
       ->join('countries', 'users.country', '=', 'countries.id')
        ->where(function ($query) use ($get_search) {
          $query->where('appl_id', 'like', "%$get_search%")
           ->orWhere('countries.title_en', 'like', "%$get_search%");
        })
        ->when(!empty($status), function ($query) use ($status) {
          return  $query->Where('paid', '=', "$status");
        })
        ->when(!empty($agent), function ($query) use ($agent) {
          return  $query->Where('users.id', '=', "$agent");
        })
        ->select('users.first_name', 'users.last_name', 'users.email', 'agent_commission.*')
        ->offset($start)
        ->limit($limit)
        ->orderBy($order, $dir)
        ->get();

      $totalFiltered = AgentCommission::join('users', 'agent_commission.agent_id', '=', 'users.id')
       ->join('countries', 'users.country', '=', 'countries.id')
         ->where(function ($query) use ($get_search) {
        $query->where('appl_id', 'like', "%$get_search%")
              ->orWhere('countries.title_en', 'like', "%$get_search%");
        })
        ->when(!empty($status), function ($query) use ($status) {
          return  $query->Where('paid', '=', "$status");
        })
        ->when(!empty($agent), function ($query) use ($agent) {
          return  $query->Where('users.id', '=', "$agent");
        })
        ->count();
    }
    $data = array();
    if (!empty($users)) {
      foreach ($users as $page) {

        $currency = $university_name = $course_name = "";
        $universityApplication = UniversityApplication::where("appl_id", $page->appl_id)->first();
         if ($universityApplication) {
            if ($universityApplication->university) {
                $university_name = $universityApplication->university;
             }
            if ($universityApplication->course) {
                $course_name = $universityApplication->course;
           }
         }
        
        $currency = $page->currency . " " . get_currency_symbol($page->currency);
        $customResult['created_at'] = date("d M Y", strtotime($page->created_at));
        $customResult['agent'] = $page->first_name . " " . $page->last_name . "<br /><span class='text-smaller text-muted'>" . $page->email . "</small>";
        $customResult['applId'] = $page->appl_id."<br>".$university_name . '<br>' .$course_name;
        $customResult['commission_notes'] = $page->agent_commission_notes;
        $customResult['commission'] =  $currency . " " . $page->amount;
        $customResult['deduction'] = $currency . " " . $page->deduction;
        $customResult['net_pay'] = $currency . " " . $page->net_pay;
        $customResult['trans_details'] = $page->notes;
        $customResult['date_paid'] = get_date_time_formated($page->paid_date);

        if ($page->paid != '1') {
          $customResult['actions'] = '<button data-currency="' . $currency . '" data-comm="' . $page->amount . '" data-id="' . $page->appl_id . '" data-agent="' . $page->first_name . " " . $page->last_name . '" class="btn btn-sm btn-primary edit_commission">Edit</button>';
        } else {
          $customResult['actions'] = "Paid";
        }


        $data[] = $customResult;
      }
    }

    $json_data = array(
      "draw"            => intval($request->input('draw')),
      "recordsTotal"    => intval($totalData),
      "recordsFiltered" => intval($totalFiltered),
      "data"            => $data,
      "status"          =>  $status
    );

    echo json_encode($json_data);
  }

  public function update_commission(Request $request)
  {
    $request->validate([
      'amount'     => 'required|numeric|gt:0',
      'deduction'  => 'required|numeric',
    ]);
    $commdata = AgentCommission::where('appl_id', $request->appl_id)->first();
    if ($commdata && $commdata->paid == 0) {
      $commdata->paid_by = \Auth::user()->id;
      $commdata->amount = $request->amount;
      $commdata->deduction = $request->deduction;
      $commdata->notes = $request->notes;
      $commdata->net_pay = $request->net_pay_val;
      $commdata->save();

      return redirect()->route('admin.agents.commission')->withFlashSuccess("Agent commission updated successfully");
    } else {
      return redirect()->route('admin.agents.commission')->withFlashDanger("Agent commission payout already done.");
    }
  }


  public function payout()
  {
    if (!(\Auth::user()->isAdmin() || \Auth::user()->can('commission'))) {
      return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
    }
  
    $payouts = AgentCommission::join('users', 'agent_commission.agent_id', '=', 'users.id')
        ->join('university_application', 'agent_commission.appl_id', '=', 'university_application.id')
        ->select('users.first_name', 'users.last_name', 'users.email', 'agent_commission.*','university_application.university','university_application.course')
        ->where('agent_commission.paid',0)
        ->get();
    return view('backend.agents.payout', compact('payouts'));
  }

  // public function payout_data(Request $request)
  // {
  //   $columns = array(
  //     0 => 'created_at',
  //     1 => 'agent_id',
  //     2 => 'appl_id',
  //     4 => 'amount',
  //     5 => 'deduction',
  //     6 => 'net_pay',
  //     7 => 'notes',
  //     8 => 'paid_date',
  //   );

  //   $totalData = AgentCommission::count();

  //   $totalFiltered = $totalData;

  //   $agent = $request->input('agent');
  //   $limit = $request->input('length');
  //   $start = $request->input('start');
  //   $order = $columns[$request->input('order.0.column')];
  //   $status = $request->input('status');
  //   $dir = $request->input('order.0.dir');

  //   if (empty($request->input('search.value'))) {
  //     $users = AgentCommission::join('users', 'agent_commission.agent_id', '=', 'users.id')
  //       ->select('users.first_name', 'users.last_name', 'users.email', 'agent_commission.*')
  //       ->where('agent_commission.paid',0)
  //       ->offset($start)
  //       ->limit($limit)
  //       ->orderBy($order, $dir)
  //       ->get();
  //   } else {
  //     $get_search = $request->input('search.value');
  //     $users = AgentCommission::join('users', 'agent_commission.agent_id', '=', 'users.id')
  //       ->where(function ($query) use ($get_search) {
  //         $query->where('appl_id', 'like', "%$get_search%");
  //       })
  //       ->when(!empty($agent), function ($query) use ($agent) {
  //         return  $query->Where('users.id', '=', "$agent");
  //       })
  //       ->select('users.first_name', 'users.last_name', 'users.email', 'agent_commission.*')
  //       ->where('agent_commission.paid',0)
  //       ->offset($start)
  //       ->limit($limit)
  //       ->orderBy($order, $dir)
  //       ->get();

  //     $totalFiltered = AgentCommission::join('users', 'agent_commission.agent_id', '=', 'users.id')
  //       ->where(function ($query) use ($get_search) {
  //         $query->where('appl_id', 'like', "%$get_search%");
  //       })
  //       ->where('agent_commission.paid',0)
  //       ->when(!empty($agent), function ($query) use ($agent) {
  //         return  $query->Where('users.id', '=', "$agent");
  //       })
  //       ->count();
  //   }
  //   $data = array();
  //   if (!empty($users)) {
  //     foreach ($users as $page) {

  //       $currency = $university_name = $course_name = "";
  //       $universityApplication = UniversityApplication::where("id", $page->appl_id)->first();
  //        if ($universityApplication) {
  //           if ($universityApplication->university) {
  //               $university_name = $universityApplication->university;
  //            }
  //           if ($universityApplication->course) {
  //               $course_name = $universityApplication->course;
  //          }
  //        }
        
  //       $currency = $page->currency . " " . get_currency_symbol($page->currency);
       
  //       $customResult['id'] = $page->appl_id;
  //       $customResult['created_at'] = date("d M Y", strtotime($page->created_at));
  //       $customResult['agent'] = $page->first_name . " " . $page->last_name . "<br /><span class='text-smaller text-muted'>" . $page->email . "</small>";
  //       $customResult['applId'] = get_application_id($page->appl_id)."<br>".$university_name . '<br>' . $course_name;
  //       $customResult['commission_notes'] = $page->agent_commission_notes;
  //       $customResult['commission'] =  $currency . " " . $page->amount;
  //       $customResult['deduction'] = $currency . " " . $page->deduction;
  //       $customResult['net_pay'] = $currency . " " . $page->net_pay;
  //       $customResult['trans_details'] = $page->notes;

  //       $data[] = $customResult;
  //     }
  //   }

  //   $json_data = array(
  //     "draw"            => intval($request->input('draw')),
  //     "recordsTotal"    => intval($totalData),
  //     "recordsFiltered" => intval($totalFiltered),
  //     "data"            => $data,
  //     "status"          =>  $status
  //   );

  //   echo json_encode($json_data);
  // }

  public function process_payment_step_two(Request $request)
  {
   // dd($request->all());
    if (!(\Auth::user()->isAdmin() || \Auth::user()->can('commission'))) {
      return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
    }

        $id_arr = explode(',', $request->input('id_arr'));
        $commdata = AgentCommission::join('users', 'agent_commission.agent_id', '=', 'users.id')
                ->whereIn('agent_commission.id',$id_arr)
                ->groupBy('agent_commission.agent_id')
                ->get(['agent_commission.*','users.first_name', 'users.last_name', 'users.email']);
          
          $data= collect($commdata)->map(function($commdata, $key) {
           $collect = (object)$commdata;
           return [
                'comm_id' => $collect->id,
                'agent_id' => $collect->agent_id,
                'currency' => get_currency_by_user_country($collect->agent_id),
                'email' => $collect->email,
                'agent_name' => $collect->first_name." ".$collect->last_name,
                'total_net_pay' => $this->modify_commision($key,$collect->agent_id)
           ];
       });
    return view('backend.agents.payout_step_two', ['commdata_arr' => $data, 'comm_ids' => $request->input('id_arr')]);
  }

  public function modify_commision($key,$agent_id) {
    $total_pay = 0;
    $paydata = AgentCommission::where('agent_commission.agent_id',$agent_id)->get(['agent_commission.*']);
    foreach($paydata as $data){
      $total_pay += currency_convert($data->net_pay, $data->currency, get_currency_by_user_country($agent_id));
    }
    return $total_pay;
  }


  public function process_payment_save(Request $request){
    
    $id_arr = explode(',', $request->input('comm_ids'));
    
    foreach($id_arr as $comm_id){
      $agent_comm = AgentCommission::find($comm_id);
      $agent_comm->amount_base = currency_convert($agent_comm->amount, $agent_comm->currency, config('app.currency'));
      $agent_comm->deduction_base = currency_convert($agent_comm->deduction, $agent_comm->currency, config('app.currency'));
      $agent_comm->net_pay_base = currency_convert($agent_comm->net_pay, $agent_comm->currency, config('app.currency'));
      $agent_comm->net_pay_agent = currency_convert($agent_comm->net_pay, $agent_comm->currency, get_currency_by_user_country($agent_comm->agent_id));
      $agent_comm->paid = 1;
      $agent_comm->paid_date = now(); 
      $agent_comm->save();	
      //send email to agent
      $mail_tpl = MailTemplates::find("AGENT_COMMISSION_PAYOUT");
      if ($mail_tpl && $mail_tpl->status) {
        \Log::debug("Sending Agent commission payout mail.....");
        $email_message = $mail_tpl->message;
        $full_name = $request->get('first_name') . ' ' . $request->get('last_name');
        $email_message = str_ireplace("#sitename#", app_name(), $email_message);
        $email_message = str_ireplace("#siteurl#", url(''), $email_message);
        $email_message = str_ireplace("#first_name#", $request->get('first_name'), $email_message);
        $email_message = str_ireplace("#full_name#", $full_name, $email_message);
        $email_message = str_ireplace("#email#", $request->get('email'), $email_message);
        $email_message = str_ireplace("#mobile#", $request->get('dialcode') . ' ' . $request->get('mobile'), $email_message);
        $email_message = str_ireplace("#currency#", get_currency_symbol(get_currency_by_user_country($agent_comm->agent_id)), $email_message);
        $email_message = str_ireplace("#payout_amount#", currency_convert($agent_comm->net_pay, $agent_comm->currency, get_currency_by_user_country($agent_comm->agent_id)), $email_message);
        try {
          $user = (new UserRepository())->where('id', $agent_comm->agent_id)->first();
          user_notify($user, $mail_tpl->subject, $email_message);
          \Log::debug("Agent: commission payout mail- " . $email_message);
        } catch (\Exception $e) {
          $message = $e->getMessage();
          \Log::debug("Mail Error: (Agent: commission payout) " . $message);
        }
      }
    }
    return redirect()->route('admin.agents.payout')->withFlashSuccess('Agent payout Done!');
}

  public function payout_second_stepdata(Request $request)
  {

    $columns = array(
      0 => 'university_application.id',
      1 => 'agent_id',
      2 => 'appl_id',
      3 => 'university_application.country',
      // 4 => 'details',
      5 => 'university_application.status',
      6 => 'university_application.total_commission',
      // 7 => 'actions'
    );

    $totalData = User::with('roles')
      ->join('users as agent_info', 'agent_info.id', '=', 'users.agent_id')
      ->join('university_application', 'university_application.userid', '=', 'users.id')
      ->where('university_application.commission_paid', 0)
      ->whereHas('roles', function ($q) {
        $q->whereIn('name', ['user']);
      })->count();

    $totalFiltered = $totalData;

    $agent = $request->input('agent');
    $limit = $request->input('length');
    $start = $request->input('start');
    $order = $columns[$request->input('order.0.column')];
    $status = $request->input('appl_status');
    $dir = $request->input('order.0.dir');

    if (empty($request->input('search.value')) && empty($agent) && empty($status)) {
      $users = User::with('roles')
        ->join('users as agent_info', 'agent_info.id', '=', 'users.agent_id')
        ->join('university_application', 'university_application.userid', '=', 'users.id')
        ->where('university_application.commission_paid', 0)
        ->whereHas('roles', function ($q) {
          $q->whereIn('name', ['user']);
        })
        ->when(!empty($agent), function ($query) use ($agent) {
          return  $query->Where('agent_id', '=', "$agent");
        })
        ->select('users.*', 'agent_info.first_name as agent_first_name', 'agent_info.last_name as agent_last_name', 'agent_info.email as agent_email', 'university_application.appl_id', 'university_application.country', 'university_application.status', 'university_application.total_commission', 'university_application.agent_percent', 'university_application.university', 'university_application.course')
        ->offset($start)
        ->limit($limit)
        ->orderBy($order, $dir)
        ->get();
    } else {
      $get_search = $request->input('search.value');
      $users = User::with('roles')
        ->join('users as agent_info', 'agent_info.id', '=', 'users.agent_id')
        ->join('university_application', 'university_application.userid', '=', 'users.id')
        ->where('university_application.commission_paid', 0)
        ->where(function ($query) use ($get_search) {
          $query->where('users.first_name', 'like', "%$get_search%")
            ->orWhere('users.last_name', 'like', "%$get_search%")
            ->orWhere('university_application.country', 'like', "%$get_search%")
            ->orWhere('university_application.appl_id', 'like', "%$get_search%");
        })
        ->whereHas('roles', function ($q) {
          $q->whereIn('name', ['user']);
        })
        ->when(!empty($status), function ($query) use ($status) {
          return  $query->Where('university_application.status', '=', "$status");
        })
        ->when(!empty($agent), function ($query) use ($agent) {
          return  $query->Where('users.agent_id', '=', "$agent");
        })
        ->select('users.*', 'agent_info.first_name as agent_first_name', 'agent_info.last_name as agent_last_name', 'agent_info.email as agent_email', 'university_application.appl_id', 'university_application.country', 'university_application.status', 'university_application.total_commission', 'university_application.agent_percent', 'university_application.university', 'university_application.course')
        ->offset($start)
        ->limit($limit)
        ->orderBy($order, $dir)
        ->get();

      $totalFiltered = User::with('roles')
        ->join('users as agent_info', 'agent_info.id', '=', 'users.agent_id')
        ->join('university_application', 'university_application.userid', '=', 'users.id')
        ->where(function ($query) use ($get_search) {
          $query->where('users.first_name', 'like', "%$get_search%")
            ->orWhere('users.last_name', 'like', "%$get_search%")
            ->orWhere('university_application.country', 'like', "%$get_search%")
            ->orWhere('university_application.appl_id', 'like', "%$get_search%");
        })
        ->whereHas('roles', function ($q) {
          $q->whereIn('name', ['user']);
        })
        ->when(!empty($status), function ($query) use ($status) {
          return  $query->Where('university_application.status', '=', "$status");
        })
        ->when(!empty($agent), function ($query) use ($agent) {
          return  $query->Where('users.agent_id', '=', "$agent");
        })
        ->count();
    }

    $data = array();
    if (!empty($users)) {
      foreach ($users as $page) {
        $customResult['id'] = $page->appl_id;
        $customResult['agent'] = "<i class='fas fa-user-tie text-sub'></i> " . $page->agent_first_name . " " . $page->agent_last_name . "<br /><span class='text-smaller text-muted'>" . $page->agent_email . "</small>";
        $customResult['applId'] = get_application_id($page->appl_id);
        $customResult['country'] = $page->country;
        $customResult['details'] = $page->university . '<br>' . $page->course . '<br>' . $page->first_name . ' ' . $page->last_name;
        $customResult['applStatus'] = $page->status;
        $customResult['expComm'] = config('app.currency_symbol') . $page->total_commission * $page->agent_percent / 100;
        $customResult['actions'] = '<a href="#" data-id="' . $page->appl_id . '" class="btn btn-sm btn-primary show_confirm">Pay</a>';

        $data[] = $customResult;
      }
    }

    $json_data = array(
      "draw"            => intval($request->input('draw')),
      "recordsTotal"    => intval($totalData),
      "recordsFiltered" => intval($totalFiltered),
      "data"            => $data
    );

    echo json_encode($json_data);
  }

  public function pay($applid)
  {
    if (!(\Auth::user()->isAdmin() || \Auth::user()->can('commission'))) {
      return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
    }
    // if (!(\Auth::user()->isAdmin() || \Auth::user()->can('lead-assignment'))) {
    //     return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
    // }
    $appldata = UniversityApplication::where('appl_id', $applid)->first();
    if ($appldata->total_commission > 0) {
      $appldata->commission_paid = '1';
      $appldata->save();

      $agent = User::where('id', $appldata->userid)->first('agent_id');

      $comm = new AgentCommission;
      $comm->agent_id = $agent->agent_id;
      $comm->appl_id = $applid;
      $comm->amount = $appldata->total_commission * $appldata->agent_percent / 100;
      $comm->net_pay = $appldata->total_commission * $appldata->agent_percent / 100;
      $comm->created_by = \Auth::user()->id;
      $comm->save();

      return redirect()->route('admin.agents.commission')->withFlashSuccess("Agent commission paid successfully");
    } else {
      return redirect()->route('admin.agents.commission')->withFlashDanger("Commission amount must be greater than 0");
    }
  }

  public function referrals()
  {
    if (!(\Auth::user()->isAdmin() || \Auth::user()->can('agent management'))) {
      return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
    }
    $search = "";
    $filters = array("search" => $search, "agent" => "");
    $userRepo = new UserRepository();
    $agents = $userRepo->getAgents();
    return view('backend.agents.referrals', compact('filters', 'agents'));
  }

  public function referralsdata(Request $request)
  {

    $columns = array(
      0 => 'first_name',
      1 => 'email',
      2 => 'mobile',
      // 3 => 'first_name',
      4 => 'users.created_at',
      // 5 => 'agent'
    );

    $totalData = User::with('roles')
      ->join('users as agent_info', 'agent_info.id', '=', 'users.agent_id')
      ->whereHas('roles', function ($q) {
        $q->whereIn('name', ['user']);
      })->count();

    $totalFiltered = $totalData;

    $agent = $request->input('agent');
    $limit = $request->input('length');
    $start = $request->input('start');
    $order = $columns[$request->input('order.0.column')];
    $dir = $request->input('order.0.dir');

    if (empty($request->input('search.value')) && empty($agent)) {
      $users = User::with('roles')
        ->join('users as agent_info', 'agent_info.id', '=', 'users.agent_id')
        ->whereHas('roles', function ($q) {
          $q->whereIn('name', ['user']);
        })
        ->when(!empty($agent), function ($query) use ($agent) {
          return  $query->Where('agent_id', '=', "$agent");
        })
        ->select('users.*', 'agent_info.first_name as agent_first_name', 'agent_info.last_name as agent_last_name', 'agent_info.email as agent_email')
        ->offset($start)
        ->limit($limit)
        ->orderBy($order, $dir)
        ->get();
    } else {
      $get_search = $request->input('search.value');
      $users = User::with('roles')
        ->join('users as agent_info', 'agent_info.id', '=', 'users.agent_id')
        ->where(function ($query) use ($get_search) {
          $query->where('users.first_name', 'like', "%$get_search%")
            ->orWhere('users.last_name', 'like', "%$get_search%")
            ->orWhereRaw("CONCAT(users.first_name, ' ', users.last_name) LIKE ?", ["%$get_search%"])
            ->orWhere('users.overseas_status', 'like', "%$get_search%")
            ->orWhere('users.email', 'like', "%$get_search%")
            ->orWhere('users.mobile', 'like', "%$get_search%");
        })
        ->whereHas('roles', function ($q) {
          $q->whereIn('name', ['user']);
        })
        ->when(!empty($agent), function ($query) use ($agent) {
          return  $query->Where('users.agent_id', '=', "$agent");
        })
        ->select('users.*', 'agent_info.first_name as agent_first_name', 'agent_info.last_name as agent_last_name', 'agent_info.email as agent_email')
        ->offset($start)
        ->limit($limit)
        ->orderBy($order, $dir)
        ->get();

      $totalFiltered = User::with('roles')
        ->join('users as agent_info', 'agent_info.id', '=', 'users.agent_id')
        ->where(function ($query) use ($get_search) {
          $query->where('users.first_name', 'like', "%$get_search%")
            ->orWhere('users.last_name', 'like', "%$get_search%")
            ->orWhereRaw("CONCAT(users.first_name, ' ', users.last_name) LIKE ?", ["%$get_search%"])
            ->orWhere('users.email', 'like', "%$get_search%")
            ->orWhere('users.mobile', 'like', "%$get_search%")
            ->orWhere('users.overseas_status', 'like', "%$get_search%");
        })
        ->whereHas('roles', function ($q) {
          $q->whereIn('name', ['user']);
        })
        ->when(!empty($agent), function ($query) use ($agent) {
          return  $query->Where('users.agent_id', '=', "$agent");
        })
        ->count();
    }

    $data = array();
    if (!empty($users)) {
      foreach ($users as $page) {
        $customResult['name'] = "<a href='" . route('admin.auth.user.show', $page->id) . "'><strong><span style='white-space:nowrap;'><i class='fas fa-user text-sub'></i> " . $page->first_name . " " . $page->last_name . "</span></strong></a>";
        $customResult['email'] = $page->email;
        $customResult['mobile'] = $page->mobile;

        //Status //////

        // $overseas_status = $course_status = "";
        // if ($page->student_type == 'study_abroad' || $page->student_type == 'study_abroad_course') {
        //     //Study abroad
        //     if ($page->overseas_status == 'Profile') {
        //         $overseas_status = '<span style="font-size: 12px;font-weight: 400;" class="badge badge-success">Profile Complete</span>';
        //     } elseif ($page->overseas_status == 'Course Preferences') {
        //         $overseas_status = '<span style="font-size: 12px;font-weight: 400;" class="badge badge-success">Course Preference Added</span>';
        //     } elseif ($page->overseas_status == 'Documents') {
        //         $overseas_status = '<span style="font-size: 12px;font-weight: 400;" class="badge badge-success">Documents Uploaded</span>';
        //     } elseif ($page->overseas_status == 'Application Processing') {
        //         $overseas_status = '<span style="font-size: 12px;font-weight: 400;" class="badge badge-success">Application Processing</span>';
        //         //appl count
        //         $appl_count = DB::table('university_application')->where('userid', $page->id)->count();
        //         $overseas_status .= "<br /><span class='text-primary'>Total ".$appl_count." Application</span>";

        //     } elseif ($page->overseas_status == 'Application Rejected/Cancelled') {
        //         $overseas_status = '<span style="font-size: 12px;font-weight: 400;" class="badge badge-danger">Application Rejected/Cancelled</span>';
        //     } elseif ($page->overseas_status == 'Admission Complete') {
        //         $overseas_status = '<span style="font-size: 12px;font-weight: 400;" class="badge badge-success">Admission Complete</span>';
        //     } else {
        //         $overseas_status = '<span style="font-size: 12px;font-weight: 400;" class="badge badge-danger">Profile Incomplete</span>';
        //     }

        //     //Course
        //     if ($page->coaching_status == 'Enrolled to Course') {
        //         $course_status = '<span style="font-size: 12px;font-weight: 400;" class="badge badge-success">Enrolled to Course</span>';
        //     } elseif($page->coaching_status == 'Course Completed'){
        //         $course_status = '<span style="font-size: 12px;font-weight: 400;" class="badge badge-default">Course Completed</span>';
        //     }
        // } elseif($page->student_type == 'course') {
        //     //Course
        //     if ($page->coaching_status == 'Not Enrolled') {
        //         $course_status = '<span style="font-size: 12px;font-weight: 400;" class="badge badge-danger">Not Enrolled</span>';
        //     } elseif ($page->coaching_status == 'Enrolled to Course') {
        //         $course_status = '<span style="font-size: 12px;font-weight: 400;" class="badge badge-success">Enrolled to Course</span>';
        //     } elseif($page->coaching_status == 'Course Completed'){
        //         $course_status = '<span style="font-size: 12px;font-weight: 400;" class="badge badge-default">Course Completed</span>';
        //     }
        // } elseif ($page->student_type == 'abroad_job') {
        //     if ($page->overseas_status == 'Profile') {
        //         $overseas_status = '<span style="font-size: 12px;font-weight: 400;" class="badge badge-success">Profile Complete</span>';
        //     } elseif ($page->overseas_status == 'Course Preferences') {
        //         $overseas_status = '<span style="font-size: 12px;font-weight: 400;" class="badge badge-success">Course Preference Added</span>';
        //     } elseif ($page->overseas_status == 'Documents') {
        //         $overseas_status = '<span style="font-size: 12px;font-weight: 400;" class="badge badge-success">Documents Uploaded</span>';
        //     } else {
        //         $overseas_status = '<span style="font-size: 12px;font-weight: 400;" class="badge badge-danger">'.$page->overseas_status.'</span>';
        //     }
        // } elseif ($page->student_type == 'visa') {
        //     if ($page->overseas_status == 'Pending') {
        //         $overseas_status = '<span style="font-size: 12px;font-weight: 400;" class="badge badge-danger">Payment Pending</span>';
        //     } elseif ($page->overseas_status == 'Application Processing') {
        //         $overseas_status = '<span style="font-size: 12px;font-weight: 400;" class="badge badge-default">Application Processing</span>';
        //     } elseif($page->overseas_status == 'Application Rejected/Cancelled') {
        //         $overseas_status = '<span style="font-size: 12px;font-weight: 400;" class="badge badge-default">Rejected/Cancelled</span>';
        //     } elseif($page->overseas_status == 'Complete') {
        //         $overseas_status = '<span style="font-size: 12px;font-weight: 400;" class="badge badge-success">Visa Completed</span>';
        //     }
        // }

        // if ($page->student_type == 'study_abroad') {
        //     $customResult['status'] = $overseas_status;    
        // } elseif($page->student_type == 'course') {
        //     $customResult['status'] = $course_status;
        // } elseif($page->student_type == 'study_abroad_course') {
        //     $customResult['status'] = $overseas_status. " ". $course_status;
        // } else {
        //     $customResult['status'] = $overseas_status;
        // }

        $customResult['status'] = '<span class="badge p-1" style="font-size: 14px!important;font-weight: 700; color:' . get_overseas_status_color($page->overseas_status) . ';">' . $page->overseas_status . '</span>';

        $user_details = UserDetails::where('userid', $page->id)->first();
        if ($user_details) {
          if ($user_details->student_disqualified_reason && $page->overseas_status == get_disqualified_student_status())
            $customResult['status'] .= '<p class="mt-1" style="font-size: 11px">' . $user_details->student_disqualified_reason . '</p>';
        }

        ///////////////

        $customResult['created_at'] = get_date_time_formated($page->created_at);
        $customResult['agent'] = "<i class='fas fa-user-tie text-sub'></i> " . $page->agent_first_name . " " . $page->agent_last_name . "<br /><span class='text-smaller text-muted'>" . $page->agent_email . "</small>";

        $data[] = $customResult;
      }
    }

    $json_data = array(
      "draw"            => intval($request->input('draw')),
      "recordsTotal"    => intval($totalData),
      "recordsFiltered" => intval($totalFiltered),
      "data"            => $data
    );

    echo json_encode($json_data);
  }

  public function destroy(ManageUserRequest $request, User $user)
  {
    if (!(\Auth::user()->isAdmin() || \Auth::user()->can('agent management'))) {
      return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
    }
    // if (!(\Auth::user()->isAdmin() || \Auth::user()->can('delete-agent'))) {
    //     return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
    // }
    $userRepo = new UserRepository();
    $userRepo->deleteById($user->id);

    event(new UserDeleted($user));

    return redirect()->route('admin.agents.index')->withFlashSuccess("Agent deleted successfully");
  }

  public function deleted()
  {
    if (!(\Auth::user()->isAdmin() || \Auth::user()->can('agent management'))) {
      return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
    }
    return view('backend.agents.agents_deleted');
  }

  public function agents_deleted_list()
  {

    $data = User::with('roles')
      ->whereHas('roles', function ($q) {
        $q->whereIn('name', ['agent']);
      })
      ->onlyTrashed()
      ->where('removed', "0")
      ->latest()
      ->get();

    return Datatables::of($data)
      //->addIndexColumn()
      ->addColumn('action', function ($data) {
        return view("backend.agents.actions_btn_deleted", compact('data'));
      })
      ->rawColumns(['action'])
      ->make(true);
  }
  // Agent Side Resources

  public function agentResourcesIndex()
  {
    $detail = DB::table('agent_resource_category')
      ->select('agent_resource_category.category', 'agent_resource_category.id')
      ->get();

    return view('backend.agents.agent_resource_index', ['detail' => $detail]);
  }
  public function agentResourcesCategory($category)
  {
    $folders = array();
    $rows = DB::table('agent_file_upload')->select('filename', 'title', 'id', 'country')
      ->where('category', $category)
      ->get();
    if ($rows) {
      foreach ($rows as $row) {
        $folders[] = array("id" => $row->id, "title" => $row->title, "country" => $row->country, "filename" => substr($row->filename, -3), "file" => $row->filename);
      }
    }
    return view('backend.agents.agent-resources', ['folders' => $folders]);
  }

  //New functions
  public function agent_details($agent_id)
  {
    if (!(\Auth::user()->isAdmin() || \Auth::user()->can('agent management'))) {
      return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
    }

    $agent = User::find($agent_id);
    if(!$agent->hasRole('agent')) {
      return redirect()->route('admin.agents.index')->withFlashDanger('Agent Not Found!');
    }

    // $agent_details = AgentDetails::where('agent_id', $agent_id)->first();
    if (!AgentDetails::where('agent_id', $agent_id)->exists()) {
      AgentDetails::create([
        'agent_id' => $agent_id
      ]);
    }
    $agent_details = AgentDetails::where('agent_details.agent_id', $agent_id)
      ->join('users', 'users.id', '=', 'agent_details.agent_id')
      ->join('branch', 'branch.id', '=', 'users.branch')
      ->select('agent_details.*', 'users.first_name','users.last_name', 'users.email', 'users.country',  'users.state', 'users.address_1', 'users.whatsapp_dialcode', 'users.whatsapp', 'users.mobile', 'users.active', 'users.avatar_location', 'users.id as user_id','branch.branch_name as branch' )
      ->first();

    $docusignSentDocs = UserDocuEsign::where('user_id', $agent_id)->get();

    $token = get_site_config('docusign_api_token');
    $baseUrl = rtrim(get_site_config('docusign_api_url'), '/');
    $templates = [];
    if( !empty($token) && !empty($baseUrl)) {
        $url = "{$baseUrl}/api/get-templates";
        $response = Http::post($url, ['token' => $token]);
        // Log the response for debugging
        \Log::debug('DocuSign API response: ' . $response->body());
        $data = $response->json();
            $templates = $data['template'];
    }
    // dd($agent_details);   
    return view('backend.agents.agent_show', ['agent_details' => $agent_details,'user' => $agent, 'docusignSentDocs' => $docusignSentDocs,'templates'=>$templates]);
  }


  public function my_referrals(Request $request)

  {
    // dd($agent_id);
    if (!(\Auth::user()->isAdmin() || \Auth::user()->can('agent management'))) {

      return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
    }
    return $this->getDatatableMyReferralslist($request);
  }


  public function getDatatableMyReferralslist(Request $request)
  {
    // dd($request->all());
    $columns = array(
      0 => 'users.first_name',
      1 => 'users.overseas_status',
      2 => 'users.created_at',
    );

    $totalData = User::with('roles')
      ->join('users as agent_info', 'agent_info.id', '=', 'users.agent_id')
      ->whereHas(
        'roles',
        function ($q) {
          $q->whereIn('name', ['user']);
        }
      )
      ->where('users.agent_id', $request->agent_id)
      ->count();

    $totalFiltered = $totalData;

    $agent = $request->input('agent');
    $limit = $request->input('length');
    $start = $request->input('start');
    $order = $columns[$request->input('order.0.column')];
    $dir = $request->input('order.0.dir');
    $startDate = $endDate = "";
    // dd($request->input('search_value'));

    if (empty($request->input('search.value')) && empty($request->input('student_status')) && empty($agent) && empty($request->input('created_date'))) {
      // dd($request->input('created_date'));
      $get_search = $request->input('search_value');
      $get_createdate = $request->input('created_date'); // Assuming this is something like "2024-10-09 - 2024-11-20"

      // Split the date range into start and end dates
      $dateRange = explode(' - ', $get_createdate);
      $startDate = $dateRange[0] ?? null;
      $endDate = $dateRange[1] ?? null;

      $users = User::with('roles')
        ->join('users as agent_info', 'agent_info.id', '=', 'users.agent_id')
        ->whereHas('roles', function ($q) {
          $q->whereIn('name', ['user']);
        })
        ->when(!empty($agent), function ($query) use ($agent) {
          return  $query->Where('agent_id', '=', "$agent");
        })
        ->when(!empty($get_search), function ($query) use ($get_search) {
          return  $query->Where('users.overseas_status', 'like', "%$get_search%");
        })
        ->when(!empty($startDate) && !empty($endDate), function ($query) use ($startDate, $endDate) {
          return $query->whereBetween('users.created_at', [$startDate, $endDate]);
        })
        ->where('users.agent_id', $request->agent_id)

        ->select('users.*', 'agent_info.first_name as agent_first_name', 'agent_info.last_name as agent_last_name', 'agent_info.email as agent_email')
        ->offset($start)
        ->limit($limit)
        ->orderBy($order, $dir)
        ->get();
    } else {

      $get_search = $request->input('search.value');
      $student_status = $request->input('student_status');
      $getdate = $request->input('created_date');
      // dd($request->all());

      // Check if $getdate exists and split it into start and end date
      if ($getdate) {
        // Assuming the format is "YYYY-MM-DD - YYYY-MM-DD"
        // [$startDate, $endDate] = explode(' - ', $getdate);

        $created_dts_str = $getdate;
        $created_dts = explode(" - ", $created_dts_str);
        $startDate = date("Y-m-d 00:00:00", strtotime($created_dts[0]));
        $endDate = date("Y-m-d 23:59:59", strtotime($created_dts[1]));
        \Log::debug('User created at: ' . $startDate . ' to ' . $endDate);

        // dd($startDate);
        // dd($startDate);


      }
      $users = User::with('roles')
        ->join('users as agent_info', 'agent_info.id', '=', 'users.agent_id')
        ->where(function ($query) use ($get_search) {
          $query->where('users.first_name', 'like', "%$get_search%")
            ->orWhere('users.last_name', 'like', "%$get_search%")
            ->orWhere('users.overseas_status', 'like', "%$get_search%")
            ->orWhere('users.email', 'like', "%$get_search%")
            ->orWhere('users.mobile', 'like', "%$get_search%");
          // Only apply date filter if $startDate and $endDate exist
        })
        ->whereHas('roles', function ($q) {
          $q->whereIn('name', ['user']);
        })
        //   ->when(!empty($startDate) && !empty($endDate), function ($query) use ($startDate, $endDate) {
        //     // dd("vdsgdg");
        //     return $query->where(function($q) use ($startDate, $endDate) {
        //       if ($startDate && $endDate) {
        //           $q->where('users.created_at', '>=', $startDate)
        //             ->orWhere('users.created_at', '<=', $endDate); // This makes it an "OR" condition
        //       }
        //   });
        //  }) 
        ->when(!empty($startDate), function ($query) use ($startDate) {
          return  $query->Where('users.created_at', '>=', "$startDate");
        })
        ->when(!empty($endDate), function ($query) use ($endDate) {
          return  $query->Where('users.created_at', '<=', "$endDate");
        })
        ->when(!empty($agent), function ($query) use ($agent) {
          return  $query->Where('users.agent_id', '=', "$agent");
        })
        ->where('users.agent_id', $request->agent_id)
        ->select('users.*', 'agent_info.first_name as agent_first_name', 'agent_info.last_name as agent_last_name', 'agent_info.email as agent_email')
        ->offset($start)
        ->limit($limit)
        ->orderBy($order, $dir)
        ->get();

      $totalFiltered = User::with('roles')
        ->join('users as agent_info', 'agent_info.id', '=', 'users.agent_id')
        ->where(function ($query) use ($get_search) {
          $query->where('users.first_name', 'like', "%$get_search%")
            ->orWhere('users.last_name', 'like', "%$get_search%")
            ->orWhere('users.overseas_status', 'like', "%$get_search%")
            ->orWhere('users.email', 'like', "%$get_search%")
            ->orWhere('users.mobile', 'like', "%$get_search%");
        })
        ->whereHas('roles', function ($q) {
          $q->whereIn('name', ['user']);
        })
        //     ->when(!empty($startDate) && !empty($endDate), function ($query) use ($startDate, $endDate) {
        //       // dd("vdsgdg");
        //       return $query->where(function($q) use ($startDate, $endDate) {
        //         if ($startDate && $endDate) {
        //             $q->where('users.created_at', '>=', $startDate)
        //               ->orWhere('users.created_at', '<=', $endDate); // This makes it an "OR" condition
        //         }
        //     });
        //    }) 
        ->when(!empty($startDate), function ($query) use ($startDate) {
          return  $query->Where('users.created_at', '>=', "$startDate");
        })
        ->when(!empty($endDate), function ($query) use ($endDate) {
          return  $query->Where('users.created_at', '<=', "$endDate");
        })
        ->when(!empty($agent), function ($query) use ($agent) {
          return  $query->Where('users.agent_id', '=', "$agent");
        })
        ->where('users.agent_id', $request->agent_id)
        ->count();
    }

    $data = array();
    if (!empty($users)) {
      foreach ($users as $page) {



        $customResult['student_name'] = "<a href='" . route('admin.auth.user.show', $page->id) . "'><strong><span style='white-space:nowrap;'><i class='fas fa-user text-sub'></i>  &nbsp;" . $page->first_name . " " . $page->last_name . "</span></strong></a>";
        $customResult['student_name'] .= "<br><b>ID:</b> &nbsp;" . $page->id;

        $customResult['status'] = '<span class="badge p-1" style="font-size: 14px!important;font-weight: 700; color:' . get_overseas_status_color($page->overseas_status) . ';">' . $page->overseas_status . '</span>';

        $user_details = UserDetails::where('userid', $page->id)->first();
        if ($user_details) {
          if ($user_details->student_disqualified_reason && $page->overseas_status == get_disqualified_student_status())
            $customResult['status'] .= '<p class="mt-1" style="font-size: 11px">' . $user_details->student_disqualified_reason . '</p>';
        }

        $customResult['created_at'] ='<p>'.  \Carbon\Carbon::parse($page->created_at)->format(env('DATE_TIME_FORMAT', 'd-m-Y h:i a')) .'</p>';

        $data[] = $customResult;
      }
    }
    $json_data = array(
      "draw"            => intval($request->input('draw')),
      "recordsTotal"    => intval($totalData),
      "recordsFiltered" => intval($totalFiltered),
      "data"            => $data
    );

    echo json_encode($json_data);
  }





  public function myagent_comission(Request $request)

  {
    if (!(\Auth::user()->isAdmin() || \Auth::user()->can('commission'))) {
      return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
    }
    $search = "";
    $filters = array("search" => $search, "agent" => "");
    $userRepo = new UserRepository();
    $agents = $userRepo->getAgents();
    // $statuses = get_agent_application_statuses();
    return view('backend.agents.agent_commission', compact('filters', 'agents'));
  }



  public function myagent_commissiondata(Request $request)
  {


    $columns = array(
      0 => 'created_at',
      1 => 'agent_id',
      2 => 'appl_id',
      4 => 'amount',
      5 => 'deduction',
      6 => 'net_pay',
      7 => 'notes',
      8 => 'paid_date',
      // 8 => 'actions'
    );

    $totalData = AgentCommission::count();

    $totalFiltered = $totalData;
$agent_id=$request->input('agent_id');
    $agent = $request->input('agent');
    $limit = $request->input('length');
    $start = $request->input('start');
    $order = $columns[$request->input('order.0.column')];
    $status = $request->input('status');
    $dir = $request->input('order.0.dir');

    if (empty($request->input('search.value')) && empty($agent) && empty($status)) {
      $users = AgentCommission::join('users', 'agent_commission.agent_id', '=', 'users.id')
        ->select('users.first_name', 'users.last_name', 'users.email', 'agent_commission.*')
        ->offset($start)
       ->Where('agent_commission.agent_id', '=',$agent_id)
        ->limit($limit)
        ->orderBy($order, $dir)
        ->get();
        // dd($users);
    } else {

      $get_search = $request->input('search.value');
      $users = AgentCommission::join('users', 'agent_commission.agent_id', '=', 'users.id')
       ->Where('agent_commission.agent_id', '=',$agent_id)
       ->when(!empty($get_search), function ($query) use ($get_search) {
          return $query->Where('appl_id', 'like', "%$get_search%");
        })
        ->when(!empty($status), function ($query) use ($status) {
if($status == "pending"){
  return  $query->Where('paid', '=', "0");
  
}else{
          return  $query->Where('paid', '=', "$status");}
        })

        ->when(!empty($agent), function ($query) use ($agent) {
          return  $query->Where('users.id', '=', "$agent");
        })
        ->select('users.first_name', 'users.last_name', 'users.email', 'agent_commission.*')
        ->offset($start)
        ->limit($limit)
        ->orderBy($order, $dir)
        ->get();
        // dd($users);

      $totalFiltered = AgentCommission::join('users', 'agent_commission.agent_id', '=', 'users.id')
        ->where(function ($query) use ($get_search) {
         return $query->where('appl_id', 'like', "%$get_search%");
        })
        ->when(!empty($status), function ($query) use ($status) {
          return  $query->Where('paid', '=', "$status");
        })
        ->when(!empty($agent), function ($query) use ($agent) {
          return  $query->Where('users.id', '=', "$agent");
        })
        ->count();
    }
    $data = array();
    if (!empty($users)) {
      foreach ($users as $page) {

        $universityApplication = UniversityApplication::where("id", $page->appl_id)->first();
        $currency = "";
        if ($universityApplication) {
          if ($universityApplication->fees_currency) {
            $currency = $universityApplication->fees_currency . " " . get_currency_symbol($universityApplication->fees_currency);
          }
        }
        if ($page->paid_date) {
         $datepaid= get_date_time_formated($page->paid_date);
        }else{
          $datepaid="<div class=btn btn-success>Pending</div>";
        }

        $customResult['created_at'] = date("d M Y", strtotime($page->created_at));
        $customResult['agent'] = $page->first_name . " " . $page->last_name . "<br /><span class='text-smaller text-muted'>" . $page->email . "</small>";
        $customResult['applId'] = $page->appl_id;
        $customResult['commission_notes'] = $page->agent_commission_notes;
        $customResult['commission'] =  $currency . " " . $page->amount;
        $customResult['deduction'] = $currency . " " . $page->deduction;
        $customResult['net_pay'] = $currency . " " . $page->net_pay;
        $customResult['trans_details'] = $page->notes;
        $customResult['date_paid'] = $datepaid;

        

        $data[] = $customResult;
      }
    }

    $json_data = array(
      "draw"            => intval($request->input('draw')),
      "recordsTotal"    => intval($totalData),
      "recordsFiltered" => intval($totalFiltered),
      "data"            => $data,
      "status"          =>  $status
    );

    echo json_encode($json_data);
  }
}
