<?php

namespace App\Http\Controllers\Backend;

use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Repositories\Backend\Auth\UserRepository;
use App\Models\Auth\Lead;
use App\Models\Auth\Branch;
use App\Models\Auth\User;
use App\Models\UserActivity;
use Illuminate\Database\Eloquent\Model;
use Log;

class AnalyticsController extends Controller
{

    private $reports = array(
        "report_leads" => "Leads Report",
        "report_students" => "Students Report",
        "report_student_referrals" => "Student Referral Report",
        "report_staff" => "Staff Report",
        "report_applications" => "Applications Report",
        "report_agents" => "Agents Report",
        "report_sales" => "Sales Report",
        // "Report - University Applications" => "University Applications Report",
        // "Report - Coaching Class Batches" => "Coaching Class Batches Report",
        // "Report - Online Class Enrollments" => "Online Class Enrollments Report",
        // "Report - Invoices" => "Invoices Report",
        // "Report - Student Attendance" => "Student Attendance Report",
        // "Report - Exam Scores" => "Exam Scores Report",
        // "Report - Student Rating" => "Student Rating Report",
        // "graph_students" => "Student Signup Graph",
        // "graph_leads" => "Leads Graph",
        // "graph_sales" => "Sales Graph",
    );

    protected $userModel;

    public function __construct()
    {
        $this->middleware(function ($request, $next) {
            if (!(\Auth::user()->isAdmin() || \Auth::user()->can('analytics'))) {
                return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
            }

            $userModel = app()->make(User::class);
            if (! $userModel instanceof Model) {
                throw new GeneralException("Class {$this->model()} must be an instance of ".Model::class);
            }
            $this->userModel = $userModel;            

            return $next($request);
        });
    }

    public function index(Request $request)
    {
        $reports = $this->reports;
        $show_report = '';
        $branch = get_staff_branch(\Auth::user()->id);
        $current_branch = $request->input('branch');
        $daterange = '';
        return view('backend.analytics.analytics_index', compact('current_branch','reports', 'show_report','branch','daterange'));
    }

    public function leadsReport(Request $request)
    {
        //DB::enableQueryLog();

        $reports = $this->reports;
        $show_report = 'report_leads';
        
        $branch = get_staff_branch(\Auth::user()->id);
        $current_branch = $request->input('branch');        
        
        $daterange = $request->input('daterange');

        

        if($request->input('subtype')) {
            $subtype = $request->input('subtype');
        } else {
            $subtype = 'summary';
        }

        $date_from = "";
        $date_to = "";        
        if(!empty($request->input('daterange'))) {
            $dts_str = $request->input('daterange');
            $dts = explode(" to ", $dts_str);
            $date_from = date("Y-m-d 00:00:00", strtotime($dts[0]));
            $date_to = date("Y-m-d 23:59:59", strtotime($dts[1]));
        }else{
            $date_from = date("Y-m-d", strtotime(now()->subDays(29)));
            $date_to = date("Y-m-d", strtotime(now()));
            $daterange = date("d-m-Y", strtotime(now()->subDays(29)))." to ".date("d-m-Y", strtotime(now()));
        }

        $my_branch = "";
        if ($request->input('branch')) {
            $my_branch = array($request->input('branch'));
        } else {
            $my_branch = get_staff_branch_ids(\Auth::user()->id);            
        }

        $lead_status = "";
        if(!empty($request->input('lead_status'))) {
            $lead_status = $request->input('lead_status');
        }

        if($subtype == 'summary') {

        if ($request->input('branch') =="branch-wise") {
            $branches = DB::table('branch')
            ->select(DB::Raw('branch.branch_name, branch.id'))
            ->get();
            return view('backend.analytics.leads_report', 
                compact('current_branch','reports', 'show_report','subtype','branch','daterange',
                'branches'
            ));
        }
         else{   
            $leads_created = $this->userModel
            ->when(!empty($my_branch), function ($query) use ($my_branch) {
                return $query->whereIn('branch', $my_branch);
            })            
            ->when(!empty($date_from), function ($query) use ($date_from) {
                return  $query->Where('created_at', '>=', "$date_from");
            })
            ->when(!empty($date_to), function ($query) use ($date_to) {
                return  $query->Where('created_at', '<=', "$date_to");
            })
            ->where(function ($query) {
                return $query->where('is_lead', 1)
                ->orWhere(function ($query) {
                    return $query->where('is_lead', 0)->whereNotNull('lead_converted_date');
                });
            })
            ->count();
            
            $leads_converted = $this->userModel
            ->when(!empty($my_branch), function ($query) use ($my_branch) {
                return $query->whereIn('branch', $my_branch);
            })            
            ->when(!empty($date_from), function ($query) use ($date_from) {
                return  $query->Where('lead_converted_date', '>=', "$date_from");
            })
            ->when(!empty($date_to), function ($query) use ($date_to) {
                return  $query->Where('lead_converted_date', '<=', "$date_to");
            })
            ->where('is_lead', 0)
            ->count();

            $followups_count = UserActivity::where('type', 'followup')
            ->join('users', 'users.id', '=', 'user_activity.user_id')
            ->when(!empty($my_branch), function ($query) use ($my_branch) {
                return $query->whereIn('users.branch', $my_branch);
            })            
            ->when(!empty($date_from), function ($query) use ($date_from) {
                return  $query->Where('users.created_at', '>=', "$date_from");
            })
            ->when(!empty($date_to), function ($query) use ($date_to) {
                return  $query->Where('users.created_at', '<=', "$date_to");
            })
            ->count();

            $counselling_count = UserActivity::where('type', 'counselling')
            ->join('users', 'users.id', '=', 'user_activity.user_id')
            ->when(!empty($my_branch), function ($query) use ($my_branch) {
                return $query->whereIn('users.branch', $my_branch);
            })            
            ->when(!empty($date_from), function ($query) use ($date_from) {
                return  $query->Where('users.created_at', '>=', "$date_from");
            })
            ->when(!empty($date_to), function ($query) use ($date_to) {
                return  $query->Where('users.created_at', '<=', "$date_to");
            })
            ->count();

            $leads_cnt = UserActivity::whereIn('type', ['followup','counselling'])
            ->join('users', 'users.id', '=', 'user_activity.user_id')
            ->when(!empty($my_branch), function ($query) use ($my_branch) {
                return $query->whereIn('users.branch', $my_branch);
            })            
            ->when(!empty($date_from), function ($query) use ($date_from) {
                return  $query->Where('users.created_at', '>=', "$date_from");
            })
            ->when(!empty($date_to), function ($query) use ($date_to) {
                return  $query->Where('users.created_at', '<=', "$date_to");
            })
            ->distinct()
            ->count('user_id');
            if($followups_count) {
                $avg_followups = round(($followups_count + $counselling_count) / $leads_cnt, 1);
            } else {
                $avg_followups = 0;
            }

            // $queries = DB::getQueryLog();
            // Log::debug(end($queries));            

            return view('backend.analytics.leads_report', 
                compact('current_branch','reports', 'show_report','subtype','branch','daterange',
                'leads_created','leads_converted','followups_count','counselling_count','avg_followups'
            ));
         }
        } elseif($subtype == 'stats_by_status') {

            if ($request->input('branch') =="branch-wise") {
                $branches = DB::table('branch')
                ->select(DB::Raw('branch.branch_name, branch.id'))
                ->get();
               
                $lead_status = lead_status_list();
                return view('backend.analytics.leads_stats_by_status', 
                compact('current_branch','reports', 'show_report','subtype','branch','daterange',
                'branches','lead_status'));
            }
            else{ 
            $lead_status_stats = [];
            foreach(lead_status_list() as $lead_status) {
                $lead_status_stats[$lead_status->status] = $this->userModel
                ->when(!empty($my_branch), function ($query) use ($my_branch) {
                    return $query->whereIn('branch', $my_branch);
                })
                ->where('lead_status', $lead_status->status)
                ->count();
            }

            return view('backend.analytics.leads_stats_by_status', 
                compact('current_branch','reports', 'show_report','subtype','branch','daterange',
                'lead_status_stats'));
            }
        }
        elseif($subtype == 'all_leads') {
            $leads = $this->userModel
            ->join('user_details', 'users.id', '=', 'user_details.userid')
            // ->leftJoin('user_activity', 'users.id', '=', 'user_activity.user_id')
            ->when(!empty($my_branch), function ($query) use ($my_branch) {
                return $query->whereIn('branch', $my_branch);
            })            
            ->when(!empty($date_from), function ($query) use ($date_from) {
                return  $query->Where('users.created_at', '>=', "$date_from");
            })
            ->when(!empty($date_to), function ($query) use ($date_to) {
                return  $query->Where('users.created_at', '<=', "$date_to");
            })
            ->where(function ($query) {
                return $query->where('is_lead', 1);
                // ->orWhere(function ($query) {
                //     return $query->where('is_lead', 0)->whereNotNull('lead_converted_date');
                // });
            })
            ->withTrashed()
            ->where('removed', "0")
            ->get(['users.*','user_details.intake_year','user_details.intake_month','user_details.qualification_score','user_details.highest_level_education','user_details.passout_year','user_details.current_course','user_details.work_experience','user_details.toefl','user_details.ielts','user_details.pte','user_details.gre','user_details.duolingo','user_details.accommodation','user_details.study_permit','user_details.education_loan','user_details.lead_source_detail','user_details.lead_campaign_name','user_details.lead_ad_name','user_details.lead_form_name','user_details.lead_lead_id','user_details.lead_notes']);

            // $followups = DB::table('user_activity')
            // ->where('user_id',)
            // ->get();


            return view('backend.analytics.leads_report', 
                compact('current_branch','reports', 'show_report','subtype','branch','daterange',
                'leads'));
            
        }
        
        elseif($subtype == 'stats_by_staff') {

            $staff_leads = $this->userModel
            ->join('users as staff','staff.id', '=', 'users.assigned_to')
            ->select(DB::Raw('staff.id, staff.first_name, staff.last_name, staff.email, count(*) as cnt'))
            ->when(!empty($my_branch), function ($query) use ($my_branch) {
                return $query->whereIn('users.branch', $my_branch);
            })            
            ->when(!empty($lead_status), function ($query) use ($lead_status) {
                return  $query->Where('users.lead_status', $lead_status);
            })
            ->where('users.is_lead', 1)
            ->groupBy('staff.id')
            ->get();

            return view('backend.analytics.leads_stats_by_staff', 
            compact('current_branch','reports', 'show_report','subtype','branch','daterange',
            'lead_status','staff_leads'));            


        }
        elseif($subtype == 'converted_staff_wise') {

            $converted_staff_wise = $this->userModel
            ->join('users as staff','staff.id', '=', 'users.lead_converted_by')
            ->select(DB::Raw('staff.id, staff.first_name, staff.last_name, staff.email, count(*) as cnt'))
            ->when(!empty($my_branch), function ($query) use ($my_branch) {
                return $query->whereIn('users.branch', $my_branch);
            })   
            ->when(!empty($date_from), function ($query) use ($date_from) {
                return  $query->Where('users.lead_converted_date', '>=', "$date_from");
            })
            ->when(!empty($date_to), function ($query) use ($date_to) {
                return  $query->Where('users.lead_converted_date', '<=', "$date_to");
            }) 
            ->Where('users.lead_status', get_converted_lead_status())
            ->groupBy('staff.id')
            ->get();

            return view('backend.analytics.leads_stats_by_staff', 
            compact('current_branch','reports', 'show_report','subtype','branch','daterange',
            'lead_status','converted_staff_wise'));            


        }
        
        elseif($subtype == 'charts') {

            //Leads created
            $leads_created_labels = [];
            $leads_created_data = [];
            for($i=12;$i>=0;$i--) {
                $cur_mth = date("Y-m-", strtotime("-".$i." months"));

                $leads_created = $this->userModel
                ->when(!empty($my_branch), function ($query) use ($my_branch) {
                    return $query->whereIn('branch', $my_branch);
                })            
                ->Where('created_at', 'like', "$cur_mth%")
                ->where(function ($query) {
                    return $query->where('is_lead', 1)
                    ->orWhere(function ($query) {
                        return $query->where('is_lead', 0)->whereNotNull('lead_converted_date');
                    });
                })
                ->count();
                $leads_created_labels[] = "'".date("m/Y", strtotime("-".$i." months"))."'";
                $leads_created_data[] = $leads_created;
            }

            //Lead followups
            $leads_followups_labels = [];
            $leads_followups_data = [];
            for($i=12;$i>=0;$i--) {
                $cur_mth = date("Y-m-", strtotime("-".$i." months"));

                $leads_followups = DB::table('user_activity')
                ->join('users', 'users.id', '=', 'user_activity.user_id')
                ->when(!empty($my_branch), function ($query) use ($my_branch) {
                    return $query->whereIn('branch', $my_branch);
                })            
                ->WhereIn('type', ['followup'])
                ->Where('user_activity.created_at', 'like', "$cur_mth%")
                ->Where('user_activity.lead_followup', 1)
                ->count();
                $leads_followups_labels[] = "'".date("m/Y", strtotime("-".$i." months"))."'";
                $leads_followups_data[] = $leads_followups;
            }

            //Lead counselling
            $leads_counselling_labels = [];
            $leads_counselling_data = [];
            for($i=12;$i>=0;$i--) {
                $cur_mth = date("Y-m-", strtotime("-".$i." months"));

                $leads_counselling = DB::table('user_activity')
                ->join('users', 'users.id', '=', 'user_activity.user_id')
                ->when(!empty($my_branch), function ($query) use ($my_branch) {
                    return $query->whereIn('branch', $my_branch);
                })            
                ->WhereIn('type', ['followup'])
                ->Where('user_activity.created_at', 'like', "$cur_mth%")
                ->Where('user_activity.lead_followup', 1)
                ->count();
                $leads_counselling_labels[] = "'".date("m/Y", strtotime("-".$i." months"))."'";
                $leads_counselling_data[] = $leads_counselling;
            }


            //Lead converted
            $leads_converted_labels = [];
            $leads_converted_data = [];
            for($i=12;$i>=0;$i--) {
                $cur_mth = date("Y-m-", strtotime("-".$i." months"));

                $leads_converted = $this->userModel
                ->when(!empty($my_branch), function ($query) use ($my_branch) {
                    return $query->whereIn('branch', $my_branch);
                })            
                ->Where('lead_converted_date', 'like', "$cur_mth%")
                ->count();
                $leads_converted_labels[] = "'".date("m/Y", strtotime("-".$i." months"))."'";
                $leads_converted_data[] = $leads_converted;
            }


            return view('backend.analytics.leads_charts', 
            compact('current_branch','reports', 'show_report','subtype','branch','daterange',
            'leads_created_labels', 'leads_created_data', 
            'leads_followups_labels', 'leads_followups_data',
            'leads_counselling_labels', 'leads_counselling_data',
            'leads_converted_labels', 'leads_converted_data'
        ));            

        }
        //return view('backend.analytics.leads_report', compact('current_branch','reports', 'show_report','subtype','branch'));
    }

//Applications Report
public function applicationsReport(Request $request)
{
    //DB::enableQueryLog();

    $reports = $this->reports;
    $show_report = 'report_applications';
    
    $branch = get_staff_branch(\Auth::user()->id);
    $current_branch = $request->input('branch');        
    
    $daterange = $request->input('daterange');

    if($request->input('subtype')) {
        $subtype = $request->input('subtype');
    } else {
        $subtype = 'summary';
    }

    $date_from = "";
    $date_to = "";        
    if(!empty($request->input('daterange'))) {
        $dts_str = $request->input('daterange');
        $dts = explode(" to ", $dts_str);
        $date_from = date("Y-m-d 00:00:00", strtotime($dts[0]));
        $date_to = date("Y-m-d 23:59:59", strtotime($dts[1]));
    }else{
        $date_from = date("Y-m-d", strtotime(now()->subDays(29)));
        $date_to = date("Y-m-d", strtotime(now()));
        $daterange = date("d-m-Y", strtotime(now()->subDays(29)))." to ".date("d-m-Y", strtotime(now()));
    }

    $my_branch = "";
    if ($request->input('branch')) {
        $my_branch = array($request->input('branch'));
    } else {
        $my_branch = get_staff_branch_ids(\Auth::user()->id);            
    }

    $appl_status = "";
    if(!empty($request->input('appl_status'))) {
        $appl_status = $request->input('appl_status');
    }

    if($subtype == 'summary') {

    if ($request->input('branch') =="branch-wise") {
        $branches = DB::table('branch')
        ->select(DB::Raw('branch.branch_name, branch.id'))
        ->get();
        return view('backend.analytics.applications_report', 
            compact('current_branch','reports', 'show_report','subtype','branch','daterange',
            'branches'
        ));
    }
     else{   
        $appls_created = DB::table('university_application')
        ->join('users', 'users.id', '=', 'university_application.userid')
        ->when(!empty($my_branch), function ($query) use ($my_branch) {
            return $query->whereIn('branch', $my_branch);
        })            
        ->when(!empty($date_from), function ($query) use ($date_from) {
            return  $query->Where('university_application.created_at', '>=', "$date_from");
        })
        ->when(!empty($date_to), function ($query) use ($date_to) {
            return  $query->Where('university_application.created_at', '<=', "$date_to");
        })
        ->count();
        
        $appls_completed = DB::table('university_application')
        ->join('users', 'users.id', '=', 'university_application.userid')
        ->when(!empty($my_branch), function ($query) use ($my_branch) {
            return $query->whereIn('branch', $my_branch);
        })            
        ->when(!empty($date_from), function ($query) use ($date_from) {
            return  $query->Where('university_application.case_closed_ts', '>=', "$date_from");
        })
        ->when(!empty($date_to), function ($query) use ($date_to) {
            return  $query->Where('university_application.case_closed_ts', '<=', "$date_to");
        })
        ->where('status', 'Case Closed - Admission Complete')
        ->where('case_closed','1')
        ->where('case_closed_reason','admission_complete')
        ->count();

        $appls_percent = DB::table('university_application')
        ->join('users', 'users.id', '=', 'university_application.userid')
        ->when(!empty($my_branch), function ($query) use ($my_branch) {
            return $query->whereIn('users.branch', $my_branch);
        })            
        ->when(!empty($date_from), function ($query) use ($date_from) {
            return  $query->Where('university_application.created_at', '>=', "$date_from");
        })
        ->when(!empty($date_to), function ($query) use ($date_to) {
            return  $query->Where('university_application.created_at', '<=', "$date_to");
        })
        ->select(DB::Raw('count(university_application.id) * 100 / (select count(*) from university_application) as university_application_percent'))
        ->whereNotIn('status', ['Pending', 'Documents Ready', 'Admission Criteria Met', 'Application Fee Paid','Application Submitted','Case Closed - Rejected By Institution','Case Closed - Visa Rejected','Case Closed - Other reasons'])
        ->get();


        // $queries = DB::getQueryLog();
        // Log::debug(end($queries));            

        return view('backend.analytics.applications_report', 
            compact('current_branch','reports', 'show_report','subtype','branch','daterange',
            'appls_created','appls_completed','appls_percent'
        ));
     }
    } elseif($subtype == 'stats_by_status') {

        if ($request->input('branch') =="branch-wise") {
            $branches = DB::table('branch')
            ->select(DB::Raw('branch.branch_name, branch.id'))
            ->get();
           
            $appls_status = get_application_statuses();
            return view('backend.analytics.applications_stats_by_status', 
            compact('current_branch','reports', 'show_report','subtype','branch','daterange',
            'branches','appls_status'));
        }
        else{ 
        $applications_status_stats = [];
        foreach(get_application_statuses() as $appls_status) {
            
            $applications_status_stats[$appls_status->status_name] = DB::table('university_application')
            ->join('users', 'users.id', '=', 'university_application.userid')
            ->when(!empty($my_branch), function ($query) use ($my_branch) {
                return $query->whereIn('branch', $my_branch);
            })
            ->where('status', $appls_status->status_name)
            ->count();
        }
        return view('backend.analytics.applications_stats_by_status', 
            compact('current_branch','reports', 'show_report','subtype','branch','daterange',
            'applications_status_stats'));
        }
    } 
    elseif($subtype == 'all_apps') {
        $applications = DB::table('university_application')
        ->join('users', 'users.id', '=', 'university_application.userid')
        ->when(!empty($my_branch), function ($query) use ($my_branch) {
            return $query->whereIn('branch', $my_branch);
        })            
        ->when(!empty($date_from), function ($query) use ($date_from) {
            return  $query->Where('university_application.created_at', '>=', "$date_from");
        })
        ->when(!empty($date_to), function ($query) use ($date_to) {
            return  $query->Where('university_application.created_at', '<=', "$date_to");
        })
        ->select(DB::raw('university_application.*, users.first_name, users.last_name, users.email, users.mobile'))
        ->get();

        return view('backend.analytics.applications_report', 
            compact('current_branch','reports', 'show_report','subtype','branch','daterange',
            'applications'));
    }
    elseif($subtype == 'stats_by_staff') {

        $staff_applications = DB::table('university_application')
        ->join('users', 'users.id', '=', 'university_application.userid')
        ->join('users as staff','staff.id', '=', 'users.assigned_to')
        ->select(DB::Raw('staff.id, staff.first_name, staff.last_name, staff.email, count(*) as cnt'))
        ->when(!empty($my_branch), function ($query) use ($my_branch) {
            return $query->whereIn('users.branch', $my_branch);
        })
        ->when(!empty($appl_status), function ($query) use ($appl_status) {
            return  $query->Where('university_application.status', $appl_status);
        })
        ->groupBy('staff.id')
        ->get();
        return view('backend.analytics.applications_stats_by_staff', 
        compact('current_branch','reports', 'show_report','subtype','branch','daterange',
        'appl_status','staff_applications'));            


    } elseif($subtype == 'charts') {

        //Appls created
        $appls_created_labels = [];
        $appls_created_data = [];
        for($i=12;$i>=0;$i--) {
            $cur_mth = date("Y-m-", strtotime("-".$i." months"));

            $appls_created = DB::table('university_application')
            ->join('users', 'users.id', '=', 'university_application.userid')
            ->when(!empty($my_branch), function ($query) use ($my_branch) {
                return $query->whereIn('branch', $my_branch);
            })            
            ->Where('university_application.created_at', 'like', "$cur_mth%")
            ->count();
            $appls_created_labels[] = "'".date("m/Y", strtotime("-".$i." months"))."'";
            $appls_created_data[] = $appls_created;
        }

        //Appls Completed
        $appls_completed_labels = [];
        $appls_completed_data = [];
        for($i=12;$i>=0;$i--) {
            $cur_mth = date("Y-m-", strtotime("-".$i." months"));

            $appls_completed = DB::table('university_application')
            ->join('users', 'users.id', '=', 'university_application.userid')
            ->when(!empty($my_branch), function ($query) use ($my_branch) {
                return $query->whereIn('branch', $my_branch);
            })            
            ->where('status', 'Case Closed - Admission Complete')
            ->where('case_closed','1')
            ->where('case_closed_reason','admission_complete')
            ->Where('university_application.case_closed_ts', 'like', "$cur_mth%")
            ->count();
            $appls_completed_labels[] = "'".date("m/Y", strtotime("-".$i." months"))."'";
            $appls_completed_data[] = $appls_completed;
        }

        return view('backend.analytics.applications_charts', 
        compact('current_branch','reports', 'show_report','subtype','branch','daterange',
        'appls_created_labels', 'appls_created_data', 
        'appls_completed_labels', 'appls_completed_data',
    ));            

    }
}

    /*public function leadsReportData(Request $request)
    {
        if (!$request->ajax()) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this page.');
        }

        if (empty($request->input('branch'))) {
            $my_branch = get_staff_branch_ids(\Auth::user()->id);
        }
        else if (!empty($request->input('branch')) && $request->input('branch') == 'all') {
            $my_branch = get_staff_branch_ids(\Auth::user()->id);
        }
        else{
            $my_branch = array($request->input('branch'));
        }

        $columns = array(
            0 => 'first_name',
            1 => 'last_name',
            2 => 'email',
            3 => 'mobile',
            5 => 'lead_status',
            7 => 'gender',
            9 => 'location',
            // 11 => 'lead_type',
            11 => 'student_type',
            13 => 'course_interested',
            // 9 => 'country_interested',
            16 => 'lead_source',
            17 => 'source_details',
            20 => 'notes',
            21 => 'assigned_to',  
            22 => 'last_followup',          
            23 => 'created_at',
            // 0 => 'first_name',
            // 1 => 'last_name',
            // 2 => 'email',
            // 3 => 'mobile',
            // 4 => 'status',
            // 5 => 'gender',
            // 6 => 'location',
            // 7 => 'lead_type',
            // 8 => 'course_interested',
            // 9 => 'country_interested',
            // 10 => 'lead_source',
            // 11 => 'source_details',
            // 12 => 'notes',
            // 13 => 'assigned_to',  
            // 14 => 'last_followup',          
            // 15 => 'created_at',
        );

        // $totalData = DB::table('leads')
        $totalData = DB::table('users')
        ->select('users.*','user_details.current_course','user_details.accommodation','user_details.study_permit','user_details.lead_source_detail','user_details.lead_notes')
        ->join('user_details', 'user_details.userid', '=', 'users.id')
        ->where('is_lead',1)
        ->when(!empty($my_branch), function ($query) use ($my_branch) {
                return $query->whereIn('branch', $my_branch);
            })
        ->count();
        $totalFiltered = $totalData;
        $limit = $request->input('length');
        $start = $request->input('start');
        $order = $columns[$request->input('order.0.column')];
        $dir = $request->input('order.0.dir');
        $date_from = $request->input('date_from');
        $date_to = $request->input('date_to');

        if (empty($date_from) && empty($date_to)) {
            //Log::debug("AnalyticsController: no date filters");
            // $leads = DB::table('leads')
            $leads = DB::table('users')
            ->select('users.*','user_details.current_course','user_details.accommodation','user_details.study_permit','user_details.lead_source_detail','user_details.lead_notes')
            ->join('user_details', 'user_details.userid', '=', 'users.id')
            ->where('is_lead',1)
                ->when(!empty($my_branch), function ($query) use ($my_branch) {
                    return $query->whereIn('branch', $my_branch);
                })
                ->offset($start)
                ->limit($limit)
                ->orderBy($order, $dir)
                ->get();
        } else {
            //Log::debug("AnalyticsController: has date filters");
            // $leads = DB::table('leads')
            $leads = DB::table('users')
            ->select('users.*','user_details.current_course','user_details.accommodation','user_details.study_permit','user_details.lead_source_detail','user_details.lead_notes')
            ->join('user_details', 'user_details.userid', '=', 'users.id')
            ->where('is_lead',1)
                ->when(!empty($my_branch), function ($query) use ($my_branch) {
                    return $query->whereIn('branch', $my_branch);
                })
                ->when(!empty($date_from), function ($query) use ($date_from) {
                    return  $query->Where('created_at', '>=', "$date_from");
                })
                ->when(!empty($date_to), function ($query) use ($date_to) {
                    return  $query->Where('created_at', '<=', "$date_to");
                })
                ->offset($start)
                ->limit($limit)
                ->orderBy($order, $dir)
                ->get();

            // $totalFiltered = DB::table('leads')
            $totalFiltered = DB::table('users')
            ->select('users.*','user_details.current_course','user_details.accommodation','user_details.study_permit','user_details.lead_source_detail','user_details.lead_notes')
            ->join('user_details', 'user_details.userid', '=', 'users.id')
            ->where('is_lead',1)
                ->when(!empty($my_branch), function ($query) use ($my_branch) {
                    return $query->whereIn('branch', $my_branch);
                })
                ->when(!empty($date_from), function ($query) use ($date_from) {
                    return  $query->Where('created_at', '>=', "$date_from");
                })
                ->when(!empty($date_to), function ($query) use ($date_to) {
                    return  $query->Where('created_at', '<=', "$date_to");
                })
                ->count();
        }

        $data = $customResult = array();
        if (!empty($leads)) {
            $count = 1;
            foreach ($leads as $row) {
                // $customResult['id'] = $count++;
                $customResult = array();
                $customResult['first_name'] = $row->first_name;
                $customResult['last_name'] = $row->last_name;
                $customResult['email'] = $row->email;
                $customResult['mobile'] = ($row->dialcode!='')? $row->dialcode.'-'.$row->mobile:$row->mobile;
                $customResult['alternate_contact'] = $row->alternate_contact;
                $customResult['country_of_citizenship'] = get_country_name($row->country);
                // $customResult['dob'] = $row->date_of_birth;
                $customResult['dob'] = $row->birthday;
                $customResult['current_course'] = $row->current_course;
                // $customResult['study_level'] = $row->study_level;
                $customResult['overseas_study_level'] = $row->overseas_study_level;
                if($row->accommodation==1){
                    $customResult['accommodation'] = "Yes";
                }else{
                 $customResult['accommodation'] = "No";
                }
                if($row->study_permit==1){
                    $customResult['study_permit'] = "Yes";
                }else{
                 $customResult['study_permit'] = "No";
                }
                $customResult['gender'] = $row->gender;
                // $customResult['location'] = $row->address;
                $customResult['location'] = $row->address_1;
                $customResult['branch'] = branch_name($row->branch);

                if ($row->student_type == 'Coaching')
                    $customResult['student_type'] = '<span class="badge badge-default">Coaching</span>';
                elseif ($row->student_type == 'Study Abroad')
                    $customResult['student_type'] = '<span class="badge badge-primary">Study Abroad</span>';
                elseif($row->student_type == 'Both')
                    $customResult['student_type'] = '<span class="badge badge-warning">Coaching & Study Abroad</span>';
                else 
                    $customResult['student_type'] = '<span class="badge badge-warning">'.$row->student_type.'</span>';

                $customResult['course_interested'] = '';                    
                if($row->coaching_course_interested) {
                    $course = Course::find($row->coaching_course_interested);
                    if($course) {
                        $customResult['course_interested'] = $course->title;
                    }
                }

                // $customResult['country_interested'] = str_replace(",", ", ",$row->country);
                $customResult['lead_source'] = $row->lead_source;
                $customResult['source_details'] = $row->lead_source_detail;
                $customResult['lead_notes'] = $row->lead_notes;
                $customResult['assigned_to'] = '';
                if($row->assigned_to) {
                    $user = User::find($row->assigned_to);
                    if($user) {
                        $customResult['assigned_to'] = $user->first_name." ".$user->last_name;
                    }
                }
                $customResult['last_followup'] = $row->last_followup;
                $customResult['created_at'] = $row->created_at;
                $customResult['counselling_date'] = $row->counselling_date ? date('d M, g:ia', strtotime($row->counselling_date)) : "--";

                if ($row->lead_status == 'Open')
                    $customResult['lead_status'] = '<span class="tag" style="background-color: #16b4d4 !important">' . $row->lead_status . '</span>';
                elseif ($row->lead_status == 'Hot')
                    $customResult['lead_status'] = '<span class="tag" style="background-color: #dc3545 !important">' . $row->lead_status . '</span>';
                elseif ($row->lead_status == 'Warm')
                    $customResult['lead_status'] = '<span class="tag" style="background-color: #ed7228 !important">' . $row->lead_status . '</span>';
                elseif ($row->lead_status == 'Cold')
                    $customResult['lead_status'] = '<span class="tag" style="background-color: #ffc107 !important">' . $row->lead_status . '</span>';
                elseif ($row->lead_status == 'Long term nurture')
                    $customResult['lead_status'] = '<span class="tag" style="background-color: #ae9c7a !important">' . $row->lead_status . '</span>';
                elseif ($row->lead_status == 'Converted')
                    $customResult['lead_status'] = '<span class="tag" style="background-color: #28a745 !important">' . $row->lead_status . '</span>';
                elseif ($row->lead_status == 'Disqualified')
                    $customResult['lead_status'] = '<span class="tag" style="background-color: #6c757d !important">' . $row->lead_status . '</span>';
                else
                    $customResult['lead_status'] = '<span class="tag">' . $row->lead_status . '</span>';

                $data[] = $customResult;
            }
        }

        $json_data = array(
            "draw"            => intval($request->input('draw')),
            "recordsTotal"    => intval($totalData),
            "recordsFiltered" => intval($totalFiltered),
            "data"            => $data
        );

        echo json_encode($json_data);
    }*/

    public function studentReferralsReport(Request $request)
    {
        DB::enableQueryLog();
        $reports = $this->reports;
        $show_report = 'report_student_referrals';
        $branch = get_staff_branch(\Auth::user()->id);
        $current_branch = $request->input('branch');        
        $daterange = $request->input('daterange');
        if($request->input('subtype')) {
            $subtype = $request->input('subtype');
        } else {
            $subtype = 'student_referrals';
        }

        $date_from = "";
        $date_to = "";        
        if(!empty($request->input('daterange'))) {
            $dts_str = $request->input('daterange');
            $dts = explode(" to ", $dts_str);
            $date_from = date("Y-m-d 00:00:00", strtotime($dts[0]));
            $date_to = date("Y-m-d 23:59:59", strtotime($dts[1]));
        }
        else{
            $date_from = date("Y-m-d", strtotime(now()->subDays(29)));
            $date_to = date("Y-m-d", strtotime(now()));
            $daterange = date("d-m-Y", strtotime(now()->subDays(29)))." to ".date("d-m-Y", strtotime(now()));
        }
        //dd($daterange);
        $my_branch = "";
        if ($request->input('branch')) {
            $my_branch = array($request->input('branch'));
        } else {
            $my_branch = get_staff_branch_ids(\Auth::user()->id);            
        }

        if($subtype == 'student_referrals') {

            if ($request->input('branch') =="branch-wise") {
                $branches = DB::table('branch')
                ->select(DB::Raw('branch.branch_name, branch.id'))
                ->get();
                return view('backend.analytics.student_referrals_report', 
                    compact('current_branch','reports', 'show_report','subtype','branch','daterange',
                    'branches'
                ));
            }
            else{
                $student_referrals = User::join('users as referer_info', 'referer_info.id', '=', 'users.ref_by')
                ->select("users.id as uid", DB::raw("concat(users.first_name,' ',users.last_name) as user_full_name"),"users.ref_status","users.created_at", DB::raw("concat(referer_info.first_name,' ',referer_info.last_name) as referred_by"), "referer_info.id as rid")
                ->when(!empty($my_branch), function ($query) use ($my_branch) {
                    return $query->whereIn('users.branch', $my_branch);
                })
                ->when(!empty($date_from), function ($query) use ($date_from) {
                    return  $query->Where('users.created_at', '>=', "$date_from");
                })
                ->when(!empty($date_to), function ($query) use ($date_to) {
                    return  $query->Where('users.created_at', '<=', "$date_to");
                })
                ->orderBy('users.created_at','desc')
                ->get();

                return view('backend.analytics.student_referrals_report', 
                    compact('current_branch','reports', 'show_report','subtype','branch','daterange',
                    'student_referrals'));
            }

        } elseif($subtype == 'referral_commission_payout') {

            // $referrals_payouts = User::join('users as referer_info', 'referer_info.id', '=', 'users.ref_by')
            //     ->select("users.id as uid", DB::raw("concat(users.first_name,' ',users.last_name) as user_full_name"),"users.ref_status","users.created_at", DB::raw("concat(referer_info.first_name,' ',referer_info.last_name) as referred_by"), "referer_info.id as rid")
            //     ->when(!empty($my_branch), function ($query) use ($my_branch) {
            //         return $query->whereIn('users.branch', $my_branch);
            //     })
            //     ->when(!empty($date_from), function ($query) use ($date_from) {
            //         return  $query->Where('users.created_at', '>=', "$date_from");
            //     })
            //     ->when(!empty($date_to), function ($query) use ($date_to) {
            //         return  $query->Where('users.created_at', '<=', "$date_to");
            //     })
            //     ->orderBy('users.created_at','desc')
            //     ->get();

$referrals_payouts = DB::table('referral_commission')
                ->join('users as referer_info', 'referer_info.id', '=', 'referral_commission.user_id')
                ->select(DB::raw("concat(referer_info.first_name,' ',referer_info.last_name) as ref_full_name, referer_info.id as rid, sum(debit) as total_reward"), 'referral_commission.*')
                ->when(!empty($my_branch), function ($query) use ($my_branch) {
                    return $query->whereIn('referer_info.branch', $my_branch);
                })
                ->when(!empty($date_from), function ($query) use ($date_from) {
                    return  $query->Where('referral_commission.created_at', '>=', "$date_from");
                })
                ->when(!empty($date_to), function ($query) use ($date_to) {
                    return  $query->Where('referral_commission.created_at', '<=', "$date_to");
                })
                ->where('referral_commission.paid',1)
                ->groupBy('referral_commission.user_id')
                ->get();

                return view('backend.analytics.student_referrals_report', 
                    compact('current_branch','reports', 'show_report','subtype','branch','daterange',
                    'referrals_payouts'));
        }
    }



    public function studentsReport(Request $request)
    {
        DB::enableQueryLog();

        $reports = $this->reports;
        $show_report = 'report_students';
        
        $branch = get_staff_branch(\Auth::user()->id);
        $current_branch = $request->input('branch');        
        
        $daterange = $request->input('daterange');

        if($request->input('subtype')) {
            $subtype = $request->input('subtype');
        } else {
            $subtype = 'summary';
        }

        $date_from = "";
        $date_to = "";        
        if(!empty($request->input('daterange'))) {
            $dts_str = $request->input('daterange');
            $dts = explode(" to ", $dts_str);
            $date_from = date("Y-m-d 00:00:00", strtotime($dts[0]));
            $date_to = date("Y-m-d 23:59:59", strtotime($dts[1]));
        }
        else{
            $date_from = date("Y-m-d", strtotime(now()->subDays(29)));
            $date_to = date("Y-m-d", strtotime(now()));
            $daterange = date("d-m-Y", strtotime(now()->subDays(29)))." to ".date("d-m-Y", strtotime(now()));
        }
        //dd($daterange);
        $my_branch = "";
        if ($request->input('branch')) {
            $my_branch = array($request->input('branch'));
        } else {
            $my_branch = get_staff_branch_ids(\Auth::user()->id);            
        }

        // $lead_status = "";
        // if(!empty($request->input('lead_status'))) {
        //     $lead_status = $request->input('lead_status');
        // }

        if($subtype == 'summary') {

            if ($request->input('branch') =="branch-wise") {
                $branches = DB::table('branch')
                ->select(DB::Raw('branch.branch_name, branch.id'))
                ->get();
                return view('backend.analytics.students_report', 
                    compact('current_branch','reports', 'show_report','subtype','branch','daterange',
                    'branches'
                ));
            }
            else{
            
                $student_status_stats = [];
                foreach(get_overseas_status_list() as $status_name) {
                    $student_status_stats[$status_name->status] = $this->userModel
                    ->when(!empty($my_branch), function ($query) use ($my_branch) {
                        return $query->whereIn('branch', $my_branch);
                    })
                    ->where('overseas_status', $status_name->status)
                    ->when(!empty($date_from), function ($query) use ($date_from) {
                        return  $query->Where('student_created_at', '>=', "$date_from");
                    })
                    ->when(!empty($date_to), function ($query) use ($date_to) {
                        return  $query->Where('student_created_at', '<=', "$date_to");
                    })
                    ->where('is_lead', 0)
                    ->count();

                    $queries = DB::getQueryLog();
                    Log::debug(end($queries));                 
                }

                return view('backend.analytics.students_report', 
                    compact('current_branch','reports', 'show_report','subtype','branch','daterange',
                    'student_status_stats'));
            }

        } elseif($subtype == 'stats_by_staff') {

            $staff_students = $this->userModel
            ->join('users as staff','staff.id', '=', 'users.assigned_to')
            ->select(DB::Raw('staff.id, staff.first_name, staff.last_name, staff.email, count(*) as cnt'))
            ->when(!empty($my_branch), function ($query) use ($my_branch) {
                return $query->whereIn('users.branch', $my_branch);
            })            
            // ->when(!empty($lead_status), function ($query) use ($lead_status) {
            //     return  $query->Where('users.lead_status', $lead_status);
            // })
            ->where('users.is_lead', 0)
            ->groupBy('staff.id')
            ->get();

            return view('backend.analytics.students_stats_by_staff', 
            compact('current_branch','reports', 'show_report','subtype','branch','daterange',
            'staff_students')); 
        }
        elseif($subtype == 'all_students') {
            $students = $this->userModel->join('user_details', 'users.id', '=', 'user_details.userid')
            ->when(!empty($my_branch), function ($query) use ($my_branch) {
                return $query->whereIn('branch', $my_branch);
            })            
            ->when(!empty($date_from), function ($query) use ($date_from) {
                return  $query->Where('users.student_created_at', '>=', "$date_from");
            })
            ->when(!empty($date_to), function ($query) use ($date_to) {
                return  $query->Where('users.student_created_at', '<=', "$date_to");
            })
            ->where(function ($query) {
                return $query->where('is_lead', 0);
            })
            ->withTrashed()
            ->where('removed', "0")
            ->get(['users.*','user_details.intake_year','user_details.intake_month','user_details.qualification_score','user_details.highest_level_education','user_details.passout_year','user_details.current_course','user_details.work_experience','user_details.toefl','user_details.ielts','user_details.pte','user_details.gre','user_details.duolingo','user_details.accommodation','user_details.study_permit','user_details.education_loan','user_details.lead_source_detail','user_details.lead_campaign_name','user_details.lead_ad_name','user_details.lead_form_name','user_details.lead_lead_id','user_details.lead_notes']);


            return view('backend.analytics.students_report', 
                compact('current_branch','reports', 'show_report','subtype','branch','daterange',
                'students'));
            
        }
        elseif($subtype == 'charts') {

            //All Students
            $all_students_labels = [];
            $all_students_data = [];
            for($i=12;$i>=0;$i--) {
                $cur_mth = date("Y-m-", strtotime("-".$i." months"));

                $all_students = $this->userModel
                ->when(!empty($my_branch), function ($query) use ($my_branch) {
                    return $query->whereIn('branch', $my_branch);
                })            
                ->Where('student_created_at', 'like', "$cur_mth%")
                ->where('is_lead', 0)
                // ->where(function ($query) {
                //     return $query->where('is_lead', 1)
                //     ->orWhere(function ($query) {
                //         return $query->where('is_lead', 0)->whereNotNull('lead_converted_date');
                //     });
                // })
                ->count();
                $all_students_labels[] = "'".date("m/Y", strtotime("-".$i." months"))."'";
                $all_students_data[] = $all_students;
            }


            //Students Added
            $students_added_labels = [];
            $students_added_data = [];
            for($i=12;$i>=0;$i--) {
                $cur_mth = date("Y-m-", strtotime("-".$i." months"));

                $students_added = $this->userModel
                ->when(!empty($my_branch), function ($query) use ($my_branch) {
                    return $query->whereIn('branch', $my_branch);
                })            
                ->Where('created_at', 'like', "$cur_mth%")
                ->where('is_lead', 0)
                ->whereNull('lead_converted_date')
                ->count();
                $students_added_labels[] = "'".date("m/Y", strtotime("-".$i." months"))."'";
                $students_added_data[] = $students_added;
            }

            //Lead to Student
            $lead_to_student_labels = [];
            $lead_to_student_data = [];
            for($i=12;$i>=0;$i--) {
                $cur_mth = date("Y-m-", strtotime("-".$i." months"));

                $lead_to_student = $this->userModel
                ->when(!empty($my_branch), function ($query) use ($my_branch) {
                    return $query->whereIn('branch', $my_branch);
                })            
                ->Where('created_at', 'like', "$cur_mth%")
                ->where('is_lead', 0)
                ->whereNotNull('lead_converted_date')
                ->count();
                $lead_to_student_labels[] = "'".date("m/Y", strtotime("-".$i." months"))."'";
                $lead_to_student_data[] = $lead_to_student;
            }


            return view('backend.analytics.students_charts', 
            compact('current_branch','reports', 'show_report','subtype','branch','daterange',
            'all_students_labels', 'all_students_data',
            'students_added_labels', 'students_added_data',
            'lead_to_student_labels', 'lead_to_student_data'
        ));            

        }

    }

    public function staffReport(Request $request)
    {
        DB::enableQueryLog();

        $reports = $this->reports;
        $show_report = 'report_staff';
        
        $branch = get_staff_branch(\Auth::user()->id);
        $current_branch = $request->input('branch');        
        
        $daterange = $request->input('daterange');

        if($request->input('subtype')) {
            $subtype = $request->input('subtype');
        } else {
            $subtype = 'summary';
        }

        $date_from = "";
        $date_to = "";        
        if(!empty($request->input('daterange'))) {
            $dts_str = $request->input('daterange');
            $dts = explode(" to ", $dts_str);
            $date_from = date("Y-m-d 00:00:00", strtotime($dts[0]));
            $date_to = date("Y-m-d 23:59:59", strtotime($dts[1]));
        }
        else{
            $date_from = date("Y-m-d", strtotime(now()->subDays(29)));
            $date_to = date("Y-m-d", strtotime(now()));
        }

        $my_branch = "";
        if ($request->input('branch')) {
            $my_branch = array($request->input('branch'));
        } else {
            $my_branch = get_staff_branch_ids(\Auth::user()->id);            
        }

        if($subtype == 'summary') {

            if ($request->input('branch') =="branch-wise") {
                $branches = DB::table('branch')
                ->select(DB::Raw('branch.branch_name, branch.id'))
                ->get();
                return view('backend.analytics.staff_report', 
                    compact('current_branch','reports', 'show_report','subtype','branch','daterange',
                    'branches'
                ));
            }
            else{
            
                    // $staff_stats = $this->userModel
                    // ->join('users', 'users.id', '=', 'user_activity.staff')
                    // ->when(!empty($my_branch), function ($query) use ($my_branch) {
                    //     return $query->whereIn('branch', $my_branch);
                    // })
                    // ->count();

                    $staff_stats = DB::table('user_activity')
                    ->leftjoin('users','user_activity.staff','=','users.id')
                    ->when(!empty($my_branch), function ($query) use ($my_branch) {
                        return $query->whereIn('branch', $my_branch);
                    })
                    ->when(!empty($date_from), function ($query) use ($date_from) {
                        return  $query->Where('user_activity.created_at', '>=', "$date_from");
                    })
                    ->when(!empty($date_to), function ($query) use ($date_to) {
                        return  $query->Where('user_activity.created_at', '<=', "$date_to");
                    })
                    ->select(DB::raw('users.first_name, users.last_name'))
                    ->addSelect(DB::raw('sum(case when type = "followup" then 1 else 0 end) as followup_count'))
                    ->addSelect(DB::raw('sum(case when type = "counselling" then 1 else 0 end) as counselling_count'))
                    ->groupBy('user_activity.staff')
                    ->get();

                    $queries = DB::getQueryLog();
                    //dd($queries);
                    Log::debug(end($queries));                 
                

                return view('backend.analytics.staff_report', 
                    compact('current_branch','reports', 'show_report','subtype','branch','daterange',
                    'staff_stats'));
            }

        }
        elseif($subtype == 'follow_up') {
            $activities = DB::table('user_activity')
            ->leftjoin('users','user_activity.user_id','=','users.id')
            ->when(!empty($my_branch), function ($query) use ($my_branch) {
                return $query->whereIn('branch', $my_branch);
            })            
            ->when(!empty($date_from), function ($query) use ($date_from) {
                return  $query->Where('user_activity.created_at', '>=', "$date_from");
            })
            ->when(!empty($date_to), function ($query) use ($date_to) {
                return  $query->Where('user_activity.created_at', '<=', "$date_to");
            })
            ->where('type','followup')
            ->select(DB::raw('users.*,user_activity.staff,user_activity.detail,user_activity.communication_mode,user_activity.application_status '))
            ->get();

            return view('backend.analytics.staff_report', 
                compact('current_branch','reports', 'show_report','subtype','branch','daterange',
                'activities'));
            
        }
        // elseif($subtype == 'stats_by_staff') {

        //     $staff_students = $this->userModel
        //     ->join('users as staff','staff.id', '=', 'users.assigned_to')
        //     ->select(DB::Raw('staff.id, staff.first_name, staff.last_name, staff.email, count(*) as cnt'))
        //     ->when(!empty($my_branch), function ($query) use ($my_branch) {
        //         return $query->whereIn('users.branch', $my_branch);
        //     })            
        //     ->where('users.is_lead', 0)
        //     ->groupBy('staff.id')
        //     ->get();

        //     return view('backend.analytics.students_stats_by_staff', 
        //     compact('current_branch','reports', 'show_report','subtype','branch','daterange',
        //     'staff_students')); 
        // }

        // elseif($subtype == 'charts') {

        //     //All Students
        //     $all_students_labels = [];
        //     $all_students_data = [];
        //     for($i=12;$i>=0;$i--) {
        //         $cur_mth = date("Y-m-", strtotime("-".$i." months"));

        //         $all_students = $this->userModel
        //         ->when(!empty($my_branch), function ($query) use ($my_branch) {
        //             return $query->whereIn('branch', $my_branch);
        //         })            
        //         ->Where('created_at', 'like', "$cur_mth%")
        //         ->where('is_lead', 0)
              
        //         ->count();
        //         $all_students_labels[] = "'".date("m/Y", strtotime("-".$i." months"))."'";
        //         $all_students_data[] = $all_students;
        //     }


        //     //Students Added
        //     $students_added_labels = [];
        //     $students_added_data = [];
        //     for($i=12;$i>=0;$i--) {
        //         $cur_mth = date("Y-m-", strtotime("-".$i." months"));

        //         $students_added = $this->userModel
        //         ->when(!empty($my_branch), function ($query) use ($my_branch) {
        //             return $query->whereIn('branch', $my_branch);
        //         })            
        //         ->Where('created_at', 'like', "$cur_mth%")
        //         ->where('is_lead', 0)
        //         ->whereNull('lead_converted_date')
        //         ->count();
        //         $students_added_labels[] = "'".date("m/Y", strtotime("-".$i." months"))."'";
        //         $students_added_data[] = $students_added;
        //     }

        //     //Lead to Student
        //     $lead_to_student_labels = [];
        //     $lead_to_student_data = [];
        //     for($i=12;$i>=0;$i--) {
        //         $cur_mth = date("Y-m-", strtotime("-".$i." months"));

        //         $lead_to_student = $this->userModel
        //         ->when(!empty($my_branch), function ($query) use ($my_branch) {
        //             return $query->whereIn('branch', $my_branch);
        //         })            
        //         ->Where('created_at', 'like', "$cur_mth%")
        //         ->where('is_lead', 0)
        //         ->whereNotNull('lead_converted_date')
        //         ->count();
        //         $lead_to_student_labels[] = "'".date("m/Y", strtotime("-".$i." months"))."'";
        //         $lead_to_student_data[] = $lead_to_student;
        //     }

        //     return view('backend.analytics.staff_charts', 
        //     compact('current_branch','reports', 'show_report','subtype','branch','daterange',
        //     'all_students_labels', 'all_students_data',
        //     'students_added_labels', 'students_added_data',
        //     'lead_to_student_labels', 'lead_to_student_data'
        // ));            

        // }

    }


    //agents report
    public function agentsReport(Request $request)
    {
        DB::enableQueryLog();

        $reports = $this->reports;
        $show_report = 'report_agents';
        
        $branch = get_staff_branch(\Auth::user()->id);
        $current_branch = $request->input('branch');        
        
        $daterange = $request->input('daterange');

        if($request->input('subtype')) {
            $subtype = $request->input('subtype');
        } else {
            $subtype = 'summary';
        }

        $date_from = "";
        $date_to = "";        
        if(!empty($request->input('daterange'))) {
            $dts_str = $request->input('daterange');
            $dts = explode(" to ", $dts_str);
            $date_from = date("Y-m-d 00:00:00", strtotime($dts[0]));
            $date_to = date("Y-m-d 23:59:59", strtotime($dts[1]));
        }
        else{
            $date_from = date("Y-m-d", strtotime(now()->subDays(29)));
            $date_to = date("Y-m-d", strtotime(now()));
        }

        $my_branch = "";
        if ($request->input('branch')) {
            $my_branch = array($request->input('branch'));
        } else {
            $my_branch = get_staff_branch_ids(\Auth::user()->id);            
        }

        if($subtype == 'summary') {

                    $agents_stats = $this->userModel::with('roles')
                    ->whereHas('roles', function ($q) {
                      $q->whereIn('name', ['agent']);
                    })
                    ->when(!empty($my_branch), function ($query) use ($my_branch) {
                        return $query->whereIn('users.branch', $my_branch);
                    })
                    ->select(DB::Raw('users.id, users.first_name, users.last_name, users.email'))
                    ->get();

                    $queries = DB::getQueryLog();
                    //dd($queries);
                    Log::debug(end($queries));                 
                

                return view('backend.analytics.agents_report', 
                    compact('current_branch','reports', 'show_report','subtype','branch','daterange',
                    'agents_stats'));

        } 
        elseif($subtype == 'charts') {

            //Students Created
            $students_created_labels = [];
            $students_created_data = [];
            for($i=12;$i>=0;$i--) {
                $cur_mth = date("Y-m-", strtotime("-".$i." months"));
                $students_created = $this->userModel
                ->when(!empty($my_branch), function ($query) use ($my_branch) {
                    return $query->whereIn('branch', $my_branch);
                })            
                ->Where('created_at', 'like', "$cur_mth%")
                ->where('agent_id',"!=", 0)
              
                ->count();
                $students_created_labels[] = "'".date("m/Y", strtotime("-".$i." months"))."'";
                $students_created_data[] = $students_created;
            }


            //Applications Created
            $applications_created_labels = [];
            $applications_created_data = [];
            for($i=12;$i>=0;$i--) {
                $cur_mth = date("Y-m-", strtotime("-".$i." months"));

                $applications_created = DB::table('university_application')
                ->join('users', 'users.id', '=', 'university_application.userid')
                ->where('agent_id',"!=", 0)
                ->when(!empty($my_branch), function ($query) use ($my_branch) {
                    return $query->whereIn('branch', $my_branch);
                })            
                ->Where('university_application.created_at', 'like', "$cur_mth%")
                ->count();
                $applications_created_labels[] = "'".date("m/Y", strtotime("-".$i." months"))."'";
                $applications_created_data[] = $applications_created;
            }


            return view('backend.analytics.agents_charts', 
            compact('current_branch','reports', 'show_report','subtype','branch','daterange',
            'students_created_labels', 'students_created_data',
            'applications_created_labels', 'applications_created_data',
        ));            

        }

    }

    public function studentsReportData(Request $request)
    {
    //     if (!$request->ajax()) {
    //         return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this page.');
    //     }

    //     if (empty($request->input('branch'))) {
    //         $my_branch = get_staff_branch_ids(\Auth::user()->id);
    //     }
    //     else if (!empty($request->input('branch')) && $request->input('branch') == 'all') {
    //         $my_branch = get_staff_branch_ids(\Auth::user()->id);
    //     }
    //     else{
    //         $my_branch = array($request->input('branch'));
    //     }
    //     $columns = array(
    //         0 => 'first_name',
    //         1 => 'middle_name',
    //         2 => 'last_name',
    //         3 => 'email',
    //         4 => 'mobile',
    //         5 => 'title',
    //         6 => 'student_type',
    //         7 => 'country_interested',
    //         8 => 'address_1',
    //         9 => 'post_code',
    //         10 => 'city',
    //         11 => 'state',
    //         12 => 'country',
    //         13 => 'birthday',
    //         14 => 'created_at',
    //         15 => 'active',
    //         16 => 'deleted_at',
    //     );

    //     $totalData = User::when(!empty($my_branch), function ($query) use ($my_branch) {
    //             return $query->whereIn('users.branch', $my_branch);
    //         })
    //         ->active()
    //             ->whereHas('roles', function ($q) {
    //                 $q->whereIn('name', ['user']);
    //             })
    //     // ->where('removed', '0')
    //     ->where('is_lead',0)
    //     ->count();
    //     $totalFiltered = $totalData;
    //     $limit = $request->input('length');
    //     $start = $request->input('start');
    //     $order = $columns[$request->input('order.0.column')];
    //     $dir = $request->input('order.0.dir');
    //     $date_from = $date_to = "";
    //     if(!empty($request->input('date_from'))){
    //         $date_from = date('Y-m-d',strtotime($request->input('date_from')));
    //     }
    //     if(!empty($request->input('date_to'))){
    //         $date_to = date('Y-m-d',strtotime($request->input('date_to')));
    //     }
    //     if (empty($date_from) && empty($date_to)) {
    //         Log::debug("AnalyticsController: no date filters");
    //         $users = User::when(!empty($my_branch), function ($query) use ($my_branch) {
    //                     return $query->whereIn('users.branch', $my_branch);
    //                 })
    //                 ->active()
    //                     ->whereHas('roles', function ($q) {
    //                         $q->whereIn('name', ['user']);
    //                     })
    //             // ->where('removed', '0')
    //             ->where('is_lead',0)
    //             ->offset($start)
    //             ->limit($limit)
    //             ->orderBy($order, $dir)
    //             ->get();
    //     } else {
    //         Log::debug("AnalyticsController: has date filters");
    //             $users = User::when(!empty($my_branch), function ($query) use ($my_branch) {
    //                 return $query->whereIn('users.branch', $my_branch);
    //             })
    //             ->active()
    //                 ->whereHas('roles', function ($q) {
    //                     $q->whereIn('name', ['user']);
    //                 })
    //         // ->where('removed', '0')
    //             ->where('is_lead',0)
    //             ->when(!empty($date_from), function ($query) use ($date_from) {
    //                 return  $query->Where('created_at', '>=', "$date_from");
    //             })
    //             ->when(!empty($date_to), function ($query) use ($date_to) {
    //                 return  $query->Where('created_at', '<=', "$date_to");
    //             })
    //             ->offset($start)
    //             ->limit($limit)
    //             ->orderBy($order, $dir)
    //             ->get();

    //         $totalFiltered = User::when(!empty($my_branch), function ($query) use ($my_branch) {
    //                     return $query->whereIn('users.branch', $my_branch);
    //                 })
    //                 ->active()
    //                     ->whereHas('roles', function ($q) {
    //                         $q->whereIn('name', ['user']);
    //                     })
    //             // ->where('removed', '0')
    //             ->where('is_lead',0)
    //             ->when(!empty($date_from), function ($query) use ($date_from) {
    //                 return  $query->Where('created_at', '>=', "$date_from");
    //             })
    //             ->when(!empty($date_to), function ($query) use ($date_to) {
    //                 return  $query->Where('created_at', '<=', "$date_to");
    //             })
    //             ->count();
    //     }

    //     $data = $customResult = array();
    //     if (!empty($users)) {
    //         $count = 1;
    //         foreach ($users as $row) {
    //             // $customResult['id'] = $count++;
    //             $customResult = array();
    //             $customResult['first_name'] = $row->first_name;
    //             $customResult['middle_name'] = $row->middle_name;
    //             $customResult['last_name'] = $row->last_name;
    //             $customResult['email'] = $row->email;
    //             $customResult['mobile'] = $row->mobile;
    //             $customResult['title'] = $row->title;
    //             $customResult['branch'] = branch_name($row->branch);
    //             $customResult['country_interested'] = str_replace(",", ", ",$row->country_interested);
    //             // $customResult['address_1'] = $row->address_1 ." ". $row->address_2;
    //             $customResult['address'] = $row->address_1 ." ". $row->address_2;
    //             $customResult['post_code'] = $row->post_code;
    //             $customResult['city'] = $row->city;
    //             $customResult['state'] = $row->state;
    //             $customResult['country'] = get_country_name($row->country);
    //             $customResult['birthday'] = get_date_formated($row->birthday);
    //             $customResult['created_at'] = get_date_time_formated($row->created_at);
    //             $customResult['active'] = $row->active;
    //             $customResult['deleted_at'] = $row->deleted_at;


    //             if ($row->student_type == 'course')
    //                 $customResult['student_type'] = '<span class="badge badge-default">Coaching</span>';
    //             elseif ($row->student_type == 'study_abroad')
    //                 $customResult['student_type'] = '<span class="badge badge-primary">Study Abroad</span>';
    //             elseif($row->student_type == 'study_abroad_course')
    //                 $customResult['student_type'] = '<span class="badge badge-warning">Coaching & Study Abroad</span>';
    //             else 
    //                 $customResult['student_type'] = '';

    //             $data[] = $customResult;
    //         }
    //     }

    //     $json_data = array(
    //         "draw"            => intval($request->input('draw')),
    //         "recordsTotal"    => intval($totalData),
    //         "recordsFiltered" => intval($totalFiltered),
    //         "data"            => $data,
    //         "date"              => $date_from."---".$date_to
    //     );

    //     echo json_encode($json_data);
    }


    public function studentsGraph(Request $request) {
        // $reports = $this->reports;
        // $show_report = 'graph_students';
        // $branch = get_staff_branch(\Auth::user()->id);
        // $current_branch = $request->input('branch');
        // if (empty($request->input('branch'))) {
        //     $my_branch = get_staff_branch_ids(\Auth::user()->id);
        // }
        // else if (!empty($request->input('branch')) && $request->input('branch') == 'all') {
        //     $my_branch = get_staff_branch_ids(\Auth::user()->id);
        // }
        // else{
        //     $my_branch = array($request->input('branch'));
        // }
        // //@todo: If date difference is more than 90 days, we can group by month-wise

        // $day_sec = 24 * 60 * 60;
        // $find_key = ["{","}",":","\"","-"];
        // $replace_val = ["[","]",",","",","];

        // $date_from = date("Y-m-d 00:00:00", strtotime("-30 days"));
        // $date_to = date("Y-m-d H:i:s");
        // if($request->input('date_from')) {
        //     if(strtotime($request->input('date_from'))) {
        //         $date_from = date("Y-m-d 00:00:00", strtotime($request->input('date_from')));
        //     }
        // }
        // if($request->input('date_to')) {
        //     if(strtotime($request->input('date_to'))) {
        //         $date_to = date("Y-m-d 00:00:00", strtotime($request->input('date_to')));
        //     }
        // }
        
        // $date_to_ts = strtotime($date_to);
                
        // Log::debug("Chart students data: From:".$date_from." To:".$date_to);
        // //signup
        // $data = [];
        // $i = 0;
        // for($i= strtotime($date_from);$i<=$date_to_ts;$i+=$day_sec) {
        //     $dt = date("Y-m-d", $i);
        //     $cnt = DB::table('users')->where('created_at', 'like', $dt.'%')
        //     ->when(!empty($my_branch), function ($query) use ($my_branch) {
        //             return $query->whereIn('branch', $my_branch);
        //         })
        //     ->count();    
        //     $js_date = substr($dt, 0, 4)."-";
        //     $js_mth = abs(substr($dt, 5, 2)) - 1;
        //     $js_date .= ($js_mth < 10) ? "0".$js_mth : $js_mth;
        //     $js_date .= "-".substr($dt, 8, 4);    
        //     $data[] = array('new Date('.$js_date.')' => $cnt);
        //     if($i++ == 90) {
        //         break;//max 90 days data
        //     }
        // }
        // $data_js = json_encode($data, JSON_NUMERIC_CHECK);
        // $data_js = str_replace($find_key, $replace_val, $data_js);

        // $date_from = substr($date_from, 0, 10);
        // $date_to = substr($date_to, 0, 10);
        // return view('backend.analytics.students_graph', compact('current_branch','branch','reports', 'show_report', 'date_from', 'date_to', 'data_js'));
    }

    public function leadsGraph(Request $request) {
        // $reports = $this->reports;
        // $show_report = 'graph_leads';
        // $branch = get_staff_branch(\Auth::user()->id);
        // $current_branch = $request->input('branch');
        // if (empty($request->input('branch'))) {
        //     $my_branch = get_staff_branch_ids(\Auth::user()->id);
        // }
        // else if (!empty($request->input('branch')) && $request->input('branch') == 'all') {
        //     $my_branch = get_staff_branch_ids(\Auth::user()->id);
        // }
        // else{
        //     $my_branch = array($request->input('branch'));
        // }
        // //@todo: If date difference is more than 90 days, we can group by month-wise

        // $day_sec = 24 * 60 * 60;
        // $find_key = ["{","}",":","\"","-"];
        // $replace_val = ["[","]",",","",","];

        // $date_from = date("Y-m-d 00:00:00", strtotime("-30 days"));
        // $date_to = date("Y-m-d H:i:s");
        // if($request->input('date_from')) {
        //     if(strtotime($request->input('date_from'))) {
        //         $date_from = date("Y-m-d 00:00:00", strtotime($request->input('date_from')));
        //     }
        // }
        // if($request->input('date_to')) {
        //     if(strtotime($request->input('date_to'))) {
        //         $date_to = date("Y-m-d 00:00:00", strtotime($request->input('date_to')));
        //     }
        // }
        
        // $date_to_ts = strtotime($date_to);
                
        // Log::debug("Chart leads data: From:".$date_from." To:".$date_to);
        // //signup
        // $data = [];
        // $i = 0;
        // for($i= strtotime($date_from);$i<=$date_to_ts;$i+=$day_sec) {
        //     $dt = date("Y-m-d", $i);
        //     // $cnt = DB::table('leads')->where('created_at', 'like', $dt.'%')
        //     $cnt = DB::table('users')->where('created_at', 'like', $dt.'%')
        //     ->when(!empty($my_branch), function ($query) use ($my_branch) {
        //             return $query->whereIn('branch', $my_branch);
        //         })
        //     ->count();    
        //     $js_date = substr($dt, 0, 4)."-";
        //     $js_mth = abs(substr($dt, 5, 2)) - 1;
        //     $js_date .= ($js_mth < 10) ? "0".$js_mth : $js_mth;
        //     $js_date .= "-".substr($dt, 8, 4);    
        //     $data[] = array('new Date('.$js_date.')' => $cnt);
        //     if($i++ == 90) {
        //         break;//max 90 days data
        //     }
        // }
        // $data_js = json_encode($data, JSON_NUMERIC_CHECK);
        // $data_js = str_replace($find_key, $replace_val, $data_js);

        // $date_from = substr($date_from, 0, 10);
        // $date_to = substr($date_to, 0, 10);
        // return view('backend.analytics.leads_graph', compact('current_branch','branch','reports', 'show_report', 'date_from', 'date_to', 'data_js'));

    }

 //Sales Report
public function salesReport(Request $request)
{
    //DB::enableQueryLog();

    $reports = $this->reports;
    $show_report = 'report_sales';
    
    $branch = get_staff_branch(\Auth::user()->id);
    $current_branch = $request->input('branch');        
    
    $daterange = $request->input('daterange');
    $subtype = 'charts';
    
    $date_from = date("Y-m-d 00:00:00", strtotime("-30 days"));
    $date_to = date("Y-m-d H:i:s");   
    if(!empty($request->input('daterange'))) {
        $dts_str = $request->input('daterange');
        $dts = explode(" to ", $dts_str);
        $date_from = date("Y-m-d 00:00:00", strtotime($dts[0]));
        $date_to = date("Y-m-d 23:59:59", strtotime($dts[1]));
    }
    else{
        $date_from = date("Y-m-d", strtotime(now()->subDays(29)));
        $date_to = date("Y-m-d", strtotime(now()));
    }

    $my_branch = "";
    if ($request->input('branch')) {
        $my_branch = array($request->input('branch'));
    } else {
        $my_branch = get_staff_branch_ids(\Auth::user()->id);            
    }

if($subtype == 'charts') {

            $day_sec = 24 * 60 * 60;

            $date_to_ts = strtotime($date_to);
                        
            Log::debug("Chart sales data: From:".$date_from." To:".$date_to);
            $data = [];
            $i = 0;
            // for($i= strtotime($date_from);$i<=$date_to_ts;$i+=$day_sec) {
            //     $dt = date("Y-m-d", $i);
            //     // echo $dt."<br>";
            //     $amt1 = DB::table('invoice')->where('created_at', 'like', $dt.'%')->sum('grand_total');
            //     $amt2 = DB::table('invoice_payment')->where('created_at', 'like', $dt.'%')->sum('amount');
            //     $js_date = substr($dt, 0, 4)."-";
            //     $js_mth = abs(substr($dt, 5, 2));
            //     // echo $js_mth."<br>";
            //     $js_date .= ($js_mth < 10) ? "0".$js_mth : $js_mth;
            //     $js_date .= "-".substr($dt, 8, 4);   
            //     // echo $js_date."<br>"; 
            //     $data[] = array('new Date('.$js_date.')', $amt1, $amt2);
            //     if($i++ == 60) {
            //         break;//max 60 days data
            //     }
            // }

            $sales_labels = [];
            $sales_data = [];
            $invoice_labels = [];
            $invoice_data = [];
            for($i= strtotime($date_from);$i<=$date_to_ts;$i+=$day_sec) {
                $dt = date("Y-m-d", $i);
                $js_mth = abs(substr($dt, 5, 2));
                $amt1 = DB::table('invoice')
                ->join('users','users.id', '=', 'invoice.student_id')
                ->where('invoice.created_at', 'like', $dt.'%')
                
                ->when(!empty($my_branch), function ($query) use ($my_branch) {
                    return $query->whereIn('branch', $my_branch);
                }) 
                ->sum('grand_total');
                $sales_labels[]= "'".date("d/m", strtotime($dt))."'";
                $sales_data[]= $amt1;
                $amt2 = DB::table('invoice_payment')
                ->join('invoice','invoice.id', '=', 'invoice_payment.invoice_id')
                ->join('users','users.id', '=', 'invoice.student_id')
                ->when(!empty($my_branch), function ($query) use ($my_branch) {
                    return $query->whereIn('branch', $my_branch);
                }) 
                ->where('invoice_payment.created_at', 'like', $dt.'%')->sum('amount');
                // $invoice_labels[] = "'".date("d/m", strtotime($dt))."'";
                $invoice_data[] = $amt2;
            }

            return view('backend.analytics.sales_charts', 
            compact('current_branch','reports', 'show_report','subtype','branch','daterange',
            'sales_labels', 'sales_data',
            'invoice_data',
        ));            
        
    }
}

    public function salesGraph(Request $request) {
        $reports = $this->reports;
        $show_report = 'graph_sales';
        $branch = get_staff_branch(\Auth::user()->id);
        $current_branch = $request->input('branch');
        //@todo: If date difference is more than 60 days, we can group by month-wise

        $day_sec = 24 * 60 * 60;
        $find_key = ["{","}",":","\"","-"];
        $replace_val = ["[","]",",","",","];

        $date_from = date("Y-m-d 00:00:00", strtotime("-30 days"));
        $date_to = date("Y-m-d H:i:s");
        if($request->input('date_from')) {
            if(strtotime($request->input('date_from'))) {
                $date_from = date("Y-m-d 00:00:00", strtotime($request->input('date_from')));
            }
        }
        if($request->input('date_to')) {
            if(strtotime($request->input('date_to'))) {
                $date_to = date("Y-m-d 00:00:00", strtotime($request->input('date_to')));
            }
        }
        
        $date_to_ts = strtotime($date_to);
                
        Log::debug("Chart sales data: From:".$date_from." To:".$date_to);
        //signup
        $data = [];
        $i = 0;
        for($i= strtotime($date_from);$i<=$date_to_ts;$i+=$day_sec) {
            $dt = date("Y-m-d", $i);
            $amt1 = DB::table('invoice')->where('created_at', 'like', $dt.'%')->sum('grand_total');
            $amt2 = DB::table('invoice_payment')->where('created_at', 'like', $dt.'%')->sum('amount');
            $js_date = substr($dt, 0, 4)."-";
            $js_mth = abs(substr($dt, 5, 2)) - 1;
            $js_date .= ($js_mth < 10) ? "0".$js_mth : $js_mth;
            $js_date .= "-".substr($dt, 8, 4);    
            $data[] = array('new Date('.$js_date.')', $amt1, $amt2);
            if($i++ == 60) {
                break;//max 60 days data
            }
        }
        $data_js = json_encode($data, JSON_NUMERIC_CHECK);
        $data_js = str_replace($find_key, $replace_val, $data_js);

        $date_from = substr($date_from, 0, 10);
        $date_to = substr($date_to, 0, 10);
        return view('backend.analytics.sales_graph', compact('current_branch','branch','reports', 'show_report', 'date_from', 'date_to', 'data_js'));

    }    

}
