<?php

namespace App\Http\Controllers\Backend\Auth\User;

use Illuminate\Support\Facades\DB;
use App\Models\Auth\User;
use App\Models\Auth\Tags;
use App\Http\Controllers\Controller;
use App\Events\Backend\Auth\User\UserDeleted;
use App\Events\Backend\Auth\UserDeletedEvent;
use App\Repositories\Backend\Auth\RoleRepository;
use App\Repositories\Backend\Auth\UserRepository;
use App\Repositories\Backend\Auth\PermissionRepository;
use App\Http\Requests\Backend\Auth\User\StoreUserRequest;
use App\Http\Requests\Backend\Auth\User\StoreStudentRequest;
use App\Http\Requests\Backend\Auth\User\ManageUserRequest;
use App\Http\Requests\Backend\Auth\User\UpdateUserRequest;
use App\Notifications\Backend\UserAlert;
use Illuminate\Support\Str;
use App\Models\UserActivity;
use App\Models\MailTemplates;
use App\Models\Auth\UserNotification;
use App\Models\Auth\CountryStudy;
use App\Models\Auth\Branch;
use App\Models\Auth\Country;
use Illuminate\Http\Request;
use App\Mail\MailSend;
use Illuminate\Support\Facades\Mail;
use App\Repositories\Backend\Auth\StaffRepository;
use App\Models\Auth\UserDetails;
use App\Models\Auth\Services;
use App\Models\Auth\ServiceApplications;
use App\Models\Auth\UniversityApplication;
use Log;
use Illuminate\Validation\Rule;


/**
 * Class UserController.
 */
class UserController extends Controller
{

    /**
     * @var UserRepository
     */
    protected $userRepository;

    /**
     * UserController constructor.
     *
     * @param UserRepository $userRepository
     */
    public function __construct(UserRepository $userRepository)
    {
        $this->userRepository = $userRepository;
        $this->middleware(function ($request, $next) {
            $user = \Auth::user();
            if (!(\Auth::user()->isAdmin() || \Auth::user()->isAgent() || \Auth::user()->can('view students'))) {
                return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
            }
            return $next($request);
        });
    }


    private function hasAccessPermission(User $user)
    {
        if (\Auth::user()->isAgent() && $user->agent_id != \Auth::user()->id) {
            return false;
        }
        if (\Auth::user()->can('view assigned data only') && $user->assigned_to != \Auth::user()->id) {
            return false;
        }
        return true;
    }

    /**
     * @param ManageUserRequest $request
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function index(ManageUserRequest $request)
    {
        
        $student_type_list = get_student_types();
        // if(config('app.product_code') == 'VISA') {
        //     $overseas_status_list = get_visa_user_status_list();
        // } else {
            $overseas_status_list = get_overseas_status_list();
        // }
        $coaching_status_list = get_coaching_status_list();

        $search_status = $search = $my_data = $search_type = $followup_daterange = '';
        if ($request->has('query')) {

            $search_type = $request->input('query') != '' ? $request->input('query') : '';
            if ($search_type != '') {
                $search_mystatus_arr = explode('mystatus::', $search_type);
                $search_mynextfollowup_arr = explode('nextfollowup::', $search_type);
                if (isset($search_mystatus_arr[1]) && $search_mystatus_arr[1] != '') {
                    $search_status = $search_mystatus_arr[1];
                    $my_data = 1;
                }
                elseif(isset($search_mynextfollowup_arr[1]) && $search_mynextfollowup_arr[1] == 'today') {
                    $followup_daterange = date("d-M-Y")." to ".date("d-M-Y");
                } 
                elseif(isset($search_mynextfollowup_arr[1]) && $search_mynextfollowup_arr[1] == 'future') {
                    $followup_daterange = date("d-M-Y", strtotime("+1 day"))." to ".date("d-M-Y", strtotime("+1 year"));
                } 
                elseif(isset($search_mynextfollowup_arr[1]) && $search_mynextfollowup_arr[1] == 'past') {
                    $followup_daterange = "01-Jan-2020 to ".date("d-M-Y", strtotime("-1 day"));
                }
                elseif(isset($search_mynextfollowup_arr[1]) && $search_mynextfollowup_arr[1] == 'todayPlusPast') {
                    $followup_daterange = "01-Jan-2020 to ".date("d-M-Y");
                }
            }
        }
        //dd($followup_daterange);
        $filters = array("student_type" => "", "overseas_status" => $search_type,"user_countries" => "", "tags_filter" => "", "coaching_status" => "", "search" => "", "assigned_to" => "","followup_daterange" => $followup_daterange,"my_data" => $my_data);
        $userRepo = new UserRepository();
        $staff = $userRepo->getNonStudentsTrainers();
	    $user_countries = User::whereNotNull('country')->where('country',"!=",'')->where('is_lead', 0)->distinct()->get(['country']);
        $branch = get_staff_branch(\Auth::user()->id);
        $current_branch = $request->input('branch');

        $countriesFilter = Country::where('overseas_status', 1)->orderBy('title_en', 'ASC')->get();
        $courseFilter = User::select("overseas_course_intersted")
            ->whereNotNull("overseas_course_intersted")
            ->orderBy('overseas_course_intersted', 'ASC')
            ->distinct()
            ->get();
        return view('backend.auth.user.index', compact(
            'user_countries',
            'student_type_list', 
            'overseas_status_list', 
            'coaching_status_list', 
            'filters', 
            'staff','branch',
            'current_branch',
            'countriesFilter',
            'courseFilter',
        ));
    }


    public function ajaxuserdata(ManageUserRequest $request)
    {
        if (!$request->ajax()) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this page.');
        }
        echo $this->userRepository->getDatatableUsers($request);
    }

    public function assignStudent($studentid){
        if (!(\Auth::user()->isAdmin() || \Auth::user()->can('student-assignment'))) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        $studentdata = User::find($studentid);
        if (\Auth::user()->can('view assigned data only')) {
            $staff = User::where('id', \Auth::user()->id)->get();
        } else {
            $userRepo = new StaffRepository();
            $staff = $userRepo->getStaffUsers(get_staff_branch_ids(\Auth::user()->id));
        }

        return view('backend.auth.user.student_assign_to_staff', ['studentdata' => $studentdata, 'staff' => $staff]);
    }

    public function bulkAssignStudent(Request $request)
    {
        if (!(\Auth::user()->isAdmin() || \Auth::user()->can('student-assignment'))) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        if (\Auth::user()->can('view assigned data only')) {
            $staff = User::where('id', \Auth::user()->id)->get();
        } else {
            $userRepo = new StaffRepository();
            $staff = $userRepo->getStaffUsers(get_staff_branch_ids(\Auth::user()->id));
        }
        $studata_arr = array();
        $id_arr = explode(',', $request->input('id_arr'));
        if (is_array($id_arr)) {
            foreach ($id_arr as $stuid) {
                $studata_arr[] = User::find($stuid);
            }
        }
        return view('backend.auth.user.bulk-student-assign', ['studata_arr' => $studata_arr, 'staff' => $staff]);
    }

    public function bulkDeleteStudent(Request $request)
    {
        if (!(\Auth::user()->isAdmin() || \Auth::user()->can('delete-lead-student-application'))) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        // $userRepo = new UserRepository();
        // $staff = $userRepo->getNonStudentsTrainers();
        // $studata_arr = array();
        $id_arr = explode(',', $request->input('id_arr'));
        // print_r($id_arr);
        // exit();
        if (is_array($id_arr)) {
            foreach ($id_arr as $stuid) {
                $user = User::find($stuid);
                // Send Push App Notification
                if(\Auth::user()->isAgent()){
                    $message = _label('backend_student')." Deleted by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
                    $email_message = "Hi, <br>"._label('backend_student')." Deleted by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
                    $subject = _label('backend_student').": " . $user->full_name . " (".$user->id.") Deleted by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
                } else{
                    $message = _label('backend_student')." Deleted by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
                    $email_message = "Hi, <br>"._label('backend_student')." Deleted by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
                    $subject = _label('backend_student').": " . $user->full_name . " (".$user->id.") Deleted by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
                }

                //Sending email Notification
                $email_message .= "<br/><strong>"._label('backend_student')." Details:</strong><br/>" . $user->first_name ." ".$user->last_name. "<br/>Mob: " . $user->mobile . "<br/>Email: " . $user->email;
                send_mail_notification($user->id,"","LEAD_STUDENT_APPLICATION_COURSEPREFERENCE_OTHERS_DELETE",$subject,$email_message);

                $message .= "<br>"._label('backend_student').": " . $user->full_name." (". $user->id. ")";
                $subject = "";
                send_in_app_notification(
                    $user->id, 
                    $user->id,
                    "LEAD_STUDENT_APPLICATION_COURSEPREFERENCE_OTHERS_DELETE", 
                    $subject, 
                    $message,
                    "student"
                );
                
                $this->userRepository->deleteById($stuid);
                event(new UserDeleted($user));

            }
        }
        return redirect()->route('admin.auth.user.index')->withFlashSuccess('Students Deleted Successfully');
    }


    public function assignStudentUpdate(Request $request)
    {
        if (!(\Auth::user()->isAdmin() || \Auth::user()->can('student-assignment'))) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }

        $staffid = $request->staff;
        $studentid = $request->stu_id;
        $user = User::find($staffid);
        $student_assigned_success = false;
        $email_message_part = '';

        if (is_array($studentid)) {
            \Log::debug("Student Assigned To Staff array");
           
            foreach ($studentid as $each_studentid) {
                if (DB::table('users')->where("id", $each_studentid)->update(['assigned_to' => $staffid])) {
                    $student = User::find($each_studentid);
                    $student_assigned_success = true;

                    //Adding activity
                    $followup = new UserActivity;
                    $followup->user_id = $each_studentid;
                    $followup->type = 'student_assignment';
                    $followup->student_assigned_to = $staffid;
                    $followup->staff = \Auth::user()->id;
                    $followup->detail = "STUDENT ASSIGNED TO " . $user->name;
                    $followup->save();
                    
                    $email_message_part .= "<br> User Name: $student->first_name $student->last_name<br /><br />Mobile: $student->mobile<br /><br />Email: $student->email <br/>";

                    if($request->staff){
                        $staff = $request->staff;
                        $notification_message_part = "";
                        $notification_message_part .= "<br> Student Name: $student->first_name $student->last_name<br />Mobile: $student->mobile<br />Email: $student->email <br/>";
                        $email_message = "Hi $user->name, Student assigned to you." . $notification_message_part;
        
                        // $noti = new UserNotification;
                        // $noti->user_id = $staff;
                        // $noti->application_id = $each_studentid;
                        // $noti->application_type = 'student';
                        // $noti->message = $email_message;
                        // $noti->from_user = \Auth::user()->id;
                        // $noti->save();

                        // New way to send email
                        //Sending email Notification
                        $subject = _label('backend_student').": " . $student->full_name . " (".$student->id.") Assigned by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
                        $email_message_part .= "<br>"._label('backend_student')." Name: $student->first_name $student->last_name<br /><br />Mobile:$student->mobile<br /><br />Email: $student->email <br/><br/><a href=\"" . url("admin/leads/" . $student->id) . "\">View "._label('backend_student')." Details</a>";
                        $email_message = "Hi $user->full_name, <br>"._label('backend_student')." assigned to you.<br />" . $email_message_part;
                        send_mail_notification($student->id,"","LEAD_STUDENT_ASSIGMENT_STAFF",$subject,$email_message);

                        // Send Push App Notification
                        $message = _label('backend_student')." <strong>".$student->full_name."</strong> (". $student->id. ") assigned to <strong>".$user->full_name."</strong> by " . get_user_full_name_by_id(\Auth::user()->id);
                        $subject = "";
                        send_in_app_notification(
                            $student->id, 
                            $student->id,
                            "LEAD_STUDENT_ASSIGMENT_STAFF", 
                            $subject, 
                            $message,
                            "student"
                        );
                    }
                } else {
                    return redirect()->route('admin.auth.user.index')->withFlashDanger('Something Went Wrong!');
                }
            }

            if ($student_assigned_success) {
                if($user){
                    // $notify_sms = get_sms_template('student_assignment');
                    // if ($notify_sms) {
                    //     $notify_sms->message = str_replace("{#var#}", "%s", $notify_sms->message);
                    //     $notify_sms->message = sprintf($notify_sms->message, $user->name);
                    //     $numbers = [$user->mobile];
                    //     send_sms($numbers, $notify_sms);
                    // }
                    return redirect()->route('admin.auth.user.index')->withFlashSuccess('User Assigned to Staff!');
                } else{
                    return redirect()->route('admin.auth.user.index')->withFlashSuccess('Removed Staff Assignment!');
                }
            } else {
                return redirect()->route('admin.auth.user.index')->withFlashDanger('Something Went Wrong!');
            }
        } else {
            \Log::debug("Student Assigned To Staff Single");
            if (DB::table('users')->where("id", $studentid)->update(['assigned_to' => $staffid])) {
                $student = User::find($studentid);

                //Adding activity
                $followup = new UserActivity;
                $followup->user_id = $student->id;
                $followup->type = 'student_assignment';
                $followup->student_assigned_to = $staffid;
                $followup->staff = \Auth::user()->id;
                $followup->detail = "STUDENT ASSIGNED TO " . $user->name;
                $followup->save();

                if($user){
                    // $subject = "New Student Assigned";
                    // $email_message = "Hi $user->name, New student assigned to you.<br /><br />User Name: $student->first_name $student->last_name<br /><br />Mobile: $student->mobile<br /><br />Email: $student->email <br/><br/>";
                    // try{
                    //     Mail::to($user->email)->send(new MailSend($subject, $email_message));
                    // }catch (\Exception $e) {
                    //     $message = $e->getMessage();
                    //     \Log::debug("Mail Error: (New Student Assigned) ".$message);   
                    // }

                    // $notify_sms = get_sms_template('student_assignment');
                    // if ($notify_sms) {
                    //     $notify_sms->message = str_replace("{#var#}", "%s", $notify_sms->message);
                    //     $notify_sms->message = sprintf($notify_sms->message, $user->name);
                    //     $numbers = [$user->mobile];
                    //     send_sms($numbers, $notify_sms);
                    // }

                    if($request->staff){
                        $staff = $request->staff;

                        // New way to send email
                        //Sending email Notification
                        $subject = _label('backend_student').": " . $student->full_name . " (".$student->id.") Assigned by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
                        $email_message_part .= "<br>"._label('backend_student')." Name: $student->first_name $student->last_name<br /><br />Mobile:$student->mobile<br /><br />Email: $student->email <br/><br/><a href=\"" . url("admin/leads/" . $student->id) . "\">View "._label('backend_student')." Details</a>";
                        $email_message = "Hi $user->full_name, <br>"._label('backend_student')." assigned to you.<br />" . $email_message_part;
                        send_mail_notification($student->id,"","LEAD_STUDENT_ASSIGMENT_STAFF",$subject,$email_message);

                        // Send Push App Notification
                        $message = _label('backend_student')." <strong>".$student->full_name."</strong> (". $student->id. ") assigned to <strong>".$user->full_name."</strong> by " . get_user_full_name_by_id(\Auth::user()->id);
                        $subject = "";
                        send_in_app_notification(
                            $student->id, 
                            $student->id,
                            "LEAD_STUDENT_ASSIGMENT_STAFF", 
                            $subject, 
                            $message,
                            "student"
                        );
                    }

                    // return redirect()->route('admin.auth.user.index')->withFlashSuccess('User Assigned!');
                    return response()->json([
                        'success' => 1, 
                        'message' => 'User Assigned to Staff!'
                    ]);
                }else{
                    // return redirect()->route('admin.auth.user.index')->withFlashSuccess('Removed User Assigned!');
                    return response()->json([
                        'success' => 1, 
                        'message' => 'Removed Staff Assignment!'
                    ]);
                }
            } else {
                // return redirect()->route('admin.auth.user.index')->withFlashDanger('Something Went Wrong!');
                return response()->json([
                    'success' => 0, 
                    'message' => 'Something went Wrong'
                ]);
            }
        }
    }

    // public function search(ManageUserRequest $request)
    // {
    //     return view('backend.auth.user.index')
    //         ->withFilter($request->input('q'))
    //         ->withUsers($this->userRepository->getFilteredPaginated($request->input('q'), 10, 'id', 'desc'));
    // }

    public function staff(ManageUserRequest $request)
    {
        if (!\Auth::user()->isAdmin() && !\Auth::user()->can('staff')) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this page.');
        }
        return view('backend.auth.user.staff')
            ->withUsers($this->userRepository->getStaffPaginated(10, 'id', 'desc'));
    }

    /**
     * @param ManageUserRequest    $request
     * @param RoleRepository       $roleRepository
     * @param PermissionRepository $permissionRepository
     *
     * @return mixed
     */
    // public function create(ManageUserRequest $request, RoleRepository $roleRepository, PermissionRepository $permissionRepository)
    // {
    //     return view('backend.auth.user.create', ['staff_mgmt' => 0])
    //         ->withRoles($roleRepository->with('permissions')->get(['id', 'name']))
    //         ->withPermissions($permissionRepository->get(['id', 'name']));
    // }

    public function studentCreate(){

        $user = \Auth::user();
        if ($user->isAgent() && !$user->active){
            return redirect()->back()->withFlashDanger('Your Account is Pending Activation. Cannot Add Student.');
        }

        $countries = Country::where('overseas_status', 1)->orderBy('title_en', 'ASC')->get();
        $country_list = Country::pluck('title_en', 'id');        
        $country_study = Country::where('overseas_status', 1)->orderBy('title_en')->get();
        // $dialcodes = Country::groupBy('tel')->orderBy('tel','asc')->get(['tel as code', 'title_en as name']);
        $dialcodes = Country::orderBy('country_code', 'asc')->get(['tel as code', 'code as country_code', 'title_en as name']);
        $branch = get_staff_branch(\Auth::user()->id);

        if (\Auth::user()->can('view assigned data only')) {
            $staff = User::where('id', \Auth::user()->id)->get();
        } else {
            $userRepo = new StaffRepository();
            $staff = $userRepo->getStaffUsers(get_staff_branch_ids(\Auth::user()->id));
        }
        $custom_attributes = DB::table('custom_attributes')->where('for_student', 1)->orderBy('display_order','asc')->get(["custom_attributes.*"]);

        $welcome_email = 0;
        $mail_tpl = MailTemplates::find("STUDENT_CREATE_WELCOME");
        if ($mail_tpl && $mail_tpl->status) {
            $welcome_email = 1;
        }

        $agents = User::whereHas('roles', function ($q) {
            $q->whereIn('name', ['agent']);
        })
        ->get();
        return view('backend.auth.user.createstudent', ['welcome_email' => $welcome_email, 'custom_attributes' => $custom_attributes,'staff' => $staff, 'countries' => $countries, 'branch' => $branch, 'country_list' => $country_list, 'country_study' => $country_study, 'dialcodes' => $dialcodes,'agents' => $agents]);
    }

    /**
     * @param StoreUserRequest $request
     *
     * @return mixed
     * @throws \Throwable
     */
    // public function store(StoreUserRequest $request)
    // {
    //     $this->userRepository->create($request->only(
    //         'first_name',
    //         'last_name',
    //         'email',
    //         'birthday',
    //         'password',
    //         'active',
    //         'confirmed',
    //         'confirmation_email',
    //         'roles',
    //         'permissions'
    //     ));

    //     return redirect()->route('admin.auth.user.index')->withFlashSuccess(__('alerts.backend.users.created'));
    // }

    public function studentstore(StoreStudentRequest $request)
    {
        if (get_site_config('student_create_admin_mandatory') == 'email') {
            $request->validate([
                'email' => 'required|email:filter,filter_unicode|unique:users',
                'mobile' => ['nullable', 'numeric', 'digits_between:' . config('app.USER_MOBILE_MIN_LENGTH') . ',' . config('app.USER_MOBILE_MAX_LENGTH')],
            ]);
        } elseif (get_site_config('student_create_admin_mandatory') == 'email_and_mobile') {
            $request->validate([
                'email' => 'required|email:filter,filter_unicode|unique:users',
                'mobile' => ['required', 'numeric', 'digits_between:' . config('app.USER_MOBILE_MIN_LENGTH') . ',' . config('app.USER_MOBILE_MAX_LENGTH'), Rule::unique('users')],
            ]);
        }
        // elseif(get_site_config('student_create_admin_mandatory') == 'email_or_mobile'){
        //     if($request->get('email') == "" && $request->get('mobile') == ""){
        //         return redirect()->back()->withFlashDanger('Email or Mobile is Required');
        //     } else{
        //         $request->validate([
        //             'email' => 'nullable|email|unique:users',
        //             'mobile' => ['nullable', 'numeric', 'digits_between:'.config('app.USER_MOBILE_MIN_LENGTH').','.config('app.USER_MOBILE_MAX_LENGTH'), Rule::unique('users')],
        //         ]);
        //     }
        // } elseif(get_site_config('student_create_admin_mandatory') == 'mobile'){
        //     $request->validate([
        //         'email' => 'nullable|email|unique:users',
        //         'mobile' => ['required', 'numeric', 'digits_between:'.config('app.USER_MOBILE_MIN_LENGTH').','.config('app.USER_MOBILE_MAX_LENGTH'), Rule::unique('users')],
        //     ]);            
        // }

        //Checking custom attributes
        $custom_attributes = DB::table('custom_attributes')->where('for_student', 1)->orderBy('display_order', 'asc')->get(["custom_attributes.*"]);
        if (count($custom_attributes)) {
            foreach ($custom_attributes as $attrs) {
                if ($attrs->laravel_rules) {
                    $request->validate([
                        "$attrs->attr_name" => "$attrs->laravel_rules",
                    ]);
                }
            }
        }
        //Checking custom attributes

        $user = $this->userRepository->create($request->all());

        $notify_emails = get_site_config('event_notification_email');
        $email_message = "New Student added by Staff<br /><br>Student Name: " . $user->full_name . "<br />Email: " . $user->email . "<br />Mobile: " . $user->dialcode . ' ' . $user->mobile;
        if ($notify_emails) {
            send_event_notification_email(explode(",", $notify_emails), "New Student Created", $email_message);
        }
        
        // Send Email, App Notification
        if (!\Auth::user()->isAgent()) {
            
            $subject = _label('backend_student') . ": " . $user->full_name . " (" . $user->id . ") added by staff: " . auth()->user()->full_name;
            $email_message = "Hi, <br><br>" . _label('backend_student') . " added by staff: " . auth()->user()->full_name;
            $email_message .= "<br><br>" . _label('backend_student') . ": " . $user->full_name . " (" . $user->id . ") <br /><br />Mobile:$user->mobile<br /><br />Email: $user->email <br/><br/><a href=\"" . url("admin/auth/user/" . $user->id) . "\">View " . _label('backend_student') . " Details</a>";
            send_mail_notification($user->id, "", "LEAD_STUDENT_CREATED", $subject, $email_message);

            $message = _label('backend_student') . " <strong>" . $user->full_name . "</strong> (" . $user->id . ")  created by staff";
            send_in_app_notification(
                $user->id,
                $user->id,
                "LEAD_STUDENT_CREATED",
                $subject,
                $message,
                "student"
            );

            if ($user->assigned_to) {

                //Sending email Notification
                $subject = "New "._label('backend_student') . ": " . $user->full_name . " (" . $user->id . ") assigned to: " . get_user_full_name_by_id($user->assigned_to);
                $email_message = "Hi " . get_user_full_name_by_id($user->assigned_to) . ", <br><br>New " . _label('backend_student') . " assigned to you by staff: " . auth()->user()->fullname . "<br /><br> " . _label('backend_student') . " Name: $user->first_name $user->last_name<br /><br />Mobile:$user->mobile<br /><br />Email: $user->email <br/><br/><a href=\"" . url("admin/auth/user" . $user->id) . "\">View " . _label('backend_student') . " Details</a>";
                send_mail_notification($user->id, "", "LEAD_STUDENT_ASSIGMENT_STAFF", $subject, $email_message);

                // Send Push App Notification
                $message = "New "._label('backend_student') . " <strong>" . $user->full_name . "</strong> (" . $user->id . ") assigned by staff <strong>" . auth()->user()->fullname . "</strong> to <strong>" . get_user_full_name_by_id($user->assigned_to) . "</strong>";
                $subject = "";
                send_in_app_notification(
                    $user->id,
                    $user->id,
                    "LEAD_STUDENT_ASSIGMENT_STAFF",
                    $subject,
                    $message,
                    "student"
                );
            }
        } else {
            //Admin Notification if added by agent
            $msg_details = "New " . _label('backend_student') . " added by Agent.<br /><br />Agent Name: " . get_user_full_name_by_id($user->agent_id) . '<br>Student Name: ' . $user->full_name . "<br />email: " . $user->email . "<br />Mobile: " . $user->dialcode . ' ' . $user->mobile . "<br />Gender: " . $user->gender;

            $notify_emails = get_site_config('event_notification_email');
            if ($notify_emails) {
                send_event_notification_email(explode(",", $notify_emails), "New " . _label('backend_student') . " added by Agent", $msg_details);
            }

            // Send Email, Push App Notification
            $subject = _label('backend_student') . ": " . $user->full_name . " (" . $user->id . ") added by agent: " . auth()->user()->full_name;
            $email_message = "Hi, <br><br>" . _label('backend_student') . " added by agent: " . auth()->user()->full_name;
            $email_message .= "<br><br>" . _label('backend_student') . ": " . $user->full_name . " (" . $user->id . ") <br /><br />Mobile:$user->mobile<br /><br />Email: $user->email <br/><br/><a href=\"" . url("admin/auth/user/" . $user->id) . "\">View " . _label('backend_student') . " Details</a>";
            send_mail_notification($user->id, "", "LEAD_STUDENT_CREATED", $subject, $email_message);

            $message = _label('backend_student') . " <strong>" . $user->fullname . "</strong> (" . $user->id . ") created by agent <strong>" . auth()->user()->full_name . "</strong>";
            send_in_app_notification(
                $user->id,
                $user->id,
                "LEAD_STUDENT_CREATED",
                $subject,
                $message,
                "student"
            );
        }

        return redirect()->route('admin.auth.user.index')->withFlashSuccess(__('alerts.backend.users.created'));
    }

    /**
     * @param ManageUserRequest $request
     * @param User              $user
     *
     * @return mixed
     */
    public function show(ManageUserRequest $request, User $user) {
        if (!$this->hasAccessPermission($user)) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }        
        $data = array();
        $cv = "";
        $stu_photo = "";
        $passports = array();
        $sops = array();
        $lors = array();
        $internships = array();
        $offerletters = array();
        $experience = array();
        $otherdocs = array();

        if (!DB::table('user_documents')->where('userid', $user->id)->exists()) {
            DB::table('user_documents')->insert(['userid' => $user->id]);
        }
        if (!DB::table('user_details')->where('userid', $user->id)->exists()) {
            DB::table('user_details')->insert(['userid' => $user->id]);
        }

        $row = DB::table('user_documents')->where('userid', $user->id)->first();
        $data['class10'] = $row->class10;
        $data['class10_marks_sheet'] = $row->class10_marks_sheet;
        $data['class12'] = $row->class12;
        $data['class12_marks_sheet'] = $row->class12_marks_sheet;
        $data['diploma'] = $row->diploma;
        $data['diploma_marks_sheet'] = $row->diploma_marks_sheet;
        $data['grad_all_sem'] = $row->grad_all_sem;
        $data['grad_degree'] = $row->grad_degree;
        $data['grad_provisional'] = $row->grad_provisional;
        $data['grad_transcripts'] = $row->grad_transcripts;
        $data['pg_all_sem'] = $row->pg_all_sem;
        $data['pg_provisional'] = $row->pg_provisional;
        $data['cmm'] = $row->cmm;
        $data['pg_degree'] = $row->pg_degree;
        $data['pg_transcripts'] = $row->pg_transcripts;
        $data['lang_ielts'] = $row->lang_ielts;
        $data['lang_pte'] = $row->lang_pte;
        $data['gre'] = $row->gre;
        $data['gmat'] = $row->gmat;
        $data['toefl'] = $row->toefl;
        $data['duolingo'] = $row->duolingo;

        $rows = DB::table('user_docs_other')->where('userid', $user->id)->get();
        if ($rows) {
            foreach ($rows as $row) {
                if ($row->doc_type == 'cv') {
                    $cv = $row->file;
                }
                if ($row->doc_type == 'stu_photo') {
                    $stu_photo = $row->file;
                }
                 elseif ($row->doc_type == 'passport') {
                    $passports[] = $row;
                } elseif ($row->doc_type == 'sop') {
                    $sops[] = $row;
                } elseif ($row->doc_type == 'letter_of_recommendation') {
                    $lors[] = $row;
                } elseif ($row->doc_type == 'internship') {
                    $internships[] = $row->file;
                } elseif ($row->doc_type == 'offer') {
                    $offerletters[] = $row->file;
                } elseif ($row->doc_type == 'experience') {
                    $experience[] = $row;
                } elseif ($row->doc_type == 'payslip') {
                    $payslips[] = $row->file;
                } elseif ($row->doc_type == 'other') {
                    $otherdocs[] = $row;
                }
            }
        }

        $data['cv'] = $cv;
        $data['stu_photo'] = $stu_photo;
        $data['passports'] = $passports;
        $data['sops'] = $sops;
        $data['lors'] = $lors;
        $data['internships'] = $internships;
        $data['offerletters'] = $offerletters;
        $data['experience'] = $experience;
        $data['otherdocs'] = $otherdocs;

        $batches = DB::table('batch_students')
            ->join('batch', 'batch.id', '=', 'batch_students.batch')
            ->join('branch', 'branch.id', '=', 'batch.branch')
            ->join('courses', 'courses.id', '=', 'batch.course')
            ->where('batch_students.student', $user->id)
            ->select('batch.id as batch_id', 'batch.code as batch_code', 'branch.branch_name as branch', 'courses.title as course', 'batch.start_date')
            ->orderBy('batch.id', 'DESC')
            ->get();
        $countries = Country::orderBy('title_en')->where('overseas_status',1)->select('id', 'title_en as name')->get();
        $userDetails = DB::table('user_details')->where('userid', $user->id)->first();

        $uni_countries = DB::table('university_application')->where('userid', $user->id)->get(['country']);
        $user_countries = array();
        foreach($uni_countries as $country){
            $user_countries[] = $country->country;
        }
        // $additional_docs = DB::table('additional_docs_configs')
        //     ->when(count($user_countries), function ($query) use ($user_countries) {
        //         return $query->Where(function ($query) use ($user_countries) {
        //             foreach ($user_countries as $key) {
        //                 $query->orwhere('countries', 'LIKE', "%$key%");
        //             }
        //             $query->orWhere('countries','all');
        //         });
        //     })
        //     ->when(!count($user_countries), function ($query) use ($user_countries) {
        //         return $query->Where('countries','all');
        //     })
        //     ->where('status', 1)
        //     // ->orWhere('countries','all')
        //     ->get();

        $additional_docs_one = DB::table('additional_docs_configs')
            ->when(count($user_countries), function ($query) use ($user_countries) {
                return $query->Where(function ($query) use ($user_countries) {
                    foreach ($user_countries as $key) {
                        $query->orwhere('countries', 'LIKE', "%$key%");
                    }
                    $query->orWhere('countries','all');
                });
            })
            ->when(!count($user_countries), function ($query) use ($user_countries) {
                return $query->Where('countries','all');
            })
            ->where('status', 1)
            ->where('section_id', 1)
            ->orderBy('display_order')
            ->get();

        $additional_docs_two = DB::table('additional_docs_configs')
            ->when(count($user_countries), function ($query) use ($user_countries) {
                return $query->Where(function ($query) use ($user_countries) {
                    foreach ($user_countries as $key) {
                        $query->orwhere('countries', 'LIKE', "%$key%");
                    }
                    $query->orWhere('countries','all');
                });
            })
            ->when(!count($user_countries), function ($query) use ($user_countries) {
                return $query->Where('countries','all');
            })
            ->where('status', 1)
            ->where('section_id', 2)
            ->orderBy('display_order')
            ->get();

        $additional_docs_three = DB::table('additional_docs_configs')
            ->when(count($user_countries), function ($query) use ($user_countries) {
                return $query->Where(function ($query) use ($user_countries) {
                    foreach ($user_countries as $key) {
                        $query->orwhere('countries', 'LIKE', "%$key%");
                    }
                    $query->orWhere('countries','all');
                });
            })
            ->when(!count($user_countries), function ($query) use ($user_countries) {
                return $query->Where('countries','all');
            })
            ->where('status', 1)
            ->where('section_id', 3)
            ->orderBy('display_order')
            ->get();

        $services = ServiceApplications::where('user_id', $user->id)->get();
        $universityApplId = UniversityApplication::where('userid', $user->id)->get();
        // $allUniversities = DB::table('universities')->orderBy('university_name')->get();

        $invoice = DB::table('invoice')
            ->select('invoice.*', 'users.first_name', 'users.last_name', 'users.mobile','users.id as userid')
            ->join('users', 'users.id', '=', 'invoice.student_id')
            ->orderBy('invoice.created_at', "DESC")
            ->where('student_id', $user->id)
            ->whereNull('deleted_at')
            ->get();

        $tickets = DB::table('tickets')
            ->select('tickets.*','users.id','users.branch')
            ->join('users','users.id','=','tickets.userid')
            ->WhereIn('status', ['new', 'customer_reply', 'staff_reply', 'inprogress'])
            ->where('userid', $user->id)
            // ->orWhere('status', 'customer_reply')
            ->get();

        $notes = DB::table('user_notes')
            ->where('user_id', $user->id)
            ->orderBy('id','DESC')
            ->get();

        if (\Auth::user()->can('view assigned data only')) {
            $staff = User::where('id', \Auth::user()->id)->get();
        } else {
            $userRepo = new StaffRepository();
            $staff = $userRepo->getStaffUsers(get_staff_branch_ids(\Auth::user()->id));
        }

        return view('backend.auth.user.show',[
            'universityApplId' => $universityApplId, 
            'services' => $services, 
            // 'additional_docs'=>$additional_docs,
            'additional_docs_one'=>$additional_docs_one,
            'additional_docs_two'=>$additional_docs_two,
            'additional_docs_three'=>$additional_docs_three,
            'countries'=>$countries,
            'staff_mgmt'=>0,
            'notes' => $notes,
            'student_batches' => $batches,
            'userDetails' => $userDetails,
            'invoice' => $invoice,
            'tickets' => $tickets,
            'staff' => $staff,
            // 'allUniversities' => $allUniversities,
        ],$data)->withUser($user);
    }

    /**
     * @param ManageUserRequest    $request
     * @param RoleRepository       $roleRepository
     * @param PermissionRepository $permissionRepository
     * @param User                 $user
     *
     * @return mixed
     */
    public function edit(ManageUserRequest $request, RoleRepository $roleRepository, PermissionRepository $permissionRepository, User $user)
    {
        if (!$this->hasAccessPermission($user)) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }

        $country_list = Country::pluck('title_en', 'id');                     
        $branch_list = get_staff_branch(\Auth::user()->id);
        $interest_country = User::select('country_interested')->where('id', $user->id)->first();
        $interest_country_list = explode(",", $interest_country->country_interested);

        $countries = Country::where('overseas_status', 1)->orderBy('title_en', 'ASC')->get();
        $userdetails = UserDetails::where('userid', $user->id)->first();

        $all_country_list = CountryStudy::all();
        // $dialcodes = Country::groupBy('tel')->orderBy('tel','asc')->get(['tel as code', 'title_en as name']);
        $dialcodes = Country::orderBy('country_code', 'asc')->get(['tel as code', 'code as country_code', 'title_en as name']);
        $custom_attributes = DB::table('custom_attributes')->where('for_student', 1)->orderBy('display_order','asc')->get(["custom_attributes.*"]);      
        return view('backend.auth.user.edit', ['custom_attributes'=>$custom_attributes,'userdetails' => $userdetails,'countries' => $countries, 'country_list' => $country_list,'all_country_list' => $all_country_list, 'interest_country_list' => $interest_country_list, 'dialcodes' => $dialcodes, 'staff_mgmt' => 0, 'branch_list' => $branch_list])
            ->withUser($user)
            ->withRoles($roleRepository->get())
            ->withUserRoles($user->roles->pluck('name')->all())
            ->withPermissions($permissionRepository->get(['id', 'name']))
            ->withUserPermissions($user->permissions->pluck('name')->all());
    }

    /**
     * @param UpdateUserRequest $request
     * @param User              $user
     *
     * @return mixed
     * @throws \App\Exceptions\GeneralException
     * @throws \Throwable
     */
    public function update(UpdateUserRequest $request, User $user)
    {
        // $request->validate([
        //     'study_level' => 'required',
        // ]);

        if(get_site_config('student_create_admin_mandatory') == 'email'){
            $request->validate([
                'email' => 'required|email:filter,filter_unicode|unique:users,email,' . $user->id,
                'mobile' => ['nullable', 'numeric', 'digits_between:'.config('app.USER_MOBILE_MIN_LENGTH').','.config('app.USER_MOBILE_MAX_LENGTH')],
            ]);
        } elseif(get_site_config('student_create_admin_mandatory') == 'email_and_mobile'){
            $request->validate([
                'email' => 'required|email:filter,filter_unicode|unique:users,email,' . $user->id,
                'mobile' => ['required', 'numeric', 'digits_between:'.config('app.USER_MOBILE_MIN_LENGTH').','.config('app.USER_MOBILE_MAX_LENGTH'), 'unique:users,mobile,' . $user->id],
            ]);
        } 
        // elseif(get_site_config('lead_create_admin_mandatory') == 'email_or_mobile'){
        //     if($request->get('email') == "" && $request->get('mobile') == ""){
        //         return redirect()->back()->withFlashDanger('Email or Mobile is Required');
        //     } else{
        //         $request->validate([
        //             'email' => 'nullable|email|unique:users,email,' . $id,
        //             'mobile' => ['nullable', 'numeric', 'digits_between:'.config('app.USER_MOBILE_MIN_LENGTH').','.config('app.USER_MOBILE_MAX_LENGTH'), 'unique:users,mobile,' . $id,],
        //         ]);
        //     }
        // } 
        // elseif(get_site_config('lead_create_admin_mandatory') == 'mobile'){
        //     $request->validate([
        //         'email' => 'nullable|email|unique:users,email,' . $id,
        //         'mobile' => ['required', 'numeric', 'digits_between:'.config('app.USER_MOBILE_MIN_LENGTH').','.config('app.USER_MOBILE_MAX_LENGTH'), 'unique:users,mobile,' . $id,],
        //     ]);            
        // }

        if(env('DEMO_MODE') && in_array($user->id, config('app.demo_users')))
            return redirect()->route('admin.demo-user-restriction');
	        
        if (!$this->hasAccessPermission($user)) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }

        if ($user->id == 1) {
            // Cant update super admin details
            return redirect()->route('admin.dashboard.message')->withFlashDanger('This operation is not allowed');
        }

        //Checking custom attributes
        $custom_attributes = DB::table('custom_attributes')->where('for_student', 1)->orderBy('display_order','asc')->get(["custom_attributes.*"]);
        if(count($custom_attributes)){
            foreach($custom_attributes as $attrs){
                if($attrs->laravel_rules){
                    $request->validate([
                        "$attrs->attr_name" => "$attrs->laravel_rules",
                    ]);
                }
            }
        }
        //Checking custom attributes

        $this->userRepository->update($user, $request->all());

        // $this->userRepository->update($user, $request->only(
        //     'first_name',
        //     'middle_name',
        //     'last_name',
        //     'gender',
        //     'email',
        //     'birthday',
        //     'dialcode',
        //     'mobile',
        //     'whatsapp_number',
        //     'alternate_contact',
        //     'country',
        //     'state',
        //     'city',
        //     'country_interested',
        //     'study_level',
        //     'course_interested',
        //     'intake',
        //     'student_type',
        //     'qualification',
        //     'passout_year',
        //     'qualification_score',
        //     'currently_studying_course',
        //     'work_experience',
        //     'lead_source',
        //     'lead_source_detail',
        //     'campaign_id',
        //     'branch',

        //     'roles',
        //     'permissions',
        //     'title',
        // ));

        return redirect()->route('admin.auth.user.index')->withFlashSuccess(__('alerts.backend.users.updated'));
    }

    /**
     * @param ManageUserRequest $request
     * @param User              $user
     *
     * @return mixed
     * @throws \Exception
     */
    public function destroy(ManageUserRequest $request, User $user){
        // This function delete student temporarily
        if(env('DEMO_MODE') && in_array($user->id, config('app.demo_users')))
            return redirect()->route('admin.demo-user-restriction');
	        
        if (!$this->hasAccessPermission($user)) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }

        if (!(\Auth::user()->isAdmin() || \Auth::user()->can('delete-lead-student-application'))) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }

        if ($user->id == 1) {
            // Cant delete super admin
            return redirect()->route('admin.dashboard.message')->withFlashDanger('This operation is not allowed');
        }


        // Send Push App Notification
        try {
            if(\Auth::user()->isAgent()){
                $message = _label('backend_student')." <strong>".$user->full_name."</strong> (". $user->id. ") Deleted by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
                $email_message = "Hi, <br>"._label('backend_student')." Deleted by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
                $subject = _label('backend_student').": " . $user->full_name . " (".$user->id.") Deleted by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
            } else{
                $message = _label('backend_student')." <strong>".$user->full_name."</strong> (". $user->id. ") DELETED by staff";
                $email_message = "Hi, <br>"._label('backend_student')." Deleted by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
                $subject = _label('backend_student').": " . $user->full_name . " (".$user->id.") Deleted by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
            }

            //Sending email Notification
            $email_message .= "<br/><strong>"._label('backend_student')." Details:</strong><br/>" . $user->first_name ." ".$user->last_name. "<br/>Mob: " . $user->mobile . "<br/>Email: " . $user->email;
            send_mail_notification($user->id,"","LEAD_STUDENT_APPLICATION_COURSEPREFERENCE_OTHERS_DELETE",$subject,$email_message);

            $subject = "";
            send_in_app_notification(
                $user->id, 
                $user->id,
                "LEAD_STUDENT_APPLICATION_COURSEPREFERENCE_OTHERS_DELETE", 
                $subject, 
                $message,
                "student"
            );
        } catch (\Exception $e) {
            \Log::debug("Sent Notifications Error: " . $e->getMessage());
        }
        $this->userRepository->deleteById($user->id);
        event(new UserDeleted($user));

        return redirect()->route('admin.auth.user.index')->withFlashSuccess(__('alerts.backend.users.deleted'));
    }

    /**
     * @param ManageUserRequest $request
     * @param User              $user
     *
     * @return mixed
     * @throws \App\Exceptions\GeneralException
     * @throws \Throwable
     */
    public function coursesuggestion(ManageUserRequest $request, User $user)
    {
        $userid = $user->id;

        //\Log::debug("::san::  uid: $userid");
        //\Log::debug("::san:: ".var_export($data, true));

        if (!$request->input('title')) {
            return redirect()->route('admin.auth.user.show', $user->id)->withFlashDanger("Please enter a title");
        }
        if (!$request->file('file')) {
            return redirect()->route('admin.auth.user.show', $user->id)->withFlashDanger("Please select a file");
        }

        $path = $request->file('file')->storeAs('/courses', Str::random(40) . '.' . $request->file->getClientOriginalExtension(), 'public');
        \Log::debug("::savedpath::$path");
        DB::table('course_recommendation')
            ->insert(
                [
                    'upload_date' => date('Y-m-d H:i:s'), 
                    'userid' => $userid, 
                    'title' => $request->input('title'), 
                    'file' => $path
                ]);

        //Notify Student
        $user = (new UserRepository())->where('id', $userid)->first();
        //email
        $mail_tpl = MailTemplates::find("COURSE_SUGGESTION");
        if(!$user->agent_id) {
            if ($mail_tpl && $mail_tpl->status) {
                $email_message = $mail_tpl->message;
                //\Log::debug("MAIL BEFORE:".$email_message);                       
                $email_message = str_ireplace("#sitename#", app_name(), $email_message);
                $email_message = str_ireplace("#siteurl#", url(''), $email_message);
                $email_message = str_ireplace("#first_name#", $user->first_name, $email_message);
                $email_message = str_ireplace("#full_name#", $user->full_name, $email_message);
                //\Log::debug("MAIL TPL FINAL:".$email_message);
                try{
                    //$user->notify(new UserAlert($mail_tpl->subject, $email_message));
                    user_notify($user,$mail_tpl->subject, $email_message);
                    \Log::debug("Mail Success: (User - COURSE_SUGGESTION) ".$email_message);   
                }catch (\Exception $e) {
                    $message = $e->getMessage();
                    \Log::debug("Mail Error: (User - COURSE_SUGGESTION) ".$message);   
                }
            }
        }
        //sms
        $notify_sms = get_sms_template('COURSE_SUGGESTION');
        if ($notify_sms && $notify_sms->status) {
            $notify_sms->message = str_replace("#sitename#", app_name(), $notify_sms->message);
            $notify_sms->message = str_replace("#siteurl#", url('/'), $notify_sms->message);
            $notify_sms->message = str_replace("#first_name#", $user->first_name, $notify_sms->message);
            $notify_sms->message = str_replace("#full_name#", $user->full_name, $notify_sms->message);
            $user->sms($notify_sms->message,$notify_sms->template_id);
        }

        // Send Push App Notification
        if(\Auth::user()->isAgent()){
            $message = "Course Suggested by Agent: " . get_user_full_name_by_id(\Auth::user()->id) . 
            "<br>"._label('backend_student').": " . $user->full_name." (". $user->id. ")";
            $email_message = "Hi, <br>Course Suggested by Agent: " . get_user_full_name_by_id(\Auth::user()->id) . 
            "<br>"._label('backend_student').": " . $user->full_name." (". $user->id. ")";
            $subject = "Course Suggested by Agent: " . get_user_full_name_by_id(\Auth::user()->id) . 
            "<br>"._label('backend_student').": " . $user->full_name." (". $user->id. ")";
        } else{
            $message = "Course Suggested by Staff: " . get_user_full_name_by_id(\Auth::user()->id) . 
            "<br>"._label('backend_student').": " . $user->full_name." (". $user->id. ")";
            $email_message = "Hi, <br>Course Suggested by Staff: " . get_user_full_name_by_id(\Auth::user()->id) . 
            "<br>"._label('backend_student').": " . $user->full_name." (". $user->id. ")";
            $subject = "Course Suggested by Staff: " . get_user_full_name_by_id(\Auth::user()->id) . 
            "<br>"._label('backend_student').": " . $user->full_name." (". $user->id. ")";
        }

        //Notify Staff
        send_mail_notification($user->id,"","COURSE_SUGGESTED_BY_STAFF",$subject,$email_message);

        $subject = "";
        send_in_app_notification(
            $user->id, 
            $user->id, 
            "COURSE_SUGGESTED_BY_STAFF", 
            $subject, 
            $message,
            "student", 
        );

        // return redirect()->route('admin.auth.user.show', $user->id)->withFlashSuccess("Course Suggested & Student Notified Successfully");
        return redirect('admin/auth/user/'.$userid.'#coursefindersuggestion')->withFlashSuccess("Course Suggested & Student Notified Successfully");
    }

    /**
     * @param ManageUserRequest $request
     * @param User              $user
     *
     * @return mixed
     * @throws \App\Exceptions\GeneralException
     * @throws \Throwable
     */
    public function addApplication(ManageUserRequest $request, User $user)
    {
        $userid = $user->id;
        if (!$request->input('university')) {
            return redirect()->route('admin.auth.user.show', $user->id)->withFlashDanger("Please enter University name");
        }
        if (!$request->input('course')) {
            return redirect()->route('admin.auth.user.show', $user->id)->withFlashDanger("Please enter Course name");
        }
        DB::table('university_application')
            ->insert(
                ['userid' => $userid, 'created_at' => date('Y-m-d H:i:s'), 'university' => $request->input('university'), 'course' => $request->input('course'), 'country' => $request->input('country'), 'year' => $request->input('year'), 'intake' => $request->input('intake')]
            );

        return redirect()->route('admin.auth.user.show', $user->id)->withFlashSuccess("Course Application Added Successfully");
    }

    //Edit country info permissions for a member
    public function permission(ManageUserRequest $request, User $user)
    {
        $userid = $user->id;
        if (DB::table('users')->where('id', $userid)->exists()) {
            $user_row = User::where('id', $userid)->first();

            $data = array();
            $folders = DB::table('country_info_folder')->where("sharing", "private")->orderBy("id", "asc")->get();
            foreach ($folders as $folder) {
                $folder_name = $folder->name;
                if ($folder->parent != 0) {
                    $folder_name = $this->get_folder_tree($folder->id);
                }
                if (DB::table('user_folder_permission')->where("userid", $userid)->where("folderid", $folder->id)->exists())
                    $check = "checked";
                else
                    $check = "";
                $data[] = array("id" => $folder->id, "name" => $folder_name, "checked" => $check);
            }

            return view('backend.countryinfo.permission', ['userid' => $userid, 'user_name' => $user_row->first_name . " " . $user_row->last_name, 'data' => $data]);
        }
    }

    public function permissionUpdate(ManageUserRequest $request, User $user)
    {
        $userid = $user->id;
        DB::table('user_folder_permission')->where("userid", $userid)->delete();
        $folders = $request->input('folder');
        foreach ($folders as $folder) {
            DB::table('user_folder_permission')->insert(["userid" => $userid, "folderid" => $folder]);
        }
        return redirect()->route('admin.auth.user.countrypermission', $user->id)->withFlashSuccess("Folder Access Permission Updated Successfully");
    }

    private function get_folder_tree($folder_id)
    {
        $name = "";
        $row = DB::table('country_info_folder')->where("id", $folder_id)->first();
        $name = $row->name;
        $parent_id = $row->parent;
        while (1) {
            if ($parent_id == 0)
                break;
            $row = DB::table('country_info_folder')->where("id", $parent_id)->first();
            $name = $row->name . " > " . $name;
            $parent_id = $row->parent;
        }
        return $name;
    }

    public function democlasspermission(ManageUserRequest $request, User $user)
    {
        $userid = $user->id;
        if (DB::table('users')->where('id', $userid)->exists()) {
            $user_row = User::where('id', $userid)->first();

            $data = array();
            $folders = DB::table('demo_class_category')->where("sharing", "private")->orderBy("id", "asc")->get();
            foreach ($folders as $folder) {
                $folder_name = $folder->name;
                if ($folder->parent != 0) {
                    $folder_name = $this->get_democlass_folder_tree($folder->id);
                }
                if (DB::table('user_democlass_permission')->where("userid", $userid)->where("folderid", $folder->id)->exists())
                    $check = "checked";
                else
                    $check = "";
                $data[] = array("id" => $folder->id, "name" => $folder_name, "checked" => $check);
            }

            return view('backend.democlass.permission', ['userid' => $userid, 'user_name' => $user_row->first_name . " " . $user_row->last_name, 'data' => $data]);
        }
    }

    public function democlasspermissionUpdate(ManageUserRequest $request, User $user)
    {
        $userid = $user->id;
        DB::table('user_democlass_permission')->where("userid", $userid)->delete();
        $folders = $request->input('folder');
        foreach ($folders as $folder) {
            DB::table('user_democlass_permission')->insert(["userid" => $userid, "folderid" => $folder]);
        }
        return redirect()->route('admin.auth.user.democlasspermission', $user->id)->withFlashSuccess("Category Access Permission Updated Successfully");
    }

    private function get_democlass_folder_tree($folder_id)
    {
        $name = "";
        $row = DB::table('demo_class_category')->where("id", $folder_id)->first();
        $name = $row->name;
        $parent_id = $row->parent;
        while (1) {
            if ($parent_id == 0)
                break;
            $row = DB::table('demo_class_category')->where("id", $parent_id)->first();
            $name = $row->name . " > " . $name;
            $parent_id = $row->parent;
        }
        return $name;
    }

    public function my_docs(ManageUserRequest $request, User $user)
    {
        $data = array();
        return view('user.documents', $data);
    }

    public function convertStudentLead($id){
        $users = User::findOrFail($id);
        if (!$this->hasAccessPermission($users)) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        return view('backend.auth.user.show.tabs.student-lead-convert-modal', compact('users'));
    }

    public function archiveStudent($id){
        $user = User::where('id', $id)->first();
        $archived = DB::table("users")->where('id', $id)->update([
            'archived' => "1",
            'active' => "0",
            'deleted_at' => now(),
        ]);

        // Send Push App Notification
        if(\Auth::user()->isAgent()){
            $message = _label('backend_student')." Archived by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
            $email_message = "Hi, <br>"._label('backend_student')." Archived by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
            $subject = _label('backend_student').": " . $user->full_name." (". $user->id. ") Archived by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
        } else{
            $message = _label('backend_student')." Archived by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
            $email_message = "Hi, <br>"._label('backend_student')." Archived by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
            $subject = _label('backend_student').": " . $user->full_name." (". $user->id. ") Archived by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
        }

        //Sending email Notification
        $email_message .= "<br/><strong>"._label('backend_student')." Details:</strong><br/>" . $user->first_name ." ".$user->last_name. "<br/>Mob: " . $user->mobile . "<br/>Email: " . $user->email;
        send_mail_notification($user->id,"","LEAD_STUDENT_APPLICATION_COURSEPREFERENCE_OTHERS_DELETE",$subject,$email_message);

        $message .= "<br>"._label('backend_student').": " . $user->full_name." (". $user->id. ")";
        $subject = "";
        send_in_app_notification(
            $user->id, 
            $user->id,
            "LEAD_STUDENT_APPLICATION_COURSEPREFERENCE_OTHERS_DELETE", 
            $subject, 
            $message,
            "student"
        );

        if($archived){
            return redirect()->route('admin.auth.user.index')->withFlashSuccess("Student Archived Successfully!");
        }else{
            return redirect()->route('admin.auth.user.index')->withFlashDanger("Something Went Wrong!");
        }
    }

    public function archiveShowStudent(){
        $archived_student = DB::table('users')->where('archived', 1)->orderBy('created_at', 'desc')->get();
        return view('backend.auth.user.archivedStudent', ['archived_student' => $archived_student]);
    }

    public function unArchiveStudent($id){
        $unArchived = DB::table("users")->where('id', $id)->update([
            'archived' => "0",
            'active' => "1",
            'deleted_at' => NULL,
        ]);
        $user = User::where('id', $id)->first();
        // Send Push App Notification
        if(\Auth::user()->isAgent()){
            $message = _label('backend_student').": " . $user->full_name." (". $user->id. ") Unarchived by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
            $email_message = "Hi, <br>" . _label('backend_student').": " . $user->full_name." (". $user->id. ") Unarchived by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
            $subject = _label('backend_student').": " . $user->full_name." (". $user->id. ") Unarchived by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
        } else{
            $message = _label('backend_student').": " . $user->full_name." (". $user->id. ") Unarchived by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
            $email_message = "Hi, <br>" . _label('backend_student').": " . $user->full_name." (". $user->id. ") Unarchived by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
            $subject = _label('backend_student').": " . $user->full_name." (". $user->id. ") Unarchived by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
        }

        //Sending email Notification
        $email_message .= "<br/>Mob: " . $user->mobile . "<br/>Email: " . $user->email;
        send_mail_notification($user->id,"","LEAD_STUDENT_APPLICATION_COURSEPREFERENCE_OTHERS_DELETE",$subject,$email_message);

        $subject = "";
        send_in_app_notification(
            $user->id, 
            $user->id,
            "LEAD_STUDENT_APPLICATION_COURSEPREFERENCE_OTHERS_DELETE", 
            $subject, 
            $message,
            "student"
        );

        if($unArchived){
            return redirect()->route('admin.auth.user.index')->withFlashSuccess("Student Unarchived Successfully!");
        }else{
            return redirect()->route('admin.auth.user.index')->withFlashDanger("Something Went Wrong!");
        }
    }

    public function bulkArchiveStudent(Request $request){
        if (!(\Auth::user()->isAdmin() || \Auth::user()->can('delete-lead-student-application'))) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }

        $id_arr = explode(',', $request->input('id_arr'));
        if (is_array($id_arr)) {
            foreach ($id_arr as $stu_id) {
                $user = User::where('id', $stu_id)->first();
                $unArchived = DB::table("users")->where('id', $stu_id)->update([
                    'archived' => "1",
                    'active' => "0",
                    'deleted_at' => now(),
                ]);

                // Send Push App Notification
                if(\Auth::user()->isAgent()){
                    $message = _label('backend_student')." Archived by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
                    $email_message = "Hi, <br>"._label('backend_student')." Archived by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
                    $subject = _label('backend_student').": " . $user->full_name." (". $user->id. ") Archived by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
                } else{
                    $message = _label('backend_student')." Archived by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
                    $email_message = "Hi, <br>"._label('backend_student')." Archived by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
                    $subject = _label('backend_student').": " . $user->full_name." (". $user->id. ") Archived by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
                }

                //Sending email Notification
                $email_message .= "<br/><strong>"._label('backend_student')." Details:</strong><br/>" . $user->first_name ." ".$user->last_name. "<br/>Mob: " . $user->mobile . "<br/>Email: " . $user->email;
                send_mail_notification($user->id,"","LEAD_STUDENT_APPLICATION_COURSEPREFERENCE_OTHERS_DELETE",$subject,$email_message);

                $message .= "<br>"._label('backend_student').": " . $user->full_name." (". $user->id. ")";
                $subject = "";
                send_in_app_notification(
                    $user->id, 
                    $user->id,
                    "LEAD_STUDENT_APPLICATION_COURSEPREFERENCE_OTHERS_DELETE", 
                    $subject, 
                    $message,
                    "student"
                );
            }
        }
        return redirect()->route('admin.auth.user.index')->withFlashSuccess('Students Archive Successfully');
    }


    public function change_overseas_status(Request $request){
        
        $stuid = $request->stu_id;
        $status = $request->student_status;

        $users = User::findOrFail($stuid);
        $old_status = $users->overseas_status;
        if (!$this->hasAccessPermission($users)) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        $users->overseas_status = $status;
        $new_status = $status;
        $users->save();

        $followup = new UserActivity;
        $followup->user_id = $stuid;
        $followup->type = 'student_status_change';
        $followup->student_status_changed_to = $status;
        $followup->staff = \Auth::user()->id;
            
        $followup->detail = "STUDENT STATUS CHANGED TO: " . $status . ", Previous Status: " . $old_status;
        $followup->save();

        // Send Push App Notification
        if(\Auth::user()->isAgent()){
            $message = _label('backend_student')." Status Changed by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
            $email_message = "Hi, <br>"._label('backend_student')." Status Changed by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
        } else{
            $message = _label('backend_student')." Status changed by staff.";
            $email_message = "Hi, <br>"._label('backend_student')." Status Changed by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
        }

        // Sending email Notification
        $subject = _label('backend_student').": " . $users->full_name . " (".$users->id.") Status Changed to: $new_status";
        $email_message .= "<br><strong>Student Details:</strong>
        <br>" . $users->first_name ." ".$users->last_name. 
        "<br>Old Status: " . $old_status . 
        "<br>New Status: " . $new_status . 
        "<br/>Mob: " . $users->mobile . 
        "<br/>Email: " . $users->email;
        send_mail_notification($users->id,"","STUDENT_STATUS_CHANGE",$subject,$email_message);

        $message .= " "._label('backend_student').": <strong>" . $users->full_name." (". $users->id. ")</strong>";
        $subject = "";
        if($new_status){
            $message .= ", Old Status: " . $old_status . ", New Status: <strong>" . $new_status."</strong>";
        }
        send_in_app_notification(
            $users->id, 
            $users->id,
            "STUDENT_STATUS_CHANGE", 
            $subject, 
            $message,
            "student"
        );

        //return "Student Status Updated!";
        return redirect()->route('admin.auth.user.show', $stuid)->withFlashSuccess('Student Status Changed Successfully');
        //return response()->json(["status"=>'200',"res"=>'Student Status Updated!',"updated_status"=>'<span class="badge" style="font-size: 13px!important;background-color:'.get_overseas_status_color($status).';">'.$status.'</span>']);
    }

    public function student_disqualified($id){
        $student = User::findOrFail($id);
        return view('backend.auth.user.show.tabs.student_disqualified_modal', compact('student'));
    }

    public function updateStudentToDisqualified(Request $request){
        $request->validate([
            'student_disqualified_reason' => 'required',
        ]);

        $student = User::findOrFail($request->get('stu_id'));
        $old_status = $student->overseas_status;
        if (DB::table('users')->where("id", $student->id)->update(['overseas_status' => get_disqualified_student_status()])){
            $leadDetails = UserDetails::firstOrNew(array('userid' => $student->id));
            $leadDetails->student_disqualified_reason = $request->student_disqualified_reason;
            $leadDetails->save();
            $followup = new UserActivity;
            $followup->user_id = $student->id;
            $followup->type = 'student_status_change';
            $followup->student_status_changed_to = get_disqualified_student_status();
            $followup->staff = \Auth::user()->id;

            $new_status = get_disqualified_student_status();
            
            $followup->detail = "STUDENT STATUS CHANGED TO: " . get_disqualified_student_status()."<br>Reason: ".$request->student_disqualified_reason;
            $followup->save();

            // Send Push App Notification
            if(\Auth::user()->isAgent()){
                $message = _label('backend_student')." Status Changed by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
                $email_message = "Hi, <br>"._label('backend_student')." Status Changed by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
            } else{
                $message = _label('backend_student')." Status Changed by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
                $email_message = "Hi, <br>"._label('backend_student')." Status Changed by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
            }

            // Sending email Notification
            $subject = _label('backend_student').": " . $student->full_name . " (".$student->id.") Status Changed to: $new_status";
            $email_message .= "<br><strong>Student Details:</strong>
            <br>" . $student->first_name ." ".$student->last_name. 
            "<br>Old Status: " . $old_status . 
            "<br>New Status: " . $new_status . 
            "<br/>Mob: " . $student->mobile . 
            "<br/>Email: " . $student->email;
            send_mail_notification($student->id,"","STUDENT_STATUS_CHANGE",$subject,$email_message);

            $message .= "<br>"._label('backend_student').": " . $student->full_name." (". $student->id. ")";
            $message .= "<br>Old Status: " . $old_status . "<br>New Status: " . get_disqualified_student_status();
            $subject = "";
            send_in_app_notification(
                $student->id, 
                $student->id,
                "STUDENT_STATUS_CHANGE", 
                $subject, 
                $message,
                "student"
            );

            return redirect()->route('admin.auth.user.show', $student->id)->withFlashSuccess('Student Status Changed Successfully');
        } else {
            return redirect()->route('admin.auth.user.index')->withFlashDanger('This student status change not allowed');
        }
    }

    public function addStudentToAgent($id){
        $user = User::findOrFail($id);
        if (!$this->hasAccessPermission($user)) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        
        $agents = User::whereHas('roles', function ($q) {
                $q->whereIn('name', ['agent']);
            })
            ->get();

        return view('backend.auth.user.add-student-to-agent', ['user' => $user, 'agents' => $agents]);
    }

    public function addStudentToAgentSave(Request $request, $user){
        $request->validate([
            'agent_id' => 'required'
        ]);

        $user = User::findOrFail($request->student_id);
        $user->agent_id = $request->agent_id;
        $user->save();

        if($user->is_lead){
            // Sending email Notification
            $email_message = "Lead Added to Agent by Staff: " . get_user_full_name_by_id(\Auth::user()->id) . 
            "<br>Lead: " . $user->full_name." (". $user->id. ") <br/>Mob: " . $user->mobile . "<br/>Email: " . $user->email;
            $subject = "Lead: " . $user->full_name . " (".$user->id.") Added to Agent by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
            send_mail_notification($user->id,"","LEAD_STUDENT_LINK_AGENT",$subject,$email_message);

            // Send Push App Notification
            $message = "Lead Added to Agent by Staff: " . get_user_full_name_by_id(\Auth::user()->id) . 
            "<br>Lead: " . $user->full_name." (". $user->id. ")";
            $subject = "";
            send_in_app_notification(
                $user->id, 
                $user->id, 
                "LEAD_STUDENT_LINK_AGENT", 
                $subject, 
                $message,
                "lead"
            );

            return redirect()->route('admin.leadmanagementdetails')->withFlashSuccess('Lead Added to Agent Successfully');
        } else{
            // Send Push App Notification
            $email_message = _label('backend_student')." Added to Agent by Staff: " . get_user_full_name_by_id(\Auth::user()->id) . 
            "<br>"._label('backend_student').": " . $user->full_name." (". $user->id. ") <br/>Mob: " . $user->mobile . "<br/>Email: " . $user->email;
            $subject = _label('backend_student').": " . $user->full_name . " (".$user->id.") Added to Agent by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
            send_mail_notification($user->id,"","LEAD_STUDENT_LINK_AGENT",$subject,$email_message);

            // Send Push App Notification
            $message = _label('backend_student')." Added to Agent by Staff: " . get_user_full_name_by_id(\Auth::user()->id) . 
            "<br>"._label('backend_student').": " . $user->full_name." (". $user->id. ")";
            $subject = "";
            send_in_app_notification(
                $user->id, 
                $user->id, 
                "LEAD_STUDENT_LINK_AGENT", 
                $subject, 
                $message,
                "student"
            );

            return redirect()->route('admin.auth.user.show', $request->student_id)->withFlashSuccess("Student Added to Agent Successfully");
        }

    }

    public function unlinkAgent(Request $request, $user, $agent){
        $user = User::findOrFail($user);
        if($user){
            if($user->agent_id == $agent){
                $user->agent_id = 0;
                $user->save();
                return response()->json([
                    'success' => 1, 
                    'message' => 'Agent unlinked from Student'
                ]);
            }
            else{
                return response()->json([
                    'success' => 0, 
                    'message' => 'Agent is not linked to the selected Student!'
                ]);
            }
        }else{
            return response()->json([
                'success' => 0, 
                'message' => 'Invalid User!'
            ]);
        }
    }

    public function add_tag(Request $request){
        $client = User::findOrFail($request->get('client_id'));
        if($client){
            if($client->tags!=''){
                $old_tags = explode(',',$client->tags);
                $merged_values= array_unique(array_merge($old_tags, $request->client_tags));
                $new_values = implode(',',$merged_values);
            }else{
                $merged_values= $request->client_tags;
                $new_values = implode(',',$merged_values);
            }
            if (DB::table('users')->where("id", $client->id)->update(['tags' => $new_values])){
            foreach($merged_values as $tags)
                {
                $tagsData = Tags::firstOrNew(['tag_name' => $tags]);
                $tagsData->tag_name = $tags;
                $tagsData->save();
                }
                //return redirect()->route('admin.auth.user.index')->withFlashSuccess('Client Tags Updated');
                return response()->json([
                    'success' => 1, 
                    'message' => 'Tags Updated!'
                ]);
            }
            else {
                return response()->json([
                    'success' => 0,
                    'message' => 'Tag already exists!'
                ]);
                //return redirect()->route('admin.auth.user.index')->withFlashDanger('This Client tag change not allowed');
            }
        }
    }

    public function remove_tag(Request $request){
		$client = User::findOrFail($request->get('client_id'));
        if($client){
            if($client->tags!=''){
                $old_tags = explode(',',$client->tags);
                $tag_to_remove = $request->get('tag_name');
                $key = array_search($tag_to_remove, $old_tags);
                if (($key) !== false)
                {
                    unset($old_tags[$key]);
                    $new_values = implode(',',$old_tags);
                    if (DB::table('users')->where("id", $client->id)->update(['tags' => $new_values])){
                        return response()->json([
                            'success' => 1, 
                            'message' => 'Tag Removed'
                        ]);
                    }else{
                        return response()->json([
                            'success' => 0, 
                            'message' => 'Caanot remove tag'
                        ]);
                    }
                }
            }
            else{
                return response()->json([
                    'success' => 0, 
                    'message' => 'No tag found for this Client'
                ]);
            }
        }
        else{
			return response()->json([
				'success' => 0, 
				'message' => 'Invalid Client'
			]);
		}

	}

    public function bulkDeletedStudentAction(Request $request){
        if (!(\Auth::user()->isAdmin() || \Auth::user()->can('delete-lead-student-application'))) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }

        $request->validate([
            'selected_ids' => 'required|array',
            'action' => 'required|in:restore,delete',
        ]);

        $ids = $request->selected_ids;

        if ($request->action === 'restore') {
            User::withTrashed()->whereIn('id', $ids)->restore();

            foreach ($ids as $id) {
                $user = User::withTrashed()->find($id);
                //event(new \App\Events\Backend\Auth\UserRestored($user));
            }

            return redirect()->route('admin.auth.user.deleted')->withFlashSuccess('Selected users restored successfully.');
        }

        if ($request->action === 'delete') {
            foreach ($ids as $id) {       
                $user = User::onlyTrashed()->find($id);

                $message = _label('backend_student')." Deleted by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
                $email_message = "Hi, <br>"._label('backend_student')." Deleted by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
                $subject = _label('backend_student').": " . $user->full_name . " (".$user->id.") Deleted by Staff: " . get_user_full_name_by_id(\Auth::user()->id);

                //Sending email Notification
                $email_message .= "<br/><strong>"._label('backend_student')." Details:</strong><br/>" . $user->first_name ." ".$user->last_name. "<br/>Mob: " . $user->mobile . "<br/>Email: " . $user->email;
                send_mail_notification($user->id,"","LEAD_STUDENT_APPLICATION_COURSEPREFERENCE_OTHERS_DELETE",$subject,$email_message);

                $message .= "<br>"._label('backend_student').": " . $user->full_name." (". $user->id. ")";
                $subject = "";
                send_in_app_notification(
                    $user->id, 
                    $user->id,
                    "LEAD_STUDENT_APPLICATION_COURSEPREFERENCE_OTHERS_DELETE", 
                    $subject, 
                    $message,
                    "student"
                );

                $this->userRepository->eductrlDelete($user);
            }

            return redirect()->route('admin.auth.user.deleted')->withFlashSuccess(__('alerts.backend.users.deleted_permanently'));
        }

        return back()->withFlashDanger('Invalid bulk action selected.');
    }
}
