<?php

namespace App\Http\Controllers\Backend\Auth\User;

use Illuminate\Support\Facades\DB;
use App\Models\Auth\User;
use App\Http\Controllers\Controller;
use App\Models\MailTemplates;
use App\Notifications\Backend\UserAlert;
use App\Repositories\Backend\Auth\UserRepository;
use App\Http\Requests\Backend\Auth\User\ManageUserRequest;

/**
 * Class UserStatusController.
 */
class UserStatusController extends Controller
{
    /**
     * @var UserRepository
     */
    protected $userRepository;

    /**
     * @param UserRepository $userRepository
     */
    public function __construct(UserRepository $userRepository)
    {
        $this->userRepository = $userRepository;
    }

    /**
     * @param ManageUserRequest $request
     *
     * @return mixed
     */
    public function getDeactivated(ManageUserRequest $request)
    {
        return view('backend.auth.user.deactivated',['staff_mgmt'=>0])
            ->withUsers($this->userRepository->getInactivePaginated(25));
    }

    /**
     * @param ManageUserRequest $request
     *
     * @return mixed
     */
    public function getDeleted(ManageUserRequest $request)
    {
        return view('backend.auth.user.deleted',['staff_mgmt'=>0])
            ->withUsers($this->userRepository->getDeletedPaginated(25));
    }

    /**
     * @param ManageUserRequest $request
     * @param User              $user
     * @param                   $status
     *
     * @return mixed
     * @throws \App\Exceptions\GeneralException
     */
    public function mark(ManageUserRequest $request, User $user, $status)
    {
       if(env('DEMO_MODE') && in_array($user->id, config('app.demo_users')))
            return redirect()->route('admin.demo-user-restriction');    

        $this->userRepository->mark($user, $status);

        return redirect()->route(
            $status == 1 ?
            'admin.auth.user.index' :
            'admin.auth.user.deactivated'
        )->withFlashSuccess(__('alerts.backend.users.updated'));
    }

    
    
    /**
     * @param ManageUserRequest $request
     * @param User              $user
     * @param                   $status
     *
     * @return mixed
     * @throws \App\Exceptions\GeneralException
     */
    public function markStaff(ManageUserRequest $request, User $user, $status)
    {

        if(env('DEMO_MODE') && in_array($user->id, config('app.demo_users')))
            return redirect()->route('admin.demo-user-restriction');    
        
        $this->userRepository->mark($user, $status);

        return redirect()->route(
            $status == 1 ?
            'admin.auth.staff' :
            'admin.auth.staff.deactivated'
        )->withFlashSuccess(__('alerts.backend.users.updated'));
    }


    public function markAgent(ManageUserRequest $request, User $user, $status)
    {
        $this->userRepository->mark($user, $status);

        if($status==1){
            $mail_tpl = MailTemplates::find("AGENT_WELCOME");
            if($mail_tpl && $mail_tpl->status){
                $email_message = $mail_tpl->message;                        
                $email_message = str_ireplace("#sitename#", app_name(), $email_message);
                $email_message = str_ireplace("#siteurl#", url(''), $email_message);
                $email_message = str_ireplace("#first_name#", $user->first_name, $email_message);
                $email_message = str_ireplace("#full_name#", $user->full_name, $email_message);
                try{
                    //$user->notify(new UserAlert($mail_tpl->subject, $email_message));
                    user_notify($user,$mail_tpl->subject, $email_message);
                }catch (\Exception $e) {
                    $message = $e->getMessage();
                    \Log::debug("Mail Error: (Agent Approval - Notify Agent) ".$message);   
                }
            }
            $notify_sms = get_sms_template('AGENT_WELCOME');
            if ($notify_sms && $notify_sms->status) {
                $notify_sms->message = str_replace("#sitename#", app_name(), $notify_sms->message);
                $notify_sms->message = str_replace("#siteurl#", url('/'), $notify_sms->message);
                $notify_sms->message = str_replace("#first_name#", $user->first_name, $notify_sms->message);
                $notify_sms->message = str_replace("#full_name#", $user->full_name, $notify_sms->message);
                $user->sms($notify_sms->message, $notify_sms->template_id);
            }
        }
        return redirect()->back()->withFlashSuccess(__('alerts.backend.users.updated'));
    }
    
    /**
     * @param ManageUserRequest $request
     * @param User              $deletedUser
     *
     * @return mixed
     * @throws \App\Exceptions\GeneralException
     * @throws \Throwable
     */
    //public function delete(ManageUserRequest $request, User $deletedUser)
    //{
    //    $this->userRepository->forceDelete($deletedUser);
    //
    //    return redirect()->route('admin.auth.user.deleted')->withFlashSuccess(__('alerts.backend.users.deleted_permanently'));
    //}
    
    public function delete(ManageUserRequest $request, $deletedUser)
    {
        $User = User::onlyTrashed()->find($deletedUser);
        //$this->userRepository->forceDelete($User);
        $this->userRepository->eductrlDelete($User);
        
        return redirect()->route('admin.auth.user.deleted')->withFlashSuccess(__('alerts.backend.users.deleted_permanently'));
    }
    
    
    

    /**
     * @param ManageUserRequest $request
     * @param User              $deletedUser
     *
     * @return mixed
     * @throws \App\Exceptions\GeneralException
     */
    
    //public function restore(ManageUserRequest $request, User $deletedUser)
    //{
    //    $this->userRepository->restore($deletedUser);
    //
    //    return redirect()->route('admin.auth.user.index')->withFlashSuccess(__('alerts.backend.users.restored'));
    //}
    //
    
    public function restore(ManageUserRequest $request, $deletedUser)
    {
        $User = User::onlyTrashed()->find($deletedUser);
        $this->userRepository->restore($User);

        // Send Push App Notification
        if(\Auth::user()->isAgent()){
            $message = "Student  by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
        } else{
            $message = "Student Restored by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
        }

        // Send Push App Notification
        if(\Auth::user()->isAgent()){
            $message = _label('backend_student').": " . $User->full_name." (". $User->id. ") Restored by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
            $email_message = "Hi, <br>" . _label('backend_student').": " . $User->full_name." (". $User->id. ") Restored by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
            $subject = _label('backend_student').": " . $User->full_name." (". $User->id. ") Restored by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
        } else{
            $message = _label('backend_student').": " . $User->full_name." (". $User->id. ") Restored by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
            $email_message = "Hi, <br>" . _label('backend_student').": " . $User->full_name." (". $User->id. ") Restored by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
            $subject = _label('backend_student').": " . $User->full_name." (". $User->id. ") Restored by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
        }

        //Sending email Notification
        $email_message .= "<br/>Mob: " . $User->mobile . "<br/>Email: " . $User->email;
        send_mail_notification($User->id,"","LEAD_STUDENT_APPLICATION_COURSEPREFERENCE_OTHERS_DELETE",$subject,$email_message);

        $message .= "<br>"._label('backend_student').": " . $User->full_name." (". $User->id. ")";
        $subject = "";
        send_in_app_notification(
            $User->id, 
            $User->id,
            "LEAD_STUDENT_APPLICATION_COURSEPREFERENCE_OTHERS_DELETE",  
            $subject, 
            $message,
            "student"
        );

        return redirect()->route('admin.auth.user.index')->withFlashSuccess(__('alerts.backend.users.restored'));
    }
    
    
    
    
    
    /**
     * @param ManageUserRequest $request
     * @param User              $deletedUser
     *
     * @return mixed
     * @throws \App\Exceptions\GeneralException
     */
    //public function restoreStaff(ManageUserRequest $request, User $deletedUser)
    //{
    //    $this->userRepository->restore($deletedUser);
    //
    //    return redirect()->route('admin.auth.staff')->withFlashSuccess(__('alerts.backend.users.restored'));
    //}
    
    public function restoreStaff(ManageUserRequest $request, $deletedUser)
    {
        $User = User::onlyTrashed()->find($deletedUser);
        $this->userRepository->restore($User);
        return redirect()->route('admin.auth.staff')->withFlashSuccess(__('alerts.backend.users.restored'));
    }
    
    public function restoreAgent(ManageUserRequest $request, $deletedUser)
    {
        $User = User::onlyTrashed()->find($deletedUser);
        $this->userRepository->restore($User);
        return redirect()->route('admin.agents.index')->withFlashSuccess(__('alerts.backend.users.restored'));
    }
    
    public function restoreTrainer(ManageUserRequest $request, $deletedUser)
    {
        $User = User::onlyTrashed()->find($deletedUser);
        $this->userRepository->restore($User);
        return redirect()->route('admin.trainers.index')->withFlashSuccess(__('alerts.backend.users.restored'));
    }    
    
    /**
     * @param ManageUserRequest $request
     * @param User              $deletedUser
     *
     * @return mixed
     * @throws \App\Exceptions\GeneralException
     * @throws \Throwable
     */
    //public function deleteStaff(ManageUserRequest $request, User $deletedUser)
    //{
    //    $this->userRepository->forceDelete($deletedUser);
    //
    //    return redirect()->route('admin.auth.staff.deleted')->withFlashSuccess(__('alerts.backend.users.deleted_permanently'));
    //}

    public function deleteStaff(ManageUserRequest $request, $deletedUser){
        $User = User::onlyTrashed()->find($deletedUser);
        //$this->userRepository->forceDelete($User);
        $this->userRepository->eductrlDelete($User);
        return redirect()->route('admin.auth.staff.deleted')->withFlashSuccess(__('alerts.backend.users.deleted_permanently'));
    }
    
    public function deleteAgent(ManageUserRequest $request, $deletedUser)
    {
        $User = User::onlyTrashed()->find($deletedUser);
        //$this->userRepository->forceDelete($User);
        $this->userRepository->eductrlDelete($User);
        return redirect()->route('admin.agents.deleted')->withFlashSuccess(__('alerts.backend.users.deleted_permanently'));
    }
    
    public function deleteTrainer(ManageUserRequest $request, $deletedUser)
    {
        $User = User::onlyTrashed()->find($deletedUser);
        //$this->userRepository->forceDelete($User);
        $this->userRepository->eductrlDelete($User);
        return redirect()->route('admin.trainers.deleted')->withFlashSuccess(__('alerts.backend.users.deleted_permanently'));
    }    

    public function markStaffLock(ManageUserRequest $request, User $user, $status)
    {
        // dd($user->user_unlock , $status);
        if(env('DEMO_MODE') && in_array($user->id, config('app.demo_users')))
            return redirect()->route('admin.demo-user-restriction');    
        
        $this->userRepository->marklock($user, $status);
        //  dd($user->user_unlock , $status);
        return redirect()->route(
            'admin.auth.staff'
        )->withFlashSuccess(__('alerts.backend.users.updated'));
    }
    
}
