<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\File;
use ZipArchive;

/**
 * Class BackupController.
 */

class BackupController extends Controller
{

    public function __construct()
    {
        $this->middleware(function ($request, $next) {
            $user = \Auth::user();
            if (!\Auth::user()->isAdmin()) {
                return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
            }
            return $next($request);
        });
    }    
    public function index(){ return view('backend.backups.backup_index');}
    
    
    public function compressBackupToZip($backupPath)
    {
        // Define the path for the zip file
        $backupZipPath = str_replace('.sql', '.zip', $backupPath);
    
        // Initialize ZipArchive
        $zip = new ZipArchive();
        
        // Open the zip file for writing
        if ($zip->open($backupZipPath, ZipArchive::CREATE | ZipArchive::OVERWRITE) === true) {
            // Add the SQL file to the zip
            $zip->addFile($backupPath, basename($backupPath));
            
            // Close the zip file to finalize it
            $zip->close();
    
            // Optionally delete the original .sql file if you only need the .zip
            File::delete($backupPath);
    
            return $backupZipPath; // Return the path of the created zip file
        } else {
            throw new \Exception("Failed to create zip file for backup.");
        }
    }
    
    public function exportDatabase()
    {
        if (env('DEMO_MODE')) {
            return redirect()->back()->with('error', 'Database export is disabled in demo mode.');
        }
    
        $dbName = env('DB_DATABASE');
        $dbUser = env('DB_USERNAME');
        $dbPassword = env('DB_PASSWORD');
        $dbHost = env('DB_HOST');
        $backupPath = storage_path("app/private/{$dbName}-backup-".date("Y_m_d").".sql");
    
        try {
            $pdo = new \PDO("mysql:host=$dbHost;dbname=$dbName", $dbUser, $dbPassword);
    
            $tables = $pdo->query("SHOW TABLES")->fetchAll(\PDO::FETCH_COLUMN);
    
            $backupSql = "SET FOREIGN_KEY_CHECKS=0;\n\n";
            foreach ($tables as $table) {
                // Fetch the create statement for the table
                $createTable = $pdo->query("SHOW CREATE TABLE `$table`")->fetch(\PDO::FETCH_ASSOC);
                $backupSql .= $createTable['Create Table'] . ";\n\n";
            
                // Fetch all rows from the table
                $rows = $pdo->query("SELECT * FROM `$table`")->fetchAll(\PDO::FETCH_ASSOC);
                foreach ($rows as $row) {
                    $columns = array_map(function ($column) {
                        return "`$column`";
                    }, array_keys($row));
            
                    // Map the values and handle NULLs
                    $values = array_map(function ($value) use ($pdo) {
                        return is_null($value) ? "NULL" : $pdo->quote($value);
                    }, array_values($row));
            
                    // Build the insert statement
                    $backupSql .= "INSERT INTO `$table` (" . implode(", ", $columns) . ") VALUES (" . implode(", ", $values) . ");\n";
                }
                $backupSql .= "\n\n";
            }
    
            // Save backup to file
            File::put($backupPath, $backupSql);
            $backupZipPath = $this->compressBackupToZip($backupPath);
    
            return response()->download($backupZipPath)->deleteFileAfterSend(true);
        } catch (\Exception $e) {
            \Log::error('Database backup failed', ['error' => $e->getMessage()]);
            return redirect()->back()->with('error', 'Failed to export database.');
        }
    }
    

}
