<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\Auth\Chapters;
use App\Models\Auth\ChaptersQuiz;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class ChaptersQuizController extends Controller
{

    public function __construct()
    {
        $this->middleware(function ($request, $next) {
            $user = \Auth::user();
            if (!(\Auth::user()->isAdmin() || \Auth::user()->can('trainings configs'))) {
                return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
            }            
            return $next($request);
        });
    }
        
    public function create()
    {
        $chapterId = (int)\Request::get('chapter_id', 0);
        if ($chapterId > 0) {
            $chapterObj = Chapters::getChapterById($chapterId);
            if (empty($chapterObj)) {
                return redirect()->route('admin.chapters.index')->withFlashDanger('Chapter id is invalid!');
            }
            return view('backend.chapters.quiz.add', ['chapter_id' => $chapterId, 'chapterObj' => $chapterObj]);
        }
        return view('backend.chapters.quiz.add');
    }

    public function store(Request $request)
    {

        $request->validate([
            'chapter_id' => 'required|numeric',
            'quiz_question' => 'required',
            'answer_type' => 'required',
            'answers_options' => 'required|array',
            'correct_answers' => 'required|array',
            'display_order' => 'required|numeric'
        ]);
        $chapterId = $request->get('chapter_id');
        $chapterObj = Chapters::getChapterById($chapterId);
        if(empty($chapterObj)){
            return redirect()->route('admin.quiz.create')->withFlashDanger('Chapter id is invalid!');
        }
        $questionFile = '';
        if ($request->hasFile('quiz_attachment')) {
            $file = $request->quiz_attachment->getClientOriginalName();
            $filename = pathinfo($file, PATHINFO_FILENAME);
            if (strlen($filename) > 125)
                $filename = substr($filename, 0, 125);

            $filename = $filename . Str::random(20) . "." . $request->quiz_attachment->getClientOriginalExtension();
            $questionFile = $request->file('quiz_attachment')->storeAs('/images', $filename, 'public');
        }
        $data = array(
            'chapter_id' => $request->get('chapter_id'),
            'question' => $request->get('quiz_question'),
            'question_file' => $questionFile,
            'answer_type' => $request->get('answer_type'),
            'answer_options' => $request->get('answers_options'),
            'answers' => $request->get('correct_answers'),
            'display_order' => $request->get('display_order'),
            'status' => $request->get('status')
        );
        ChaptersQuiz::addOrUpdateChapterQuiz($data);
        return redirect()->route('admin.chapters.edit',$request->get('chapter_id'))->withFlashSuccess('Quiz Added Successfully!');
    }

    public function edit($id)
    {
        $quiz = ChaptersQuiz::with('chapterObj')->where('id','=',$id)->first();
        $correctAnswers = !empty($quiz['answers']) ? explode(',',$quiz['answers']) : array();
        return view('backend.chapters.quiz.edit', ['quiz' => $quiz,'correct_answers'=>$correctAnswers]);
    }

    public function update(Request $request, $id)
    {
        $request->validate([
            'chapter_id' => 'required|numeric',
            'quiz_question' => 'required',
            'answer_type' => 'required',
            'answers_options' => 'required|array',
            'correct_answers' => 'required|array',
            'display_order' => 'required|numeric'
        ]);
        $quizObj = ChaptersQuiz::getChaptersQuizById($id);
        if(empty($quizObj)){
            return redirect()->route('admin.quiz.index')->withFlashDanger('Quiz id is invalid!');
        }
        $chapterId = $request->get('chapter_id');
        $chapterObj = Chapters::getChapterById($chapterId);
        if(empty($chapterObj)){
            return redirect()->route('admin.quiz.create')->withFlashDanger('Chapter id is invalid!');
        }
        $questionFile = '';
        if ($request->hasFile('quiz_attachment')) {
            $oldImagePath = $quizObj->question_file;
            if(!empty($oldImagePath)){
                $oldFilePath = storage_path('app/public').'/'.$oldImagePath;
                if(file_exists($oldFilePath)){
                    unlink($oldFilePath);
                }
            }
            $file = $request->quiz_attachment->getClientOriginalName();
            $filename = pathinfo($file, PATHINFO_FILENAME);
            if (strlen($filename) > 125)
                $filename = substr($filename, 0, 125);

            $filename = $filename . Str::random(20) . "." . $request->quiz_attachment->getClientOriginalExtension();
            $questionFile = $request->file('quiz_attachment')->storeAs('/images', $filename, 'public');
        }
        $data = array(
            'chapter_id' => $request->get('chapter_id'),
            'question' => $request->get('quiz_question'),
            'question_file' => $questionFile,
            'answer_type' => $request->get('answer_type'),
            'answer_options' => $request->get('answers_options'),
            'answers' => $request->get('correct_answers'),
            'display_order' => $request->get('display_order'),
            'status' => $request->get('status')
        );
        ChaptersQuiz::addOrUpdateChapterQuiz($data,$id);
        return redirect()->route('admin.chapters.edit',$request->get('chapter_id'))->withFlashSuccess('Quiz Information saved!');
    }
    
    public function destroy($id)
    {
        $chapterQuizObj = ChaptersQuiz::getChaptersQuizById($id);
        $chapterId = (int)\Request::get('chapter_id');
        if(!empty($chapterQuizObj)){
            $chapterQuizObj->status = 0;
            $chapterQuizObj->save();
            return redirect()->route('admin.chapters.edit',$chapterId)->withFlashSuccess('Quiz Deleted Successfully');
        }
        return redirect()->route('admin.chapters.edit',$chapterId)->withFlashDanger('Quiz Not Deleted Successfully');
    }
}
