<?php

namespace App\Http\Controllers\Backend;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use App\Http\Controllers\Controller;
use App\Http\Requests\Backend\CountryInfo\AddFolderRequest;
use App\Http\Requests\Backend\CountryInfo\AddDocRequest;
use Illuminate\Support\Str;

/**
 * Class CountryInfoController.
 */
class CountryInfoController extends Controller {
   
    public function __construct()
    {
        $this->middleware(function ($request, $next) {
            $user = \Auth::user();
            if (!(\Auth::user()->isAdmin() || \Auth::user()->can('country info'))) {
                return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
            }            
            return $next($request);
        });
    }
        
    /**
     * @return \Illuminate\View\View
     */
    public function index($folderid = 0) {
        if (!\Auth::user()->isAdmin() && !\Auth::user()->can('country info')) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this page.');
        }
        
        $folders = array();
        $rows = DB::table('country_info_folder')->where('parent', $folderid)->get();
        if ($rows) {
            foreach ($rows as $row) {
                $folders[] = array("id" => $row->id, "name" => $row->name, "sharing" => $row->sharing, 'image' => $row->image);
            }
        }

        $cur_folder = $folderid;
        $folder_breadcrumb = array();
        $i = 0;
        while (true) {
            if ($cur_folder == 0) {
                $folder_breadcrumb[$i++] = array('id' => $cur_folder, 'name' => 'Top Level');
                break;
            } else {
                $row = DB::table('country_info_folder')->where('id', $cur_folder)->first();
                $folder_breadcrumb[$i++] = array('id' => $row->id, 'name' => $row->name);
                $cur_folder = $row->parent;
            }
        }

        //docs
        $docs = array();
        if ($folderid > 0) {
            $rows = DB::table('country_info_doc')->where('folder', $folderid)->get();
            if ($rows) {
                foreach ($rows as $row) {
                    $description = $row->description;
                    if (strlen($description) > 200)
                        $description = substr($description, 0, 200) . "...";
                    $filepath = url('storage/' . $row->file);
                    $docs[] = array("id" => $row->id, "title" => $row->title, "file" => $filepath, "description" => $description);
                }
            }
        }

        return view('backend.countryinfo', ['folders' => $folders, 'docs' => $docs, 'folder_breadcrumb' => $folder_breadcrumb, 'current_folder' => $folderid]);
    }

    public function addfolder(AddFolderRequest $request) {
        //\Log::debug("::addfolder::".$request->input('name')." parent:".$request->input('parent'));
        $path = NULL;
        if($request->file('image')){
            $path = $request->file('image')->storeAs('/countryinfo', Str::random(40) . '.' . $request->image->getClientOriginalExtension(), 'public');
        }
        DB::table('country_info_folder')
                ->insert(
                        ['name' => $request->input('name'), 'sharing' => $request->input('sharing'), 'parent' => $request->input('parent'),'image' =>$path]
        );

        return redirect()->back()->withFlashSuccess("Folder added successfully");
    }

    public function removefolder($id) {
        if (DB::table('country_info_folder')->where('id', $id)->exists()) {
            if (DB::table('country_info_doc')->where('folder', $id)->exists()) {
                return redirect()->back()->withFlashDanger("This folder has documents. First remove documents.");
            } else {
                if (DB::table('country_info_folder')->where('parent', $id)->exists()) {
                    return redirect()->back()->withFlashDanger("This folder has Sub Folder. First remove Sub Folder.");
                }

                DB::table('country_info_folder')->where('id', $id)->delete();
                return redirect()->back()->withFlashSuccess("Folder removed successfully");
            }
        }
    }

    public function adddoc(AddDocRequest $request) {
        $max_allowed = get_php_max_upload_size_kb();
        $request->validate([
            'file' => 'required|mimes:jpg,jpeg,png,docx,pdf,doc,mp4,mov,ogg,mp3,3gp,mpeg,zip|max:'.$max_allowed
        ],[
            'file.required' => 'Please select a file',
            'file.mimes' => 'Only jpg, jpeg, png, docx, pdf, doc, mp4, mov, ogg, mp3, 3gp, zip mpeg files are allowed',
        ]);
        // if (!$request->file('file')) {
        //     return redirect()->route('admin.countryinfo', $request->parent)->withFlashDanger("Please select a file");
        // }
        if (!$request->input('parent') > 0) {
            return redirect()->route('admin.countryinfo', $request->parent)->withFlashDanger("Uploading documents to Top Level not allowed");
        }

        $file = $request->file->getClientOriginalName();
        $filename = pathinfo($file, PATHINFO_FILENAME);
        if (strlen($filename) > 125)
            $filename = substr($filename, 0, 125);

        $filename = $filename . Str::random(20) . "." . $request->file->getClientOriginalExtension();

        $path = $request->file('file')->storeAs('/countryinfo', $filename, 'public');
        \Log::debug("::countryinfo doc savedpath::$path");
        DB::table('country_info_doc')
                ->insert(
                        ['title' => $request->input('title'), 'file' => $path, 'folder' => $request->input('parent'), 'description' => $request->input('description')]
        );
        return redirect()->back()->withFlashSuccess("Document added successfully");
    }

    public function removedoc($id) {
        if (DB::table('country_info_doc')->where('id', $id)->exists()) {
            $row = DB::table('country_info_doc')->where('id', $id)->first();
            $file_path = storage_path() . '/app/public/' . $row->file;
            DB::table('country_info_doc')->where('id', $id)->delete();
            \Log::debug("::countryinfo removedoc id::$id file::$file_path");
            //Storage::delete($file);
            unlink($file_path);

            return redirect()->back()->withFlashSuccess("Document removed successfully");
        }
    }


}
