<?php

namespace App\Http\Controllers\Backend;

use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Auth\User;
use App\Models\Auth\Invoice;
use App\Models\Auth\InvoiceItems;
use App\Models\Auth\InvoicePayment;
use App\Models\Auth\UniInvoice;
use App\Models\Auth\UniInvoiceDetail;
use App\Models\Auth\AgentCommission;
use App\Models\Auth\Batch;
use App\Models\Auth\TaxRates;
use App\Models\Auth\UniApplyAgents;
use Illuminate\Support\Facades\Input;
use App\Models\Auth\Universities;
use App\Models\Auth\UniversityApplication;
use App\Models\Auth\UniInvoicePayment;
use App\Repositories\Backend\Auth\UserRepository;
use App\Repositories\Backend\Auth\StaffRepository;
use PDF;
use Illuminate\Support\Facades\Mail;

class InvoiceController extends Controller
{

    protected $userRepository;

    /**
     * UserController constructor.
     *
     * @param UserRepository $userRepository
     */
    public function __construct(UserRepository $userRepository)
    {
        $this->userRepository = $userRepository;
        $this->middleware(function ($request, $next) {
          if (!(\Auth::user()->isAdmin() || \Auth::user()->can('invoice'))) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
          }
          return $next($request);
        });
    }

    public function university_invoices(Request $request){
        if(!env('UNIVERSITY_INVOICES_ENABLE')){
            return redirect()->route('admin.dashboard.message')->withFlashDanger('University Invoices is disabled.');
        }

        $data = $request->all();
        $filters = array("status" => "");
        if (isset($data['search']) && strlen($data['search']) && $data['status']) {
            $filters['search'] = $data['search'];
            $filters['status'] = $data['status'];
            $university = "%" . $data['search'] . "%";
            $pages = DB::table('uni_invoice')->select('uni_invoice.*')
                ->join('universities', 'uni_invoice.university_id', '=', 'universities.id')
                ->where('status', '=', $data['status'])
                ->when(!empty($university), function ($query) use ($university) {
                    return $query->Where(function ($query) use ($university) {
                        $query->Where('universities.university_name', 'like', "$university")
                        ->orWhere('invoice_num', 'like', $university);
                    });
                })
                ->orderBy('status', "ASC")
                ->orderBy('updated_at', "DESC")
                ->get();
        } elseif (isset($data['search']) && strlen($data['search'])) {
            $filters['search'] = $data['search'];
            $university = "%" . $data['search'] . "%";
            $pages = DB::table('uni_invoice')->select('uni_invoice.*')
                ->join('universities', 'uni_invoice.university_id', '=', 'universities.id')
                ->when(!empty($university), function ($query) use ($university) {
                    return $query->Where(function ($query) use ($university) {
                        $query->Where('universities.university_name', 'like', "$university")
                        ->orWhere('invoice_num', 'like', $university);
                    });
                })
                ->orderBy('status', "ASC")
                ->orderBy('updated_at', "DESC")
                ->get();
        } elseif (isset($data['status'])) {
            $filters['status'] = $data['status'];
            $pages = DB::table('uni_invoice')->select('uni_invoice.*')
                ->where('status', '=', $data['status'])
                ->orderBy('status', "ASC")
                ->orderBy('updated_at', "DESC")
                ->get();
        } else {
            $pages = DB::table('uni_invoice')
                ->select('uni_invoice.*')
                ->orderBy('uni_invoice.created_at', "DESC")
                ->get();
        }
        return view('backend.invoice.uni_invoice_list', ['pages' => $pages, 'filters' => $filters]);
    }

    public function generate_enrolled_student_invoice(){
        $universities = Universities::orderBy('university_name')->get(["university_name", "id"]);
        $app_statuses = get_application_statuses();
        return view('backend.invoice.generate_enrolled_invoice', ['universities' => $universities,'app_statuses' => $app_statuses]);
    }

    public function generate_enrolled_student_invoice_step_two(Request $request)
    {
        $appdata_arr = array();
        $has_fees = 1;
        $id_arr = explode(',', $request->input('id_arr'));
        if (is_array($id_arr)) {
            foreach ($id_arr as $leadid) {
                $appdata_arr[] = $app_arr = UniversityApplication::join('users', 'users.id', '=', 'university_application.userid')->where('appl_id',$leadid)->first(["university_application.*","users.first_name","users.last_name"]);
                if(get_course_details_by_pref_id($app_arr->course_pref_id)->total_tuition_fee == ""){
                    $has_fees = 0;
                }
            }
        }
        if($request->uni_id){
            $uni_id = get_university_id_by_name($request->uni_id);
            $universityData = Universities::where('id',$uni_id)->first(["university_name", "id","commission_percent","commission_count","tax_percent","tax_type"]);
        }
        if($request->get('business')!=""){
            $current_business = $request->get('business');
        }else{
            $current_business = "1";
        }
        return view('backend.invoice.generate_enrolled_invoice_step_two', ['appdata_arr' => $appdata_arr,'universityData' => $universityData,'has_fees' => $has_fees,'current_business' => $current_business]);
    }

    public function generate_enrolled_student_invoice_save(Request $request){
        $validator =  \Validator::make($request->all(), [
            'student_id' => 'required',
            'app_id' => 'required',
            'payment_stage' => 'required',
            'total_commission' => 'required',
            'grand_total' => 'required',
        ]);
        if($validator->fails()) {
            return redirect()->route('admin.invoice.generate-enrolled-student-invoice')->withErrors($validator);
        }
        $due_date = date('Y-m-d',strtotime(now()));
        $inv_id = "";
        $inv_count =  UniInvoice::latest('invoice_num')->first();
        //$inv_count = UniInvoice::count();
        if($inv_count){
            $inv_num = $inv_count->invoice_num;
        }else{
            $inv_num = 100000;
        }
        $inv_id = $inv_num + 1;

        $payment_stage = implode(',', $request->get('payment_stage'));
        if($request->get('total_commission')){
            $sub_total = $request->get('total_commission');
        } else{
            $sub_total = 0;
        }
        if($request->get('grand_total')){
            $grand_total = $request->get('grand_total');
        } else{
            $grand_total = 0;
        }
        $terms = get_business_details($request->get('business_id'))->terms_text;
        $record = new UniInvoice([
            'invoice_num' => $inv_id,
            'payment_stage' => $payment_stage,
            'business_id' => $request->get('business_id'),
            'university_id' => $request->get('university_id'),
            'due_date' => $due_date,
            'sub_total' => $sub_total,
            'currency' => $request->get('currency'),
            'tax' => $request->get('total_tax'),
            'tax_rate' => $request->get('tax_percent'),
            'tax_type' => $request->get('tax_type'),
            'commission_rate' => $request->get('commission_percent'),
            'terms_conditions' => $terms,
            'total' => $grand_total,
        ]);
        $record->save();
        $app_id = $request->get('app_id');
        for($i = 0; $i < count($app_id); $i++){
                $invoice_item = new UniInvoiceDetail;
                $invoice_item->invoice_id = $record->id;
                $invoice_item->student_id = $request->student_id[$i];
                $invoice_item->commission = $request->commission[$i];
                $invoice_item->uni_application_id = $app_id[$i];
                $invoice_item->tax = $request->tax[$i];
                $invoice_item->total = $request->total_inline[$i];
                $invoice_item->save();	
        }
        return redirect()->route('admin.invoice.university-invoices')->withFlashSuccess('University Invoice Generated!');
    }

    public function getApplications(Request $request)
    {
        $columns = array(
            0 => 'appl_id',
            2 => 'created_at',
            3 => 'student',
            4 => 'country',
            9 => 'status',
        );
        $limit = $request->input('length');
        $start = $request->input('start');
        $order = $columns[$request->input('order.0.column')];
        $dir = $request->input('order.0.dir');
        $university = $request->input('university');
        $status = $request->input('app_status');
        $intake_year = $request->input('intake_year');
        $get_search = $request->input('search.value');
            $applications = DB::table('university_application')
                ->join('users', 'users.id', '=', 'university_application.userid')
                ->select(DB::raw('university_application.*, users.first_name, users.last_name, users.email, users.mobile'))
                ->where(function ($query) use ($get_search) {
                    $query->Where('users.first_name', 'like', "%$get_search%")
                        ->orWhere('users.last_name', 'like', "%$get_search%")
                        ->orWhereRaw("CONCAT(`first_name`, ' ', `last_name`) LIKE ?", ['%'.$get_search.'%'])
                        ->orWhere('users.email', 'like', "%$get_search%")
                        ->orWhere('users.mobile', '=', "$get_search")
                        ->orWhere('university_application.appl_id', '=', "$get_search")
                        ->orWhere('university_application.country', 'like', "%$get_search%")
                        ->orWhere('university_application.course', 'like', "%$get_search%")
                        ->orWhere('university_application.university', 'like', "%$get_search%");
                })
                ->when(!empty($university), function ($query) use ($university) {
                    return  $query->Where('university_application.university', '=', "$university");
                })
                ->when(!empty($status), function ($query) use ($status) {
                    return  $query->Where('university_application.status',$status);
                })
                ->when(!empty($intake_year), function ($query) use ($intake_year) {
                    return  $query->Where('university_application.year', '=', "$intake_year");
                })
                ->whereNull('users.deleted_at')
                ->where('users.archived', "0")
                ->offset($start)
                ->limit($limit)
                ->orderBy($order, $dir)
                ->get();

        $totalData = $totalFiltered = count($applications);
        $data = $customResult = array();
        if (!empty($applications)) {
            $count = 1;
            foreach ($applications as $appl) {
                $customResult['id'] = $appl->appl_id;
                $customResult['appl_id'] = '<a href="'.route('admin.applicationdetail', [$appl->id]).'" title="Details" class="" style="font-size:13px;"><b>'.$appl->appl_id.'</b></a>';
                $customResult['export_id'] = '<span>'.$appl->appl_id.'</span>';
                $customResult['student'] = "<a target='_blank' href='" . route('admin.auth.user.show', $appl->userid) . "'><strong><span style='white-space:nowrap;'>".$appl->first_name . ' ' . $appl->last_name."</a>";
                $customResult['country'] = $appl->country;

                if($appl->type == 'visa') {
                    $visa_pref = DB::table("visa_preference")->where('id', $appl->visa_pref_id)->first();
                    $customResult['details'] = get_visa_service_type_name($visa_pref->service_type)."<br />".get_visa_letter_type_name($visa_pref->letter_type);
                    $customResult['status'] = show_visa_application_status($appl->visa_status);
                } elseif($appl->type == 'abroad_job') {
                    $visa_pref = DB::table("abroad_job_preference")->where('id', $appl->abroad_job_pref_id)->first();
                    $customResult['details'] = "";
                    $customResult['status'] = show_abroad_job_application_status($appl->abroad_job_status);
                } else {
                    $customResult['details'] = "<span class='text-sub'>University:</span> ".$appl->university."<br /><span class='text-sub'>Course:</span> ".$appl->course."<br /><span class='text-sub'>Intake:</span> ".$appl->year."-".get_intake_month_name($appl->intake);
                    $customResult['status'] = show_application_status($appl->status);
                }

                $customResult['university'] = $appl->university;
                $customResult['course'] =  $appl->course;
                $customResult['intake'] =  $appl->year."-".get_intake_month_name($appl->intake);
                $customResult['created_at'] = date("d-m-Y", strtotime($appl->created_at));
                $data[] = $customResult;
            }
        }
        $json_data = array(
            "draw"            => intval($request->input('draw')),
            "recordsTotal"    => intval($totalData),
            "recordsFiltered" => intval($totalFiltered),
            "data"            => $data,
            "status"   =>   $status,
        );
        return json_encode($json_data);
    }

    public function view_uni_invoice($id)
    {
        $invoice = UniInvoice::find($id);
        $invoiceDetails = UniInvoiceDetail::join('university_application', 'uni_invoice_detail.uni_application_id', '=', 'university_application.id')->where('invoice_id',$invoice->id)->get(['uni_invoice_detail.*','university_application.intake','university_application.year','university_application.course_pref_id','university_application.appl_id']);

        $uniData = Universities::find($invoice->university_id);
        $payments = UniInvoicePayment::where('invoice_id', $invoice->id)->get();
        $total_payment = UniInvoicePayment::where('invoice_id', $invoice->id)->sum('amount');

        return view('backend.invoice.uni_invoice_detail', compact('invoice','invoiceDetails', 'uniData', 'payments', 'total_payment'));
    }

    public function add_uni_payment($id)
    {
        $invoice = UniInvoice::find($id);
        $uniData = Universities::find($invoice->university_id);
        $payments = UniInvoicePayment::where('invoice_id', $invoice->id)->get();
        $total_payment = UniInvoicePayment::where('invoice_id', $invoice->id)->sum('amount');
        $staff = $this->userRepository->getNonStudentsTrainers();
        return view('backend.invoice.add_uni_payment', compact('invoice', 'uniData', 'payments', 'total_payment', 'staff'));
    }

    public function store_uni_payment(Request $request)
    {
        $request->validate([
            'amount' => 'required',
            'payment_date' => 'required',
            'payment_mode' => 'required',
            'transaction_id' => 'required',
        ]);
        $payment_date = "";
          if(!empty($request->get('payment_date'))){
              $payment_date = date('Y-m-d',strtotime($request->get('payment_date')));
          }
        $record = new UniInvoicePayment([
            'invoice_id' => $request->get('invoice_id'),
            'amount' => $request->get('amount'),
            'payment_date' => $payment_date,
            'pay_mode' => $request->get('payment_mode'),
            'trans_id' => $request->get('transaction_id')
        ]);
        $record->save();
        $invoice = UniInvoice::find($request->get('invoice_id'));
        $total_payment = UniInvoicePayment::where('invoice_id', $request->get('invoice_id'))->sum('amount');
        if ($invoice->total - $total_payment <= 0) {
            $invoice->status = "paid";
            $invoice->save();
        }
        $invoiceDetails = UniInvoiceDetail::join('university_application', 'uni_invoice_detail.uni_application_id', '=', 'university_application.id')->where('invoice_id',$invoice->id)->whereNotNull('my_commission_percent')->get(['uni_invoice_detail.*','university_application.agent_commission_percent','university_application.agent_commission','university_application.my_commission_percent','university_application.userid','university_application.fees_currency']);
        if(count($invoiceDetails)){
                foreach($invoiceDetails as $inv_details){
                    $userData = get_user_by_id($inv_details->userid);
                    if($userData->agent_id != 0){
                        //agent commission
                            $agentCommission = new AgentCommission([
                            'appl_id' => $inv_details->uni_application_id,
                            'agent_id' => $userData->agent_id,
                            'currency' => $inv_details->fees_currency,
                            'amount' => $inv_details->total * ($inv_details->agent_commission_percent / $inv_details->my_commission_percent),
                            //'agent_commission_notes' => $request->agent_commission_notes,
                            'created_by' => \Auth::user()->id,
                            ]);
                            $agentCommission->save();
                        //agent commission
                    }
                }
        }
        return redirect()->route('admin.invoice.view-uni-invoice', $request->get('invoice_id'))->withFlashSuccess('University Payment Record Added Successfully');
    }

    public function generate_ad_hoc_invoice(Request $request){
        $universities = Universities::orderBy('university_name')->get(["university_name", "id"]);
        if($request->get('business')!=""){
            $current_business = $request->get('business');
        }else{
            $current_business = "1";
        }
        return view('backend.invoice.generate_adhoc_invoice', ['universities' => $universities, 'current_business' => $current_business]);
    }

    public function generate_ad_hoc_invoice_save(Request $request){
        $request->validate([
            'university' => 'required',
            'tax_amount' => 'required',
        ]);

        $due_date = date('Y-m-d',strtotime(now()));
        $inv_id = "";
        $inv_count =  UniInvoice::latest('invoice_num')->first();
        //$inv_count = UniInvoice::count();
        if($inv_count){
            $inv_num = $inv_count->invoice_num;
        }else{
            $inv_num = 100000;
        }
        $inv_id = $inv_num + 1;
         
        if($request->get('sub_total')){
            $sub_total = $request->get('sub_total');
        } else{
            $sub_total = 0;
        }
        if($request->get('grand_total')){
            $grand_total = $request->get('grand_total');
        } else{
            $grand_total = 0;
        }

        $record = new UniInvoice([
            'invoice_num' => $inv_id,
            'business_id' => $request->get('business_id'),
            'university_id' => $request->get('university'),
            'due_date' => $due_date,
            'ad_hoc' => 1,
            'item1' => $request->get('item1'),
            'item1_amount' => $request->get('item1amount'),
            'sub_total' => $sub_total,
            'currency' => $request->get('currency'),
            'tax' => $request->get('tax_amount'),
            'tax_rate' => $request->get('tax_percent'),
            'tax_type' => $request->get('tax_type'),
            'terms_conditions' => $request->get('terms_conditions'),
            'total' => $grand_total,
        ]);
        $record->save();
    
        return redirect()->route('admin.invoice.university-invoices')->withFlashSuccess('Ad-hoc Invoice Generated!');
    }

    public function search_uni_apply_agent_data(Request $request)
    {
        $agentData = UniApplyAgents::where('id', $request->id)->first();
        return ['agentData' => $agentData];
    }

    public function search_university_data(Request $request)
    {
        $uniData = Universities::where('id', $request->id)->first();
        $uni_currency = config('app.currency_symbol');
        if($uniData){
            $country = DB::table('countries')->where('id', '=', $uniData->country)->first(['currency']);
            $uni_currency = get_currency_symbol($country->currency);
            $currency_code = $country->currency;
        }
        return ['uni_currency' => $uni_currency,'currency_code'=> $currency_code, 'uniData' => $uniData];
    }

    public function view_ad_hoc_invoice($id)
    {
        $invoice = UniInvoice::find($id);
        $uniData = Universities::find($invoice->university_id);
        $payments = UniInvoicePayment::where('invoice_id', $invoice->id)->get();
        $total_payment = UniInvoicePayment::where('invoice_id', $invoice->id)->sum('amount');
        return view('backend.invoice.adhoc_invoice_detail', compact('invoice','uniData', 'payments', 'total_payment'));
    }

    public function add_ad_hoc_payment($id)
    {
        $invoice = UniInvoice::find($id);
        $uniData = Universities::find($invoice->university_id);
        $payments = UniInvoicePayment::where('invoice_id', $invoice->id)->get();
        $total_payment = UniInvoicePayment::where('invoice_id', $invoice->id)->sum('amount');
        $staff = $this->userRepository->getNonStudentsTrainers();
        return view('backend.invoice.add_adhoc_payment', compact('invoice', 'uniData', 'payments', 'total_payment', 'staff'));
    }

    public function store_ad_hoc_payment(Request $request)
    {
        $request->validate([
            'amount' => 'required',
            'payment_date' => 'required',
            'payment_mode' => 'required',
            'transaction_id' => 'required',
        ]);

        $payment_date = "";
          if(!empty($request->get('payment_date'))){
              $payment_date = date('Y-m-d',strtotime($request->get('payment_date')));
          }
        $record = new UniInvoicePayment([
            'invoice_id' => $request->get('invoice_id'),
            'amount' => $request->get('amount'),
            'payment_date' => $payment_date,
            'pay_mode' => $request->get('payment_mode'),
            'trans_id' => $request->get('transaction_id')
        ]);
        $record->save();
        $invoice = UniInvoice::find($request->get('invoice_id'));
        $total_payment = UniInvoicePayment::where('invoice_id', $request->get('invoice_id'))->sum('amount');
        if ($invoice->total - $total_payment <= 0) {
            $invoice->status = "paid";
            $invoice->save();
        }
        return redirect()->route('admin.invoice.view-ad-hoc-invoice', $request->get('invoice_id'))->withFlashSuccess('Ad-hoc Payment Record Added Successfully');
    }

    //uni apply agent invoice
    public function generate_uni_apply_agent_invoice(){
        $universities = Universities::orderBy('university_name')->get(["university_name", "id"]);
        $app_statuses = get_application_statuses();
        return view('backend.invoice.generate_uni_apply_agent_invoice', ['universities' => $universities,'app_statuses' => $app_statuses]);
    }

    public function getAgentApplications(Request $request)
    {
        $columns = array(
            0 => 'appl_id',
            2 => 'created_at',
            3 => 'student',
            4 => 'country',
            9 => 'status',
        );
        $limit = $request->input('length');
        $start = $request->input('start');
        $order = $columns[$request->input('order.0.column')];
        $dir = $request->input('order.0.dir');
        $university = $request->input('university');
        $status = $request->input('app_status');
        $intake_year = $request->input('intake_year');
    
        $get_search = $request->input('search.value');

            $applications = DB::table('university_application')
                ->join('users', 'users.id', '=', 'university_application.userid')
                ->select(DB::raw('university_application.*, users.first_name, users.last_name, users.email, users.mobile'))
                ->where(function ($query) use ($get_search) {
                    $query->Where('users.first_name', 'like', "%$get_search%")
                        ->orWhere('users.last_name', 'like', "%$get_search%")
                        ->orWhereRaw("CONCAT(`first_name`, ' ', `last_name`) LIKE ?", ['%'.$get_search.'%'])
                        ->orWhere('users.email', 'like', "%$get_search%")
                        ->orWhere('users.mobile', '=', "$get_search")
                        ->orWhere('university_application.appl_id', '=', "$get_search")
                        ->orWhere('university_application.country', 'like', "%$get_search%")
                        ->orWhere('university_application.course', 'like', "%$get_search%")
                        ->orWhere('university_application.university', 'like', "%$get_search%");
                })
                ->when(!empty($university), function ($query) use ($university) {
                    return  $query->WhereIn('university_application.university', $university);
                })
                ->when(!empty($status), function ($query) use ($status) {
                    return  $query->Where('university_application.status',$status);
                })
                ->when(!empty($intake_year), function ($query) use ($intake_year) {
                    return  $query->Where('university_application.year', '=', "$intake_year");
                })
                ->whereNull('users.deleted_at')
                ->where('users.archived', "0")
                ->offset($start)
                ->limit($limit)
                ->orderBy($order, $dir)
                ->get();

        $totalData = $totalFiltered = count($applications);

        $data = $customResult = array();
        if (!empty($applications)) {
            $count = 1;
            foreach ($applications as $appl) {
                $customResult['id'] = $appl->appl_id;
                $customResult['appl_id'] = '<a href="'.route('admin.applicationdetail', [$appl->id]).'" title="Details" class="" style="font-size:13px;"><b>'.$appl->appl_id.'</b></a>';
                $customResult['export_id'] = '<span>'.$appl->appl_id.'</span>';
                $customResult['student'] = "<a target='_blank' href='" . route('admin.auth.user.show', $appl->userid) . "'><strong><span style='white-space:nowrap;'>".$appl->first_name . ' ' . $appl->last_name."</a>";
                $customResult['country'] = $appl->country;

                if($appl->type == 'visa') {
                    $visa_pref = DB::table("visa_preference")->where('id', $appl->visa_pref_id)->first();
                    $customResult['details'] = get_visa_service_type_name($visa_pref->service_type)."<br />".get_visa_letter_type_name($visa_pref->letter_type);
                    $customResult['status'] = show_visa_application_status($appl->visa_status);
                } elseif($appl->type == 'abroad_job') {
                    $visa_pref = DB::table("abroad_job_preference")->where('id', $appl->abroad_job_pref_id)->first();
                    $customResult['details'] = "";
                    $customResult['status'] = show_abroad_job_application_status($appl->abroad_job_status);
                } else {
                    $customResult['details'] = "<span class='text-sub'>University:</span> ".$appl->university."<br /><span class='text-sub'>Course:</span> ".$appl->course."<br /><span class='text-sub'>Intake:</span> ".$appl->year."-".get_intake_month_name($appl->intake);
                    $customResult['status'] = show_application_status($appl->status);
                }
                $customResult['university'] = $appl->university;
                $customResult['course'] =  $appl->course;
                $customResult['intake'] =  $appl->year."-".get_intake_month_name($appl->intake);
                $customResult['created_at'] = date("d-m-Y", strtotime($appl->created_at));
                $data[] = $customResult;
            }
        }

        $json_data = array(
            "draw"            => intval($request->input('draw')),
            "recordsTotal"    => intval($totalData),
            "recordsFiltered" => intval($totalFiltered),
            "data"            => $data,
            "status"   =>   $status,
        );

        return json_encode($json_data);
    }

    public function generate_uni_apply_agent_invoice_step_two(Request $request)
    {
        $currency_single = 0;
        $has_fees = 1;
        $appdata_arr = $currency_arr = array();
        $id_arr = explode(',', $request->input('id_arr'));
        if (is_array($id_arr)) {
            foreach ($id_arr as $leadid) {
                $appdata_arr[] = $app_arr = UniversityApplication::join('users', 'users.id', '=', 'university_application.userid')
                ->join('universities', 'universities.university_name', '=', 'university_application.university')
                ->where('appl_id',$leadid)
                ->first(["university_application.*","users.first_name","users.last_name",'universities.commission_percent']);
                $currency_arr[] = get_course_details_by_pref_id($app_arr->course_pref_id)->currency;
                if(get_course_details_by_pref_id($app_arr->course_pref_id)->total_tuition_fee == ""){
                    $has_fees = 0;
                }
            }
            if(count(array_unique($currency_arr)) == 1){
                $currency_single = 1;
            }else{
                $currency_single = 0;
            }
        }
        $uni_apply_agents = UniApplyAgents::orderBy('name','asc')->get();
        if($request->get('business')!=""){
            $current_business = $request->get('business');
        }else{
            $current_business = "1";
        }
        return view('backend.invoice.generate_uni_apply_agent_invoice_step_two', ['uni_apply_agents'=> $uni_apply_agents, 'appdata_arr' => $appdata_arr, 'currency_single' => $currency_single,'has_fees' => $has_fees,'current_business' => $current_business]);
    }

    public function generate_uni_apply_agent_invoice_save(Request $request){
        $validator =  \Validator::make($request->all(), [
            'student_id' => 'required',
            'app_id' => 'required',
            'payment_stage' => 'required',
            'total_commission' => 'required',
            'grand_total' => 'required',
        ]);
        if($validator->fails()) {
            return redirect()->route('admin.invoice.generate-uni-apply-agent-invoice')->withErrors($validator);
        }

        $due_date = date('Y-m-d',strtotime(now()));
        $inv_id = "";
        $inv_count =  UniInvoice::latest('invoice_num')->first();
        //$inv_count = UniInvoice::count();
        if($inv_count){
            $inv_num = $inv_count->invoice_num;
        }else{
            $inv_num = 100000;
        }
        $inv_id = $inv_num + 1;

        $payment_stage = implode(',', $request->get('payment_stage'));
        if($request->get('total_commission')){
            $sub_total = $request->get('total_commission');
        } else{
            $sub_total = 0;
        }
        if($request->get('grand_total')){
            $grand_total = $request->get('grand_total');
        } else{
            $grand_total = 0;
        }
        $terms = get_business_details($request->get('business_id'))->terms_text;
        $record = new UniInvoice([
            'invoice_num' => $inv_id,
            'payment_stage' => $payment_stage,
            'business_id' => $request->get('business_id'),
            'uni_apply_agent_id' => $request->get('uni_apply_agent'),
            'due_date' => $due_date,
            'sub_total' => $sub_total,
            'currency' => $request->get('currency'),
            'tax' => $request->get('total_tax'),
            'tax_rate' => $request->get('agent_tax_percent'),
            'tax_type' => $request->get('agent_tax_type'),
            'terms_conditions' => $terms,
            'total' => $grand_total,
        ]);
        $record->save();
        $app_id = $request->get('app_id');
        for($i = 0; $i < count($app_id); $i++){
                $invoice_item = new UniInvoiceDetail;
                $invoice_item->invoice_id = $record->id;
                $invoice_item->student_id = $request->student_id[$i];
                $invoice_item->commission_rate = $request->commission_percent[$i];
                $invoice_item->commission = $request->commission[$i];
                $invoice_item->uni_application_id = $app_id[$i];
                $invoice_item->tax = $request->tax[$i];
                $invoice_item->total = $request->total_inline[$i];
                $invoice_item->save();	
        }
        return redirect()->route('admin.invoice.university-invoices')->withFlashSuccess('Uni Apply Agent Invoice Generated!');
    }

    public function view_uni_apply_agent_invoice($id)
    {
        $invoice = UniInvoice::find($id);
        $invoiceDetails = UniInvoiceDetail::join('university_application', 'uni_invoice_detail.uni_application_id', '=', 'university_application.id')->where('invoice_id',$invoice->id)->get(['uni_invoice_detail.*','university_application.intake','university_application.year','university_application.course_pref_id','university_application.appl_id','university_application.university']);

        $agentData = UniApplyAgents::where('id', $invoice->uni_apply_agent_id)->first();
        $payments = UniInvoicePayment::where('invoice_id', $invoice->id)->get();
        $total_payment = UniInvoicePayment::where('invoice_id', $invoice->id)->sum('amount');
        return view('backend.invoice.uni_apply_agent_invoice_detail', compact('invoice','invoiceDetails', 'agentData', 'payments', 'total_payment'));
    }
//uni apply agent invoice

    public function student_invoices(Request $request)
    {
        $data = $request->all();
        $filters = array("status" => "");

        if (isset($data['search']) && strlen($data['search']) && $data['status']) {
            $filters['search'] = $data['search'];
            $filters['status'] = $data['status'];
            $student = "%" . $data['search'] . "%";
            $pages = DB::table('invoice')->select('invoice.*', 'users.first_name', 'users.last_name', 'users.mobile','users.id as userid')
                ->join('users', 'users.id', '=', 'invoice.student_id')
                ->where('status', '=', $data['status'])
                ->when(!empty($student), function ($query) use ($student) {
                    return $query->Where(function ($query) use ($student) {
                        $query->Where('first_name', 'like', "$student")
                            ->orWhere('users.last_name', 'like', $student)
                            ->orWhereRaw("CONCAT(`first_name`, ' ', `last_name`) LIKE ?", [$student])
                            ->orWhere('invoice.invoice_num', 'like', $student);
                    });
                })
                ->orderBy('status', "ASC")
                ->orderBy('updated_at', "DESC")
                ->whereNull('deleted_at')
                ->get();
        } elseif (isset($data['search']) && strlen($data['search'])) {
            $filters['search'] = $data['search'];
            $student = "%" . $data['search'] . "%";
            $pages = DB::table('invoice')->select('invoice.*', 'users.first_name', 'users.last_name', 'users.mobile','users.id as userid')
                ->join('users', 'users.id', '=', 'invoice.student_id')
                // ->where('invoice_num', '=', $data['search'])
                ->when(!empty($student), function ($query) use ($student) {
                    return  $query->Where('users.first_name', 'like', $student)
                            ->orWhere('users.last_name', 'like', $student)
                            ->orWhereRaw("CONCAT(`first_name`, ' ', `last_name`) LIKE ?", [$student])
                            ->orWhere('invoice_num', 'like', $student);
                })
                // ->orWhere('first_name', 'like', $student)
                // ->orWhere('last_name', 'like', $student)
                ->orderBy('status', "ASC")
                ->orderBy('updated_at', "DESC")
                ->whereNull('deleted_at')
                ->get();
        } elseif (isset($data['status'])) {
            $filters['status'] = $data['status'];
            $pages = DB::table('invoice')->select('invoice.*', 'users.first_name', 'users.last_name', 'users.mobile','users.id as userid')
                ->join('users', 'users.id', '=', 'invoice.student_id')
                ->where('status', '=', $data['status'])
                ->orderBy('status', "ASC")
                ->orderBy('updated_at', "DESC")
                ->whereNull('deleted_at')
                ->get();
        } else {
            $pages = DB::table('invoice')
                ->select('invoice.*', 'users.first_name', 'users.last_name', 'users.mobile','users.id as userid')
                ->join('users', 'users.id', '=', 'invoice.student_id')
                ->orderBy('invoice.created_at', "DESC")
                ->whereNull('deleted_at')
                ->get();
        }
        return view('backend.invoice.student_invoice_list', ['pages' => $pages, 'filters' => $filters]);
    }

    public function view_student_invoice($id)
    {
        $invoice = Invoice::find($id);
        $invoice_items = InvoiceItems::where("invoice_id", $invoice->id)->get();
        $stuData = User::find($invoice->student_id);
        $payments = InvoicePayment::
            //select(\DB::raw("SUM(amount) AS total_paid"), 'payment_date','pay_mode','trans_id','staff_id','amount')
            where('invoice_id', $invoice->id)
            ->get();

        $total_payment = InvoicePayment::where('invoice_id', $invoice->id)->sum('amount');

        return view('backend.invoice.student_invoice_detail', compact('invoice','invoice_items', 'stuData', 'payments', 'total_payment'));
    }

    public function student_invoice_add_payment($id)
    {
        $invoice = Invoice::find($id);

        $stuData = User::find($invoice->student_id);

        $payments = InvoicePayment::where('invoice_id', $invoice->id)->get();

        $total_payment = InvoicePayment::where('invoice_id', $invoice->id)->sum('amount');

        $staff = $this->userRepository->getNonStudentsTrainers();

        return view('backend.invoice.student_invoice_add_payment', compact('invoice', 'stuData', 'payments', 'total_payment', 'staff'));
    }

    public function student_invoice_store_payment(Request $request)
    {
        $request->validate([
            'amount' => 'required|numeric|gt:0',
            'payment_date' => 'required',
            'payment_mode' => 'required',
            'transaction_id' => 'required',
            'staff_id' => 'required'
        ]);

        $payment_date = "";
          if(!empty($request->get('payment_date'))){
              $payment_date = date('Y-m-d',strtotime($request->get('payment_date')));
          }

        $record = new InvoicePayment([
            'invoice_id' => $request->get('invoice_id'),
            'amount' => $request->get('amount'),
            'payment_date' => $payment_date,
            'pay_mode' => $request->get('payment_mode'),
            'trans_id' => $request->get('transaction_id'),
            'staff_id' => $request->get('staff_id')
        ]);
        $record->save();

        $invoice = Invoice::find($request->get('invoice_id'));
        $total_payment = InvoicePayment::where('invoice_id', $request->get('invoice_id'))->sum('amount');
        if ($invoice->grand_total - $total_payment <= 0) {
            $invoice->status = "paid";
            $invoice->save();
        }

        $student = User::where("id", $invoice->student_id)->first();
        // Send Push App Notification
        if(\Auth::user()->isAgent()){
            // $message = "Payment Record Added by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
            // if(is_staff_agent($invoice->student_id)){
            //     $message .= "Agent: " . $student->first_name." ". $student->last_name . " Email: " . $student->email;
            // } elseif(is_staff_user($invoice->student_id)){
            //     $message .= "Staff: " . $student->first_name." ". $student->last_name . " Email: " . $student->email;
            // } else{
            //     $message .= "Student: " . $student->first_name." ". $student->last_name." ID: ". $student->id . " Email: " . $student->email;
            // }
        } else{
            $message = "Payment Record for Invoice # ".$invoice->invoice_num." Added by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
            $email_message = "Payment Record for Invoice # ".$invoice->invoice_num." Added by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
            if(is_staff_agent($invoice->student_id)){
                $message .= "<br>Agent: " . $student->full_name;
                $email_message .= "<br>Agent: " . $student->full_name;
            } elseif(is_staff_user($invoice->student_id)){
                $message .= "<br>Staff: " . $student->full_name;
                $email_message .= "<br>Staff: " . $student->full_name;
            } else{
                $message .= "<br>"._label('backend_student').": " . $student->full_name." (". $student->id. ")";
                $email_message .= "<br>"._label('backend_student').": " . $student->full_name." (". $student->id. ")";
            }
        }

        //Sending email Notification
        $subject = "Payment Record for Invoice # ".$invoice->invoice_num;
        $email_message .= "<br>Amount: " .$request->get('amount');
        $email_message .= "<br>Payment Mode: " .$request->get('payment_mode');
        send_mail_notification(
            $student->id,
            "",
            "INVOICE_CREATE_AND_INVOICE_PAYMENT",
            $subject,
            $email_message
        );
        
        $message .= "<br>Amount: " .$request->get('amount');
        $message .= "<br>Payment Mode: " .$request->get('payment_mode');
        $subject = "";
        send_in_app_notification(
            $student->id, 
            $invoice->id,
            "INVOICE_CREATE_AND_INVOICE_PAYMENT", 
            $subject, 
            $message,
            "invoice"
        );

        return redirect()->route('admin.invoice.view-student-invoice', $request->get('invoice_id'))->withFlashSuccess('Payment Record Added Successfully');
    }

    public function add_uni_apply_agent_payment($id)
    {
        $invoice = UniInvoice::find($id);
        $agentData = UniApplyAgents::where('id', $invoice->uni_apply_agent_id)->first();
        $payments = UniInvoicePayment::where('invoice_id', $invoice->id)->get();
        $total_payment = UniInvoicePayment::where('invoice_id', $invoice->id)->sum('amount');

        $staff = $this->userRepository->getNonStudentsTrainers();

        return view('backend.invoice.add_uni_apply_agent_payment', compact('invoice', 'agentData', 'payments', 'total_payment', 'staff'));
    }

    public function store_uni_apply_agent_payment(Request $request)
    {
        $request->validate([
            'amount' => 'required',
            'payment_date' => 'required',
            'payment_mode' => 'required',
            'transaction_id' => 'required',
        ]);

        $payment_date = "";
          if(!empty($request->get('payment_date'))){
              $payment_date = date('Y-m-d',strtotime($request->get('payment_date')));
          }

        $record = new UniInvoicePayment([
            'invoice_id' => $request->get('invoice_id'),
            'amount' => $request->get('amount'),
            'payment_date' => $payment_date,
            'pay_mode' => $request->get('payment_mode'),
            'trans_id' => $request->get('transaction_id')
        ]);
        $record->save();

        $invoice = UniInvoice::find($request->get('invoice_id'));
        $total_payment = UniInvoicePayment::where('invoice_id', $request->get('invoice_id'))->sum('amount');
        if ($invoice->total - $total_payment <= 0) {
            $invoice->status = "paid";
            $invoice->save();
        }
        
        $invoiceDetails = UniInvoiceDetail::join('university_application', 'uni_invoice_detail.uni_application_id', '=', 'university_application.id')->where('invoice_id',$invoice->id)->whereNotNull('my_commission_percent')->get(['uni_invoice_detail.*','university_application.agent_commission_percent','university_application.agent_commission','university_application.my_commission_percent','university_application.userid']);

        if(count($invoiceDetails)){
                foreach($invoiceDetails as $inv_details){
                    $userData = get_user_by_id($inv_details->userid);
                    if($userData->agent_id != 0){
                        //agent commission
                            $agentCommission = new AgentCommission([
                            'appl_id' => $inv_details->uni_application_id,
                            'agent_id' => $userData->agent_id,
                            'amount' => $inv_details->total * ($inv_details->agent_commission_percent / $inv_details->my_commission_percent),
                            'created_by' => \Auth::user()->id,
                            ]);
                            $agentCommission->save();
                        //agent commission
                    }
                }
        }

        return redirect()->route('admin.invoice.view-uni-apply-agent-invoice', $request->get('invoice_id'))->withFlashSuccess('Uni Apply Agent Invoice Payment Record Added Successfully');
    }

    public function search_student(Request $request)
    {
        $my_branch = get_staff_branch_ids(\Auth::user()->id);
        $q =  $request->input('q', '');
        $users = User::where(function ($query) use ($q) {
                $query->where('first_name', 'LIKE', $q . '%')
                ->orwhere('last_name', 'LIKE', $q . '%')
                ->orWhere('email', '=', $q)
                ->orWhere('mobile', '=', $q);
            })
            ->whereIn('branch', $my_branch)
            ->select('id', DB::raw("concat(IF(first_name IS NOT NULL, CONCAT(first_name), ''), ' ', IF(last_name IS NOT NULL, CONCAT(last_name), ''), ' | ', IF(last_name IS NOT NULL, CONCAT(email), '')) as text"))
            ->get();
        return ['results' => $users,'branch' => $my_branch];
    }

    public function search_student_data(Request $request)
    {
        $users = User::where('id', $request->id)
            ->first();

        $address = "";
        if($users->address_1){
            $address .= $users->address_1 . ", ";
        }
        if($users->address_2){
            $address .= $users->address_2 . ", " ;
        }
        if($users->city){
            $address .= $users->city . ", " ;
        }
        if($users->state){
            $address .= get_state_name($users->state) . ", " ;
        }
        if($users->country){
            $address .= get_country_name($users->country) . ", ";
        }
        if($users->post_code){
            $address .= "-". $users->post_code ;
        }

        $user_currency = get_currency_by_user_country($users->id);
        $users["user_currency"] = get_currency_symbol($user_currency);

        $batch = Batch::join('batch_students', 'batch.id', '=', 'batch_students.batch')
            ->where('student', $request->id)
            ->get(['batch.id', 'batch.code']);

        return ['results' => $users, 'batch' => $batch,  'address' => $address];
    }

    public function getTax(Request $request){
		if($request->user_id && $request->total_amount){
			$tax_one_amt = $tax_two_amt = 0;
			$tax_one_name = $tax_two_name = $tax_one_rate = $tax_two_rate =  $tax_type = "";		
			$userData = User::where('id',$request->user_id)->first();
			$toti = $request->total_amount;
			$tax = 0;
			$taxData = TaxRates::where('country',$userData->country)->where('state',$userData->state)
				->orWhere(function ($query) use ($userData)  {
					$query->where('country',$userData->country)->where('state',0);
				})
				->orWhere(function ($query) use ($userData)  {
					$query->where('country',0)->where('state',$userData->state);
				})
				->orWhere(function ($query) {
					$query->where('country',0)->where('state',0);
				})
				->get();
			if(get_site_config('tax_exclusive')){
				if(count($taxData)){
					$tax_one_amt = round($toti * $taxData[0]->tax_rate/100,2);
					$tax_one_name = $taxData[0]->display_name;
					$tax_one_rate = $taxData[0]->tax_rate;
					if(isset($taxData[1])){
						$tax_two_amt = round($toti * $taxData[1]->tax_rate/100,2);
						$tax_two_name = $taxData[1]->display_name;
						$tax_two_rate = $taxData[1]->tax_rate;
					}
					$tax = round($tax_one_amt+$tax_two_amt,2);
				}else{
					$tax = 0;
					$tax_one_name = $tax_two_name = $tax_one_rate = $tax_two_rate = "";
					$tax_one_amt = $tax_two_amt = 0;
				}
				$tax_type = "exclusive";
			}
			if(get_site_config('tax_inclusive')){
				if(count($taxData)){
					$tax_one_amt = round($toti * $taxData[0]->tax_rate/100,2);
					$tax_one_name = $taxData[0]->display_name;
					$tax_one_rate = $taxData[0]->tax_rate;
					if(isset($taxData[1])){
						$tax_two_amt = round($toti * $taxData[1]->tax_rate/100,2);
						$tax_two_name = $taxData[1]->display_name;
						$tax_two_rate = $taxData[1]->tax_rate;
					}
					$tax = round($tax_one_amt+$tax_two_amt,2);
				}else{
					$tax = 0;
					$tax_one_name = $tax_two_name = $tax_one_rate = $tax_two_rate = "";
					$tax_one_amt = $tax_two_amt = 0;
				}
				$tax_type = "inclusive";
			}
		}else{
			$tax = 0;
			$tax_one_name = $tax_two_name = $tax_one_rate = $tax_two_rate = "";
			$tax_one_amt = $tax_two_amt = 0;
		}
		$users_currency = "";
		if($userData){
			$users_currency = get_currency_symbol(get_currency_by_user_country($userData->id));
		}
		
		return response()->json(['tax' => $tax, 'tax_one_amt' => $tax_one_amt,'tax_two_amt' => $tax_two_amt,'tax_one_name' => $tax_one_name,'tax_two_name' => $tax_two_name, 'tax_one_rate' => $tax_one_rate, 'tax_two_rate' =>$tax_two_rate,'users_currency' => $users_currency,'tax_type' => $tax_type]);
	}

    public function get_course(Request $request)
    {
        $batch = Batch::join('courses', 'batch.course', '=', 'courses.id')
            ->where('batch.id', $request->id)
            ->first(['courses.title', 'courses.fees', 'batch.code']);
        return $batch;
    }


    public function generate_student_invoice(Request $request, $user_id=null){
        $user = "";
        if($user_id){
            $user = User::find($user_id);
        }
        if($request->get('business')!=""){
            $current_business = $request->get('business');
        }else{
            $current_business = "1";
        }
        return view('backend.invoice.generate_invoice', ['user' => $user, 'current_business' => $current_business]);
    }

    public function store_student_invoice(Request $request){
        // $request->validate([
        //     'student_id' => 'required',
        //     'discount' => 'required',
        // ]);
        // $due_date = "";
        //   if(!empty($request->get('due_date'))){
        //       $due_date = date('Y-m-d',strtotime($request->get('due_date')));
        //   }
        // $inv_id = "";
        // while(1){
        //     $inv_id = random_int(100000, 999999);
        //     $allInv = DB::table('invoice')->where('invoice_num',$inv_id)->first();
        //     if($allInv){
        //         continue;
        //     }else{
        //         break;
        //     }
        // }

        // if($request->get('sub_total')){
        //     $sub_total = $request->get('sub_total');
        // } else{
        //     $sub_total = 0;
        // }
        // if($request->get('cart_total')){
        //     $cart_total = $request->get('cart_total');
        // } else{
        //     $cart_total = 0;
        // }

        // $record = new Invoice([
        //     'invoice_num' => $inv_id,
        //     'item1' => $request->get('item1'),
        //     'item2' => $request->get('item2'),
        //     'item3' => $request->get('item3'),
        //     'item4' => $request->get('item4'),
        //     'item5' => $request->get('item5'),
        //     'item6' => $request->get('item6'),
        //     'item7' => $request->get('item7'),
        //     'item8' => $request->get('item8'),
        //     'item9' => $request->get('item9'),
        //     'item10' => $request->get('item10'),
        //     'item11' => $request->get('item11'),
        //     'item12' => $request->get('item12'),
        //     'item13' => $request->get('item13'),
        //     'item14' => $request->get('item14'),
        //     'item15' => $request->get('item15'),
        //     'item16' => $request->get('item16'),
        //     'item17' => $request->get('item17'),
        //     'item18' => $request->get('item18'),
        //     'item19' => $request->get('item19'),
        //     'item20' => $request->get('item20'),
        //     'item21' => $request->get('item21'),

        //     'item1_amount' => $request->get('item1amount'),
        //     'item2_amount' => $request->get('item2amount'),
        //     'item3_amount' => $request->get('item3amount'),
        //     'item4_amount' => $request->get('item4amount'),
        //     'item5_amount' => $request->get('item5amount'),
        //     'item6_amount' => $request->get('item6amount'),
        //     'item7_amount' => $request->get('item7amount'),
        //     'item8_amount' => $request->get('item8amount'),
        //     'item9_amount' => $request->get('item9amount'),
        //     'item10_amount' => $request->get('item10amount'),
        //     'item11_amount' => $request->get('item11amount'),
        //     'item12_amount' => $request->get('item12amount'),
        //     'item13_amount' => $request->get('item13amount'),
        //     'item14_amount' => $request->get('item14amount'),
        //     'item15_amount' => $request->get('item15amount'),
        //     'item16_amount' => $request->get('item16amount'),
        //     'item17_amount' => $request->get('item17amount'),
        //     'item18_amount' => $request->get('item18amount'),
        //     'item19_amount' => $request->get('item19amount'),
        //     'item20_amount' => $request->get('item20amount'),
        //     'item21_amount' => $request->get('item21amount'),

        //     'tax1_amount' => $request->get('tax1_amount'),
        //     'tax2_amount' => $request->get('tax2_amount'),
        //     'tax1_name' => $request->get('tax1_name'),
        //     'tax2_name' => $request->get('tax2_name'),

        //     'student_id' => $request->get('student_id'),
        //     'batch_id' => $request->get('batch_id') ?? 0,
        //     'due_date' => $due_date,
        //     'discount' => $request->get('discount'),
        //     'sub_total' => $sub_total,
        //     'cart_total' => $cart_total,
        //     'terms_conditions' => $request->get('terms_conditions')
        // ]);

        // $record->save();
//-------------------------------------New code--------------------------------------------
        $request->validate([
            'business_id' => 'required',
            'student_id' => 'required',
            'discount' => 'required',
            'grand_total' => 'required',
        ]);

        $due_date = "";
        if(!empty($request->get('due_date'))){
            $due_date = date('Y-m-d',strtotime($request->get('due_date')));
        }
        $inv_id = "";
        while(1){
            $inv_id = random_int(100000, 999999);
            $allInv = DB::table('invoice')->where('invoice_num',$inv_id)->first();
            if($allInv){
                continue;
            }else{
                break;
            }
        }

        if($request->get('sub_total')){
            $sub_total = $request->get('sub_total');
        } else{
            $sub_total = 0;
        }

        $user_currency = get_currency_by_user_country($request->student_id);
        $invoice_count = $request->invoiceCount;
        $total_tax = $request->get('tax_one_amt') + $request->get('tax_two_amt');
        
        $record = new Invoice([
            'invoice_num' => $inv_id,
            'business_id' => $request->get('business_id'),
            'student_id' => $request->get('student_id'),
            'currency' => $user_currency,
            'tax_type' => $request->tax_type,
            'tax1_value' => $request->get('tax_one_amt'),
            'tax1_name' => $request->get('tax_one_name'),
            'tax1_rate' => $request->get('tax_one_rate'),
            'tax1_value_base' => currency_convert($request->get('tax_one_amt'), $user_currency, config('app.currency')),

            'tax2_value' => $request->get('tax_two_amt'),
            'tax2_name' => $request->get('tax_two_name'),
            'tax2_rate' => $request->get('tax_two_rate'),
            'tax2_value_base' => currency_convert($request->get('tax_two_amount'), $user_currency, config('app.currency')),

            'total_tax' => $total_tax,
            'total_tax_base' => currency_convert($total_tax, $user_currency, config('app.currency')),

            'due_date' => $due_date,

            'discount' => $request->get('discount'),
            'discount_base' => currency_convert($request->get('discount'), $user_currency, config('app.currency')),

            'sub_total' => $sub_total,
            'sub_total_base' => currency_convert($sub_total, $user_currency, config('app.currency')),

            'grand_total' => $request->get('grand_total'),
            'grand_total_base' => currency_convert($request->get('grand_total'), $user_currency, config('app.currency')),
            'terms_conditions' => $request->get('terms_conditions')
        ]);
        $record->save();

        $no_of_items = 0;
        $email_order_content = "";
        for($i = 1; $i <= $invoice_count; $i++){
            ${"item$i"} = "";
            ${"item$i"."_amount_base"} = currency_convert($request->{"item$i"."amount"}, $user_currency, config('app.currency'));
            ${"item$i"."qty"} = 1;
            ${"item$i"."_amount"} = $request->{"item$i"."amount"};
            if(isset($request->{"item$i"}) && !empty($request->{"item$i"})){                
                $invoice_item = new InvoiceItems;
                $invoice_item->invoice_id = $record->id;
                $invoice_item->currency = $user_currency;
                $invoice_item->item = $request->{"item$i"};
                $invoice_item->item_amount = $request->{"item$i"."amount"};
                $invoice_item->item_amount_base = ${"item$i"."_amount_base"};
                $invoice_item->save();	
                $email_order_content .= "Item".$i.": " . ${"item$i"} . "<br>Item".$i." Price: "  . $request->user_currency . ${"item$i"."_amount"} . "<br><br>";
                $no_of_items += 1;
            }
        }
//-----------------------------------------New code--------------------------------------------

        $student = User::where("id", $request->get('student_id'))->first();
        // Send Push App Notification
        if(\Auth::user()->isAgent()){
            // $message = "Invoice Created by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
            // if(is_staff_agent($request->get('student_id'))){
            //     $message .= "Agent: " . $student->first_name." ". $student->last_name . " Email: " . $student->email;
            // } elseif(is_staff_user($request->get('student_id'))){
            //     $message .= "Staff: " . $student->first_name." ". $student->last_name . " Email: " . $student->email;
            // } else{
            //     $message .= "Student: " . $student->first_name." ". $student->last_name." ID: ". $student->id . " Email: " . $student->email;
            // }
        } else{
            $message = "Invoice #".$inv_id." Created by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
            $email_message = "Invoice #".$inv_id." Created by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
            if(is_staff_agent($request->get('student_id'))){
                $message .= "<br>Agent: " . $student->full_name;
                $email_message .= "<br>Agent: " . $student->full_name;
            } elseif(is_staff_user($request->get('student_id'))){
                $message .= "<br>Staff: " . $student->full_name;
                $email_message .= "<br>Staff: " . $student->full_name;
            } else{
                $message .= "<br>"._label('backend_student').": " . $student->full_name." (". $student->id. ")";
                $email_message .= "<br>"._label('backend_student').": " . $student->full_name." (". $student->id. ")";
            }
        }

        //Sending email Notification
        $subject = "Invoice #".$inv_id." Created";
        $email_message .= "<br>Invoice Detail: " . $email_order_content . "<br>Amount: " .$record->sub_total;
        send_mail_notification(
            $student->id,
            "",
            "INVOICE_CREATE_AND_INVOICE_PAYMENT",
            $subject,
            $email_message
        );

        $message .= "<br>Invoice Detail: " . $email_order_content . "<br>Amount: " .$record->sub_total;
        $subject = "";
        send_in_app_notification(
            $student->id, 
            $record->id,
            "INVOICE_CREATE_AND_INVOICE_PAYMENT", 
            $subject, 
            $message,
            "invoice"
        );

        if ($request->input('send_invoice')) {
            $stuData = User::find($request->get('student_id'));
            $payments = InvoicePayment::where('invoice_id', $record->id)->get();
            $total_payment = InvoicePayment::where('invoice_id', $record->id)->sum('amount');
            $invoice_items = InvoiceItems::where("invoice_id", $record->id)->get();

            $filename = "Invoice_" . $record->invoice_num;

            if ($stuData->email) {
                $user['to'] = $stuData->email;
                $user['subject'] = "Invoice #" . $record->invoice_num . " from " . get_site_config('invoice_business_name');
                $data = ['invoice' => $record, 'stuData' => $stuData, 'payments' => $payments, 'total_payment' => $total_payment, 'invoice_items' => $invoice_items];
                $pdf = PDF::loadView('backend.invoice_pdf', $data);
                try{
                Mail::send('email.invoice', $data, function ($messages) use ($user, $filename, $pdf) {
                    $messages->to($user['to']);
                    $messages->subject($user['subject']);
                    $messages->attachData($pdf->output(), $filename . ".pdf");
                });
                }catch (\Exception $e) {
                    $message = $e->getMessage();
                    \Log::debug("Mail Error: (Email Invoice) ".$message);   
                }
                return redirect()->route('admin.invoice.student-invoices')->withFlashSuccess('Invoice Generated and sent to client!');
            }
        }

        return redirect()->route('admin.invoice.student-invoices')->withFlashSuccess('Invoice Generated!');
    }

    public function delete_student_invoice($id){
        if (\Auth::user()->id != 1) {
            return redirect()->back()->withFlashDanger('You do not have permission to delete invoice.');
        }

        $invoice = DB::table('invoice')->where('id',$id)->first();
        $invoice_delete = DB::table('invoice')->where('id',$id)->delete();

        if($invoice_delete){
            // Send Push App Notification
            $student = User::where("id", $invoice->student_id)->first();
            if(\Auth::user()->isAgent()){
                // $message = "Invoice #".$invoice->invoice_num." Deleted by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
                // $message .= ", Student: " . $student->first_name." ". $student->last_name . " Email: " . $student->email;
            } else{
                $message = "Invoice #".$invoice->invoice_num." Deleted by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
                $message .= "<br>"._label('backend_student').": " . $student->full_name." (". $student->id. ")";

                $email_message = "Invoice #".$invoice->invoice_num." Deleted by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
                $email_message .= "<br>"._label('backend_student').": " . $student->full_name." (". $student->id. ")";
            }

            //Sending email Notification
            $subject = "Invoice #".$invoice->invoice_num." Deleted";
            send_mail_notification(
                $student->id,
                "",
                "INVOICE_DELETE",
                $subject,
                $email_message
            );

            $subject = "";
            send_in_app_notification(
                $student->id, 
                $invoice->id,
                "INVOICE_DELETE", 
                $subject, 
                $message,
                "invoice"
            );
        }

        if($invoice){
            InvoicePayment::where('invoice_id',$id)->delete();
            return redirect()->route('admin.invoice.student-invoices')->withFlashSuccess('Student Invoice Deleted Successfully');          
        }     
    }

    public function edit_student_invoice($id){
        if(!(\Auth::user()->isAdmin() || \Auth::user()->can('edit-invoice'))){
            return redirect()->back()->withFlashDanger('You do not have permission to Edit invoice.');
        }

        $invoice = DB::table('invoice')->where('id',$id)->first();
        $stuData = User::find($invoice->student_id);
        $invoice_items = InvoiceItems::where("invoice_id", $invoice->id)->get();
        if($invoice->business_id){
            $current_business = $invoice->business_id;
            $get_business_details = get_business_details($invoice->business_id);
        }else{
            $current_business = "1";
            $get_business_details = get_business_details(1);
        }

        return view('backend.invoice.edit_invoice', [
            'invoice' => $invoice, 'stuData' => $stuData, "invoice_items" => $invoice_items, 
            'current_business' => $current_business, 'get_business_details' => $get_business_details,
        ]);
    }

    public function edit_student_invoice_store(Request $request){
        // dd($request->all());
        if(!(\Auth::user()->isAdmin() || \Auth::user()->can('edit-invoice'))){
            return redirect()->back()->withFlashDanger('You do not have permission to Edit invoice.');
        }
        
        $request->validate([
            'business_id' => 'required',
            'student_id' => 'required',
            'discount' => 'required',
            'grand_total' => 'required',
        ]);

        $due_date = "";
        if(!empty($request->get('due_date'))){
            $due_date = date('Y-m-d',strtotime($request->get('due_date')));
        }

        if($request->get('sub_total')){
            $sub_total = $request->get('sub_total');
        } else{
            $sub_total = 0;
        }

        $user_currency = get_currency_by_user_country($request->student_id);
        $invoice_count = $request->invoiceCount;
        $total_tax = $request->get('tax_one_amt') + $request->get('tax_two_amt');
        
        Invoice::where('id', $request->invoice_id)->update([
            'business_id' => $request->get('business_id'),
            'student_id' => $request->get('student_id'),
            'currency' => $user_currency,
            'tax_type' => $request->tax_type,
            'tax1_value' => $request->get('tax_one_amt'),
            'tax1_name' => $request->get('tax_one_name'),
            'tax1_rate' => $request->get('tax_one_rate'),
            'tax1_value_base' => currency_convert($request->get('tax_one_amt'), $user_currency, config('app.currency')),

            'tax2_value' => $request->get('tax_two_amt'),
            'tax2_name' => $request->get('tax_two_name'),
            'tax2_rate' => $request->get('tax_two_rate'),
            'tax2_value_base' => currency_convert($request->get('tax_two_amount'), $user_currency, config('app.currency')),

            'total_tax' => $total_tax,
            'total_tax_base' => currency_convert($total_tax, $user_currency, config('app.currency')),

            'due_date' => $due_date,

            'discount' => $request->get('discount'),
            'discount_base' => currency_convert($request->get('discount'), $user_currency, config('app.currency')),

            'sub_total' => $sub_total,
            'sub_total_base' => currency_convert($sub_total, $user_currency, config('app.currency')),

            'grand_total' => $request->get('grand_total'),
            'grand_total_base' => currency_convert($request->get('grand_total'), $user_currency, config('app.currency')),
            'terms_conditions' => $request->get('terms_conditions')
        ]);

        $record = Invoice::where('id', $request->invoice_id)->first();
        $inv_id = $record->invoice_id;
        InvoiceItems::where('invoice_id', $record->id)->delete();

        $no_of_items = 0;
        $email_order_content = "";
        for($i = 1; $i <= $invoice_count; $i++){
            ${"item$i"} = "";
            ${"item$i"."_amount_base"} = currency_convert($request->{"item$i"."amount"}, $user_currency, config('app.currency'));
            ${"item$i"."qty"} = 1;
            ${"item$i"."_amount"} = $request->{"item$i"."amount"};
            if(isset($request->{"item$i"}) && !empty($request->{"item$i"})){                
                $invoice_item = new InvoiceItems;
                $invoice_item->invoice_id = $record->id;
                $invoice_item->currency = $user_currency;
                $invoice_item->item = $request->{"item$i"};
                $invoice_item->item_amount = $request->{"item$i"."amount"};
                $invoice_item->item_amount_base = ${"item$i"."_amount_base"};
                $invoice_item->save();	
                $email_order_content .= "Item".$i.": " . ${"item$i"} . "<br>Item".$i." Price: "  . $request->user_currency . ${"item$i"."_amount"} . "<br><br>";
                $no_of_items += 1;
            }
        }

        $student = User::where("id", $request->get('student_id'))->first();
        // Send Push App Notification
        $message = "Invoice #".$inv_id." Updated by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
        $email_message = "Invoice #".$inv_id." Updated by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
        if(is_staff_agent($request->get('student_id'))){
            $message .= "<br>Agent: " . $student->full_name;
            $email_message .= "<br>Agent: " . $student->full_name;
        } elseif(is_staff_user($request->get('student_id'))){
            $message .= "<br>Staff: " . $student->full_name;
            $email_message .= "<br>Staff: " . $student->full_name;
        } else{
            $message .= "<br>"._label('backend_student').": " . $student->full_name." (". $student->id. ")";
            $email_message .= "<br>"._label('backend_student').": " . $student->full_name." (". $student->id. ")";
        }

        //Sending email Notification
        $subject = "Invoice #".$inv_id." Updated";
        $email_message .= "<br>Invoice Detail: " . $email_order_content . "<br>Amount: " .$record->sub_total;
        send_mail_notification(
            $student->id,
            "",
            "INVOICE_CREATE_AND_INVOICE_PAYMENT",
            $subject,
            $email_message
        );

        $message .= "<br>Invoice Detail: " . $email_order_content . "<br>Amount: " .$record->sub_total;
        $subject = "";
        send_in_app_notification(
            $student->id, 
            $record->id,
            "INVOICE_CREATE_AND_INVOICE_PAYMENT", 
            $subject, 
            $message,
            "invoice"
        );

        if ($request->input('send_invoice')) {
            $stuData = User::find($request->get('student_id'));
            $payments = InvoicePayment::where('invoice_id', $record->id)->get();
            $total_payment = InvoicePayment::where('invoice_id', $record->id)->sum('amount');
            $invoice_items = InvoiceItems::where("invoice_id", $record->id)->get();

            $filename = "Invoice_" . $record->invoice_num;

            if ($stuData->email) {
                $user['to'] = $stuData->email;
                $user['subject'] = "Invoice #" . $record->invoice_num . " from " . get_site_config('invoice_business_name');
                $data = ['invoice' => $record, 'stuData' => $stuData, 'payments' => $payments, 'total_payment' => $total_payment, 'invoice_items' => $invoice_items];
                $pdf = PDF::loadView('backend.invoice_pdf', $data);
                try{
                    Mail::send('email.invoice', $data, function ($messages) use ($user, $filename, $pdf) {
                        $messages->to($user['to']);
                        $messages->subject($user['subject']);
                        $messages->attachData($pdf->output(), $filename . ".pdf");
                    });
                }catch (\Exception $e) {
                    $message = $e->getMessage();
                    \Log::debug("Mail Error: (Email Invoice) ".$message);   
                }

                return redirect()->route('admin.invoice.student-invoices')->withFlashSuccess('Invoice Updated and sent to client!');
            }
        }

        return redirect()->route('admin.invoice.student-invoices')->withFlashSuccess('Invoice Updated Successfully');  
    }

    public function remove_student_invoice_payment_record($id){
        $invoicePayment = InvoicePayment::where('id', $id)->first();
        if($invoicePayment){
            $invoice = Invoice::find($invoicePayment->invoice_id);
            $total_payment = InvoicePayment::where('invoice_id', $invoice->id)->sum('amount');
            $total_payment -= $invoicePayment->amount; 
            if (($invoice->grand_total - $total_payment) > 0) {
                $invoice->status = "due";
                $invoice->save();
            }
            $invoicePayment->delete();
            return redirect()->route('admin.invoice.view-student-invoice', $invoice->id)->withFlashSuccess('Payment Record Deleted Successfully');
        } else{
            return redirect()->route('admin.invoice.view-student-invoice', $id)->withFlashDanger('Payment Record Not Found!');
        }
    }

    public function remove_uni_payment($id){
        $invoicePayment = UniInvoicePayment::where('id', $id)->first();
        if($invoicePayment){
            $invoice = UniInvoice::find($invoicePayment->invoice_id);
            $total_payment = UniInvoicePayment::where('invoice_id', $invoice->id)->sum('amount');
            $total_payment -= $invoicePayment->amount; 
            if (($invoice->total - $total_payment) > 0) {
                $invoice->status = "due";
                $invoice->save();
            }
            $invoicePayment->delete();
            return redirect()->route('admin.invoice.view-uni-invoice', $invoice->id)->withFlashSuccess('Payment Record Deleted Successfully');
        } else{
            return redirect()->route('admin.invoice.view-uni-invoice', $id)->withFlashDanger('Payment Record Not Found!');
        }
    }

    public function remove_uni_apply_agent_payment($id){
        $invoicePayment = UniInvoicePayment::where('id', $id)->first();
        if($invoicePayment){
            $invoice = UniInvoice::find($invoicePayment->invoice_id);
            $total_payment = UniInvoicePayment::where('invoice_id', $invoice->id)->sum('amount');
            $total_payment -= $invoicePayment->amount; 
            if (($invoice->total - $total_payment) > 0) {
                $invoice->status = "due";
                $invoice->save();
            }
            $invoicePayment->delete();
            return redirect()->route('admin.invoice.view-uni-apply-agent-invoice', $invoice->id)->withFlashSuccess('Payment Record Deleted Successfully');
        } else{
            return redirect()->route('admin.invoice.view-uni-apply-agent-invoice', $id)->withFlashDanger('Payment Record Not Found!');
        }
    }

    public function remove_ad_hoc_payment($id){
        $invoicePayment = UniInvoicePayment::where('id', $id)->first();
        if($invoicePayment){
            $invoice = UniInvoice::find($invoicePayment->invoice_id);
            $total_payment = UniInvoicePayment::where('invoice_id', $invoice->id)->sum('amount');
            $total_payment -= $invoicePayment->amount; 
            if (($invoice->total - $total_payment) > 0) {
                $invoice->status = "due";
                $invoice->save();
            }
            $invoicePayment->delete();
            return redirect()->route('admin.invoice.view-ad-hoc-invoice', $invoice->id)->withFlashSuccess('Payment Record Deleted Successfully');
        } else{
            return redirect()->route('admin.invoice.view-ad-hoc-invoice', $id)->withFlashDanger('Payment Record Not Found!');
        }
    }

    public function deleteUniInvoice($id){
        if (\Auth::user()->id != 1) {
            return redirect()->back()->withFlashDanger('You do not have permission to delete invoice.');
        }
        $invoice = DB::table('uni_invoice')->where('id',$id)->first();
        if($invoice){
            DB::table('uni_invoice')->where('id',$id)->delete();
            UniInvoicePayment::where('invoice_id',$id)->delete();
            return redirect()->route('admin.invoice.university-invoices')->withFlashSuccess('University Invoice Deleted Successfully');          
        }     
    }

    public function deleteUniApplyAgentInvoice($id){
        if (\Auth::user()->id != 1) {
            return redirect()->back()->withFlashDanger('You do not have permission to delete invoice.');
        }
        $invoice = DB::table('uni_invoice')->where('id',$id)->first();
        if($invoice){
            DB::table('uni_invoice')->where('id',$id)->delete();
            UniInvoicePayment::where('invoice_id',$id)->delete();
            return redirect()->route('admin.invoice.university-invoices')->withFlashSuccess('Uni Apply Agent Invoice Deleted Successfully');          
        }     
    }

    public function deleteAdhocInvoice($id){
        if (\Auth::user()->id != 1) {
            return redirect()->back()->withFlashDanger('You do not have permission to delete invoice.');
        }
        $invoice = DB::table('uni_invoice')->where('id',$id)->first();
        if($invoice){
            DB::table('uni_invoice')->where('id',$id)->delete();
            UniInvoicePayment::where('invoice_id',$id)->delete();
            return redirect()->route('admin.invoice.university-invoices')->withFlashSuccess('Ad-hoc Invoice Deleted Successfully');          
        }     
    }
}
