<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\Auth\User;
use App\Models\Auth\StaffLeaves;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use Validator;
use DateTime;
use App\Mail\MailSend;
use Illuminate\Support\Facades\Mail;
use Log;


/**
 * Class LeavesManagementController.
 */
class LeavesManagementController extends Controller
{

    public function __construct()
    {
        // $this->middleware(function ($request, $next) {
        //     $user = \Auth::user();
        //     if (!\Auth::user()->isAdmin()) {
        //         return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        //     }
        //     return $next($request);
        // });
        
        // $this->middleware(function ($request, $next) {
        //     $user = \Auth::user();
        //     if (!(\Auth::user()->isAdmin() || \Auth::user()->can('staff leaves'))) {
        //         return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        //     }
        //     return $next($request);
        // });
    }

    public function index()
    {
        $my_leaves = StaffLeaves::where('staff_id', \Auth::user()->id)->where('leave_date', 'like', date("Y-") . '%')->get();
        $total_leaves = StaffLeaves::where('staff_id', \Auth::user()->id)
            ->where('leave_date', 'like', date("Y-") . '%')
            ->where('approved', 1)
            ->sum('days');
        $year = date("Y");
        return view('backend.staff_leaves.my_leaves', compact('my_leaves', 'total_leaves', 'year'));
    }

    public function apply()
    {
        return view('backend.staff_leaves.apply');
    }

    public function store(Request $request)
    {
        $my_id = \Auth::user()->id;
        $my_name = \Auth::user()->full_name;

        \Log::debug("LeaveManagementController:store:$my_id:" . var_export($request->all(), true));

        $messages = [
            'date_from.required_if' => 'The Date From field is required when duration is Multiple Days',
            'date_to.required_if' => 'The Date To field is required when duration is Multiple Days',
        ];

        $validator = Validator::make($request->all(), [
            'duration' => 'required',
            'reason' => 'required|max:200',
            'type' => 'required',
            'date' => 'nullable|required_if:duration,single|date',
            'date_from' => 'nullable|required_if:duration,multi|date',
            'date_to' => 'nullable|required_if:duration,multi|date|after:date_from',
        ], $messages);

        if ($validator->fails()) {
            return redirect(route('admin.my-leaves.apply'))
                ->withErrors($validator)
                ->withInput();
        }

        $req_id = StaffLeaves::max('request_id');
        if ($req_id) {
            $req_id++;
        } else {
            $req_id = 1;
        }

        $total_days = 0;

        if ($request->input('duration') == 'multi') {
            //Multiple days leave
            $from = new DateTime($request->input('date_from'));
            $to = new DateTime($request->input('date_to'));
            $interval = $from->diff($to, true);
            $days = $interval->format('%a');
            if ($days > 31) {
                return redirect(route('admin.my-leaves.apply'))
                    ->withFlashDanger("Date difference cannot exceed 31 days")
                    ->withInput();
            }

            $today = new DateTime();
            $interval = $today->diff($to, true);
            $days = $interval->format('%a');
            if ($days > 90) {
                return redirect(route('admin.my-leaves.apply'))
                    ->withFlashDanger("Date must be within next 90 days")
                    ->withInput();
            }

            //save
            for ($i = $from; $i <= $to; $i->modify('+1 day')) {
                \Log::debug("Checking Date:" . $i->format("Y-m-d"));

                //check if weekly off or holiday
                if (in_array($i->format("l"), explode(',',env('WEEKLY_OFF')) )) {
                    \Log::debug("-Weekly off. skip");
                    continue;
                }
                //check if holiday
                if (DB::table('holidays')->where('date', $i->format("Y-m-d"))->exists()) {
                    \Log::debug("-Holidy. skip");
                    continue;
                }

                if ($i == $from && $request->input('multi_duration') == 'second_half') {
                    $when = $request->input('multi_duration');
                    $days = 0.5;
                } elseif ($i == $to && $request->input('multi_duration') == 'first_half') {
                    $when = $request->input('multi_duration');
                    $days = 0.5;
                } else {
                    $when = 'full_day';
                    $days = 1;
                }

                $StaffLeaves = StaffLeaves::create([
                    "staff_id" => $my_id,
                    "leave_date" => $i->format("Y-m-d"),
                    "days" => $days,
                    "when" => $when,
                    "reason" => $request->input('reason'),
                    "type" => $request->input('type'),
                    "request_id" => $req_id,
                ]);
                $total_days += $days;

                // Send Push App Notification
                //$staffLeaves = StaffLeaves::latest();
                //dd($newLeave->id);
                $subject = "";
                $message = "New Leave Request by Staff: " . get_user_full_name_by_id(\Auth::user()->id) .
                    " No. of Days: " . $total_days;
                send_in_app_notification(
                    $my_id, 
                    $StaffLeaves->id, 
                    "STAFF_LEAVES_APPLY", 
                    $subject, 
                    $message,
                    "staff_leaves", 
                );
            }

            //Sending email Notification
            $subject = "New Leave Requested by: " . $my_name;
            $email_message = "Dear Admin,<br />New Leave Request by: $my_name<br />No. of Days: $total_days";
            send_mail_notification($my_id,"","STAFF_LEAVES_APPLY",$subject,$email_message);
            // $this->notifyAdmin($message);

            return redirect()->route('admin.my-leaves')->withFlashSuccess("Leave applied for $total_days days.");
        } elseif ($request->input('duration') == 'single') {
            //Single day leave
            $dt = new DateTime($request->input('date'));

            //check if weekly off or holiday
            if (in_array($dt->format("l"),  explode(',',env('WEEKLY_OFF')) )) {
                \Log::debug("-Weekly off. skip");
                return redirect(route('admin.my-leaves.apply'))->withFlashDanger("The Date is a weekly off day")->withInput();
            }
            //check if holiday
            if (DB::table('holidays')->where('date', $dt->format("Y-m-d"))->exists()) {
                \Log::debug("-Holiday. skip");
                return redirect(route('admin.my-leaves.apply'))->withFlashDanger("The Date is a holiday")->withInput();
            }

            $when = $request->input('single_duration');
            if ($when == 'first_half' || $when == 'second_half') {
                $days = 0.5;
            } else {
                $days = 1;
            }
            $StaffLeaves = StaffLeaves::create([
                "staff_id" => $my_id,
                "leave_date" => $dt->format("Y-m-d"),
                "days" => $days,
                "when" => $when,
                "reason" => $request->input('reason'),
                "type" => $request->input('type'),
                "request_id" => $req_id,
            ]);

            // Send Push App Notification
            //$StaffLeaves = StaffLeaves::latest('id')->first();
            $subject = "";
            $message = "New Leave Request by Staff: " . get_user_full_name_by_id(\Auth::user()->id) .
                " No. of Days: " . $total_days;
            send_in_app_notification(
                $my_id, 
                $StaffLeaves->id, 
                "STAFF_LEAVES_APPLY", 
                $subject, 
                $message,
                "staff_leaves", 
            );

            //Sending email Notification
            $subject = "New Leave Requested by: " . $my_name;
            $email_message = "Dear Admin,<br />New Leave Request by: $my_name<br />No. of Days: $total_days";
            send_mail_notification($my_id,"","STAFF_LEAVES_APPLY",$subject,$email_message);
            // $this->notifyAdmin($message);

            return redirect()->route('admin.my-leaves')->withFlashSuccess("Leave applied for $days day.");
        }
    }


    private function notifyStaff($staff, $message)
    {
        $user = User::find($staff);
        $message = "Dear " . $user->first_name . ",<br />" . $message;
        try{
            //Mail::to($user->email)->send(new MailSend("Your Leave Approved", $message));
            user_notify($user->email,"Your Leave Approved", $message);
        }
        catch (\Exception $e) {
            $message = $e->getMessage();
            \Log::debug("Mail Error: (Leaves Management STAFF_EMAIL: ) ".$message);   
        }
    }

    public function staffLeaves(){
        if (!(\Auth::user()->isAdmin() || \Auth::user()->can('staff leaves'))) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }

        if(!env('STAFF_LEAVES_AND_ATTENDANCE_ENABLE')){
            return redirect()->route('admin.dashboard.message')->withFlashDanger('Staff Leaves is disabled.');
        }
        $filters = array("status" => "", "search" => "", "daterange" => "");
        return view('backend.staff_leaves.index', compact('filters'));
    }

    public function ajaxstaffleaves(Request $request)
    {        
        if (!$request->ajax()) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this page.');
        }
        $columns = array(
            0 => 'created_at',
            1 => 'first_name',
            2 => 'leave_dates',
            3 => 'days',
            4 => 'type',
            5 => 'reason',
            6 => 'actions',
        );

        $totalData = DB::table('staff_leaves')->groupBy('request_id')->get()->count();
        //$totalFiltered = $totalData;
        $limit = $request->input('length');
        $start = $request->input('start');
        $order = $columns[$request->input('order.0.column')];
        $dir = $request->input('order.0.dir');
        $status = $request->input('status');
        $daterange = $request->input('daterange');

        DB::enableQueryLog();

        $get_search = $request->input('search.value');
        $datarows = DB::table('staff_leaves')
            ->select('staff_leaves.*', 'users.first_name', 'users.last_name')
            ->join('users', 'users.id', '=', 'staff_leaves.staff_id')
            ->when($status != "", function ($query) use ($status) {
                return $query->Where(function ($query) use ($status) {
                    $query->where('approved', $status);
                });
            })
            ->when(!empty($get_search), function ($query) use ($get_search) {
                return $query->Where(function ($query) use ($get_search) {
                     $formatted_date = null;
                    if (preg_match('/\d{2}-[A-Za-z]{3}-\d{4}/', $get_search)) {
                        $formatted_date = date('Y-m-d', strtotime($get_search));
                    }
                    $query->Where('first_name', 'like', "%$get_search%")
                        ->orWhere('last_name', 'like', "%$get_search%")
                        ->orWhereRaw("CONCAT(first_name, ' ', last_name) LIKE ?", ["%$get_search%"])
                        ->orWhere('mobile', '=', "$get_search")
                        ->orWhere('email', 'like', "%$get_search%")
                        ->orWhere('type', '=', "$get_search")
                        ->orWhere('staff_leaves.created_at', 'like', "$get_search%");
                         if ($formatted_date) {
                            $query->orWhere('staff_leaves.leave_date', 'like', "$formatted_date%");
                        }
                });
            })
            ->when(!empty($daterange), function ($query) use ($daterange) {
                return $query->Where(function ($query) use ($daterange) {
                    $da = explode('to', $daterange);
                    $todate = $da[1];
                    $tdate = str_replace('/', '-', $todate);
                    $newtoDate = date("Y-m-d", strtotime($tdate));
                    $frmdate = $da[0];
                    $fdate = str_replace('/', '-', $frmdate);
                    $newfrmDate = date("Y-m-d", strtotime($fdate));
                    $query->whereBetween('leave_date', [$newfrmDate, $newtoDate]);
                });
            })
            ->groupBy('request_id')
            ->offset($start)
            ->limit($limit)
            ->orderBy($order, $dir)
            ->get();

        $queries = DB::getQueryLog();
        Log::debug(end($queries));

        $totalFiltered = DB::table('staff_leaves')
            ->join('users', 'users.id', '=', 'staff_leaves.staff_id')
            ->when($status != "", function ($query) use ($status) {
                return $query->Where(function ($query) use ($status) {
                    $query->where('approved', $status);
                });
            })
            ->when(!empty($get_search), function ($query) use ($get_search) {
                return $query->Where(function ($query) use ($get_search) {
                    $query->Where('first_name', 'like', "%$get_search%")
                        ->orWhere('last_name', 'like', "%$get_search%")
                        ->orWhereRaw("CONCAT(first_name, ' ', last_name) LIKE ?", ["%$get_search%"])
                        ->orWhere('mobile', '=', "$get_search")
                        ->orWhere('email', 'like', "%$get_search%")
                        ->orWhere('type', '=', "$get_search")
                        ->orWhere('staff_leaves.created_at', 'like', "$get_search%");
                });
            })
            ->when(!empty($daterange), function ($query) use ($daterange) {
                return $query->Where(function ($query) use ($daterange) {
                    $da = explode('to', $daterange);
                    $todate = $da[1];
                    $tdate = str_replace('/', '-', $todate);
                    $newtoDate = date("Y-m-d", strtotime($tdate));
                    $frmdate = $da[0];
                    $fdate = str_replace('/', '-', $frmdate);
                    $newfrmDate = date("Y-m-d", strtotime($fdate));
                    $query->whereBetween('leave_date', [$newfrmDate, $newtoDate]);
                });
            })
            ->groupBy('request_id')
            ->get()
            ->count();


        $data = $customResult = array();
        if (!empty($datarows)) {
            $count = 1;
            foreach ($datarows as $row) {
                // $customResult['id'] = $count++;
                $customResult['date'] = get_date_time_formated($row->created_at);
                $customResult['staff'] = $row->first_name . " " . $row->last_name;
                if ($row->approved)
                    $status = '<span class="badge badge-success">Approved</span>';
                else
                    $status = '<span class="badge badge-danger">Pending Approval</span>';

                $customResult['status'] = $status;

                $request_id = $row->request_id;
                $days = 0;
                $leave_dates = "";
                $leaves_rows = DB::table('staff_leaves')->where('request_id', $request_id)->orderBy('leave_date')->get();
                foreach ($leaves_rows as $leave_row) {
                    $leave_dates .= date("d-M-Y", strtotime($leave_row->leave_date));
                    if ($leave_row->when != 'full_day') {
                        $leave_dates .= " (" . ucfirst(str_replace("_", " ", $leave_row->when)) . ")";
                    }
                    $leave_dates .= ", ";
                    $days += $leave_row->days;
                }
                $customResult['leave_dates'] = $leave_dates;
                $customResult['days'] = $days;
                $customResult['type'] = $leave_row->type;
                $customResult['reason'] = $leave_row->reason;

                $actions = "";
                if ($leave_row->approved == 0) {
                    if (\Auth::user()->isAdmin() || \Auth::user()->can('Approve Leaves')){
                        $actions = '<button class="btn btn-sm btn-primary" onclick="doApprove(' . $request_id . ')" style="margin:2px;">Approve</button>';
                        $actions .= '<button class="btn btn-sm btn-danger" onclick="doReject(' . $request_id . ')" style="margin:2px;">Cancel</a>';
                    }else{
                        $actions = '<button class="btn btn-sm btn-secondary disabled" style="cursor: not-allowed" onclick="doApprove(' . $request_id . ')" style="margin:2px;">Approve</button>';
                        $actions .= '<button class="btn btn-sm btn-secondary disabled" onclick="doReject(' . $request_id . ')" style="margin:2px;">Cancel</a>';
                    }
                }
                $customResult['actions'] = $actions;

                $data[] = $customResult;
            }
        }

        $json_data = array(
            "draw"            => intval($request->input('draw')),
            "recordsTotal"    => intval($totalData),
            "recordsFiltered" => intval($totalFiltered),
            "data"            => $data,
        );

        echo json_encode($json_data);
    }

    public function approve(Request $request)
    {
        if (!(\Auth::user()->isAdmin() || \Auth::user()->can('Approve Leaves'))){
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }        
        if (StaffLeaves::where('request_id', $request->input('id'))->where('approved', 0)->count()) {
            $total_leaves = StaffLeaves::where('request_id', $request->input('id'))->sum('days');
            $row = StaffLeaves::where('request_id', $request->input('id'))->first();
            StaffLeaves::where('request_id', $request->input('id'))->update(['approved' => 1]);
            $this->notifyStaff($row->staff_id, "Your leave request for $total_leaves days is approved!");
            return redirect()->route("admin.staff-leaves")->withFlashSuccess("Leaves approved");
        } else {
            return redirect()->route("admin.staff-leaves")->withFlashDanger("Invalid request");
        }
    }

    public function reject(Request $request)
    {
        if (!(\Auth::user()->isAdmin() || \Auth::user()->can('Approved Leaves'))){
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }        
        if (StaffLeaves::where('request_id', $request->input('id'))->where('approved', 0)->count()) {
            StaffLeaves::where('request_id', $request->input('id'))->delete();
            return redirect()->route("admin.staff-leaves")->withFlashSuccess("Leaves declined");
        } else {
            return redirect()->route("admin.staff-leaves")->withFlashDanger("Invalid request");
        }
    }
}
