<?php

namespace App\Http\Controllers\Backend;

use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Auth\OnlineCourse;
use Illuminate\Support\Str;

class OnlineCoursesController extends Controller
{

    public function __construct()
    {
        $this->middleware(function ($request, $next) {
            $user = \Auth::user();
            if (!(\Auth::user()->isAdmin() || \Auth::user()->can('trainings configs'))) {
                return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
            }
            return $next($request);
        });
    }

    public function index()
    {
        $courses = OnlineCourse::with(['chapters' => function ($query) {
            $query->active();
        }])->orderBy("display_order", 'ASC')->paginate(10);
        return view('backend.online_course.coursesdetails', ['courses' => $courses]);
    }

    public function create()
    {
        return view('backend.online_course.addcoursepage');
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required',
            'short_description' => 'required',
            'display_order' => 'required|numeric',
            'image' => 'required|mimes:jpg,jpeg,gif,png',
            'price' => 'required|numeric',
            'offer_price' => 'required|numeric',
        ]);
        $imagePath = '';
        if ($request->file('image')) {
            $imagePath = OnlineCourse::uploadCourseImage($request->file('image'));
        }
        $data = array(
            'title' => $request->get('title'),
            'description' => $request->get('description'),
            'short_description' => $request->get('short_description'),
            'display_order' => $request->get('display_order'),
            'status' => $request->get('status'),
            'image' => $imagePath,
            'price' => $request->input('price'),
            'offer_price' => $request->input('offer_price')
        );
        OnlineCourse::addOrUpdateOnlineCourse($data);
        return redirect()->route('admin.onlinecourses')->withFlashSuccess('Online Course Added Successfully!');
    }

    public function edit($id)
    {
        $course = OnlineCourse::getCourseWithChaptersObjById($id);
        return view('backend.online_course.editcourse', ['course' => $course]);
    }

    public function update(Request $request, $id)
    {
        $request->validate([
            'title' => 'required',
            'short_description' => 'required',
            'display_order' => 'required|numeric',
            'price' => 'required|numeric',
            'offer_price' => 'required|numeric'
        ]);
        $courseObj = OnlineCourse::getCourseObjById($id);
        if (empty($courseObj)) {
            return redirect()->route('admin.onlinecourses')->withFlashDanger('Online Course id is invalid!');
        }
        $imagePath = $courseObj->image;
        if ($request->file('image')) {
            $oldImagePath = $courseObj->image;
            if (!empty($oldImagePath)) {
                $oldFilePath = storage_path('app/public') . '/' . $oldImagePath;
                if (file_exists($oldFilePath)) {
                    unlink($oldFilePath);
                }
            }
            $imagePath = OnlineCourse::uploadCourseImage($request->file('image'));
        }
        $data = array(
            'title' => $request->get('title'),
            'description' => $request->get('description'),
            'short_description' => $request->get('short_description'),
            'type' => $request->get('type'),
            'display_order' => $request->get('display_order'),
            'status' => $request->get('status'),
            'image' => $imagePath,
            'price' => $request->input('price'),
            'offer_price' => $request->input('offer_price')            
        );
        OnlineCourse::addOrUpdateOnlineCourse($data, $id);
        return redirect()->route('admin.onlinecourses')->withFlashSuccess('Online Course Information saved!');
    }
    public function destroy($id)
    {
        $course = OnlineCourse::getCourseObjById($id);
        if (!empty($course)) {
            $chapters = DB::table('online_course_chapters')->where('online_course_id',$course->id)->first();
            // if($course->chapters()) {
            if($chapters) {
                return redirect()->route('admin.onlinecourses')->withFlashDanger('This course has chapters, Could not delete Course');
            } else {
                $course->delete();
            }
            return redirect()->route('admin.onlinecourses')->withFlashSuccess('Online Course Deleted Successfully');
        }
        return redirect()->route('admin.onlinecourses')->withFlashDanger('Could not delete Course');
    }
}
