<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\Auth\Services;
use App\Models\Auth\VisaServicesActivity;
use App\Models\Auth\ServiceApplications;
use App\Models\Auth\ServiceApplicationMessage;
use App\Models\Auth\UniversityApplication;
use App\Models\Auth\Country;
use App\Models\Auth\Reminders;
use App\Repositories\Backend\Auth\StaffRepository;
use App\Models\Auth\User;
use App\Repositories\Backend\Auth\UserRepository;
use App\Mail\MailSend;
use Illuminate\Support\Facades\Mail;
use App\Recipients\AdminRecipient;
use App\Notifications\Frontend\AdminAlert;
use Illuminate\Support\Str;
use App\Models\Auth\ServiceStages;
use App\Models\Auth\UserNotification;
use App\Notifications\Backend\UserAlert;
use App\Models\UserActivity;

class ServiceController extends Controller{

    public function __construct(){
        $this->middleware(function ($request, $next) {
            $user = \Auth::user();
            if (!(\Auth::user()->isAdmin() || \Auth::user()->isAgent() || \Auth::user()->can('delete-visa-service-application'))) {
                return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
            }  
            
            if(!env('ADDITIONAL_SERVICES_ENABLE')){
                return redirect()->route('admin.dashboard.message')->withFlashDanger('Additional Services is disabled.');
            }

            return $next($request);
        });
    }

    // Settings ------------------------------------
    public function showServicesList(){
        $stages = DB::table('services')->OrderBy('display_order')->get();
        return view('backend.settings.service_list', compact('stages'));
    }

    public function servicesListStore(Request $request){
        // dd($request->all());
        $request->validate([
            'service_name' => 'required',
            'country_field' => 'required',
            'display_order' => 'required',
        ]);

        $services = new Services([
            'service_name' => $request->service_name,
            // 'visa_service' => $request->visa_service,
            'country_field' => $request->country_field,
            'amount' => $request->amount,
            'display_order' => $request->display_order,
        ]);
        
        if($services->save()){
            $data = [
                ['status_name'=>'New', 'service_id'=> $services->id, 'default_stage'=> 1, 'case_close_stage'=> 0, 'color'=> '#0AD7FE', 'stage_order'=> 99],
                ['status_name'=>'Processing', 'service_id'=> $services->id, 'default_stage'=> 0, 'case_close_stage'=> 0, 'color'=> '#FCB802', 'stage_order'=> 99],
                ['status_name'=>'Completed', 'service_id'=> $services->id, 'default_stage'=> 0, 'case_close_stage'=> 1, 'color'=> '#08B333', 'stage_order'=> 99],
            ];
            ServiceStages::insert($data);
        }

        return redirect()->route('admin.settings.services.list')->withFlashSuccess("Service Added Successfully!");
    }

    public function fetchStatus(Request $request)
    {
        $status = ServiceStages::where('service_id', $request->service_id)
        ->orderBy('stage_order')
        ->get(["status_name"]);
         if(count($status)>0){
          $data['status'] = $status;
          return response()->json($data);
         }else{
          $data['status'] = "No Status Found!";
          return response()->json($data);
         } 
    }

    public function servicesListEditStore(Request $request){
        $request->validate([
            'service_name' => 'required',
            'country_field' => 'required',
            'display_order' => 'required',
        ]);

        $services = Services::where('id', $request->service_id)->first();
        $services->service_name = $request->service_name;
        $services->country_field = $request->country_field;
        $services->amount = $request->amount;
        $services->display_order = $request->display_order;
        $services->save();
        
        return redirect()->route('admin.settings.services.list')->withFlashSuccess("Service Updated Successfully!");
    }
    
    public function servicesDestroy($id){
        $services = Services::where('id', $id)->first();   
        if($services){
            //check if application is present
            $aapl = DB::table('service_applications')->where('service_id', $services->id)->first();
            if($aapl){
                return redirect()->route('admin.settings.services.list')->withFlashDanger('Application is present for the service "'.get_services($services->id).'". Cannot delete this service!');
            }
            ServiceStages::where('service_id', $id)->delete();
            $services->delete();
            return redirect()->route('admin.settings.services.list')->withFlashSuccess("Service Deleted Successfully!");
        }else{
            return redirect()->route('admin.settings.services.list')->withFlashDanger("Service Not Found!");
        }
    }

    // ------------------------------ End Settings

    // Start User detail service page ----------------------------------------
    public function addServiceRequestStore(Request $request){
        $request->validate([
            'service_id' => 'required',
        ]);

        $service = DB::table('services')->where('id', $request->service_id)->first();
        if($service->country_field){
            $request->validate([
                'country' => 'required',
            ]);
        }

        $serviceApplications = new ServiceApplications([
            'user_id' => $request->user_id,
            'service_id' => $request->service_id,
            'country' => $request->country,
            'college_appl_id' => $request->college_appl_id,
            'amount' => $request->amount,
            'description' => $request->description,
        ]);
        
        // if($request->service_id == get_study_visa()){
        //     $college_appl_id = UniversityApplication::where('userid', $request->user_id)->first();
        //     if($college_appl_id){
        //         $serviceApplications->college_appl_id = $college_appl_id->id;
        //     }
        // }
        $applId = null;
        while(1){
            $applId = random_int(100000, 999999);
            $allServiceApplId = DB::table('service_applications')->where('appl_id', $applId)->first();
            if($allServiceApplId){
                continue;
            }else{
                break;
            }
        }
        $serviceApplications->appl_id = $applId;
        $serviceApplications->save();

        $student = User::where("id", $serviceApplications->user_id)->first();
        //Sending email Notification
        $subject = "Visa & Service Application #".$serviceApplications->appl_id." Created by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
        $email_message = "Hi, <br><br>Visa & Service Application #".$serviceApplications->appl_id." Created by Staff: " . get_user_full_name_by_id(\Auth::user()->id) . 
        "<br>Service: " . get_services($serviceApplications->service_id).
        "<br>Country: " . get_country_name($serviceApplications->country).
        "<br>"._label('backend_student').": " . $student->full_name." (". $student->id. ")";
        send_mail_notification($serviceApplications->user_id,$serviceApplications->id,"VISA_AND_SERVICES_APPLICATION_CREATED", $subject, $email_message, true);

        // Send Push App Notification
        $message = "Visa & Service Application #".$serviceApplications->appl_id." Created by Staff: " . get_user_full_name_by_id(\Auth::user()->id) . 
        "<br>"._label('backend_student').": " . $student->full_name." (". $student->id. ")";
        $subject = "";
        send_in_app_notification(
            $serviceApplications->user_id, 
            $serviceApplications->appl_id,
            "VISA_AND_SERVICES_APPLICATION_CREATED", 
            $subject, 
            $message,
            "visa", 
            true
        );

        return redirect('admin/auth/user/'.$request->user_id.'#services')->withFlashSuccess("Service Request Added Successfully!");
    }

    public function serviceRequestEdit($id){
        $service_application = ServiceApplications::where('id', $id)->first();
        $universityApplId = UniversityApplication::get();
        $countries = Country::orderBy('title_en')->where('overseas_status',1)->select('id', 'title_en as name')->get();
        return view('backend.auth.user.service-application-edit', compact('universityApplId','service_application','countries'));
    }

    public function serviceRequestEditStore(Request $request){
        $services_application = ServiceApplications::where('id', $request->service_application_id)->first();
        $services_application->user_id = $request->user_id;
        // $services_application->service_id = $request->service_id;
        $services_application->college_appl_id = $request->college_appl_id;

        // if($request->service_id == get_study_visa()){
        //     $college_appl_id = UniversityApplication::where('userid', $request->user_id)->first();
        //     if($college_appl_id){
        //         $services_application->college_appl_id = $college_appl_id->id;
        //     }
        // }

        $services_application->country = $request->country;
        $services_application->amount = $request->amount;
        $services_application->description = $request->description;
        $services_application->save();

        return redirect('admin/auth/user/'.$request->user_id.'#services')->withFlashSuccess("Service Request Edited Successfully!");
    }

    public function serviceRequestDelete($id){
        $services_application = ServiceApplications::where('id', $id)->first(); 
        if($services_application){
            $user_id = $services_application->user_id;

            // Send Push App Notification
            $user = User::find($user_id);
            if(\Auth::user()->isAgent()){
                $message = "Visa & Services Application #".$services_application->appl_id." Deleted by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
                $email_message = "Hi, <br><br>Visa & Service Application #".$services_application->appl_id." Deleted by Agent: " . get_user_full_name_by_id(\Auth::user()->id); 
                $subject = "Visa & Service Application #".$services_application->appl_id." Deleted by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
            } else{
                $message = "Visa & Services #".$services_application->appl_id." Deleted by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
                $email_message = "Hi, <br><br>Visa & Service Application #".$services_application->appl_id." Deleted by Staff: " . get_user_full_name_by_id(\Auth::user()->id); 
                $subject = "Visa & Service Application #".$services_application->appl_id." Deleted by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
            }

            //Sending email Notification
            $email_message .= "<br>Service: " . get_services($services_application->service_id).
            "<br>Country: " . get_country_name($services_application->country).
            "<br>"._label('backend_student').": " . get_user_full_name_by_id($user->id) . " (" . $user->id . ")";
            send_mail_notification($services_application->user_id,$services_application->id,"LEAD_STUDENT_APPLICATION_COURSEPREFERENCE_OTHERS_DELETE", $subject, $email_message, true);

            $message .= "<br>Service: " . get_services($services_application->service_id).
            "<br>Country: " . get_country_name($services_application->country).
            "<br>"._label('backend_student').": " . get_user_full_name_by_id($user->id) . " (" . $user->id . ")";
            $subject = "";
            send_in_app_notification(
                $services_application->user_id, 
                $services_application->appl_id, 
                "LEAD_STUDENT_APPLICATION_COURSEPREFERENCE_OTHERS_DELETE", 
                $subject, 
                $message,
                "visa", 
                true
            );

            $services_application->delete();
            return redirect('admin/auth/user/'.$user_id.'#services')->withFlashSuccess("Service Request Deleted Successfully!");
        }else{
            return redirect()->back()->withFlashDanger("Service Not Found!");
        }
    }

    // ------------------------------ End User detail service page

    public function serviceAndVisaList(){
        $country_list = DB::table('countries')->get();
        $userRepo = new StaffRepository();
        $staff = $userRepo->getStaffUsers(get_staff_branch_ids(\Auth::user()->id));
        // $userRepo = new UserRepository();
        // $staff = $userRepo->getNonStudentsTrainers();

        return view('backend.visa-and-services.visa-and-services', ["staff" => $staff, 'country_list' => $country_list]);
    }

    public function ajaxvisaapplicationdata(Request $request){
        
        if (!$request->ajax()) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this page.');
        }
        echo $this->getDatatableApplications($request);
    }

    public function getDatatableApplications($request){
        $columns = array(
            0 => 'id',
            1 => 'created_at',
            // 2 => 'service_id',
            // 3 => 'status',
            3 => 'first_name',
            5 => 'country',
            // 6 => 'details',
            // 7 => 'assigned_to',
            // 8 => 'action2',
            9 => 'created_at',
        );

        $my_branch = "";
        //DB::enableQueryLog();
        if(\Auth::user()->isAgent()){
            $my_user_id = \Auth::user()->id;
            // $my_branch = get_staff_branch_ids(\Auth::user()->id);
            // No need to check branches for agents
            $my_branch = get_all_branch_ids();
        }else{
            if(\Auth::user()->can('view assigned data only')) {
                $my_user_id = \Auth::user()->id;
            } else {
                $my_user_id = $request->input('myapps') == 1 ? \Auth::user()->id : '';        
                $my_branch = get_staff_branch_ids(\Auth::user()->id);
            }
        }
        $country = $request->input('country');
        $service_id = $request->input('service_id');
        // $service_name = $request->input('');
        $assigned_to = $request->input('assigned_to');
        $status = $request->input('status');
        $created_date = $request->input('created_date');
        $created_date_from = "";
        $created_date_to = "";   

        if(!empty($created_date)) {
            $created_dts_str = $created_date;
            $created_dts = explode(" to ", $created_dts_str);
            $created_date_from = date("Y-m-d 00:00:00", strtotime($created_dts[0]));
            $created_date_to = date("Y-m-d 23:59:59", strtotime($created_dts[1]));
            //\Log::debug('Lead created dt: '.$created_date_from. ' to '.$created_date_to);
        }
        
        $totalData = DB::table('service_applications')
            ->join('users', 'users.id', '=', 'service_applications.user_id')
            
            ->when(!empty($my_branch), function ($query) use ($my_branch) {
                return $query->whereIn('users.branch', $my_branch);
            })
            ->when(\Auth::user()->isAgent(), function ($query) use ($my_user_id) {
                return $query->where('users.agent_id', $my_user_id);
            })
            ->when(!empty($my_user_id) && !\Auth::user()->isAgent(), function ($query) use ($my_user_id) {
                return  $query->Where('users.assigned_to', '=', "$my_user_id");
            })
            ->when(!empty($service_id), function ($query) use ($service_id) {
                return  $query->Where('service_id', '=', $service_id);
            })
            ->when(!empty($country), function ($query) use ($country) {
                return  $query->Where('service_applications.country', '=', $country);
            })
            ->when(!empty($assigned_to), function ($query) use ($assigned_to) {
                return  $query->Where('service_applications.assigned_to', '=', "$assigned_to");
            })
            ->when(!empty($status), function ($query) use ($status) {
                return  $query->Where('service_applications.status', '=', "$status");
            })
            ->when(!empty($created_date_from), function ($query) use ($created_date_from) {
                return  $query->Where('service_applications.created_at', '>=', "$created_date_from");
            })
            ->when(!empty($created_date_to), function ($query) use ($created_date_to) {
                return  $query->Where('service_applications.created_at', '<=', "$created_date_to");
            })
            ->whereNull('users.deleted_at')
            ->where('users.archived', "0")
            ->count();

        $totalFiltered = $totalData;
        $limit = $request->input('length');
        $start = $request->input('start');
        $order = $columns[$request->input('order.0.column')];
        $dir = $request->input('order.0.dir');
        $country = $request->input('country');
        $status = $request->input('status');
        
        // \Log::debug("My branch(s):".print_r($my_branch, true));

        if (empty($request->input('search.value')) && empty($country) && empty($status) && empty($service_id) && empty($assigned_to)) {
            $applications = DB::table('service_applications')
                ->join('users', 'users.id', '=', 'service_applications.user_id')
                ->when(!empty($my_branch), function ($query) use ($my_branch) {
                    return $query->whereIn('users.branch', $my_branch);
                })
                ->when(\Auth::user()->isAgent(), function ($query) use ($my_user_id) {
                    return $query->where('users.agent_id', $my_user_id);
                })
                ->when(!empty($my_user_id) && !\Auth::user()->isAgent(), function ($query) use ($my_user_id) {
                    return  $query->Where('users.assigned_to', '=', "$my_user_id");
                })
                ->when(!empty($service_id), function ($query) use ($service_id) {
                    return  $query->Where('service_id', '=', $service_id);
                })
                ->when(!empty($country), function ($query) use ($country) {
                    return  $query->Where('service_applications.country', '=', $country);
                })
                ->when(!empty($assigned_to), function ($query) use ($assigned_to) {
                    return  $query->Where('service_applications.assigned_to', '=', "$assigned_to");
                })
                ->when(!empty($status), function ($query) use ($status) {
                    return  $query->Where('service_applications.status', '=', "$status");
                })
                ->when(!empty($created_date_from), function ($query) use ($created_date_from) {
                    return  $query->Where('service_applications.created_at', '>=', "$created_date_from");
                })
                ->when(!empty($created_date_to), function ($query) use ($created_date_to) {
                    return  $query->Where('service_applications.created_at', '<=', "$created_date_to");
                })
                ->whereNull('users.deleted_at')
                ->where('users.archived', "0")
                ->select(DB::raw('service_applications.*, users.first_name, users.last_name, users.email, users.mobile'))
                ->offset($start)
                ->limit($limit)
                ->orderBy($order, $dir)
                ->get();
                // $queries = DB::getQueryLog();
            // \Log::debug(end($queries));
        } else {
            \Log::info("get to know search value is not empty");
            $get_search = $request->input('search.value');
            $applications = DB::table('service_applications')
                ->join('users', 'users.id', '=', 'service_applications.user_id')
                ->join('services', 'services.id', '=', 'service_applications.service_id')
                ->leftJoin('users as assigned_user', 'assigned_user.id', '=', 'service_applications.assigned_to')
                ->join('countries','countries.id','=','service_applications.country')
                ->when(!empty($my_branch), function ($query) use ($my_branch) {
                    return $query->whereIn('users.branch', $my_branch);
                })
                ->when(\Auth::user()->isAgent(), function ($query) use ($my_user_id) {
                    return $query->where('users.agent_id', $my_user_id);
                })
                ->when(!empty($my_user_id) && !\Auth::user()->isAgent(), function ($query) use ($my_user_id) {
                    return  $query->Where('users.assigned_to', '=', "$my_user_id");
                })
                ->when(!empty($service_id), function ($query) use ($service_id) {
                    return  $query->Where('service_id', '=', $service_id);
                })
                ->when(!empty($country), function ($query) use ($country) {
                    return  $query->Where('service_applications.country', '=', $country);
                })
                ->when(!empty($assigned_to), function ($query) use ($assigned_to) {
                    return  $query->Where('service_applications.assigned_to', '=', "$assigned_to");
                })
                ->when(!empty($status), function ($query) use ($status) {
                    return  $query->Where('service_applications.status', '=', "$status");
                })
                ->when(!empty($created_date_from), function ($query) use ($created_date_from) {
                    return  $query->Where('service_applications.created_at', '>=', "$created_date_from");
                })
                ->when(!empty($created_date_to), function ($query) use ($created_date_to) {
                    return  $query->Where('service_applications.created_at', '<=', "$created_date_to");
                })
                ->select(DB::raw('service_applications.*, users.first_name, users.last_name, users.email, users.mobile,assigned_user.first_name as assigned_first_name,
                assigned_user.last_name as assigned_last_name,
                      assigned_user.email as assigned_email'))
                ->where(function ($query) use ($get_search) {
                    $query->Where('users.first_name', 'like', "%$get_search%")
                        ->orWhere('users.last_name', 'like', "%$get_search%")
                        ->orWhereRaw("CONCAT(`users`.`first_name`, ' ', `users`.`last_name`) LIKE ?", ['%'.$get_search.'%'])
                        ->orWhere('users.email', 'like', "%$get_search%")
                        ->orWhere('users.mobile', '=', "$get_search")
                        ->orWhere('services.service_name', 'like', "%$get_search%")
                        ->orWhere('service_applications.country', 'like', "%$get_search%")
                        ->orWhere('countries.title_en', 'like', "%$get_search%")
                        ->orWhere('service_applications.appl_id', 'like', "%$get_search%")
                        ->orWhere('service_applications.status', 'like', "%$get_search%")
                        ->orWhereRaw("CONCAT(`assigned_user`.`first_name`, ' ', `assigned_user`.`last_name`) LIKE ?", ['%' . $get_search . '%']);

                        // ->orWhere('service_applications.status', 'like', "%$get_search%");
                })
                ->whereNull('users.deleted_at')
                ->where('users.archived', "0")
                ->offset($start)
                ->limit($limit)
                ->orderBy($order, $dir)
                ->get();

            $totalFiltered = DB::table('service_applications')
                ->join('users', 'users.id', '=', 'service_applications.user_id')
                ->when(!empty($my_branch), function ($query) use ($my_branch) {
                    return $query->whereIn('users.branch', $my_branch);
                })
                ->when(\Auth::user()->isAgent(), function ($query) use ($my_user_id) {
                    return $query->where('users.agent_id', $my_user_id);
                })
                ->when(!empty($my_user_id) && !\Auth::user()->isAgent(), function ($query) use ($my_user_id) {
                    return  $query->Where('users.assigned_to', '=', "$my_user_id");
                })
                ->when(!empty($service_id), function ($query) use ($service_id) {
                    return  $query->Where('service_id', '=', $service_id);
                })
                ->when(!empty($country), function ($query) use ($country) {
                    return  $query->Where('service_applications.country', '=', $country);
                })
                ->when(!empty($assigned_to), function ($query) use ($assigned_to) {
                    return  $query->Where('service_applications.assigned_to', '=', "$assigned_to");
                })
                ->when(!empty($status), function ($query) use ($status) {
                    return  $query->Where('service_applications.status', '=', "$status");
                })
                ->when(!empty($created_date_from), function ($query) use ($created_date_from) {
                    return  $query->Where('service_applications.created_at', '>=', "$created_date_from");
                })
                ->when(!empty($created_date_to), function ($query) use ($created_date_to) {
                    return  $query->Where('service_applications.created_at', '<=', "$created_date_to");
                })
                ->select(DB::raw('service_applications.*, users.first_name, users.last_name, users.email, users.mobile'))
                ->where(function ($query) use ($get_search) {
                    $query->Where('users.first_name', 'like', "%$get_search%")
                        ->orWhere('users.last_name', 'like', "%$get_search%")
                        ->orWhere('users.email', 'like', "%$get_search%")
                        ->orWhere('users.mobile', '=', "$get_search")
                        ->orWhere('service_applications.country', 'like', "%$get_search%")
                        ->orWhere('service_applications.service_id', 'like', "$get_search")
                        ->orWhere('service_applications.status', 'like', "%$get_search%");
                })
                ->whereNull('users.deleted_at')
                ->where('users.archived', "0")
                ->count();
        }

        //$queries = DB::getQueryLog();
        //\Log::debug(end($queries));

        $data = $customResult = array();
        if (!empty($applications)) {
            $count = 1;
            foreach ($applications as $appl) {
                // $customResult['id'] = $count++;
                $customResult['id'] = '<a href="' . route('admin.service-visa.view', $appl->id) . '" title="Details" class="" style="font-size:13px;">'.$appl->appl_id.'</a>';
                $customResult['date'] = "<span>".get_date_formated($appl->created_at)."</span>";
                $customResult['services'] = get_services($appl->service_id);
                $customResult['name'] = "<a target='_blank' href='" . route('admin.auth.user.show', $appl->user_id) . "'><strong><span style='white-space:nowrap;'><i class='fa fa-user'></i> ".$appl->first_name . ' ' . $appl->last_name."</a>";
                if($appl->country){
                    $customResult['country'] = "<span><i class='fa fa-globe'></i> ".get_country_name($appl->country)."</span>";
                }else{
                    $customResult['country'] = "--";
                }
                if($appl->description){
                    $customResult['details'] = "<span class='text-dark'>". substr($appl->description, 0, 30) .".....</span>";
                }else{
                    $customResult['details'] = "--";
                }
                $customResult['status'] = "<span class='text-dark'>". $appl->status ."</span>";

                $customResult['assigned_to'] = "<div style='display: flex; justify-content: space-between;'>";
                $customResult['assigned_to'] .= $appl->assigned_to ? get_user_full_name_by_id($appl->assigned_to) : "--";
                if (\Auth::user()->isAdmin()) {
                    $customResult['assigned_to'] .= '<a style="padding-top: 4px;font-size:12px;float:right;" href="' . route('admin.service-visa.assignstaff', $appl->id) . '" data-toggle="tooltip" data-placement="right" title="Assign Lead" class="btn actionBtn"><i class="fas fa-user-edit"></i></a>';
                }
                $customResult['assigned_to'] .= "</div>";

                //\Log::debug("Date Creatd:".$customResult['date']);
                $customResult['created_at'] = $appl->created_at;
                $customResult['action2'] = '<div style="min-width: 130px">';
                $customResult['action2'] .= '<a href="' . route('admin.service-visa.view', $appl->id) . '" class="btn btn-primary btn-sm"><i class="fa fa-eye "></i> View </a>';
                if(\Auth::user()->isAdmin()){
                    $customResult['action2'] .= '<a href="' . route('admin.service-visa.delete', $appl->id). '" class="btn btn-sm btn-danger ms-1" data-method="delete"
                                data-trans-button-cancel="Cancel"
                                data-trans-button-confirm="Delete"
                                data-html="true"
                                data-trans-title="Are you sure want to delete this? <br><br><label style=&quot;color: red; font-size: 15px;&quot;>This action cannot be Undone!</label>"> Delete </a>';
                    
                }
                $customResult['action2'] .= '</div>';
                // else{
                //     $customResult['action2'] = '<a href="' . route('admin.service-visa.list') . '" class="btn btn-primary btn-sm mx-2"><i class="fa fa-eye "></i> View </a>';
                // }
                $data[] = $customResult;
            }
        }

        $json_data = array(
            "draw"            => intval($request->input('draw')),
            "recordsTotal"    => intval($totalData),
            "recordsFiltered" => intval($totalFiltered),
            "data"            => $data,
            "status"   =>   $status
        );

        return json_encode($json_data);
    }

    public function assignVisaApplicationStaff($id){
        if (!(\Auth::user()->isAdmin())) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }

        $serviceApplication = ServiceApplications::where('id', $id)->first();
        $userdata = User::find($serviceApplication->user_id);

        if (\Auth::user()->can('view assigned data only')) {
            $staff = User::where('id', \Auth::user()->id)->get();
        } else {
            $userRepo = new StaffRepository();
            $staff = $userRepo->getStaffUsers(get_staff_branch_ids(\Auth::user()->id));
        }

        return view('backend.visa-and-services.visa-services-assign-staff', ['serviceApplication' => $serviceApplication, 'userdata' => $userdata, 'staff' => $staff]);
    }

    public function assignVisaApplicationStaffStore(Request $request){
        if (!(\Auth::user()->isAdmin())) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }

        $staffid = $request->staff;
        $serviceApplicationId = $request->id;
        $user = User::find($staffid);
        $lead_assigned_success = false;
        $email_message_part = '';

        if (DB::table('service_applications')->where("id", $serviceApplicationId)->update(['assigned_to' => $staffid])) {
            $email_message_part = "";
            $serviceApplication = DB::table('service_applications')->where("id", $serviceApplicationId)->first();
            $serviceApplicationUser = User::find($serviceApplication->user_id);
            
            if($user){
                //Adding activity
                VisaServicesActivity::create(['appl_id' => $serviceApplication->id, 'detail' => 'ADDITIONAL SERVICES ASSIGNED TO '.get_user_full_name_by_id($staffid), 'staff' => \Auth::user()->id]);

                $subject = "Additional Services Appl #: ".$serviceApplicationUser->appl_id." Assigned by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
                $email_message_part = "";
                $email_message_part .= "<br>Service: ".get_services($serviceApplication->service_id);
                $email_message_part .= "<br>Service Country: ".get_country_name($serviceApplication->country);
                $email_message_part .= "<br>"._label('backend_student')." Name: ". $serviceApplicationUser->first_name." ".$serviceApplicationUser->last_name;
                $email_message_part .= "<br>Mobile: ".$serviceApplicationUser->mobile;
                $email_message_part .= "<br>Email: ".$serviceApplicationUser->email;
                
                // New way to send email
                //Sending email Notification
                $email_message = "Hi $user->name, <br><br>Additional Services #: ".$serviceApplicationUser->appl_id." assigned to you.<br />";
                $email_message .= $email_message_part; 
                send_mail_notification($serviceApplication->user_id, $serviceApplication->id,"VISA_OTHER_APPLICATION_ASSIGNED_STAFF",$subject,$email_message, true);

                // Send Push App Notification
                $message = "Additional Services #: ".$serviceApplicationUser->appl_id." Assigned by Staff: " . get_user_full_name_by_id(\Auth::user()->id) . " to: " .$user->full_name .
                "<br>"._label('backend_student').": " . get_user_full_name_by_id($serviceApplicationUser->id) . " (" . $serviceApplicationUser->id . ")";
                $subject = "";
                send_in_app_notification(
                    $serviceApplication->user_id, 
                    $serviceApplication->appl_id,
                    "VISA_OTHER_APPLICATION_ASSIGNED_STAFF", 
                    $subject, 
                    $message,
                    "visa", 
                    true
                );

                return redirect()->route('admin.service-visa.list')->withFlashSuccess('Staff Assigned!');
            }else{
                //Adding activity
                VisaServicesActivity::create(['appl_id' => $serviceApplication->id, 'detail' => 'ADDITIONAL SERVICES ASSIGNED REMOVED.', 'staff' => \Auth::user()->id]);
                return redirect()->route('admin.service-visa.list')->withFlashSuccess('Removed Staff Assigned!');
            }
        } else {
            return redirect()->route('admin.service-visa.list')->withFlashDanger('Something Went Wrong!');
        }
    }

    public function visaServicesDelete($id){
        $services_application = ServiceApplications::where('id', $id)->first(); 
        if($services_application){
            $user_id = $services_application->user_id;

            // Send Push App Notification
            $user = User::find($user_id);
            if(\Auth::user()->isAgent()){
                $message = "Visa & Services Application #".$services_application->appl_id." Deleted by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
                $email_message = "Hi, <br><br>Visa & Service Application #".$services_application->appl_id." Deleted by Agent: " . get_user_full_name_by_id(\Auth::user()->id); 
                $subject = "Visa & Service Application #".$services_application->appl_id." Deleted by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
            } else{
                $message = "Visa & Services Application #".$services_application->appl_id." Deleted by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
                $email_message = "Hi, <br><br>Visa & Service Application #".$services_application->appl_id." Deleted by Staff: " . get_user_full_name_by_id(\Auth::user()->id); 
                $subject = "Visa & Service Application #".$services_application->appl_id." Deleted by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
            }

            //Sending email Notification
            $email_message .= "<br>Service: " . get_services($services_application->service_id).
            "<br>Country: " . get_country_name($services_application->country).
            "<br>"._label('backend_student').": " . get_user_full_name_by_id($user->id) . " (" . $user->id . ")";
            send_mail_notification($services_application->user_id,$services_application->id,"LEAD_STUDENT_APPLICATION_COURSEPREFERENCE_OTHERS_DELETE", $subject, $email_message, true);

            $message .= "<br>"._label('backend_student').": " . get_user_full_name_by_id($user->id) . " (" . $user->id . ")";
            $subject = "";
            send_in_app_notification(
                $services_application->user_id, 
                $services_application->appl_id, 
                "LEAD_STUDENT_APPLICATION_COURSEPREFERENCE_OTHERS_DELETE", 
                $subject, 
                $message,
                "visa", 
                true
            );
            
            $services_application->delete();
            return redirect()->route('admin.service-visa.list')->withFlashSuccess("Service Deleted Successfully!");
        }else{
            return redirect()->back()->withFlashDanger("Service Not Found!");
        }
    }

    public function viewVisaAndServices($id){
        $services_application = ServiceApplications::where('id', $id)->first(); 
        if (!$services_application) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger("Service Not Found");
        }

        $user = User::find($services_application->user_id);
        if (\Auth::user()->isAgent()) {
            if ($user->agent_id != \Auth::user()->id) {
                return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
            }
        }

        //Updating the Student Application
        $studata = array();
        $cv = "";
        $passports = array();
        $sops = array();
        $lors = array();
        $internships = array();
        $offerletters = array();
        $experience = array();
        $otherdocs = array();

        if (!DB::table('user_documents')->where('userid', $services_application->user_id)->exists()) {
            DB::table('user_documents')->insert(['userid' => $services_application->user_id]);
        }

        $sturow = DB::table('user_documents')->where('userid', $services_application->user_id)->first();
        $studata['class10'] = $sturow->class10;
        $studata['class10_marks_sheet'] = $sturow->class10_marks_sheet;
        $studata['class12'] = $sturow->class12;
        $studata['class12_marks_sheet'] = $sturow->class12_marks_sheet;
        $studata['diploma'] = $sturow->diploma;
        $studata['diploma_marks_sheet'] = $sturow->diploma_marks_sheet;
        $studata['grad_all_sem'] = $sturow->grad_all_sem;
        $studata['grad_degree'] = $sturow->grad_degree;
        $studata['grad_provisional'] = $sturow->grad_provisional;
        $studata['grad_transcripts'] = $sturow->grad_transcripts;
        $studata['pg_all_sem'] = $sturow->pg_all_sem;
        $studata['pg_provisional'] = $sturow->pg_provisional;
        $studata['cmm'] = $sturow->cmm;
        $studata['pg_degree'] = $sturow->pg_degree;
        $studata['pg_transcripts'] = $sturow->pg_transcripts;
        $studata['lang_ielts'] = $sturow->lang_ielts;
        $studata['lang_pte'] = $sturow->lang_pte;
        $studata['gre'] = $sturow->gre;
        $studata['gmat'] = $sturow->gmat;
        $studata['duolingo'] = $sturow->duolingo;

        $docrows = DB::table('user_docs_other')->where('userid', $services_application->user_id)->get();
        if ($docrows) {
            foreach ($docrows as $docrow) {
                if ($docrow->doc_type == 'cv') {
                    $cv = $docrow->file;
                } elseif ($docrow->doc_type == 'passport') {
                    $passports[] = $docrow;
                } elseif ($docrow->doc_type == 'sop') {
                    $sops[] = $docrow;
                } elseif ($docrow->doc_type == 'letter_of_recommendation') {
                    $lors[] = $docrow;
                } elseif ($docrow->doc_type == 'internship') {
                    $internships[] = $docrow->file;
                } elseif ($docrow->doc_type == 'offer') {
                    $offerletters[] = $docrow->file;
                } elseif ($docrow->doc_type == 'experience') {
                    $experience[] = $docrow;
                } elseif ($docrow->doc_type == 'payslip') {
                    $payslips[] = $docrow->file;
                } elseif ($docrow->doc_type == 'other') {
                    $otherdocs[] = $docrow;
                }
            }
        }

        $studata['cv'] = $cv;
        $studata['passports'] = $passports;
        $studata['sops'] = $sops;
        $studata['lors'] = $lors;
        $studata['internships'] = $internships;
        $studata['offerletters'] = $offerletters;
        $studata['experience'] = $experience;
        $studata['otherdocs'] = $otherdocs;

        $studentUser = User::where('id', $services_application->user_id)->first(); 
        // if (\Auth::user()->isAgent()) {
        //     $msgs = DB::table('service_application_messages')->where('application_id', $services_application->id)->where('private','<>',1)->orderBy('id', 'desc')->get();
        // }else{
        //     $msgs = DB::table('service_application_messages')->where('application_id', $services_application->id)->orderBy('id','desc')->get();
        // }
        $msgs = DB::table('service_application_messages')->where('application_id', $services_application->id)->orderBy('id','desc')->get();

        $userDetails = DB::table('user_details')->where('userid', $services_application->user_id)->first();
        $visaServicesActivity = DB::table('visa_services_activity')->where('appl_id', $services_application->id)->get();
        
        $statuses = ServiceStages::where('service_id', $services_application->service_id)->orderBy('stage_order')->get(["status_name"]);
        $reminders = DB::table('reminders')->where('entity', "service_application")->where('entity_id', $id)->get();
        return view('backend.visa-and-services.visa-and-services-view', [
            "reminders" => $reminders,
            "statuses" => $statuses,
            "services_application" => $services_application, 
            "student" => $studentUser, 
            "app_messages" => $msgs, 
            "visaServicesActivity" => $visaServicesActivity, 
            "userDetails" => $userDetails
        ],$studata);
    }

    public function add_activity(Request $request){
        $request->validate([
            'activity' => 'required',
            'deadline' => 'required|date|after_or_equal:' . date('Y-m-d H:i:s'),
        ]);
        if (\Auth::user()->isAgent()) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        $reminder = new Reminders;
        $reminder->entity_id = $request->input('appl_id');
        $reminder->activity = $request->input('activity');
        $reminder->entity = "service_application";
        $reminder->deadline = date('Y-m-d',strtotime($request->input('deadline')));
        $reminder->created_by = \Auth::user()->id;
        $reminder->save();
        return redirect()->route('admin.service-visa.view', $request->input('appl_id'))->withFlashSuccess("Reminder added Successfully");
    }

    public function delete_activity(Request $request){
        if (\Auth::user()->isAgent()) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        if (DB::table('reminders')->where("id", $request->id)->delete()) {
            return response()->json(['message'=>'Activity removed']);
        } else{
            return response()->json(['message'=>'Activity Not Found']);
        }
        
    }
    public function activity_done(Request $request){
        if (\Auth::user()->isAgent()) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        if($request->action == 1){
            DB::table('reminders')->where("id", $request->id)->update(["done"=>1,"done_date"=>now(),"done_by"=>\Auth::user()->id]);
            return response()->json(['message'=>'Activity marked as Done.']);
        }
        else if($request->action == 0){
            DB::table('reminders')->where("id", $request->id)->update(["done"=>0,"done_date"=>NULL,"done_by"=>NULL]);
            return response()->json(['message'=>'Activity marked as Not Done.']);
        }
        else{
            return response()->json(['message'=>'Activity Not Found']);
        }
        
    }

    public function visaServiceUpdateNotes(Request $request){
        $request->validate([
            'notes' => 'max:500'
        ]);
        $applData = ServiceApplications::find($request->id);
        $applData->notes = $request->notes;
        $applData->save();
        VisaServicesActivity::create(['appl_id' => $request->id, 'detail' => 'ADDITIONAL SERVICES NOTES UPDATED', 'staff' => \Auth::user()->id]);
        return redirect()->route('admin.service-visa.view', $request->id)->withFlashSuccess('Notes Updated!');
    }

    public function visaServiceMessageSend(Request $request){
        $request->validate([
            'message' => 'required',
            'attachment.*' => 'mimes:jpg,jpeg,docx,pdf,doc,png|max:8000'
        ],[
            'attachment.*.mimes' => 'Only pdf, doc, docx, jpeg, png files are allowed',
            'image_file.*.max' => 'Sorry! Maximum allowed file size is 8MB',
            ]
        );

        $data = $request->all();
        //$data['insert_timestamp'] = date("Y-m-d H:i:s");
        $msg = new ServiceApplicationMessage;

        $msg->application_id = $request->app_id;
        $msg->user_id = \Auth::user()->id;
        $msg->message = $request->message;

        if ($request->file('attachment')) {
            $cnt = 1;
            foreach ($request->file('attachment') as $file){
                $filename = $file->getClientOriginalName();
                $filename = str_replace(" ", "-", $filename);
                $filename = preg_replace('/[^A-Za-z0-9\-_]+/', '', $filename);
                if (strlen($filename) > 20) {
                    $filename = substr($filename, 0, 20);
                    }
                    $filename .= Str::random(10);
                    $path = $file->storeAs('/app_messages', $filename . "." . $file->getClientOriginalExtension(), 'public');
                    if($cnt == 1){
                        $msg->attachment_1 = $path;
                    }
                    elseif($cnt == 2){
                        $msg->attachment_2 = $path;
                    }
                    elseif($cnt == 3){
                        $msg->attachment_3 = $path;
                    }
                    elseif($cnt == 4){
                        $msg->attachment_4 = $path;
                    }
                    $cnt++;
                }
        }
        if(isset($request->notify_staff)){
            $noti_staff = array();
            foreach($request->notify_staff as $staff){
                //echo "<br>".$staff;
                $noti = new UserNotification;
                $noti->user_id = $staff;
                $noti->application_id = $request->app_id;
                $noti->application_type = 'visa';
                $noti->message = $request->message;
                $noti->from_user = \Auth::user()->id;
                $noti->save();
                $noti_staff[] = $staff;
            }
            $staff_list = implode(',',$noti_staff);
            $msg->notified_staff = $staff_list;
        }
        $msg->save();
        
        $application = ServiceApplications::find($request->app_id);
        $student = User::find($application->user_id);
        $email_message = $request->message;
        if (!\Auth::user()->isAgent()) {                    
            if($student->agent_id) {
                //We must notify agent instead
                $agent = User::find($student->agent_id);
                $email_message = "Dear ".$agent->full_name.",<br />New message for your visa and service #".$application->appl_id."<br/>" . $email_message;
                try{
                    //$agent->notify(new UserAlert("New message for your visa and service #".$application->appl_id, $email_message));
                    user_notify($agent, "New message for your visa and service #".$application->appl_id, $email_message); 
                }catch (\Exception $e) {
                    $message = $e->getMessage();
                    \Log::debug("Mail Error: (Uni Visa and Service - (Agent) Send Message) ".$message);   
                }
                //store event in user activity
                UserActivity::create(['user_id' => $agent->id, 'type' => 'email_sent', 'title' => "Email Notification", 'detail' => "Subject:" . "New message for your visa and service #".$application->appl_id . "<br>Message:" . $email_message, 'staff' => \Auth::user()->id]);
                VisaServicesActivity::create(['appl_id' => $request->app_id, 'detail' => 'ADDITIONAL SERVICES MESSAGE SENT', 'staff' => \Auth::user()->id]);
            } else {
                try{
                    //$student->notify(new UserAlert("New message for your visa and service #".$application->appl_id, $email_message));
                    user_notify($student, "New message for your visa and service #".$application->appl_id, $email_message); 
                }catch (\Exception $e) {
                    $message = $e->getMessage();
                    \Log::debug("Mail Error: (Uni Visa and Service. - (User) Send Message) ".$message);   
                }
                //store event in user activity
                UserActivity::create(['user_id' => $student->id, 'type' => 'email_sent', 'title' => "Email Notification", 'detail' => "Subject:" . "New message for your application #".$application->appl_id. "<br>Message:" . $email_message, 'staff' => \Auth::user()->id]);
                VisaServicesActivity::create(['appl_id' => $request->app_id, 'detail' => 'ADDITIONAL SERVICES MESSAGE SENT', 'staff' => \Auth::user()->id]);
            } 
        }else{
            try{
                // if($student->assigned_to){
                //     // User::find($row->assigned_to);
                //     $staff = DB::table('users')->where('id', $student->assigned_to)->first();
                //     Mail::to($staff->email)->send(new MailSend("New Visa and Service Message by agent", $email_message));
                // }else{
                //     // $recipient = new AdminRecipient();
                //     // $recipient->notify(new AdminAlert("New Visa and Service Message by agent", $email_message));

                // }         
                //Notify Staff
                $subject = "New Visa and Service Message by agent";
                send_mail_notification($application->user_id,$application->id,"VISA_OTHER_APPLICATION_MESSAGE_BY_STUDENT",$subject,$email_message, true);
            }catch (\Exception $e) {
                $message = $e->getMessage();
                \Log::debug("Mail Error: (New Visa and Service Message by Agent) ".$message);   
            }
        }
        
        // Send Push App Notification
        if(\Auth::user()->isAgent()){
            $message = "Visa and Service Application #".$application->appl_id.", New Message by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
            $email_message = "Hi, <br><br>Visa and Service Application #".$application->appl_id.", New Message by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
            $subject = "Visa & Service Application #".$application->appl_id.", New Message by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
        } else{
            $message = "Visa and Service Application #".$application->appl_id.", New Message by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
            $email_message = "Hi, <br><br>Visa and Service Application #".$application->appl_id.", New Message by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
            $subject = "Visa & Service Application #".$application->appl_id.", New Message by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
        }

        //Sending email Notification
        $email_message .= "<br>Service: " . get_services($application->service_id).
        "<br>Country: " . get_country_name($application->country).
        "<br>"._label('backend_student').": " . get_user_full_name_by_id($student->id) . " (" . $student->id . ")";
        send_mail_notification($application->user_id,$application->id,"VISA_OTHER_APPLICATION_MESSAGE_BY_STUDENT", $subject, $email_message, true);
        
        $message .= "<br>"._label('backend_student').": " . get_user_full_name_by_id($student->id) . " (" . $student->id . ")";
        $subject = "";
        send_in_app_notification(
            $application->user_id, 
            $application->appl_id, 
            "VISA_OTHER_APPLICATION_MESSAGE_BY_STUDENT", 
            $subject, 
            $message,
            "visa", 
            true
        );

        return redirect()->route('admin.service-visa.view', $request->app_id)->withFlashSuccess("Message sent successfully");
    }

    public function visaServiceStatusUpdate(Request $request){
        if (\Auth::user()->isAgent()) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }

        $request->validate([
            'status' => 'required'
        ]);

        $visaServices = ServiceApplications::where('id', $request->id)->first();
        if($request->input('status') == $visaServices->status){
            return redirect()->back()->withFlashDanger("Status Already Exists!");
        }
        $old_status = $visaServices->status;
        $student = User::where('id', $visaServices->user_id)->first();

        $id = $request->input('id');
        $visaServices->status = $request->input('status');
        $visaServices->save();

        VisaServicesActivity::create(['appl_id' => $request->id, 'detail' => 'ADDITIONAL SERVICES STATUS UPDATED TO '.$visaServices->status, 'staff' => \Auth::user()->id]);

        
        // Send Push App Notification
        if(\Auth::user()->isAgent()){
            $message = "Visa and Service Application #".$visaServices->appl_id.", Status Changed by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
            $email_message = "Hi, <br><br>Visa and Service Application #".$visaServices->appl_id.", Status Changed by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
            $subject = "Visa and Service Application #".$visaServices->appl_id." Status Change";
        } else{
            $message = "Visa and Service Application #".$visaServices->appl_id.", Status Changed by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
            $email_message = "Hi, <br><br>Visa and Service Application #".$visaServices->appl_id.", Status Changed by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
            $subject = "Visa and Service Application #".$visaServices->appl_id." Status Change";
        }
        
        $email_message .= "<br>New Status: " . $visaServices->status .
            "<br>Previous Status: " . $old_status .
            "<br><b>Application Details:</b><br>Application #".$visaServices->appl_id .
            "<br>Service: " . get_services($visaServices->service_id).
            "<br>Country: " .get_country_name($visaServices->country);
        send_mail_notification($visaServices->user_id,$id,"VISA_OTHER_APPLICATION_STATUS_CHANGE",$subject,$email_message, true);

        $message .= "<br>Appl ID: #".$visaServices->appl_id.
            "<br>New Status: " . $visaServices->status .
            "<br>Previous Status: " . $old_status .
            "<br>"._label('backend_student').": " . get_user_full_name_by_id($student->id) . " (" . $student->id . ")";
        $subject = "";
        send_in_app_notification(
            $visaServices->user_id, 
            $visaServices->appl_id, 
            "VISA_OTHER_APPLICATION_STATUS_CHANGE", 
            $subject, 
            $message,
            "visa", 
            true
        );

        return redirect()->route('admin.service-visa.view', $request->id)->withFlashSuccess("Service Status Changed successfully");
    }

    public function search_service_country(Request $request){
        $service = DB::table('services')->where('id', $request->services_id)->first();
        $results = [];
        $amount = 0;
        if($service){
            $results = $service->country_field;
            $amount = $service->amount;
        }
        return ['results' => $results, 'amount' => $amount];
    }

}
