<?php

namespace App\Http\Controllers\Backend;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use App\Http\Requests\Backend\Auth\User\UpdateApplicationRequest;
use App\Repositories\Backend\Auth\UserRepository;
use App\Notifications\Backend\UserAlert;
use App\Models\Auth\User;
use App\Models\MailTemplates;
use App\Models\Auth\Countries;
use App\Models\Auth\Reminders;
use App\Models\Auth\CountryStudy;
use App\Models\Auth\UserNotification;
use App\Models\Auth\ApplicationActivity;
use App\Models\Auth\ApplicationMessages;
use App\Models\Auth\UniversityApplication;
use App\Models\Auth\CollegeFees;
use App\Models\Auth\ApplicationDocRequest;
use App\Models\Auth\Commission;
use App\Models\Auth\AgentCommission;
use App\Models\UserActivity;
use App\Mail\MailSend;
use Illuminate\Support\Facades\Mail;
use App\Recipients\AdminRecipient;
use App\Notifications\Frontend\AdminAlert;
use App\Repositories\Backend\Auth\StaffRepository;
use Carbon\Carbon;
use Illuminate\Support\Facades\File;
use ZipArchive;

/**
 * Class UnivApplicationsController.
 */
class UnivApplicationsController extends Controller
{

    public function __construct()
    {
        $this->middleware(function ($request, $next) {
            $user = \Auth::user();
            if (!(\Auth::user()->isAdmin() || \Auth::user()->isAgent() || \Auth::user()->can('view university applications'))) {
                return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
            }
            return $next($request);
        });
    }

    /**
     * @return \Illuminate\View\View
     */
    public function index(Request $request)
    {
        $filters = array();
        $search_type = $request->input('q') != '' ? $request->input('q') : '';
        //$country_list = UniversityApplication::select('country')->distinct()->get();
        $country_list = CountryStudy::where('status',1)->select('country')->get();
        $statuses = get_application_statuses();
        $branch = get_staff_branch(\Auth::user()->id);

        $userRepo = new StaffRepository();
        $staff = $userRepo->getStaffUsers(get_staff_branch_ids(\Auth::user()->id));
        $task_deadline = '';
        if ($request->has('task_deadline')) {
            $task_deadline = 1;
        }

        $filters = array("appl_status" => $search_type, "search" => "", "task_deadline" =>$task_deadline);
        return view('backend.applications.applications', ['branch' => $branch, 'staff' => $staff, "appl_status" => $search_type, 'filters' => $filters, 'country_list' => $country_list, 'statuses' => $statuses]);
    }


    public function ajaxapplicationdata(Request $request)
    {
        if (!$request->ajax()) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this page.');
        }
        echo $this->getDatatableApplications($request);
    }

    public function getDatatableApplications($request)
    {
        $columns = array(
            0 => 'appl_id',
            2 => 'created_at',
            3 => 'first_name',
            4 => 'country',
            9 => 'status',
        );

        // Enable query logging
        DB::enableQueryLog();

        $my_branch = "";
        if(\Auth::user()->isAgent()){
            $my_user_id = \Auth::user()->id;
            // $my_branch = get_staff_branch_ids(\Auth::user()->id);
            // No need to check branches for agents
            $my_branch = get_all_branch_ids();
        }else{
            if(\Auth::user()->can('view assigned data only')) {
                $my_user_id = \Auth::user()->id;
            } else {
                $my_user_id = $request->input('myapps') == 1 ? \Auth::user()->id : '';        
                $my_branch = get_staff_branch_ids(\Auth::user()->id);
            }
        }

        $assigned_to = $request->input('assigned_to');
        $branch = $request->input('branch');
        $applied_with = $request->input('applied_with');
        
        $totalData = DB::table('university_application')
            ->join('users', 'users.id', '=', 'university_application.userid')
            ->leftJoin('users as agent', 'users.agent_id', '=', 'agent.id')
            ->leftJoin('users as staff', 'university_application.assign_to', '=', 'staff.id')
            ->when(!empty($my_branch), function ($query) use ($my_branch) {
                return $query->whereIn('users.branch', $my_branch);
            })
            ->when(!empty($branch), function ($query) use ($branch) {
                return $query->where('users.branch', $branch);
            })
            ->when(!empty($my_user_id) && !\Auth::user()->isAgent(), function ($query) use ($my_user_id) {
                return  $query->Where('assign_to', '=', $my_user_id);
            })
            ->when(\Auth::user()->isAgent(), function ($query) use ($my_user_id) {
                return $query->where('users.agent_id', $my_user_id);
            })
            ->when(!empty($assigned_to), function ($query) use ($assigned_to) {
                if($assigned_to == "null"){
                    return  $query->whereNull('university_application.assign_to');
                }else{
                    return  $query->Where('university_application.assign_to', '=', "$assigned_to");
                }
            })
            ->when(!empty($applied_with), function ($query) use ($applied_with) {
                return $query->where('university_application.applied_with', $applied_with);
            })
            ->whereNull('users.deleted_at')
            ->where('users.archived', "0")
            ->count();

        $totalFiltered = $totalData;
        $limit = $request->input('length');
        $start = $request->input('start');
        $order = $columns[$request->input('order.0.column')];
        $dir = $request->input('order.0.dir');
        $country = $request->input('country');
        $status = $request->input('appl_status');

        $task_deadline = $request->input('task_deadline');
        $intake = $request->input('intake');
        $intake_month = "";
        $intake_year = "";
        $created_date_from = "";
        $created_date_to = ""; 
        if(!empty($intake)){
            $inta =  explode(" ",$intake);
            $intake_month = $inta[0];
            $intake_year = $inta[1];
        }

        $created_date = $request->input('created_date');
        if(!empty($created_date)) {
            $created_dts_str = $created_date;
            $created_dts = explode(" to ", $created_dts_str);
            $created_date_from = date("Y-m-d 00:00:00", strtotime($created_dts[0]));
            $created_date_to = date("Y-m-d 23:59:59", strtotime($created_dts[1]));
            \Log::debug('Lead created dt: '.$created_date_from. ' to '.$created_date_to);
        }
        // \Log::debug("My branch(s):".print_r($my_branch, true));
        // dd($my_user_id);
        if (empty($request->input('search.value')) && empty($country) && empty($status) && empty($intake)) {
            $applications = DB::table('university_application')
                ->join('users', 'users.id', '=', 'university_application.userid')
                ->leftJoin('users as agent', 'users.agent_id', '=', 'agent.id')
                ->leftJoin('users as staff', 'university_application.assign_to', '=', 'staff.id')
                ->when(!empty($my_branch), function ($query) use ($my_branch) {
                    return $query->whereIn('users.branch', $my_branch);
                })
                ->when(!empty($branch), function ($query) use ($branch) {
                    return $query->where('users.branch', $branch);
                })
                ->when(!empty($my_user_id) && !\Auth::user()->isAgent(), function ($query) use ($my_user_id) {
                    return  $query->Where('assign_to', '=', "$my_user_id");
                })
                ->when(\Auth::user()->isAgent(), function ($query) use ($my_user_id) {
                    return $query->where('users.agent_id', $my_user_id);
                })
                ->when(isset($task_deadline) && $task_deadline == 1, function ($query) {
                            $query->Where('university_application.deadline','>=', date("Y-m-d"));
                    return  $query->Where('university_application.deadline','<=', Carbon::now()->addDays(14));
                })
                ->when(!empty($assigned_to), function ($query) use ($assigned_to) {
                    if($assigned_to == "null"){
                        return  $query->whereNull('university_application.assign_to');
                    }else{
                        return  $query->Where('university_application.assign_to', '=', "$assigned_to");
                    }
                })
                ->when(!empty($created_date_from), function ($query) use ($created_date_from) {
                    return  $query->Where('university_application.created_at', '>=', "$created_date_from");
                })
                ->when(!empty($created_date_to), function ($query) use ($created_date_to) {
                    return  $query->Where('university_application.created_at', '<=', "$created_date_to");
                })
                ->when(!empty($applied_with), function ($query) use ($applied_with) {
                    return $query->where('university_application.applied_with', $applied_with);
                })
                ->whereNull('users.deleted_at')
                ->where('users.archived', "0")
                ->select(DB::raw('university_application.*, users.first_name, users.last_name, users.email, users.mobile,users.agent_id, staff.first_name as staff_first_name, staff.last_name as staff_last_name, agent.first_name as agent_first_name, agent.last_name as agent_last_name'))
                ->offset($start)
                ->limit($limit)
                ->orderBy($order, $dir)
                ->get();
                // $queries = DB::getQueryLog();
            // \Log::debug(end($queries));
        } else {
            $get_search = $request->input('search.value');

            $applications = DB::table('university_application')
                ->join('users', 'users.id', '=', 'university_application.userid')
                ->leftJoin('users as agent', 'users.agent_id', '=', 'agent.id')
                ->leftJoin('users as staff', 'university_application.assign_to', '=', 'staff.id')
                ->when(!empty($my_branch), function ($query) use ($my_branch) {
                    return $query->whereIn('users.branch', $my_branch);
                })
                ->when(!empty($branch), function ($query) use ($branch) {
                    return $query->where('users.branch', $branch);
                })
                ->when(!empty($my_user_id) && !\Auth::user()->isAgent(), function ($query) use ($my_user_id) {
                    return  $query->Where('assign_to', '=', "$my_user_id");
                })
                ->when(\Auth::user()->isAgent(), function ($query) use ($my_user_id) {
                    return $query->where('users.agent_id', $my_user_id);
                })
                ->select(DB::raw('university_application.*, users.first_name, users.last_name, users.email, users.mobile,users.agent_id, staff.first_name as staff_first_name, staff.last_name as staff_last_name, agent.first_name as agent_first_name, agent.last_name as agent_last_name'))
                ->where(function ($query) use ($get_search) {
                    $query->Where('users.first_name', 'like', "%$get_search%")
                        ->orWhere('users.last_name', 'like', "%$get_search%")
                        ->orWhereRaw("CONCAT(`users`.`first_name`, ' ', `users`.`last_name`) LIKE ?", ['%'.$get_search.'%'])
                        ->orWhere('users.email', 'like', "%$get_search%")
                        ->orWhere('users.mobile', '=', "$get_search")
                        ->orWhere('university_application.appl_id', '=', "$get_search")
                        ->orWhere('university_application.country', 'like', "%$get_search%")
                        ->orWhere('university_application.course', 'like', "%$get_search%")
                        ->orWhere('university_application.university', 'like', "%$get_search%");
                })
                ->when(!empty($country), function ($query) use ($country) {
                    return  $query->Where('university_application.country', '=', "$country");
                })
                ->when(!empty($status), function ($query) use ($status) {
                    return  $query->WhereIn('university_application.status',$status);
                })
                ->when(!empty($intake), function ($query) use ($intake_month,$intake_year) {
                            $query->Where('university_application.intake', '=', "$intake_month");
                    return  $query->Where('university_application.year', '=', "$intake_year");
                })
                ->when(!empty($assigned_to), function ($query) use ($assigned_to) {
                    if($assigned_to == "null"){
                        return  $query->whereNull('university_application.assign_to');
                    }else{
                        return  $query->Where('university_application.assign_to', '=', "$assigned_to");
                    }
                })
                ->when(!empty($created_date_from), function ($query) use ($created_date_from) {
                    return  $query->Where('university_application.created_at', '>=', "$created_date_from");
                })
                ->when(!empty($created_date_to), function ($query) use ($created_date_to) {
                    return  $query->Where('university_application.created_at', '<=', "$created_date_to");
                })
                ->when(!empty($applied_with), function ($query) use ($applied_with) {
                    return $query->where('university_application.applied_with', $applied_with);
                })
                ->whereNull('users.deleted_at')
                ->where('users.archived', "0")
                ->offset($start)
                ->limit($limit)
                ->orderBy($order, $dir)
                ->get();

            $totalFiltered = DB::table('university_application')
                ->join('users', 'users.id', '=', 'university_application.userid')
                ->leftJoin('users as agent', 'users.agent_id', '=', 'agent.id')
                ->leftJoin('users as staff', 'university_application.assign_to', '=', 'staff.id')
                ->when(!empty($my_branch), function ($query) use ($my_branch) {
                    return $query->whereIn('users.branch', $my_branch);
                })
                ->when(!empty($branch), function ($query) use ($branch) {
                    return $query->where('users.branch', $branch);
                })
                ->when(!empty($my_user_id) && !\Auth::user()->isAgent(), function ($query) use ($my_user_id) {
                    return  $query->Where('assign_to', '=', "$my_user_id");
                })
                ->when(\Auth::user()->isAgent(), function ($query) use ($my_user_id) {
                    return $query->where('users.agent_id', $my_user_id);
                })
                ->select(DB::raw('university_application.*, users.first_name, users.last_name, users.email, users.mobile,users.agent_id, staff.first_name as staff_first_name, staff.last_name as staff_last_name, agent.first_name as agent_first_name, agent.last_name as agent_last_name'))
                ->where(function ($query) use ($get_search) {
                    $query->Where('users.first_name', 'like', "%$get_search%")
                        ->orWhere('users.last_name', 'like', "%$get_search%")
                        ->orWhereRaw("CONCAT(`users`.`first_name`, ' ', `users`.`last_name`) LIKE ?", ['%'.$get_search.'%'])
                        ->orWhere('users.email', 'like', "%$get_search%")
                        ->orWhere('users.mobile', '=', "$get_search")
                        ->orWhere('university_application.appl_id', '=', "$get_search")
                        ->orWhere('university_application.country', 'like', "%$get_search%")
                        ->orWhere('university_application.course', 'like', "%$get_search%")
                        ->orWhere('university_application.university', 'like', "%$get_search%");
                })
                ->when(!empty($country), function ($query) use ($country) {
                    return  $query->Where('university_application.country', '=', "$country");
                })
                ->when(!empty($status), function ($query) use ($status) {
                    return  $query->WhereIn('university_application.status', $status);
                })
                ->when(!empty($intake), function ($query) use ($intake_month,$intake_year) {
                            $query->Where('university_application.intake', '=', "$intake_month");
                    return  $query->Where('university_application.year', '=', "$intake_year");
                })
                ->when(!empty($assigned_to), function ($query) use ($assigned_to) {
                    if($assigned_to == "null"){
                        return  $query->whereNull('university_application.assign_to');
                    }else{
                        return  $query->Where('university_application.assign_to', '=', "$assigned_to");
                    }
                })
                ->when(!empty($created_date_from), function ($query) use ($created_date_from) {
                    return  $query->Where('university_application.created_at', '>=', "$created_date_from");
                })
                ->when(!empty($created_date_to), function ($query) use ($created_date_to) {
                    return  $query->Where('university_application.created_at', '<=', "$created_date_to");
                })
                ->when(!empty($applied_with), function ($query) use ($applied_with) {
                    return $query->where('university_application.applied_with', $applied_with);
                })
                ->whereNull('users.deleted_at')
                ->where('users.archived', "0")
                ->count();
        }

        $data = $customResult = array();
        if (!empty($applications)) {
            $count = 1;
            foreach ($applications as $appl) {
                $customResult = []; 
                // $customResult['id'] = $count++;
                $customResult['checkbox'] = '<input type="checkbox" class="application_checkbox" value="'.$appl->id.'">';
                $customResult['appl_id'] = '<a href="' . route('admin.applicationdetail', [$appl->id]) . '" title="Details" class="" style="font-size:13px;"><b>'.$appl->appl_id.'</b></a>';
                $customResult['export_id'] = '<span>'.$appl->appl_id.'</span>';
                $customResult['name'] = "<a target='_blank' href='" . route('admin.auth.user.show', $appl->userid) . "'><strong><span style='white-space:nowrap;'>".$appl->first_name . ' ' . $appl->last_name."</a>";
                $customResult['country'] = $appl->country;

                // if($appl->type == 'visa') {
                //     $visa_pref = DB::table("visa_preference")->where('id', $appl->visa_pref_id)->first();
                //     $customResult['details'] = get_visa_service_type_name($visa_pref->service_type)."<br />".get_visa_letter_type_name($visa_pref->letter_type);
                //     $customResult['status'] = show_visa_application_status($appl->visa_status);
                // } elseif($appl->type == 'abroad_job') {
                //     $visa_pref = DB::table("abroad_job_preference")->where('id', $appl->abroad_job_pref_id)->first();
                //     $customResult['details'] = "";
                //     $customResult['status'] = show_abroad_job_application_status($appl->abroad_job_status);
                // } else {
                    $customResult['details'] = "<span class='text-sub'>University:</span> ".$appl->university."<br /><span class='text-sub'>Course:</span> ".$appl->course."<br /><span class='text-sub'>Intake:</span> ".$appl->year."-".get_intake_month_name($appl->intake);
                    $customResult['status'] = $appl->status ? show_application_status($appl->status) : "--";
                //}
                if(!\Auth::user()->isAgent()){
                    if($appl->agent_id) {
                        $customResult['details'] .= "<p style='padding: 0px !important; margin: 0px !important;'><i title='Partner/Agent' class='fas fa-user-tie text-sub'></i> <span style='white-space:nowrap;' title='Partner/Agent'>" . $appl->agent_first_name . " " . $appl->agent_last_name . "</span></p>";
                    }
                }
                if($appl->applied_with && !\Auth::user()->isAgent()){
                    $customResult['details'] .= "Applied Through Agent:<b>" . get_uni_apply_agent_name_by_id($appl->applied_with) . "</b>";
                }

                //For Excel export
                $customResult['university'] = $appl->university;
                $customResult['course'] =  $appl->course;
                $customResult['intake'] =  $appl->year."-".get_intake_month_name($appl->intake);
                //For Excel export

                $customResult['assigned_to'] = "<div style='display: flex; justify-content: space-between;'>";
                $customResult['assigned_to'] .= $appl->assign_to ? $appl->staff_first_name . " " . $appl->staff_last_name : "<span style='color: red;'>Unassigned</span>";
                if (\Auth::user()->isAdmin() || \Auth::user()->can('application-assignment')) {
                    $customResult['assigned_to'] .= '<a style="padding-top: 4px;font-size:12px;float:right;" href="' . route('admin.collegeApplication.assignstaff', $appl->id) . '" data-toggle="tooltip" data-placement="right" title="Assign Application" class="btn actionBtn"><i class="fas fa-user-edit"></i></a>';
                }
                $customResult['assigned_to'] .= "</div>";

                $customResult['created_at'] = date("d-m-Y", strtotime($appl->created_at));
                $customResult['action2'] = '<a title="View Application" href="' . route('admin.applicationdetail', [$appl->id]) . '" class="btn btn-primary btn-sm"><i class="fa fa-eye "></i> View </a>';
                $data[] = $customResult;
            }
        }

        // Get the logged queries
        $queries = DB::getQueryLog();

        // Count the queries
        $queryCount = count($queries);
        \Log::debug(end($queries));
        \Log::debug("UNIVERSITY APPLICATION DATABASE QUERY COUNT: " . $queryCount);

        $json_data = array(
            "draw"            => intval($request->input('draw')),
            "recordsTotal"    => intval($totalData),
            "recordsFiltered" => intval($totalFiltered),
            "data"            => $data,
            "status"   =>   $status,
            "task_deadline" => $task_deadline,
        );

        return json_encode($json_data);
    }


    // public function search(Request $request)
    // {

    //     $data = $request->all();
    //     $find = $data['search'];
    //     $filters = array("search" => $find);
    //     $applications = DB::table('university_application')
    //         ->join('users', 'users.id', '=', 'university_application.userid')
    //         ->where('university_application.country', '=', $find)
    //         ->orWhere('university_application.course', 'like', "%$find%")
    //         ->orWhere('university_application.university', 'like', "%$find%")
    //         ->orWhere('users.first_name', 'like', "%$find%")
    //         ->orWhere('users.last_name', 'like', "%$find%")
    //         ->orWhere('users.email', '=', $find)
    //         ->orWhere('users.mobile', '=', $find)
    //         ->orderBy('university_application.id', "DESC")->select(DB::raw('university_application.*, users.first_name, users.last_name, users.email, users.mobile'))->paginate(5);
    //     return view('backend.applications', ['applications' => $applications, 'filters' => $filters]);
    // }

    public function assignCollegeApplStaff($id){
        if (!(\Auth::user()->isAdmin() || \Auth::user()->can('application-assignment'))) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        
        $collegeApplication = DB::table('university_application')->where('id', $id)->first();
        $userdata = User::find($collegeApplication->userid);
        if (\Auth::user()->can('view assigned data only')) {
            $staff = User::where('id', \Auth::user()->id)->get();
        } else {
            $userRepo = new StaffRepository();
            $staff = $userRepo->getStaffUsers(get_staff_branch_ids(\Auth::user()->id));
        }

        return view('backend.applications.college-app-assign-staff', ['collegeApplication' => $collegeApplication, 'userdata' => $userdata, 'staff' => $staff]);
    }

    public function assignCollegeApplStaffUpdate(Request $request){
        if (!(\Auth::user()->isAdmin() || \Auth::user()->can('application-assignment'))) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        
        $staffid = $request->staff;
        $collegeApplicationId = $request->id;
        $user = User::find($staffid);
        $lead_assigned_success = false;
        $email_message_part = '';

        if (DB::table('university_application')->where("id", $collegeApplicationId)->update(['assign_to' => $staffid])) {
            $email_message_part = "";
            $collegeApplication = DB::table('university_application')->where("id", $collegeApplicationId)->first();
            $collegeUser = User::find($collegeApplication->userid);

            $email_message_part .= "Application #: ".$collegeApplication->appl_id."<br>Application University:".$collegeApplication->university."<br>Application Course:".$collegeApplication->course."<br>Application Country:".$collegeApplication->country."<br> Client Name: $collegeUser->first_name $collegeUser->last_name<br /><br />Mobile:$collegeUser->mobile<br /><br />Email: $collegeUser->email <br/><br/><a href=\"" . url("admin/applications") . "\">View Application Details</a>";
            $subject = "University Application #: ".$collegeApplication->appl_id." Assigned by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
            if(isset($request->staff)){
                $staff = $request->staff;
                $email_message_part = '';
                $email_message_part .= "<br>Application University: ".$collegeApplication->university;
                $email_message_part .= "<br>Application Course: ".$collegeApplication->course;
                $email_message_part .= "<br>Application Country: ".get_country_name($collegeApplication->country);
                $email_message_part .= "<br>Client Name: ". $collegeUser->first_name." ".$collegeUser->last_name;
                $email_message_part .= "<br>Mobile: ".$collegeUser->mobile;
                $email_message_part .= "<br>Email: ".$collegeUser->email;
            }
            $email_message = "Hi $user->name, <br><br>University Application assigned to you.<br />" . $email_message_part;

            // New way to send email
            //Sending email Notification
            send_mail_notification($collegeUser->id,$collegeApplication->id,"APPLICATION_ASSIGNED_STAFF",$subject,$email_message);

            // Send Push App Notification
            $message = "University Appl #".$collegeApplication->appl_id." of <strong>".get_user_full_name_by_id($collegeUser->id) . "</strong> (" . $collegeUser->id . ") assigned to " .$user->full_name;
            $subject = "";
            send_in_app_notification(
                $collegeApplication->userid, 
                $collegeApplication->appl_id,
                "APPLICATION_ASSIGNED_STAFF", 
                $subject, 
                $message,
                "university"
            );

            //Adding activity
            UserActivity::create(['user_id' => $collegeUser->id, 'type' => 'others', 'title' => "Application asigned to staff", 'detail' => "Application #" . $collegeApplication->appl_id." Assigned to ".get_user_full_name_by_id($staffid), 'staff' => \Auth::user()->id]);

            if($user){
                return redirect()->route('admin.applications')->withFlashSuccess('Staff Assigned!');
            }else{
                return redirect()->route('admin.applications')->withFlashSuccess('Staff Assigned Removed!');
            }
        } else {
            return redirect()->route('admin.applications')->withFlashDanger('Something Went Wrong!');
        }
    }

    public function detail($id){
        $row = DB::table('university_application')
            ->join('users', 'users.id', '=', 'university_application.userid')
            ->leftJoin('users as staff', 'university_application.assign_to', '=', 'staff.id')
            ->select('university_application.*', 'users.deleted_at', 'users.archived', 'staff.first_name as staff_first_name', 'staff.last_name as staff_last_name')
            ->where('university_application.id', $id)
            ->whereNull('users.deleted_at')
            ->where('users.archived', "0")
            ->first();
        // dd($row);
        $msgs = DB::table('application_messages')->where('application_id', $id)->orderBy('id','ASC')->get();
        if (!$row) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger("University Applications Not Found");
        }

        $visa_pref = "";
        // if($row->type == 'visa') {
        //     $visa_pref = DB::table('visa_preference')->where('id', $row->visa_pref_id)->first();
        // }
        $abroad_job_pref = "";
        // if($row->type == 'abroad_job') {
        //     $abroad_job_pref = DB::table('abroad_job_preference')->where('id', $row->abroad_job_pref_id)->first();
        // }

        $user = User::find($row->userid);
        // dd($user);
        if (\Auth::user()->isAgent()) {
            if ($user->agent_id != \Auth::user()->id) {
                return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
            }
        }

        //Updating the Student Application
        $studata = array();
        $cv = "";
        $passports = array();
        $sops = array();
        $lors = array();
        $internships = array();
        $offerletters = array();
        $experience = array();
        $otherdocs = array();

        if (!DB::table('user_documents')->where('userid', $row->userid)->exists()) {
            DB::table('user_documents')->insert(['userid' => $row->userid]);
        }

        $sturow = DB::table('user_documents')->where('userid', $row->userid)->first();
        $studata['class10'] = $sturow->class10;
        $studata['class10_marks_sheet'] = $sturow->class10_marks_sheet;
        $studata['class12'] = $sturow->class12;
        $studata['class12_marks_sheet'] = $sturow->class12_marks_sheet;
        $studata['diploma'] = $sturow->diploma;
        $studata['diploma_marks_sheet'] = $sturow->diploma_marks_sheet;
        $studata['grad_all_sem'] = $sturow->grad_all_sem;
        $studata['grad_degree'] = $sturow->grad_degree;
        $studata['grad_provisional'] = $sturow->grad_provisional;
        $studata['grad_transcripts'] = $sturow->grad_transcripts;
        $studata['pg_all_sem'] = $sturow->pg_all_sem;
        $studata['pg_provisional'] = $sturow->pg_provisional;
        $studata['cmm'] = $sturow->cmm;
        $studata['pg_degree'] = $sturow->pg_degree;
        $studata['pg_transcripts'] = $sturow->pg_transcripts;
        $studata['lang_ielts'] = $sturow->lang_ielts;
        $studata['lang_pte'] = $sturow->lang_pte;
        $studata['gre'] = $sturow->gre;
        $studata['gmat'] = $sturow->gmat;
        $studata['duolingo'] = $sturow->duolingo;

        $docrows = DB::table('user_docs_other')->where('userid', $row->userid)->get();
        if ($docrows) {
            foreach ($docrows as $docrow) {
                if ($docrow->doc_type == 'cv') {
                    $cv = $docrow->file;
                } elseif ($docrow->doc_type == 'passport') {
                    $passports[] = $docrow;
                } elseif ($docrow->doc_type == 'sop') {
                    $sops[] = $docrow;
                } elseif ($docrow->doc_type == 'letter_of_recommendation') {
                    $lors[] = $docrow;
                } elseif ($docrow->doc_type == 'internship') {
                    $internships[] = $docrow->file;
                } elseif ($docrow->doc_type == 'offer') {
                    $offerletters[] = $docrow->file;
                } elseif ($docrow->doc_type == 'experience') {
                    $experience[] = $docrow;
                } elseif ($docrow->doc_type == 'payslip') {
                    $payslips[] = $docrow->file;
                } elseif ($docrow->doc_type == 'other') {
                    $otherdocs[] = $docrow;
                }
            }
        }

        $studata['cv'] = $cv;
        $studata['passports'] = $passports;
        $studata['sops'] = $sops;
        $studata['lors'] = $lors;
        $studata['internships'] = $internships;
        $studata['offerletters'] = $offerletters;
        $studata['experience'] = $experience;
        $studata['otherdocs'] = $otherdocs;

        $course_id = $uni_id = "";
        if($row->course_pref_id) {
            $course_pref = DB::table('course_preference')->where('id', $row->course_pref_id)->first(['course_id','university_id']);
            if($course_pref){
                $course_id = $course_pref->course_id;
                $uni_id = $course_pref->university_id;
            }
        }

        $userDetails = DB::table('user_details')->where('userid', $row->userid)->first();
        $appActivity = ApplicationActivity::where('application_id',$id)->orderBy('created_at','desc')->get();
        $currency = DB::table('currency')->get();
        $collegeFeePayments = DB::table('college_fees')->where('application_id', $id)->get();
        $myCommission = DB::table('commission')->where('application_id', $id)->get();
        $agentCommission = DB::table('agent_commission')->where('appl_id', $id)->get();
        $applicationDocRequest = DB::table('application_doc_request')->where('application_id', $id)->get();
        $reminders = DB::table('reminders')->where('entity', "university_application")->where('entity_id', $id)->get();
        return view('backend.applications.application_detail', [
            'reminders' => $reminders,
            'userDetails' => $userDetails,
            'student' => $user, 
            'data' => $row, 
            // 'visa_pref' => $visa_pref, 
            // 'abroad_job_pref' => $abroad_job_pref,
            'course_id'=>$course_id,
            'uni_id'=>$uni_id, 
            'app_messages' => $msgs,
            'appActivity'=>$appActivity,
            'currency' => $currency,
            'collegeFeePayments' => $collegeFeePayments,
            'myCommission' => $myCommission,
            'agentCommission' => $agentCommission,
            'applicationDocRequest' => $applicationDocRequest,
        ],$studata);
    }

    public function delete(Request $request){
        
        if (!(\Auth::user()->isAdmin() || \Auth::user()->can('delete-lead-student-application-batch'))) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }

        $id = $request->get('application_id');
        //dd($id);
        $collegeAppl = UniversityApplication::where('id', $id)->first();

        if($collegeAppl) {
            $user = User::find($collegeAppl->userid);
            if (\Auth::user()->isAgent()) {
                if ($user->agent_id != \Auth::user()->id) {
                    return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
                }
            }
            
            $application_message = ApplicationMessages::where('application_id', $collegeAppl->id)->get();
            if($application_message) {
                $application_message->each->delete();
            }

            // Send Push App Notification
            if(\Auth::user()->isAgent()){
                $message = "Application Deleted by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
                $email_message = "Application Deleted by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
                $subject = "Application #".$collegeAppl->appl_id." Deleted by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
            } else{
                $message = "Application Deleted by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
                $email_message = "Application Deleted by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
                $subject = "Application #".$collegeAppl->appl_id." Deleted by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
            }

            //Notify Staff
            $email_message .= "<br><b>Application Details:</b><br>University Application #".$collegeAppl->appl_id ."<br>Course: ".$collegeAppl->course . "<br>University: " . $collegeAppl->university . "<br>Country: " .$collegeAppl->country;
            $email_message .= "<br>"._label('backend_student').": " . $user->full_name." (". $user->id. ")";
            send_mail_notification($user->id,$id,"LEAD_STUDENT_APPLICATION_COURSEPREFERENCE_OTHERS_DELETE",$subject,$email_message);

            $message .= " University: ".$collegeAppl->university.
            "<br>Course: ".$collegeAppl->course.
            "<br>Country: ".$collegeAppl->country.
            "<br>"._label('backend_student').": " . get_user_full_name_by_id($user->id) . " (" . $user->id . ")";
            $subject = "";
            send_in_app_notification(
                $collegeAppl->userid, 
                $collegeAppl->appl_id, 
                "LEAD_STUDENT_APPLICATION_COURSEPREFERENCE_OTHERS_DELETE", 
                $subject, 
                $message,
                "university"
            );
            $collegeAppl->delete();
            
            return redirect('admin/auth/user/'.$collegeAppl->userid.'#application')->withFlashSuccess("Application Deleted Successfully.");
        } else {
            return redirect()->back()->withFlashDanger("Something went wrong!");
        }
    }

    public function updateNotes(Request $request)
    {
        if (!(\Auth::user()->isAdmin() || \Auth::user()->isAgent() || \Auth::user()->can('manage university applications'))) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        $request->validate([
            'notes' => 'max:500'
        ]);
        $applData = UniversityApplication::find($request->appl_id);
        $applData->notes = $request->notes;
        $applData->save();
        return redirect()->back()->withFlashSuccess('Notes Updated!');
    }

    public function sendMessage(Request $request)
    {
        if (!(\Auth::user()->isAdmin() || \Auth::user()->isAgent() || \Auth::user()->can('manage university applications'))) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        //dd($request->all());
        $request->validate([
            'message' => 'required',
            'attachment.*' => 'mimes:jpg,jpeg,docx,pdf,doc,png|max:8000'
        ],[
            'attachment.*.mimes' => 'Only pdf, doc, docx, jpeg, png files are allowed',
            'image_file.*.max' => 'Sorry! Maximum allowed file size is 8MB',
            ]
        );

        $data = $request->all();
        $application = UniversityApplication::find($request->app_id);
        $student = User::find($application->userid);
        //$data['insert_timestamp'] = date("Y-m-d H:i:s");
        $message = $data['message'];
        $messages = str_replace('#first_name#', $student->first_name, $message);
        $messages = str_replace('#full_name#', $student->first_name . " " . $student->last_name, $messages);
        $messages = str_replace('#sitename#', app_name(), $messages);
        $messages = str_replace('#siteurl#', config('app.url'), $messages);

        $msg = new ApplicationMessages;
        $msg->application_id = $request->app_id;
        $msg->user_id = \Auth::user()->id;
        // $msg->message = $request->message;
        $msg->message = $messages;
        $user_activity = new UserActivity;
        $user_activity->type = "message_sent_to_student";
        $user_activity->user_id = $student->id;
        $user_activity->title = "Application Message Sent by Staff";
        $user_activity->detail = "Application #".$application->appl_id.": ".get_user_full_name_by_id(\Auth::user()->id).":" . $messages;
        $user_activity->staff = \Auth::user()->id;
        $user_activity->attachment = "";

        if ($request->file('attachment')) {
            $cnt = 1;
            foreach ($request->file('attachment') as $file){
                $filename = $file->getClientOriginalName();
                $filename = str_replace(" ", "-", $filename);
                $filename = preg_replace('/[^A-Za-z0-9\-_]+/', '', $filename);
                if (strlen($filename) > 20) {
                    $filename = substr($filename, 0, 20);
                }
                $filename .= Str::random(10);
                $path = $file->storeAs('/app_messages', $filename . "." . $file->getClientOriginalExtension(), 'public');
                if($cnt == 1){
                    $msg->attachment_1 = $path;
                    $user_activity->attachment .= $path;
                }
                elseif($cnt == 2){
                    $msg->attachment_2 = $path;
                    $user_activity->attachment .= "," . $path;
                }
                elseif($cnt == 3){
                    $msg->attachment_3 = $path;
                    $user_activity->attachment .= "," . $path;
                }
                elseif($cnt == 4){
                    $msg->attachment_4 = $path;
                    $user_activity->attachment .= "," . $path;
                }
                $cnt++;
            }
        }

        if(\Auth::user()->isAgent()){
            $message = "Application #".$application->appl_id." Message by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
            $subject = "Application #".$application->appl_id." Message by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
        } else{
            $message = "Application #".$application->appl_id." Message by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
            $subject = "Application #".$application->appl_id." Message by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
        }
        $message .= " - " . $messages;
        $message .= " University: ".
            $application->university.", Course: ".
            $application->course.", Country: ".
            $application->country.", Student Email: ". 
            $student->email;

        if(isset($request->notify_staff)){
            $noti_staff = array();
            foreach($request->notify_staff as $staff){
                //echo "<br>".$staff;
                $noti = new UserNotification;
                $noti->user_id = $staff;
                $noti->application_id = $request->app_id;
                $noti->application_type = 'university';
                $noti->message = $message;
                $noti->from_user = \Auth::user()->id;
                $noti->save();
                $noti_staff[] = $staff;
            }
            $staff_list = implode(',',$noti_staff);
            $msg->notified_staff = $staff_list;
        }
        $msg->save();
        $user_activity->save();
        
        $email_message = $messages;
        if(!$student->agent_id) {
            //We must notify student if this student is not belongs to any student.
            $agent = User::find($student->agent_id);
            try{
                //$student->notify(new UserAlert("New message for University Appl#".$application->appl_id, $email_message));
                user_notify($student, "New message for University Appl#".$application->appl_id, $email_message); 
            }catch (\Exception $e) {
                $message = $e->getMessage();
                \Log::debug("Mail Error: (Uni App. - (User) Send Message) ".$message);   
            }
        } else{
            $agent = User::where("agent_id", $student->agent_id)->first();
            if($agent){
                //store event in user activity
                UserActivity::create(['user_id' => $agent->id, 'type' => 'email_sent', 'title' => "Email Notification", 'detail' => "Subject:" . "New message for your application #".$application->appl_id . "<br>Message:" . $email_message, 'staff' => \Auth::user()->id]);
            }
        }

        // Send Push App Notification
        $email_message .= "<br />Student Name: ".get_user_full_name_by_id($student->id).
        ",<br />Student Email: ".$student->email.
        ",<br />New message for University Application #".
        $application->appl_id;
        send_mail_notification($student->id,$application->id,"UNIVERSITY_APPLICATION_MESSAGE_BY_STUDENT",$subject,$email_message);

        $subject = "";
        send_in_app_notification(
            $application->userid, 
            $application->appl_id, 
            "UNIVERSITY_APPLICATION_MESSAGE_BY_STUDENT", 
            $subject, 
            $message,
            "university"
        );

        return redirect()->back()->withFlashSuccess("Message sent successfully");
    }


    public function create()
    {
        if (!(\Auth::user()->isAdmin() || \Auth::user()->isAgent() || \Auth::user()->can('manage university applications'))) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        $countries = CountryStudy::where('status', 1)->orderBy('country')->get();
        return view('backend.applications.applications_create', compact('countries'));
    }

    public function store(Request $request)
    {
        if (!(\Auth::user()->isAdmin() || \Auth::user()->isAgent() || \Auth::user()->can('manage university applications'))) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }

        $request->validate([
            'userid' => 'required',
            'country' => 'required',
            'course' => 'required',
            'university' => 'required',
            'year' => 'required',
            'intake' => 'required',
        ]);

        $data = $request->all();
        //$data['insert_timestamp'] = date("Y-m-d H:i:s");
        UniversityApplication::create($data);

        update_student_overseas_status($request->input('userid'));

        return redirect()->route('admin.applications')->withFlashSuccess("University Applications added successfully");
    }

    public function status_update_notes(Request $request){
        if (\Auth::user()->isAgent()) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        if (!(\Auth::user()->isAdmin() || \Auth::user()->isAgent() || \Auth::user()->can('manage university applications'))) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        $id = $request->input('id');
        
        $row = DB::table('university_application')->where('id', $id)->first();
        $new_status = $request->input('new_status');
        $prev_status = $row->status;
        // dd($request->all());
        if ($row) {
            $update_data = array();
            $notify_message = $sms_msg = "";
            $user = User::find($row->userid);
            /***************************************/
            $update_data['status'] = $new_status;
            $title = "APPLICATION STATUS CHANGED TO: ".$update_data['status'];
            if($request->closed=='1'){
                $update_data['case_closed'] = 1;
                $update_data['case_closed_reason'] = $request->close_appl_reason;
                $update_data['case_closed_ts'] = date("Y-m-d H:i:s");
                $title .= "<br>Reason for Closure: ".ucfirst(str_replace("_"," ",$request->close_appl_reason));
            }
            $updated = false;
            $updated = DB::table('university_application')->where("id", $id)->update($update_data);
            update_student_overseas_status($row->userid);
            
            $notes = "";
            if($request->input('notes')){
                $notes = $request->input('notes');
            }
            $chk_activity = ApplicationActivity::where('application_id', $id)->where('title', $title)->where('staff',"!=",'1')->get();
            if(count($chk_activity)){
                $title = "";
            }
            $file_array = "";
            if($request->hasFile('app_stage_file')){
                $request->validate([
                    'app_stage_file.*' => 'mimes:docx,pdf,doc,jpeg,png|max:8000'
                ],[
                    'app_stage_file.*.mimes' => 'Only pdf, doc, docx, png, jpeg files are allowed',
                    'app_stage_file.*.max' => 'Sorry! Maximum allowed file size is 8MB',
                ]);

                $file_array = array();
                foreach ($request->file('app_stage_file') as $file){
                   $filename = $file->getClientOriginalName();
                   
                   $filename = str_replace(" ", "-", $filename);
                   $filename = preg_replace('/[^A-Za-z0-9\-_]+/', '', $filename);
                   if (strlen($filename) > 20) {
                       $filename = substr($filename, 0, 20);
                       }
                       $filename .= Str::random(10);
                       $path = $file->storeAs('/app_activity', $filename . "." . $file->getClientOriginalExtension(), 'public');
                       $file_array[] = $filename . "." . $file->getClientOriginalExtension();
                   }
                   $file_array = implode(",",$file_array);
               }
            
            ApplicationActivity::create(['application_id' => $id, 'type' => 'status_update', 'title' => $title, 'detail' => $notes, 'staff' => \Auth::user()->id,'attachment' => $file_array]);

            $notify_message .= "New University Applications Status: " . $update_data['status'];
            if ($updated) {
                $app_detail = DB::table('university_application')->where("id", $id)->first();
                if (strlen($notify_message)) {
                    if(!$user->agent_id) {
                        //Notify Student
                        $user = (new UserRepository())->where('id', $row->userid)->first();
                        //email
                        // if($app_detail->type == 'visa') {
                        //     $mail_tpl = MailTemplates::find("VISA_APPLICATION_UPDATE");
                        // } elseif($app_detail->type == 'abroad_job') {
                        //     $mail_tpl = MailTemplates::find("ABROAD_JOB_APPLICATION_UPDATE");
                        // } else {
                            $mail_tpl = MailTemplates::find("UNIVERSITY_APPLICATION_UPDATE");
                        // }
                        $email_message = $mail_tpl->message;
                        //\Log::debug("MAIL BEFORE:".$email_message);                       
                        $email_message = str_ireplace("#sitename#", app_name(), $email_message);
                        $email_message = str_ireplace("#siteurl#", url(''), $email_message);
                        $email_message = str_ireplace("#first_name#", $user->first_name, $email_message);
                        $email_message = str_ireplace("#full_name#", $user->full_name, $email_message);
                        // if($app_detail->type == 'visa' || $app_detail->type == 'abroad_job') {
                        //     $email_message = str_ireplace("#application_name#", $row->country, $email_message);                        
                        // } else {
                            $email_message = str_ireplace("#application_name#", $row->course . "/" . $row->university . "/" . $row->country, $email_message);
                        // }
                        $email_message = str_ireplace("#details#", $notify_message, $email_message);
                        $email_message .= ", Previous Status: ".$prev_status;
                        
                        if ($mail_tpl && $mail_tpl->status) {
                            try{
                                //$user->notify(new UserAlert($mail_tpl->subject, $email_message));
                                user_notify($user,$mail_tpl->subject, $email_message);
                            }catch (\Exception $e) {
                                $message = $e->getMessage();
                                \Log::debug("Mail Error: (Uni App. Update - Notify User) ".$message);   
                            }
                        }
                    }

                    // Sending SMS
                    // if($app_detail->type == 'visa') {
                    //     $notify_sms = get_sms_template('VISA_STATUS_UPDATE');
                    // } elseif($app_detail->type == 'abroad_job') {
                    //     $notify_sms = get_sms_template('ABROAD_JOB_STATUS_UPDATE');
                    // } else {                    
                        $notify_sms = get_sms_template('UNIVERSITY_APPLICATION_UPDATE');
                    // }
                    if ($notify_sms && $notify_sms->status) {
                        $notify_sms->message = str_replace("#sitename#", app_name(), $notify_sms->message);
                        $notify_sms->message = str_replace("#siteurl#", url('/'), $notify_sms->message);
                        $notify_sms->message = str_replace("#first_name#", $user->first_name, $notify_sms->message);
                        $notify_sms->message = str_replace("#full_name#", $user->full_name, $notify_sms->message);

                        // if($app_detail->type == 'visa' || $app_detail->type == 'abroad_job') {
                        //     $notify_sms->message = str_replace("#application#", "Country: ".$row->country, $notify_sms->message);
                        // } else {
                            $notify_sms->message = str_replace("#application#", $row->course . "/" . $row->university . "/" . $row->country, $notify_sms->message);
                        // }
                        $notify_sms->message = str_replace("#details#", $sms_msg, $notify_sms->message);

                        if($user->agent_id) {
                            //We must notify agent instead
                            $agent = User::find($user->agent_id);
                            $agent->sms($notify_sms->message, $notify_sms->template_id);
                        } else {
                            $user->sms($notify_sms->message, $notify_sms->template_id);
                        }
                    }
                }
                
                // Send Push App Notification
                if(\Auth::user()->isAgent()){
                    $message = "Appl <strong>#".$row->appl_id."</strong> (<strong>".$user->full_name."</strong> (". $user->id. ")) Status changed by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
                    $email_message = "Hi, <br>Application #".$row->appl_id." Status Changed by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
                } else{
                    $message = "Appl <strong>#".$row->appl_id."</strong> (<strong>".$user->full_name."</strong> (". $user->id. ")) Status changed by " . get_user_full_name_by_id(\Auth::user()->id);
                    $email_message = "Hi, <br>Application #".$row->appl_id." Status Changed by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
                }
                
                //Notify Staff
                $subject = "Status Change: Appl #". $row->appl_id . " - ".$user->full_name." (". $user->id. ")";
                $email_message .= "<br><br>Previous Status: ".$prev_status."<br>New Status: ".$new_status."<br><b>Application Details:</b><br>University Application #".$row->appl_id ."<br>Course: ".$row->course . "<br>University: " . $row->university . "<br>Country: " .$row->country;
                $email_message .= "<br>"._label('backend_student').": " . $user->full_name." (". $user->id. ")";
                send_mail_notification($user->id,$id,"UNIVERSITY_APPLICATION_STATUS_CHANGE",$subject,$email_message);
                
                $message .=  ", Previous Status: ".$prev_status.", New Status: <strong>".$new_status."</strong>";
                $subject = "";
                send_in_app_notification(
                    $row->userid, 
                    $row->appl_id, 
                    "UNIVERSITY_APPLICATION_STATUS_CHANGE", 
                    $subject, 
                    $message,
                    "university"
                );

                // Send Whatsapp 
                if (get_site_config('whatsapp_enabled')) {
                    $numbers = [$user->whatsapp_dialcode.$user->whatsapp];

                    $hook = 'university_application_update';

                    $customMessage = [
                        'application' => $row->appl_id,
                        'new_status'  => $new_status
                    ];

                    send_whatsapp_message($numbers, $hook, $user->id, $customMessage);
                }

                return redirect()->route('admin.applicationdetail', $id)->withFlashSuccess("University Applications Status Updated Successfully");
            } else {
                return redirect()->route('admin.applicationdetail', $id);
            }
        } else {
            return redirect()->route('admin.applicationdetail', $id)->withFlashDanger("University Applications Not Found");
        }
    }

    public function updateStatus(UpdateApplicationRequest $request){
        if (!(\Auth::user()->isAdmin() || \Auth::user()->isAgent() || \Auth::user()->can('manage university applications'))) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        if(!empty($request->input('application_date'))){
            $request->validate([
                'open_date' => 'required|date',
                'close_date' => 'required|date|after:open_date',
            ]);
        }
        
        if (\Auth::user()->isAgent()) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        $id = $request->input('id');
        $app_open_date = $app_close_date = "";

        if(!empty($request->input('open_date'))){
            $app_open_date = date('Y-m-d',strtotime($request->input('open_date')));
        }
        if(!empty($request->input('close_date'))){
            $app_close_date =date('Y-m-d',strtotime($request->input('close_date')));
        }

        if(!empty($app_open_date)){
            $update_data['application_open_date'] = $app_open_date;
        
        }
        if(!empty($app_close_date)){
            $update_data['application_close_date'] = $app_close_date;
        }     
        
        if(!empty($request->input('applied_with'))){
            $update_data['applied_with'] = $request->input('applied_with');
        }

        $updated = false;

        if(count($update_data) > 0){
            if (DB::table('university_application')->where("id", $id)->update($update_data)) {
                return redirect()->route('admin.applicationdetail', $id)->withFlashSuccess("University Applications Date Updated Successfully");
            } else{
                return redirect()->route('admin.applicationdetail', $id)->withFlashDanger("University Applications Not Found");
            }
        }
    }

    public function closeCase($id)
    {
        if (!(\Auth::user()->isAdmin() || \Auth::user()->isAgent() || \Auth::user()->can('manage university applications'))) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        $row = DB::table('university_application')->where('id', $id)->first();
        if (!$row) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger("University Applications Not Found");
        }

        $user = User::find($row->userid);
        if (\Auth::user()->isAgent()) {
            if ($user->agent_id != \Auth::user()->id) {
                return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
            }
        }

        return view('backend.applications.application_close', ['student' => $user, 'data' => $row]);
    }

    // public function agentCommission($id)
    // {
    //     if (!(\Auth::user()->isAdmin() || \Auth::user()->can('commission')) || \Auth::user()->isAgent()) {
    //         return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
    //     }
    //     $row = DB::table('university_application')->where('id', $id)->first();
    //     $msgs = DB::table('application_messages')->where('application_id', $id)->orderBy('id','desc')->get();
    //     if (!$row) {
    //         return redirect()->route('admin.dashboard.message')->withFlashDanger("Application Not Found");
    //     }

    //     $user = User::find($row->userid);
    //     $agent = User::find($user->agent_id);
    //     return view('backend.applications.agent-commission', ['student' => $user,'agent' => $agent, 'data' => $row,'app_messages' => $msgs]);
    // }

    // public function agentCommissionUpdate(Request $request)
    // {
    //     if (!(\Auth::user()->isAdmin() || \Auth::user()->can('commission'))) {
    //         return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
    //     }
    //     $request->validate([
    //         'total_commission' => 'required',
    //         'agent_percent' => 'required',
    //     ]);

    //     $app = UniversityApplication::find($request->app_id);
    //     $app->total_commission = $request->total_commission;
    //     $app->agent_percent = $request->agent_percent;
    //     $app->save();
    //     return redirect()->route('admin.applicationdetail',$request->app_id)->withFlashSuccess("Agent commission updated successfully");
    // }

    public function search_staff(Request $request)
    {
        $my_branch = get_staff_branch_ids(\Auth::user()->id);
        $q =  $request->input('q', '');
        if(\Auth::user()->isAgent()){
            $users = User::with('roles')
                ->whereHas('roles', function ($q) {
                //   $q->whereIn('name', ['agent','staff member','administrator']);
                  $q->whereNotIn('name', ['user','agent']);
                })
                ->where(function ($query) use ($q) {
                    $query->where('first_name', 'LIKE', $q . '%')
                    ->orwhere('last_name', 'LIKE', $q . '%')
                    ->orWhere('email', '=', $q)
                    ->orWhere('mobile', '=', $q);
                })
                //->whereIn('branch', $my_branch)
                ->select('id', DB::raw("concat(first_name, ' ', last_name, ' | ', email) as text"))
                ->get();
            // return ['results' => $users,'branch' => $my_branch];
        }else{
            $users = User::with('roles')
                ->whereHas('roles', function ($q) {
                  $q->where('name', '!=', 'user');
                // $q->whereIn('name', ['agent','staff member','administrator']);
                })
                ->where(function ($query) use ($q) {
                    $query->where('first_name', 'LIKE', $q . '%')
                    ->orwhere('last_name', 'LIKE', $q . '%')
                    ->orWhere('email', '=', $q)
                    ->orWhere('mobile', '=', $q);
                })
                //->whereIn('branch', $my_branch)
                ->select('id', DB::raw("concat(first_name, ' ', last_name, ' | ', email) as text"))
                ->get();
        }
        return ['results' => $users,'branch' => $my_branch];
    }

    // ----------------------------------------------------------------------------------------------------------------------

    public function setFeeCommission(Request $request){
        if (!(\Auth::user()->isAdmin() || \Auth::user()->can('commission'))) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }

        $request->validate([
            'fees_currency' => 'required',
            'tuition_fees' => 'required|numeric',
            'my_commission_percent' => 'required|numeric',
        ]);

        $application = UniversityApplication::where("id", $request->application_id)->first();
        $userApplication = User::where("id", $application->userid)->first();
        if($userApplication->agent_id){
            $request->validate([
                'agent_commission_percent' => 'required|numeric',
            ]);
        }

        $universityApplication = UniversityApplication::where("id", $request->application_id)->first();
        $universityApplication->fees_currency = $request->fees_currency;
        $universityApplication->tuition_fees = $request->tuition_fees;

        if($userApplication->agent_id){
            $universityApplication->agent_commission_percent = $request->agent_commission_percent;
            $universityApplication->agent_commission = $request->tuition_fees * $request->agent_commission_percent / 100;
        }
        $universityApplication->my_commission_percent = $request->my_commission_percent;
        $universityApplication->my_commission = $request->tuition_fees * $request->my_commission_percent / 100;

        $universityApplication->save();
        
        // return redirect()->route('admin.applicationdetail', $request->application_id)->withFlashSuccess("Fees and commission added successfully");
        return redirect('admin/application/detail/'.$request->application_id.'#fees-and-commission')->withFlashSuccess("Fees and commission added successfully");
    }

    // ----------------------------------------------------------------------------------------------------------------------

    public function addCollegeFeePayment(Request $request){
        if (!(\Auth::user()->isAdmin() || \Auth::user()->can('commission'))) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        $request->validate([
            'date_paid' => 'required|date|before_or_equal:' . date('Y-m-d'),
            'college_fee_amount' => 'required|numeric',
            'college_fee_notes' => 'required|max:400',
        ]);

        $date_paid = substr($request->input('date_paid'), 6, 4) . "-" . substr($request->input('date_paid'), 3, 2) . "-" . substr($request->input('date_paid'), 0, 2);

        // $tution_fees = $request->tution_fees;
        // $totalAmount = CollegeFees::where('application_id', $request->application_id)->get();
        // $remainingAmount = $request->college_fee_amount;
        // foreach($totalAmount as $amount){
        //     $remainingAmount += $amount->college_fee_amount;
        // }
        // if($remainingAmount > $tution_fees){
        //     return redirect()->route('admin.applicationdetail', $request->application_id)->withFlashDanger("College Fee Payment more than Total Fees!");
        // }

        $collegeFeePayment = new CollegeFees([
            'application_id' => $request->application_id,
            'college_fee_amount' => $request->college_fee_amount,
            'notes' => $request->college_fee_notes,
            'date_paid' => $date_paid,
            'created_by' => \Auth::user()->id,
        ]);
        $collegeFeePayment->save();

        // return redirect()->route('admin.applicationdetail', $request->application_id)->withFlashSuccess("University Fee Payment added successfully");
        return redirect('admin/application/detail/'.$collegeFeePayment->application_id.'#fees-and-commission')->withFlashSuccess("College Fee Payment Updated successfully");
    }

    public function editCollegeFeePayment($id){
        $collegeFees = CollegeFees::where('id', $id)->first();
        return view('backend.applications.edit-college-fee-payment', ['collegeFees' => $collegeFees]);
    }

    public function editSaveCollegeFeePayment(Request $request){
        if (!(\Auth::user()->isAdmin() || \Auth::user()->can('commission'))) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        $request->validate([
            'date_paid' => 'required|date|before_or_equal:' . date('Y-m-d'),
            'college_fee_amount' => 'required|numeric',
            'college_fee_notes' => 'required|max:400',
        ]);

        $date_paid = substr($request->input('date_paid'), 6, 4) . "-" . substr($request->input('date_paid'), 3, 2) . "-" . substr($request->input('date_paid'), 0, 2);

        // $tution_fees = UniversityApplication::where('id', $request->application_id)->first()->tuition_fees;
        // $totalAmount = CollegeFees::where('id', '!=', $request->id)->where('application_id', $request->application_id)->get();
        // $remainingAmount = $request->college_fee_amount;
        // foreach($totalAmount as $amount){
        //     $remainingAmount += $amount->college_fee_amount;
        // }
        // if($remainingAmount > $tution_fees){
        //     return redirect()->route('admin.applicationdetail', $request->application_id)->withFlashDanger("University Fee Payment more than Total Fees!");
        // }

        $collegeFeePayment = CollegeFees::where('id', $request->id)->first();
        $collegeFeePayment->college_fee_amount = $request->college_fee_amount;
        $collegeFeePayment->notes = $request->college_fee_notes;
        $collegeFeePayment->date_paid = $date_paid;
        $collegeFeePayment->save();

        // return redirect()->route('admin.applicationdetail', $collegeFeePayment->application_id)->withFlashSuccess("University Fee Payment Updated successfully");
        return redirect('admin/application/detail/'.$collegeFeePayment->application_id.'#fees-and-commission')->withFlashSuccess("College Fee Payment Updated successfully");
    }

    public function deleteCollegeFeePayment($id, $application_id){
        if (!(\Auth::user()->isAdmin() || \Auth::user()->can('commission'))) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        $collegeFeePayment = CollegeFees::where('id', $id)->first();
        if($collegeFeePayment){
            $collegeFeePayment->delete();
            return redirect('admin/application/detail/'.$application_id.'#fees-and-commission')->withFlashSuccess("College Fee Payment deleted successfully");
        }
        return redirect('admin/application/detail/'.$application_id.'#fees-and-commission')->withFlashDanger("College Fee Payment not present!");
    }

    // ----------------------------------------------------------------------------------------------------------------------

    public function addMyCommission(Request $request){
        if (!(\Auth::user()->isAdmin() || \Auth::user()->can('commission'))) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        $request->validate([
            'date_received' => 'required|date|before_or_equal:' . date('Y-m-d'),
            'my_commission_amount' => 'required|numeric',
            'my_commission_notes' => 'nullable|max:400',
        ]);

        $date_received = substr($request->input('date_received'), 6, 4) . "-" . substr($request->input('date_received'), 3, 2) . "-" . substr($request->input('date_received'), 0, 2);

        // $my_commission = $request->my_commission;
        // $totalCommissionAmount = Commission::where('application_id', $request->application_id)->get();
        // $remainingCommissionAmount = $request->my_commission_amount;
        // foreach($totalCommissionAmount as $amount){
        //     $remainingCommissionAmount += $amount->my_commission_amount;
        // }
        // if($remainingCommissionAmount > $my_commission){
        //     return redirect()->route('admin.applicationdetail', $request->application_id)->withFlashDanger("My Commission Amount is more than Expected Commission Amount!");
        // }

        $myCommission = new Commission([
            'application_id' => $request->application_id,
            'date_received' => $date_received,
            'my_commission_amount' => $request->my_commission_amount,
            'notes' => $request->my_commission_notes,
            'created_by' => \Auth::user()->id,
        ]);
        $myCommission->save();
        
        // return redirect()->route('admin.applicationdetail', $request->application_id)->withFlashSuccess("My Commission added successfully");
        return redirect('admin/application/detail/'.$request->application_id.'#fees-and-commission')->withFlashSuccess("My Commission added successfully");
    }

    public function editMyCommission($id){
        if (!(\Auth::user()->isAdmin() || \Auth::user()->can('commission'))) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        $myCommission = Commission::where('id', $id)->first();
        return view('backend.applications.edit-my-commission', ['myCommission' => $myCommission]);
    }

    public function editSaveMyCommission(Request $request){
        if (!(\Auth::user()->isAdmin() || \Auth::user()->can('commission'))) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        $request->validate([
            'date_received' => 'required|date|before_or_equal:' . date('Y-m-d'),
            'my_commission_amount' => 'required|numeric',
            'my_commission_notes' => 'nullable|max:400',
        ]);

        $date_received = substr($request->input('date_received'), 6, 4) . "-" . substr($request->input('date_received'), 3, 2) . "-" . substr($request->input('date_received'), 0, 2);

        // $my_commission = UniversityApplication::where('id', $request->application_id)->first()->my_commission;
        // $totalCommissionAmount = Commission::where('id', '!=', $request->id)->where('application_id', $request->application_id)->get();
        // $remainingCommissionAmount = $request->my_commission_amount;
        // foreach($totalCommissionAmount as $amount){
        //     $remainingCommissionAmount += $amount->my_commission_amount;
        // }
        // if($remainingCommissionAmount > $my_commission){
        //     return redirect()->route('admin.applicationdetail', $request->application_id)->withFlashDanger("My Commission more than Commission Amount!");
        // }

        $myCommission = Commission::where('id', $request->id)->first();
        $myCommission->my_commission_amount = $request->my_commission_amount;
        $myCommission->notes = $request->my_commission_notes;
        $myCommission->date_received = $date_received;
        $myCommission->save();

        // return redirect()->route('admin.applicationdetail', $request->application_id)->withFlashSuccess("My Commission Updated successfully");
        return redirect('admin/application/detail/'.$request->application_id.'#fees-and-commission')->withFlashSuccess("My Commission Updated successfully");
    }
    
    public function deleteMyCommission($id, $application_id){
        if (!(\Auth::user()->isAdmin() || \Auth::user()->can('commission'))) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        $myCommission = Commission::where('id', $id)->first();
        if($myCommission){
            $myCommission->delete();
            return redirect('admin/application/detail/'.$application_id.'#fees-and-commission')->withFlashSuccess("My Commission deleted successfully");
        }
        return redirect('admin/application/detail/'.$application_id.'#fees-and-commission')->withFlashDanger("College Fee Payment not present!");
    }

    // ----------------------------------------------------------------------------------------------------------------------

    public function addAgentCommission(Request $request){
        // dd($request->all());
        if (!(\Auth::user()->isAdmin() || \Auth::user()->can('commission'))) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        $request->validate([
            'commission_paid_date' => 'required|date|before_or_equal:' . date('Y-m-d'),
            'agent_commission_amount' => 'required|numeric',
            'agent_commission_notes' => 'nullable|max:400',
        ]);
        
        $commission_paid_date = substr($request->input('commission_paid_date'), 6, 4) . "-" . substr($request->input('commission_paid_date'), 3, 2) . "-" . substr($request->input('commission_paid_date'), 0, 2);

        // $agent_commission = $request->agent_commission;
        // $totalCommissionAmount = AgentCommission::where('appl_id', $request->application_id)->get();
        // $remainingCommissionAmount = $request->agent_commission_amount;
        // foreach($totalCommissionAmount as $amount){
        //     $remainingCommissionAmount += $amount->amount;
        // }
        // if($remainingCommissionAmount > $agent_commission){
        //     return redirect()->route('admin.applicationdetail', $request->application_id)->withFlashDanger("Agent Commission Amount is more than Expected Commission Amount!");
        // }
        $university = DB::table("university_application")->where("id", $request->application_id)->first();
        $agentCommission = new AgentCommission([
            'appl_id' => $request->application_id,
            'agent_id' => $request->agent_id,
            'currency' => $university->fees_currency,
            'created_at' => $commission_paid_date,
            'amount' => $request->agent_commission_amount,
            'agent_commission_notes' => $request->agent_commission_notes,
            'created_by' => \Auth::user()->id,
        ]);
        $agentCommission->save();

        // Send Push App Notification
        $student = User::where("id", $university->userid)->first();

        //Notify Staff
        $subject = "Application #".$university->appl_id." Commission added by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
        $email_message = "Hi, <br>Application #".$university->appl_id." Commission added by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
        $email_message .= "<br><b>Application Details:</b><br>University Application #".$university->appl_id ."<br>Course: ".$university->course . "<br>University: " . $university->university . "<br>Country: " .$university->country;
        $email_message .= "<br>"._label('backend_student').": " . $student->full_name." (". $student->id. ")";
        send_mail_notification($student->id,$university->id,"AGENT_COMMISSION_ONLY",$subject,$email_message);

        $subject = "";
        $message = "Application Commission added by Staff: " . get_user_full_name_by_id(\Auth::user()->id) . " Application #: " . $university->appl_id;
        $message .= "<br>Application #: ".$university->appl_id.
        "<br>"._label('backend_student').": " . $student->full_name." (". $student->id. ")";
        send_in_app_notification(
            $university->userid, 
            $university->appl_id, 
            "AGENT_COMMISSION_ONLY", 
            $subject, 
            $message,
            "university"
        );
        
        // return redirect()->route('admin.applicationdetail', $request->application_id)->withFlashSuccess("Agent Commission added successfully");
        return redirect('admin/application/detail/'.$request->application_id.'#fees-and-commission')->withFlashSuccess("Agent Commission added successfully");
    }

    public function editAgentCommission($id){
        if (!(\Auth::user()->isAdmin() || \Auth::user()->can('commission'))) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        $agentCommission = AgentCommission::where('id', $id)->first();
        return view('backend.applications.edit-agent-commission', ['agentCommission' => $agentCommission]);
    }

    public function editSaveAgentCommission(Request $request){
        if (!(\Auth::user()->isAdmin() || \Auth::user()->can('commission'))) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        $request->validate([
            'commission_paid_date' => 'required|date|before_or_equal:' . date('Y-m-d'),
            'agent_commission_amount' => 'required|numeric',
            'agent_commission_notes' => 'nullable|max:400',
        ]);
        $commission_paid_date = substr($request->input('commission_paid_date'), 6, 4) . "-" . substr($request->input('commission_paid_date'), 3, 2) . "-" . substr($request->input('commission_paid_date'), 0, 2);

        // $agent_commission = UniversityApplication::where('id', $request->application_id)->first()->agent_commission;
        // $totalCommissionAmount = AgentCommission::where('id', '!=', $request->id)->where('appl_id', $request->application_id)->get();
        // $remainingCommissionAmount = $request->agent_commission_amount;
        // foreach($totalCommissionAmount as $amount){
        //     $remainingCommissionAmount += $amount->amount;
        // }
        // if($remainingCommissionAmount > $agent_commission){
        //     return redirect()->route('admin.applicationdetail', $request->application_id)->withFlashDanger("Agent Commission Amount is more than Expected Commission Amount!");
        // }

        $agentCommission = AgentCommission::where('id', $request->id)->first();
        $agentCommission->amount = $request->agent_commission_amount;
        $agentCommission->agent_commission_notes = $request->agent_commission_notes;
        $agentCommission->created_at = $commission_paid_date;
        $agentCommission->save();

        // return redirect()->route('admin.applicationdetail', $request->application_id)->withFlashSuccess("Agent Commission Updated successfully");
        return redirect('admin/application/detail/'.$request->application_id.'#fees-and-commission')->withFlashSuccess("Agent Commission Updated successfully");
    }
    
    public function deleteAgentCommission($id, $application_id){
        if (!(\Auth::user()->isAdmin() || \Auth::user()->can('commission'))) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        $agentCommission = AgentCommission::where('id', $id)->first();
        if($agentCommission){
            $agentCommission->delete();
            return redirect('admin/application/detail/'.$application_id.'#fees-and-commission')->withFlashSuccess("Agent Commission deleted successfully");
        }
        return redirect('admin/application/detail/'.$application_id.'#fees-and-commission')->withFlashDanger("Agent Commission Not Found");
    }
    
    // ----------------------------------------------------------------------------------------------------------------------
    
    public function requestDocument(Request $request){
        $request->validate([
            'title' => 'required|max:300',
        ]);
        
        $application = UniversityApplication::where("id", $request->application_id)->first();
        $student = User::where("id", $application->userid)->first();
        
        $applicationDocRequest = new ApplicationDocRequest([
            'application_id' => $request->application_id,
            'title' => $request->title,
            'staff_id' => \Auth::user()->id,
        ]);
        $applicationDocRequest->save();

        $email_message = $request->title;
        if(!$student->agent_id) {
            try{
                //$student->notify(new UserAlert("Document requested for your application #".$application->appl_id, $email_message));
                user_notify($student,"Document requested for your application #".$application->appl_id, $email_message);
            }catch (\Exception $e) {
                $message = $e->getMessage();
                \Log::debug("Mail Error: (Uni App. - (User) Send Message) ".$message);   
            }
        }
        
        //store event in user activity
        UserActivity::create(['user_id' => $student->id, 'type' => 'email_sent', 'title' => "Email Notification", 'detail' => "Subject:" . "Document requested for your application #".$application->appl_id. "<br>Documents: " . $request->title, 'staff' => \Auth::user()->id]);

        // Send Push App Notification
        if(\Auth::user()->isAgent()){
            $message = "Document Requested by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
            $email_message = "Hi, <br>Application #".$application->appl_id.", Document Requested by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
            $subject = "Application #".$application->appl_id.", Document Requested by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
        } else{
            $message = "Document Requested by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
            $email_message = "Hi, <br>Application #".$application->appl_id.", Document Requested by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
            $subject = "Application #".$application->appl_id.", Document Requested by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
        }

        //Notify Staff
        $email_message .= "<br><b>Application Details:</b><br>University Application #".$application->appl_id ."<br>Course: ".$application->course . "<br>University: " . $application->university . "<br>Country: " .$application->country;
        $email_message .= "<br>"._label('backend_student').": " . $student->full_name." (". $student->id. ")";
        send_mail_notification($student->id,$application->id,"UNIVERSITY_APPLICATION_MESSAGE_BY_STUDENT",$subject,$email_message);

        $message .= "<br>Application #: ".$application->appl_id.
        "<br>"._label('backend_student').": " . $student->full_name." (". $student->id. ")";
        $subject = "";
        send_in_app_notification(
            $application->userid, 
            $application->appl_id, 
            "UNIVERSITY_APPLICATION_MESSAGE_BY_STUDENT", 
            $subject, 
            $message,
            "university"
        );

        return redirect()->route('admin.applicationdetail', $request->application_id)->withFlashSuccess("Document requested successfully");
    }

    public function uploadRequestedDocument(Request $request){
        $request->validate([
            // 'file_upload' => 'required|mimes:pdf,doc,docx,jpeg',
            'file_upload' => 'required',
        ]); 

        $application_doc_request = ApplicationDocRequest::where('id', $request->doc_request_id)->where('type', 'university')->first();
        $file_upload = $request->file_upload;
        $allFiles = "";
        foreach($file_upload as $file){
            $filename = $application_doc_request->title . "-";
            $filename .= $file->getClientOriginalName();
            $filename = str_replace(" ", "-", $filename);
            $filename = preg_replace('/[^A-Za-z0-9\-_]+/', '', $filename);
            if (strlen($filename) > 30) {
                $filename = substr($filename, 0, 30);
            }
            $filename .= Str::random(10);

            $path = $file->storeAs('/cert', $filename . "." . $file->getClientOriginalExtension(), 'public');
            $filename = $filename.".".$file->getClientOriginalExtension();
            $allFiles .= $filename."|";
        }

        $application_doc_request->files = $allFiles;
        $application_doc_request->uploaded_at = now();
        $application_doc_request->save();

        $application = UniversityApplication::where("id", $request->application_id)->first();
        $student = User::where("id", $application->userid)->first();
        
        $email_message = $allFiles;
        $application = UniversityApplication::where("id", $request->application_id)->first();

        if(!$student->agent_id) {
            try{
                //$student->notify(new UserAlert("Requested Doc uploaded for appl#".$application->appl_id, $email_message));
                user_notify($student,"Requested Doc uploaded for appl#".$application->appl_id, $email_message);
                \Log::debug("Document requested Uploaded >> ".$email_message);
            }catch (\Exception $e) {
                $message = $e->getMessage();
                \Log::debug("Mail Error: (Uni App. - (User) Send Message) ".$message);   
            }
        }
        
        //store event in user activity
        UserActivity::create(['user_id' => $student->id, 'type' => 'admin_event', 'title' => "Email Notification", 'detail' => "Subject:" . "Document requested for your application #".$application->appl_id. " are uploaded.<br>Documents are: " . $email_message, 'staff' => \Auth::user()->id]);

        // Send Push App Notification
        if(\Auth::user()->isAgent()){
            $message = "Requested Document (".$application_doc_request->title.") are uploaded by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
            $email_message = "Hi, <br>Application #".$application->appl_id.", Requested Document (".$application_doc_request->title.") are uploaded by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
            $subject = "Application #".$application->appl_id.", Requested Document (".$application_doc_request->title.") are uploaded by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
        } else{
            $message = "Requested Document (".$application_doc_request->title.") are uploaded by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
            $email_message = "Hi, <br>Application #".$application->appl_id.", Requested Document (".$application_doc_request->title.") are uploaded by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
            $subject = "Application #".$application->appl_id.", Requested Document (".$application_doc_request->title.") are uploaded by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
        }

        //Notify Staff
        $email_message .= "<br><b>Application Details:</b><br>University Application #".$application->appl_id ."<br>Course: ".$application->course . "<br>University: " . $application->university . "<br>Country: " .$application->country;
        $email_message .= "<br>"._label('backend_student').": " . $student->full_name." (". $student->id. ")";
        send_mail_notification($student->id,$application->id,"UNIVERSITY_APPLICATION_MESSAGE_BY_STUDENT",$subject,$email_message);

        $message .= "<br>Application #: ".$application->appl_id.
        "<br>"._label('backend_student').": " . $student->full_name." (". $student->id. ")";
        $subject = "";
        send_in_app_notification(
            $application->userid, 
            $application->appl_id, 
            "UNIVERSITY_APPLICATION_MESSAGE_BY_STUDENT", 
            $subject, 
            $message,
            "university"
        );

        return redirect()->route('admin.applicationdetail', $request->application_id)->withFlashSuccess("Document requested successfully");
    }

    // ----------------------------------------------------------------------------------------------------------------------
    public function add_activity(Request $request){
        if (!(\Auth::user()->isAdmin() || \Auth::user()->isAgent() || \Auth::user()->can('manage university applications'))) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        $request->validate([
            'activity' => 'required',
            'deadline' => 'required|date',
        ]);
        if (\Auth::user()->isAgent()) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        $reminder = new Reminders;
        $reminder->entity_id = $request->input('appl_id');
        $reminder->activity = $request->input('activity');
        $reminder->entity = "university_application";
        $reminder->deadline = date('Y-m-d',strtotime($request->input('deadline')));
        $reminder->created_by = \Auth::user()->id;
        $reminder->save();
        // return redirect()->route('admin.applicationdetail', $request->input('appl_id'))->withFlashSuccess("Reminder added Successfully");
        return redirect('admin/application/detail/'.$request->input('appl_id').'#reminders')->withFlashSuccess("Reminder added Successfully");
    }

    public function delete_activity(Request $request){
        if (\Auth::user()->isAgent()) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        if (!(\Auth::user()->isAdmin() || \Auth::user()->can('delete-lead-student-application'))) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        if (DB::table('reminders')->where("id", $request->id)->delete()) {
            return response()->json(['message'=>'Activity removed']);
        } else{
            return response()->json(['message'=>'Activity Not Found']);
        }
    }

    public function activity_done(Request $request){
        if (\Auth::user()->isAgent()) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        if($request->action == 1){
            DB::table('reminders')->where("id", $request->id)->update(["done"=>1,"done_date"=>now(),"done_by"=>\Auth::user()->id]);
            return response()->json(['message'=>'Activity marked as Done.']);
        }
        else if($request->action == 0){
            DB::table('reminders')->where("id", $request->id)->update(["done"=>0,"done_date"=>NULL,"done_by"=>NULL]);
            return response()->json(['message'=>'Activity marked as Not Done.']);
        }
        // if (DB::table('reminders')->where("id", $request->id)->update(["done"=>$request->action,"done_date"=>now(),"done_by"=>\Auth::user()->id])) {
        //     return response()->json(['message'=>'Activity marked as Done.']);
        // } 
        else{
            return response()->json(['message'=>'Activity Not Found']);
        }
        
    }

    public function add_task(Request $request){
        $request->validate([
            'task' => 'required',
            'deadline' => 'required|date',
        ]);
        
        if (\Auth::user()->isAgent()) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        if (!(\Auth::user()->isAdmin() || \Auth::user()->isAgent() || \Auth::user()->can('manage university applications'))) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        $appl_id = $request->input('appl_id');
        $task = $request->input('task');
        $deadline = date('Y-m-d',strtotime($request->input('deadline')));

        if (DB::table('university_application')->where("id", $appl_id)->update(["task"=>$task,"deadline"=>$deadline])) {
           
            return redirect()->route('admin.applicationdetail', $appl_id)->withFlashSuccess("Task added Successfully");
        } else{
            return redirect()->route('admin.applicationdetail', $appl_id)->withFlashDanger("University Applications Not Found");
        }
    }

    public function clear_task($appl_id){
        if (\Auth::user()->isAgent()) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        if (!(\Auth::user()->isAdmin() || \Auth::user()->isAgent() || \Auth::user()->can('manage university applications'))) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        if (DB::table('university_application')->where("id", $appl_id)->update(["task"=>NULL,"deadline"=>NULL])) {
            return redirect()->route('admin.applicationdetail', $appl_id)->withFlashSuccess("Task removed");
        } else{
            return redirect()->route('admin.applicationdetail', $appl_id)->withFlashDanger("University Applications Not Found");
        }
        
    }

    public function courseUniversityEditView($course_pref, $byApplOrCouPref){
        if (!(\Auth::user()->isAdmin() || \Auth::user()->isAgent() || \Auth::user()->can('manage university applications'))) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        $pref = DB::table('course_preference')->where('id', $course_pref)->first();
        $applicationDetails = DB::table('university_application')->where("course_pref_id", $course_pref)->first();
        return view('backend.applications.edit-course-university',['pref' => $pref, 'byApplOrCouPref' => $byApplOrCouPref, 'applicationDetails' => $applicationDetails]);
    }

    public function courseUniversityEditSave(Request $request){
        $request->validate([
           'country'  => 'required',
           'course'  => 'required',
           'intake'  => 'required',
       ]);

       $user = User::find($request->input('user_id'));
       $userid = $user->id;

       $intake = $request->input('intake');
       $inta =  explode(" ",$intake);

       if($request->input('course') == ''){
           $course_preference_current = DB::table('course_preference')->where('id', $request->pref_id)->first();
           $course = $course_preference_current->course_id;
           $course_other = $course_preference_current->course_other;
       }else{
           if(filter_var($request->input('course'), FILTER_VALIDATE_INT) !== false){
               $course = $request->input('course');
               $course_name = get_university_course_data($request->input('course'))->course_name;
               $course_other = null;
           }else{
               $course = 0;
               $course_other = $request->input('course');
               $course_name = $request->input('course');
           }
       }
       
       if($request->input('university') == ''){
           $course_preference_current = DB::table('course_preference')->where('id', $request->pref_id)->first();
           $university = $course_preference_current->university_id;
           $university_other = $course_preference_current->university_other;
       }else{
           if(filter_var($request->input('university'), FILTER_VALIDATE_INT) !== false){
               $university = $request->input('university');
               $university_name = get_university_data($request->input('university'))->university_name;
               $university_other = null;
           }else{
               $university = 0;
               $university_other = $request->input('university');
               $university_name = $request->input('university');
           }
       }

       $country = get_country_name($request->country);

       $prev_course_pref = DB::table('course_preference')->where('id', $request->pref_id)->first();
       $old_course = "";
       $old_university = "";
       if($prev_course_pref->course_id){
           $old_course = $old_course = get_university_course_data($prev_course_pref->course_id)->course_name;
       } else{
           $old_course = $prev_course_pref->course_other;
       }
       if($prev_course_pref->university_id){
           $old_university = get_university_data($prev_course_pref->university_id)->university_name;
       } else{
           $old_university = $prev_course_pref->university_other;
       }

       DB::table('course_preference')->where('id', $request->pref_id)
           ->update(
               [
                   'intake_year' => $inta[1],
                   'intake' => $inta[0],
                   'course_other' => $course_other,
                   'course_id' => $course,
                   'university_other' => $university_other,
                   'university_id' => $university,
                   'country_id' => $request->country,
               ]
           );
       
        DB::table('university_application')->where('course_pref_id', $request->pref_id)
           ->update(
               [
                   'university' => $university_name, 
                   'course' => $course_name, 
                   'country' => $country, 
                   'year' => $inta[1], 
                   'intake' => $inta[0]
               ]
           );

        //store event in user activity
        $applicationId = DB::table('university_application')->where('course_pref_id', $request->pref_id)->first();
        ApplicationActivity::create([
            'application_id' => $applicationId->id, 
            'type' => 'others', 
            'title' => "Course and University Updated", 
            'detail' => "<br>Old Data > New data: <br>Country: ".get_country_name($prev_course_pref->country_id)." > $country <br>University: $old_university > $university_name <br>Course: $old_course > $course_name <br>Intake: $prev_course_pref->intake_year, $prev_course_pref->intake > $inta[1], $inta[0]", 
            'staff' => \Auth::user()->id
        ]);
        
        // Send Push App Notification
        $message = "Application #".$applicationId->appl_id." Edited by Staff: " . get_user_full_name_by_id(\Auth::user()->id) . 
        "<br>Student: " . get_user_full_name_by_id($user->id)." (". $user->id. ")";
        $subject = "";
        send_in_app_notification(
            $applicationId->userid, 
            $applicationId->appl_id,
            "UNIVERSITY_APPLICATION_CREATED", 
            $subject, 
            $message,
            "university"
        );

        if($request->byApplOrCouPref == "course_pref"){
            return redirect('admin/auth/user/'.$userid.'#coursepreference')->withFlashSuccess("Course and University updated successfully");
        } elseif($request->byApplOrCouPref == "apllication"){
            return redirect()->route('admin.applicationdetail', $applicationId->id)->withFlashSuccess("Course and University updated successfully");
        }
    }


    public function downloadDocsZip($id)
    {
        $records = DB::table('application_doc_request')
            ->where('application_id', $id)
            ->get();

        if ($records->isEmpty()) {
            return redirect()->back()->with('error', 'No documents found to download.');
        }

        // Get the appl_id for filename
        $application = DB::table('university_application')->where('id', $id)->first();
        $appNo = $application ? $application->appl_id : $id;

        // Collect all files
        $allFiles = [];

        foreach ($records as $record) {
            if (!empty($record->files)) {
                $files = explode('|', $record->files);
                foreach ($files as $file) {
                    if (!empty($file)) {
                        $allFiles[] = $file;
                    }
                }
            }
        }

        if (empty($allFiles)) {
            return redirect()->back()->with('error', 'No valid files found to zip.');
        }

        $zipFileName = 'documents_' . $appNo . '.zip';
        // $zipFileName = $appNo . '.zip';
        $zipPath = storage_path('app/public/' . $zipFileName);

        $zip = new \ZipArchive();
        if ($zip->open($zipPath, \ZipArchive::CREATE | \ZipArchive::OVERWRITE) === TRUE) {
            foreach ($allFiles as $file) {
                $filePath = storage_path('app/public/cert/' . $file);
                if (file_exists($filePath)) {
                    $zip->addFile($filePath, $file);
                }
            }
            $zip->close();

            return response()->download($zipPath)->deleteFileAfterSend(true);
        }

        return redirect()->back()->with('error', 'Could not create ZIP file.');

    }

    // BULK OPERATION 
    public function bulkAssignView(Request $request){
        $applicationIds = explode(',', $request->selected_ids);

        // Fetch applications with user (student) details
        $applications = DB::table('university_application as ua')
            ->join('users as u', 'ua.userid', '=', 'u.id')
            ->select('ua.id as application_id', 'ua.appl_id', 'u.first_name', 'u.last_name')
            ->whereIn('ua.id', $applicationIds)
            ->get();

        return view('backend.applications.bulk_assign', [
            'selected_ids' => $applicationIds,
            'applications' => $applications,
            'staff' => User::where('is_staff', 1)->whereNull('deleted_at')->get(),
        ]);
    }

    public function bulkAssignUpdate(Request $request){
        $request->validate([
            'application_ids' => 'required|array',
            'staff' => 'required|exists:users,id',
        ]);

        $user = User::find($request->staff);
        foreach($request->application_ids as $id) {
            $collegeApplication = UniversityApplication::find($id);
            $prevStaff = $collegeApplication->assign_to;
            $collegeApplication->assign_to = $request->staff;
            $collegeApplication->updated_at = now();
            $collegeApplication->save();

            if($prevStaff == $request->staff) continue;

            $collegeUser = User::find($collegeApplication->userid);
            $staffid = $request->staff;
            $staffName = get_user_full_name_by_id($staffid);
            //Adding activity
            // ApplicationActivity::create([
            //     'application_id' => $id, 
            //     'type' => 'others', 
            //     'title' => "Application Assignment", 
            //     'detail' => "Application asigned to staff: " . $staffName,
            //     'staff' => \Auth::user()->id
            // ]);

            UserActivity::create([
                'user_id' => $collegeUser->id, 
                'type' => 'others', 
                'title' => "Application asigned to staff", 
                'detail' => "Application #" . $collegeApplication->appl_id." Assigned to ".get_user_full_name_by_id($staffid), 
                'staff' => \Auth::user()->id
            ]);

            $email_message_part = "";
            $email_message_part .= "Application #: ".$collegeApplication->appl_id."<br>Application University:".$collegeApplication->university."<br>Application Course:".$collegeApplication->course."<br>Application Country:".$collegeApplication->country."<br> Client Name: $collegeUser->first_name $collegeUser->last_name<br /><br />Mobile:$collegeUser->mobile<br /><br />Email: $collegeUser->email <br/><br/><a href=\"" . url("admin/applications") . "\">View Application Details</a>";
            $subject = "University Application #: ".$collegeApplication->appl_id." Assigned by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
            if(isset($request->staff)){
                $staff = $request->staff;
                $email_message_part = '';
                $email_message_part .= "<br>Application University: ".$collegeApplication->university;
                $email_message_part .= "<br>Application Course: ".$collegeApplication->course;
                $email_message_part .= "<br>Application Country: ".get_country_name($collegeApplication->country);
                $email_message_part .= "<br>Client Name: ". $collegeUser->first_name." ".$collegeUser->last_name;
                $email_message_part .= "<br>Mobile: ".$collegeUser->mobile;
                $email_message_part .= "<br>Email: ".$collegeUser->email;
            }
            $email_message = "Hi $user->name, <br><br>University Application assigned to you.<br />" . $email_message_part;

            // New way to send email
            //Sending email Notification
            send_mail_notification($collegeUser->id,$collegeApplication->id,"APPLICATION_ASSIGNED_STAFF",$subject,$email_message);

            // Send Push App Notification
            $message = "University Appl #".$collegeApplication->appl_id." of <strong>".get_user_full_name_by_id($collegeUser->id) . "</strong> (" . $collegeUser->id . ") assigned to " .$user->full_name;
            $subject = "";
            send_in_app_notification(
                $collegeApplication->userid, 
                $collegeApplication->appl_id,
                "APPLICATION_ASSIGNED_STAFF", 
                $subject, 
                $message,
                "university"
            );
        }

        return redirect()->route('admin.applications')->withFlashSuccess('Applications assigned successfully.');
    }

    public function bulkStatusChangeView(Request $request){
        $ids = explode(',', $request->selected_ids);          
        $applications = DB::table('university_application as ua')
            ->join('users as u', 'ua.userid', '=', 'u.id')
            ->whereIn('ua.id', $ids)
            ->select(
                'ua.id',               
                'ua.appl_id',           
                'u.first_name',
                'u.last_name'
            )
            ->get();

        return view('backend.applications.bulk_statuschange', [
            'applications' => $applications,              
            'statusList'   => get_application_statuses(),
        ]);
    }

    public function bulkStatusUpdate(Request $request){
        $request->validate([
            'selected_ids' => 'required|array',
            'status'       => 'required|string',
            'note'         => 'nullable|string',
        ]);

        foreach($request->selected_ids as $id) {
            $row = UniversityApplication::find($id);
            $prev_status = $row->status;
            $new_status = $request->status;

            $row->status = $request->status;
            $row->notes = $request->note;
            $row->updated_at = now();
            $row->save();

            if($prev_status == $new_status) continue;

            ApplicationActivity::create([
                'application_id' => $id, 
                'type' => 'status_update', 
                'title' => "APPLICATION STATUS CHANGED TO: " . $request->status, 
                'detail' => $request->note, 
                'staff' => \Auth::user()->id
            ]);

            $user = DB::table('users')->where("id", $row->userid)->first();
            $notify_message = $sms_msg = "";
            $notify_message .= "New University Applications Status: " . $request->status;
            if (strlen($notify_message)) {
                if(!$user->agent_id) {
                    //Notify Student
                    $user = (new UserRepository())->where('id', $row->userid)->first();
                    $mail_tpl = MailTemplates::find("UNIVERSITY_APPLICATION_UPDATE");
                    $email_message = $mail_tpl->message;                      
                    $email_message = str_ireplace("#sitename#", app_name(), $email_message);
                    $email_message = str_ireplace("#siteurl#", url(''), $email_message);
                    $email_message = str_ireplace("#first_name#", $user->first_name, $email_message);
                    $email_message = str_ireplace("#full_name#", $user->full_name, $email_message);
                    $email_message = str_ireplace("#application_name#", $row->course . "/" . $row->university . "/" . $row->country, $email_message);
                    $email_message = str_ireplace("#details#", $notify_message, $email_message);
                    $email_message .= ", Previous Status: " . $prev_status;
                    
                    if ($mail_tpl && $mail_tpl->status) {
                        try{
                            user_notify($user, $mail_tpl->subject, $email_message);
                        }catch (\Exception $e) {
                            $message = $e->getMessage();
                            \Log::debug("Mail Error: (Uni App. Update - Notify User) ".$message);   
                        }
                    }
                }

                // Sending SMS
                $notify_sms = get_sms_template('UNIVERSITY_APPLICATION_UPDATE');
                if ($notify_sms && $notify_sms->status) {
                    $notify_sms->message = str_replace("#sitename#", app_name(), $notify_sms->message);
                    $notify_sms->message = str_replace("#siteurl#", url('/'), $notify_sms->message);
                    $notify_sms->message = str_replace("#first_name#", $user->first_name, $notify_sms->message);
                    $notify_sms->message = str_replace("#full_name#", $user->full_name, $notify_sms->message);

                    $notify_sms->message = str_replace("#application#", $row->course . "/" . $row->university . "/" . $row->country, $notify_sms->message);
                    $notify_sms->message = str_replace("#details#", $sms_msg, $notify_sms->message);

                    if($user->agent_id) {
                        //We must notify agent instead
                        $agent = User::find($user->agent_id);
                        $agent->sms($notify_sms->message, $notify_sms->template_id);
                    } else {
                        $user->sms($notify_sms->message, $notify_sms->template_id);
                    }
                }
            }
            
            // Send Push App Notification
            $message = "Appl <strong>#".$row->appl_id."</strong> (<strong>".$user->full_name."</strong> (". $user->id. ")) Status changed by " . get_user_full_name_by_id(\Auth::user()->id);
            $email_message = "Hi, <br>Application #".$row->appl_id." Status Changed by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
            
            //Notify Staff
            $subject = "Status Change: Appl #". $row->appl_id . " - ".$user->full_name." (". $user->id. ")";
            $email_message .= "<br><br>Previous Status: ".$prev_status."<br>New Status: ".$new_status."<br><b>Application Details:</b><br>University Application #".$row->appl_id ."<br>Course: ".$row->course . "<br>University: " . $row->university . "<br>Country: " .$row->country;
            $email_message .= "<br>"._label('backend_student').": " . $user->full_name." (". $user->id. ")";
            send_mail_notification($user->id,$id,"UNIVERSITY_APPLICATION_STATUS_CHANGE",$subject,$email_message);
            
            $message .=  ", Previous Status: ".$prev_status.", New Status: <strong>".$new_status."</strong>";
            $subject = "";
            send_in_app_notification(
                $row->userid, 
                $row->appl_id, 
                "UNIVERSITY_APPLICATION_STATUS_CHANGE", 
                $subject, 
                $message,
                "university"
            );

            // Send Whatsapp 
            if (get_site_config('whatsapp_enabled')) {
                $numbers = [$user->whatsapp_dialcode.$user->whatsapp];
                $hook = 'university_application_update';

                $customMessage = [
                    'application' => $row->appl_id,
                    'new_status'  => $new_status
                ];

                send_whatsapp_message($numbers, $hook, $user->id, $customMessage);
            }
        }

        return redirect()->route('admin.applications')->withFlashSuccess('Status updated for selected applications.');
    }

    public function bulkDeleteView(Request $request){
        $ids = explode(',', $request->selected_ids);

        $applications = DB::table('university_application as ua')
            ->join('users as u', 'ua.userid', '=', 'u.id')
            ->whereIn('ua.id', $ids)
            ->select('ua.id', 'ua.appl_id', 'u.first_name', 'u.last_name')
            ->get();

        return view('backend.applications.bulk_delete', [
            'applications' => $applications
        ]);
    }

    public function bulkDelete(Request $request){
        $request->validate([
            'selected_ids' => 'required|array',
        ]);

        DB::table('university_application')
            ->whereIn('id', $request->selected_ids)
            ->delete();

        return redirect()->route('admin.applications')->withFlashSuccess('Selected applications deleted successfully.');
    }
}
