<?php

namespace App\Http\Controllers\Backend;

use PDF;
use App\Models\Auth\User;
use App\Models\Auth\Country;
use Illuminate\Http\Request;
use App\Models\MailTemplates;
use App\Models\Auth\UserDetails;
use App\Models\Auth\Universities;
use App\Models\CourseShortlisted;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use App\Models\Auth\UniversityCourses;
use App\Models\Auth\UniversityCourseCategories;
use Illuminate\Support\Facades\Auth;
use App\Exceptions\GeneralException;
use Illuminate\Support\Str;
use File;

class UniversityCourseManagementController extends Controller {

    public function __construct(){
        $this->middleware(function ($request, $next) {
            $user = \Auth::user();
            if (!(\Auth::user()->isAdmin() || \Auth::user()->isAgent() || \Auth::user()->can('course finder'))) {
                return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
            }            
            return $next($request);
        });
    }

    private function cleanUpText($str){
        $str = str_replace('&nbsp;', ' ', $str);
        $str = html_entity_decode($str, ENT_QUOTES | ENT_COMPAT , 'UTF-8');
        $str = html_entity_decode($str, ENT_HTML5, 'UTF-8');
        $str = html_entity_decode($str);
        $str = htmlspecialchars_decode($str);
        $str = strip_tags($str);
    
        return $str;
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index() {
        $countries = get_study_countries();
        $levels = get_study_levels();
        $categories = UniversityCourseCategories::where('parent_id', '=', 0)->where('status',1)->get();
        return view('backend.course_management.index', ['countries' => $countries, 'levels' => $levels, 'categories' => $categories]);
    }

    public function indexData(Request $request){

        if (!$request->ajax()) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this page.');
        }
        echo $this->getDatatableCourseManagementlist($request);
    }

    public function getDatatableCourseManagementlist($request){
        //Query debug
        // DB::enableQueryLog();

        $columns = array(
            // 0 => 'id',
            // 1 => 'id',
            3 => 'id',
            4 => 'course_name',
            5 => 'university_name',
            6 => 'country',
            7 => 'study_level',
            8 => 'category',
            9 => 'intakes',
            10 => 'ielts_score',
            11 => 'ielts_no_band_less_than',
            12 => 'toefl_score',
            13 => 'toefl_no_band_less_than',
            14 => 'pte_score',
            15 => 'pte_no_band_less_than',
            16 => 'gre_score',
            17 => 'gmat_score',
            18 => 'status',
            19 => 'status',
            //18 => 'action2'
        );        

        $study_level = $request->input('study_level');
        $country = $request->input('country');
        $subcategory = $category = "";
        $subcategories = array();
        if($request->category){
            if(!empty($request->sub_category)){
                $subcategory = $request->sub_category;
            }else{
                // $category = $request->category;
                $subcategories = UniversityCourseCategories::where('parent_id',$request->category)->where('status',1)->pluck('id')->toArray();
                $subcategories = array_values($subcategories);
                $subcategories[] = $request->category;
                // dd($subcategories);
            }
        }

        $limit = $request->input('length');
        $start = $request->input('start');
        $order = $columns[$request->input('order.0.column')];
        $dir = $request->input('order.0.dir');

        $get_search = $request->input('search.value');

        $totalData = DB::table('university_courses')
            ->join('universities', 'university_courses.university', '=', 'universities.id')
            ->leftjoin('university_course_categories', 'university_courses.category', '=', 'university_course_categories.id')
            ->select('universities.country','universities.university_name', 'university_courses.*')
            ->when(!empty($study_level), function ($query) use ($study_level) {
                return $query->where('university_courses.study_level',$study_level);
            })
            ->when(!empty($subcategory), function ($query) use ($subcategory) {
                return $query->where('university_courses.category', $subcategory);
            })
            ->when(!empty($category), function ($query) use ($category) {
                return $query->where('university_courses.category', $category);
            })
            ->when(!empty($subcategories) && count($subcategories), function ($query) use ($subcategories) {
                return $query->whereIn('university_courses.category', $subcategories);
            })
            ->when(!empty($country), function ($query) use ($country) {
                return $query->where('universities.country', $country);
            })
            ->when(!empty($get_search), function ($query) use ($get_search) {
                return $query->Where(function ($query) use ($get_search) {
                    $query->Where('course_name', 'like', "%$get_search%")
                        ->orWhere('universities.university_name', 'like', "%$get_search%")
                        ->orWhere('universities.country', 'like', "%$get_search%")
                        ->orWhere('university_course_categories.title', 'like', "%$get_search%")
                        ->orWhere('study_level', 'like', "%$get_search%")
                        ->orWhere('university_courses.id', '=', "$get_search")
                        ->orWhere('duration', 'like', "%$get_search%") 
                        ->orWhere('campus', 'like', "%$get_search%") 
                        ->orWhere('intakes', 'like', "%$get_search%") 
                        ->orWhere('year', 'like', "%$get_search%")
                        ->orWhere('website_url', 'like', "%$get_search%") 
                        ->orWhere('scholarship_detail', 'like', "%$get_search%")
                        ->orWhere('scholarship_available', 'like', "%$get_search%")
                        ->orWhere('backlog_range', 'like', "%$get_search%")
                        ->orWhere('remarks', 'like', "%$get_search%")
                        ->orWhere('esl_elp_detail', 'like', "%$get_search%")
                        ->orWhere('applicationmode', 'like', "%$get_search%")
                        ->orWhere('detscore', '=', "$get_search")
                        ->orWhere('ielts_score', '=', "$get_search")
                        ->orWhere('ielts_no_band_less_than', '=', "$get_search")
                        ->orWhere('toefl_score', '=', "$get_search")
                        ->orWhere('toefl_no_band_less_than', '=', "$get_search")
                        ->orWhere('pte_score', '=', "$get_search")
                        ->orWhere('pte_no_band_less_than', '=', "$get_search")
                        ->orWhere('gre_score', '=', "$get_search")
                        ->orWhere('gmat_score', '=', "$get_search")
                        ->orWhere('university_courses.status', 'like', "%$get_search%");
                });
            })              
            ->count();

        $users = DB::table('university_courses')
            ->join('universities', 'university_courses.university', '=', 'universities.id')
            ->leftjoin('university_course_categories', 'university_courses.category', '=', 'university_course_categories.id')
            ->select('universities.country','universities.university_name', 'university_courses.*')
            
            ->when(!empty($study_level), function ($query) use ($study_level) {
                return $query->where('university_courses.study_level',$study_level);
            })
            ->when(!empty($subcategory), function ($query) use ($subcategory) {
                return $query->where('university_courses.category', $subcategory);
            })
            ->when(!empty($category), function ($query) use ($category) {
                return $query->where('university_courses.category', $category);
            })
            ->when(!empty($subcategories) && count($subcategories), function ($query) use ($subcategories) {
                return $query->whereIn('university_courses.category', $subcategories);
            })
            ->when(!empty($country), function ($query) use ($country) {
                return $query->where('universities.country', $country);
            })

            ->when(!empty($get_search), function ($query) use ($get_search) {
                return $query->Where(function ($query) use ($get_search) {
                    $query->Where('course_name', 'like', "%$get_search%")
                        ->orWhere('universities.university_name', 'like', "%$get_search%")
                        ->orWhere('universities.country', 'like', "%$get_search%")
                        ->orWhere('university_course_categories.title', 'like', "%$get_search%")
                        ->orWhere('study_level', 'like', "%$get_search%")
                        ->orWhere('university_courses.id', '=', "$get_search")
                        ->orWhere('duration', 'like', "%$get_search%") 
                        ->orWhere('campus', 'like', "%$get_search%") 
                        ->orWhere('intakes', 'like', "%$get_search%") 
                        ->orWhere('year', 'like', "%$get_search%")
                        ->orWhere('website_url', 'like', "%$get_search%") 
                        ->orWhere('scholarship_detail', 'like', "%$get_search%")
                        ->orWhere('scholarship_available', 'like', "%$get_search%")
                        ->orWhere('backlog_range', 'like', "%$get_search%")
                        ->orWhere('remarks', 'like', "%$get_search%")
                        ->orWhere('esl_elp_detail', 'like', "%$get_search%")
                        ->orWhere('applicationmode', 'like', "%$get_search%")
                        ->orWhere('detscore', '=', "$get_search")
                        ->orWhere('ielts_score', '=', "$get_search")
                        ->orWhere('ielts_no_band_less_than', '=', "$get_search")
                        ->orWhere('toefl_score', '=', "$get_search")
                        ->orWhere('toefl_no_band_less_than', '=', "$get_search")
                        ->orWhere('pte_score', '=', "$get_search")
                        ->orWhere('pte_no_band_less_than', '=', "$get_search")
                        ->orWhere('gre_score', '=', "$get_search")
                        ->orWhere('gmat_score', '=', "$get_search")
                        ->orWhere('university_courses.status', 'like', "%$get_search%");
                });
            })              
            ->offset($start)
            ->limit($limit)
            ->orderBy($order, $dir)
            ->get();

            
        // $totalData = UniversityCourses::all()->sortByDesc('id')->count();
        $totalFiltered = $totalData;

        $data = $courses = array();
        if (!empty($users)) {
            $count = 1;
            foreach ($users as $page) {
                // $customResult['id'] = null;
                $customResult['id'] = $page->id;
                $customResult['course_id'] = $page->id;
                // $folow_text = $this->get_lead_followup($page->id);


                $customResult['course_name'] =  $page->course_name ? $page->course_name : "--";
                // $customResult['course_name'] = "<span style='white-space:nowrap;'><span class='text-sub'>" . $page->course_name . "</span></span><br />";
                $customResult['university'] = $page->university_name ? $page->university_name : "--";
                // $customResult['university'] = "<span style='white-space:nowrap;'><span class='text-sub'>" . $page->university_name . "</span></span><br />";
            
                $customResult['campus'] = $page->campus ? $page->campus : "--";
                // $customResult['campus'] = "<span style='white-space:nowrap;'><span class='text-sub'>" . $page->campus . "</span></span><br />";
                $customResult['category'] = $page->category ? get_university_course_parent_category_name($page->category ) : "--";
                $customResult['sub_category'] = $page->category ? get_university_course_category_name($page->category ) : "--";
                // $customResult['category'] = "<span style='white-space:nowrap;'><span class='text-sub'>" . get_university_course_category($page->category ). "</span></span><br />";
                $customResult['duration'] = $page->duration ? $page->duration : "--";
                // $customResult['duration'] = $page->duration ? "<span style='white-space:nowrap;'><span class='text-sub'>" . $page->duration . "</span></span><br />" : "--";
                // if ($page->duration) {
                //     $customResult['duration'] = $page->duration ? "<span style='white-space:nowrap;'><span class='text-sub'>" . $page->duration . "</span></span><br />" : "--";
                // }
                $customResult['country'] = $page->country ? get_country_name($page->country) : "--";
                // if ($page->country) {
                //     $customResult['country'] = $page->country ? "<span style='white-space:nowrap;'> <span class='text-sub'>" .$page->country . "</span></span><br />" : "--";
                // }
                $customResult['study_level'] = $page->study_level ? $page->study_level : "--";
                // if ($page->study_level) {
                //     $customResult['study_level'] = $page->study_level ? "<span style='white-space:nowrap;'> <span class='text-sub'>" .$page->study_level . "</span></span><br />" : "--";
                // }
                $customResult['intakes'] = $page->intakes ? $page->intakes : "--";
                // if ($page->intakes) {
                //     $customResult['intakes'] = $page->intakes ? "<span style='white-space:nowrap;'> <span class='text-sub'>" .$page->intakes . "</span></span><br />" : "--";
                // }
                $customResult['year'] = $page->year ? $page->year : "--";
                // if ($page->year) {
                //     $customResult['year'] =  $page->year ? "<span style='white-space:nowrap;'> <span class='text-sub'>" .$page->year . "</span></span><br />" : "--";
                // }
                $customResult['website_url'] = $page->website_url ? "<a href='".$page->website_url."' target='_blank'>". substr($page->website_url, 0, 100) . '...'."</a>" : "--";
                // if ($page->website_url) {
                //     $customResult['website_url'] =  $page->website_url ? "<span style='white-space:nowrap;'> <span class='text-sub'>" .$page->website_url . "</span></span><br />" : "--";
                // }
                $customResult['entry_requirements'] = $page->entry_requirements ? "<span style='word-wrap: break-word;''> <span class='text-sub'>" .$page->entry_requirements . "</span></span><br />" : "--";
                // if ($page->entry_requirements) {
                //     $customResult['entry_requirements'] =  $page->entry_requirements ? "<span style='white-space:nowrap;'> <span class='text-sub'>" .$page->entry_requirements . "</span></span><br />" : "--";
                // }
                $customResult['ielts_score'] = $page->ielts_score ? $page->ielts_score : "--";
                // if ($page->ielts_score) {
                //     $customResult['ielts_score'] =  $page->ielts_score ? "<span style='white-space:nowrap;'> <span class='text-sub'>" .$page->ielts_score . "</span></span><br />" : "--";
                // }
                $customResult['ielts_no_band_less_than'] = $page->ielts_no_band_less_than ? $page->ielts_no_band_less_than : "--";
                // if ($page->ielts_no_band_less_than) {
                //     $customResult['ielts_no_band_less_than'] =  $page->ielts_no_band_less_than ? "<span style='white-space:nowrap;'> <span class='text-sub'>" .$page->ielts_no_band_less_than . "</span></span><br />" : "--";
                // }
                $customResult['toefl_score'] = $page->toefl_score ? $page->toefl_score : "--";
                // if ($page->toefl_score) {
                //     $customResult['toefl_score'] =  $page->toefl_score ? "<span style='white-space:nowrap;'> <span class='text-sub'>" .$page->toefl_score . "</span></span><br />" : "--";
                // }
                $customResult['toefl_no_band_less_than'] = $page->toefl_no_band_less_than ? $page->toefl_no_band_less_than : "--";
                // if ($page->toefl_no_band_less_than) {
                //     $customResult['toefl_no_band_less_than'] =  $page->toefl_no_band_less_than ? "<span style='white-space:nowrap;'> <span class='text-sub'>" .$page->toefl_no_band_less_than . "</span></span><br />" : "--";
                // }
                $customResult['pte_score'] = $page->pte_score ? $page->pte_score : "--";
                // if ($page->pte_score) {                    
                //     $customResult['pte_score'] = $page->pte_score ? "<span style='white-space:nowrap;'> <span class='text-sub'>" .$page->pte_score . "</span></span><br />" : "--";
                // }
                $customResult['pte_no_band_less_than'] = $page->pte_no_band_less_than ? $page->pte_no_band_less_than : "--";
                // if ($page->pte_no_band_less_than) {
                //     $customResult['pte_no_band_less_than'] =  $page->pte_no_band_less_than ? "<span style='white-space:nowrap;'> <span class='text-sub'>" .$page->pte_no_band_less_than . "</span></span><br />" : "--";
                // }
                $customResult['gre_score'] = $page->gre_score ? $page->gre_score : "--";
                // if ($page->gre_score) {
                //     $customResult['gre_score'] =  $page->gre_score ? "<span style='white-space:nowrap;'> <span class='text-sub'>" .$page->gre_score . "</span></span><br />" : "--";
                // }
                $customResult['gmat_score'] = $page->gmat_score ? $page->gmat_score : "--";
                // if ($page->gmat_score) {
                //     $customResult['gmat_score'] =  $page->gmat_score ? "<span style='white-space:nowrap;'> <span class='text-sub'>" .$page->gmat_score . "</span></span><br />" : "--";
                // }
                $customResult['application_deadline'] = $page->application_deadline ? $page->application_deadline : "--";
                // if ($page->application_deadline) {
                //     $customResult['application_deadline'] =  $page->application_deadline ? "<span style='white-space:nowrap;'> <span class='text-sub'>" .$page->application_deadline . "</span></span><br />" : "--";
                // }
                $customResult['application_fee'] = $page->application_fee ? $page->application_fee : "--";
                // if ($page->application_fee) {
                //     $customResult['application_fee'] =  $page->application_fee ? "<span style='white-space:nowrap;'> <span class='text-sub'>" .$page->application_fee . "</span></span><br />" : "--";
                // }
                $customResult['course_details'] = $page->course_details ? $page->course_details : "--";
                $customResult['course_content'] = $page->course_content ? $page->course_content : "--";
                $customResult['practical_work'] = $page->practical_work ? $page->practical_work : "--";

                $customResult['qualify_extended_visa'] = $page->qualify_extended_visa ? $page->qualify_extended_visa : "--";
                $customResult['regional_location'] = $page->regional_location ? $page->regional_location : "--";
                $customResult['concentration'] = $page->concentration ? $page->concentration : "--";

                if($page->country_students_not_accepted){
                    $country_names = array();
                    $countries = explode(",",$page->country_students_not_accepted);
                    foreach($countries as $country){
                        $country_names[] = get_country_name($country);
                    }
                    $countries = implode(", ",$country_names);
                }
                $customResult['country_students_not_accepted'] = (isset($countries)) ? $countries : '--';
                
                $customResult['occupation_outcome'] = $page->occupation_outcome ? $page->occupation_outcome : "--";
                // $customResult['tuition_fee_onshore'] = $page->tuition_fee_onshore ? $page->tuition_fee_onshore : "--";
                $customResult['currency'] = $page->currency .' '.get_currency_symbol($page->currency) ;
                $customResult['yearly_tuition_fee'] = $page->yearly_tuition_fee ? $page->yearly_tuition_fee : "--";
                // $customResult['tuition_fee_offshore'] = $page->tuition_fee_offshore ? $page->tuition_fee_offshore : "--";
                $customResult['non_tuition_fee'] = $page->non_tuition_fee ? $page->non_tuition_fee : "--";
                $customResult['total_tuition_fee'] = $page->total_tuition_fee ? $page->total_tuition_fee : "--";

                $customResult['commission_percentage'] = $page->commission_percentage ? $page->commission_percentage : "--";
                
                // if ($page->tuition_fee) {
                //     $customResult['tuition_fee'] =  $page->tuition_fee ? "<span style='white-space:nowrap;'> <span class='text-sub'>" .$page->tuition_fee . "</span></span><br />" : "--";
                // }
                $customResult['scholarship_available'] = $page->scholarship_available ? $page->scholarship_available : "--";
                // if ($page->scholarship_available) {
                //     $customResult['scholarship_available'] =  $page->scholarship_available ? "<span style='white-space:nowrap;'> <span class='text-sub'>" .$page->scholarship_available . "</span></span><br />" : "--";
                // }
                $customResult['scholarship_detail'] = $page->scholarship_detail ?  "<a href='".$page->scholarship_detail."' target='_blank'>". substr($page->scholarship_detail, 0, 100) . '...'."</a>": "--";
                // if ($page->scholarship_detail) {
                //     $customResult['scholarship_detail'] =  $page->scholarship_detail ? "<span style='white-space:nowrap;'> <span class='text-sub'>" .$page->scholarship_detail . "</span></span><br />" : "--";
                // }
                $customResult['backlog_range'] = $page->backlog_range ? $page->backlog_range : "--";
                // if ($page->backlog_range) {
                //     $customResult['backlog_range'] =  $page->backlog_range ? "<span style='white-space:nowrap;'> <span class='text-sub'>" .$page->backlog_range . "</span></span><br />" : "--";
                // }
                $customResult['remarks'] = $page->remarks ? $page->remarks : "--";
                // if ($page->remarks) {
                //     $customResult['remarks'] =  $page->remarks ? "<span style='white-space:nowrap;'> <span class='text-sub'>" .$page->remarks . "</span></span><br />" : "--";
                // }
                $customResult['esl_elp_detail'] = $page->esl_elp_detail ? $page->esl_elp_detail : "--";
                // if ($page->esl_elp_detail) {
                //     $customResult['esl_elp_detail'] =  $page->esl_elp_detail ? "<span style='white-space:nowrap;'> <span class='text-sub'>" .$page->esl_elp_detail . "</span></span><br />" : "--";
                // }
                $customResult['applicationmode'] = $page->applicationmode ? $page->applicationmode : "--";
                // if ($page->applicationmode) {
                //     $customResult['applicationmode'] =  $page->applicationmode ? "<span style='white-space:nowrap;'> <span class='text-sub'>" .$page->applicationmode . "</span></span><br />" : "--";
                // }
                $customResult['detscore'] = $page->detscore ? $page->detscore : "--";
                // if ($page->detscore) {
                    //     $customResult['detscore'] =  $page->detscore ? "<span style='white-space:nowrap;'> <span class='text-sub'>" .$page->detscore . "</span></span><br />" : "--";
                    // }"<span style='word-wrap: break-word;''> <span class='text-sub'>" .$page->entry_requirements . "</span></span><br />" : "--";
                $customResult['status'] = $page->status ? "<span class='badge badge-success'> Enabled</span>" : "<span class='badge badge-danger'> Disabled</span>";
                // if ($page->status) {
                //     $customResult['status'] =  "<span style='white-space:nowrap;'> <span class='text-sub'>" . $page->status . "</span></span><br />";
                // }
                $customResult['actions'] =
                    '<div class="btn-group" role="group" aria-label="User Actions">                            
                            <a href="' . route('admin.coursemanagement.edit', $page->id) . '" class="btn btn-sm btn-primary "><i class="fa fa-edit"></i></a> <hr style="margin:0px;">
                            <a href="' . route('admin.coursemanagement.destroy', $page->id) . '" class="btn btn-sm btn-danger" data-method="delete"><i class="fa fa-trash"></i></a>
                            
                            ';
                $customResult['actions'] .= '</div>
                            </div>';

                // 09-12-2023 Not required as get_utf8_string() is added during import --------
                // $customResult['entry_requirements']= $this->cleanUpText($customResult['entry_requirements']);
                // $customResult['scholarship_detail']= $this->cleanUpText($customResult['scholarship_detail']);
                // $customResult['remarks']= $this->cleanUpText($customResult['remarks']);
                // $customResult['esl_elp_detail']= $this->cleanUpText($customResult['esl_elp_detail']);
                // ----------

                $data[] = $customResult;
            }
        }

        $json_data = array(
            "draw"            => intval($request->input('draw')),
            "recordsTotal"    => intval($totalData),
            "recordsFiltered" => intval($totalFiltered),
            "data"            => $data,
            "search" => $get_search,
            // "Branch" => $my_branch,
            // "user" => $my_user_id
        );

        return json_encode($json_data);
    }

    public function categories_index() {
        $categories = UniversityCourseCategories::where('parent_id', '=', 0)->get();
        $subcategories = UniversityCourseCategories::where('parent_id', '!=', 0)->get();
        return view('backend.course_management.category_index', compact('categories','subcategories'));
    }

    public function universities_index() {
        $universities = Universities::all()->sortByDesc('id');
        return view('backend.course_management.university_index', compact('universities'));
    }

    public function search_student(Request $request)
    {
        if(\Auth::user()->isAgent()){
            $my_branch = get_staff_branch_ids(\Auth::user()->id);
            $q =  $request->input('q', '');
            $users = User::whereHas('roles', function ($query) {
                $query->where('id', 3); // Role ID 3 for "user"
            })
            ->where(function ($query) use ($q) {
                $query->where('first_name', 'LIKE', $q . '%')
                    ->orWhere('last_name', 'LIKE', $q . '%')
                    ->orWhere('email', '=', $q)
                    ->orWhere('mobile', '=', $q);
            })
            ->whereIn('branch', $my_branch)
            ->where('agent_id', \Auth::user()->id)
            ->select('id', DB::raw("concat(IF(first_name IS NOT NULL, CONCAT(first_name), ''), ' ', IF(last_name IS NOT NULL, CONCAT(last_name), ''), ' | ', email) as text"))
            ->get();
        } else{
            $my_branch = get_staff_branch_ids(\Auth::user()->id);
            $q =  $request->input('q', '');
            $users = User::whereHas('roles', function ($query) {
                $query->where('id', 3); 
            })
            ->where(function ($query) use ($q) {
                $query->where('first_name', 'LIKE', $q . '%')
                    ->orWhere('last_name', 'LIKE', $q . '%')
                    ->orWhere('email', '=', $q)
                    ->orWhere('mobile', '=', $q);
            })
            ->whereIn('branch', $my_branch)
            ->select('id', DB::raw("concat(IF(first_name IS NOT NULL, CONCAT(first_name), ''), ' ', IF(last_name IS NOT NULL, CONCAT(last_name), ''), ' | ', email) as text"))
            ->get();
        }
        return ['results' => $users,'branch' => $my_branch];
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create() {
        
        $intakes = get_course_suggest_intake();
        $levels = get_study_levels();
        $universities = Universities::all();
        // $years = get_course_suggest_years();
        $countries = Country::orderBy('title_en')->select('id', 'title_en as name')->get();
        $categories = UniversityCourseCategories::where('parent_id', '=', 0)->get();
        // $currency = DB::table('currency')->orderByRaw('')->get();
        $currency = DB::table('currency')->orderByRaw('FIELD(currency_code, "USD", "EUR", "GBP", "AUD") DESC,currency_code ASC')->get();
        return view('backend.course_management.create', ['countries'=> $countries, 'universities' => $universities, 'intakes' => $intakes, 'levels' => $levels,'categories' => $categories, 'currency' => $currency]);
    }

    public function fetchSubCat(Request $request)
    {
        $cat_data = UniversityCourseCategories::where('parent_id', '=', $request->category_id)->where('status',1)->get(["title", "id"]);
         if(count($cat_data)>0){
          $data['subcategories'] = $cat_data;
          return response()->json($data);
         }else{
          $data['subcategories'] = "No Sub Category Found!";
          return response()->json($data);
         } 
    }

    public function fetchCampus(Request $request)
    {
        $campus = Universities::where('country', $request->country_id)
        ->Join('university_courses','university_courses.university',"=",'universities.id')
        ->where('status',1)
        ->whereNotNull('campus')
        ->distinct()
        ->orderBy('university_courses.campus','asc')
        ->get(["campus"]);
         if(count($campus)>0){
          $data['campus'] = $campus;
          return response()->json($data);
         }else{
          $data['campus'] = "No Campus Found!";
          return response()->json($data);
         } 
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request) {
        $request->validate([
            'title' => 'required',
            'university' => 'required',
            'category' => 'required',
            // 'sub_category' => 'required',
            'study_level' => 'required',
            'intake' => 'required',
            // 'year' => 'required'
        ]);

        if(!empty($request->get('sub_category'))){
            $cat = $request->get('sub_category');
        }else{
            $cat = $request->get('category');
        }

        $currency = "USD";
        if(!empty($request->get('currency'))){
            $currency = $request->get('currency');
        }

        $course = new UniversityCourses([
            'course_name' => $request->get('title'),
            'university' => $request->get('university'),
            'category' => $cat,
            'study_level' => $request->get('study_level'),
            'campus' => $request->get('campus'),
            'duration' => $request->get('duration'),

            'course_details' => $request->get('course_details'),
            'occupation_outcome' => $request->get('occupation_outcome'),
            'practical_work' => $request->get('practical_work'),
            'course_content' => $request->get('course_content'),
            'qualify_extended_visa' => $request->get('qualify_extended_visa'),
            'regional_location' => $request->get('regional_location'),
            'country_students_not_accepted' => ($request->get('country_students_not_accepted'))? implode(",",$request->get('country_students_not_accepted')):'',
            
            'intakes' => implode(',', $request->get('intake')),
            // 'year' => implode(',', $request->get('year')),
            'website_url' => $request->get('website_url'),
            'entry_requirements' => get_utf8_string($request->get('entry_requirements')),
            'ielts_score' => $request->get('ielts_score'),
            'ielts_no_band_less_than' => $request->get('ielts_no_band_less_than'),
            'toefl_score' => $request->get('toefl_score'),
            'toefl_no_band_less_than' => $request->get('toefl_no_band_less_than'),
            'pte_score' => $request->get('pte_score'),
            'pte_no_band_less_than' => $request->get('pte_no_band_less_than'),
            'gre_score' => $request->get('gre_score'),
            'gmat_score' => $request->get('gmat_score'),
            'application_deadline' => $request->get('application_deadline'),
            'application_fee' => $request->get('application_fee'),
            'yearly_tuition_fee' => $request->get('yearly_tuition_fee'),
            'total_tuition_fee' => $request->get('total_tuition_fee'),
            'non_tuition_fee' => $request->get('non_tuition_fee'),
            'currency' => $currency,

            'commission_percentage' => $request->get('commission_percentage'),
            'sub_agent_commission_detail' => $request->get('sub_agent_commission_detail'),

            // 'tuition_fee_offshore' => $request->get('tuition_fee_offshore'),
            // 'tuition_fee_onshore' => $request->get('tuition_fee_onshore'),
            'scholarship_available' => get_utf8_string($request->get('scholarship_available')),
            'scholarship_detail' => $request->get('scholarship_detail'),
            'backlog_range' => $request->get('backlog_range'),
            'remarks' => get_utf8_string($request->get('remarks')),
            'esl_elp_detail' => get_utf8_string($request->get('esl_elp_detail')),
            'applicationmode' => $request->get('applicationmode'),
            'detscore' => $request->get('detscore'),
            'status' => $request->get('status'),
            'concentration' => $request->get('concentration'),
        ]);
        $course->save();
        return redirect()->route('admin.coursemanagement')->withFlashSuccess('Course saved!');
    }

    public function edit($id) {
        
        $levels = get_study_levels();
        $universities = Universities::all();

        $categories = UniversityCourseCategories::where('parent_id', '=', 0)->get();
        $countries = Country::orderBy('title_en')->select('id', 'title_en as name')->get();
        $course = UniversityCourses::find($id);
        $course_cat = $course->category ? UniversityCourseCategories::where('id', $course->category)->first(['id','parent_id']) : NULL;
        $course_parent_cat = isset($course_cat) ? UniversityCourseCategories::where('id', $course_cat->parent_id)->first(['id']) : (object)[];
        $currency = DB::table('currency')->get();
        //$years = get_course_suggest_years();
        return view('backend.course_management.edit', ['currency' => $currency, 'universities' => $universities,'levels' => $levels,'categories' => $categories,'course' => $course,'course_cat' => $course_cat,'course_parent_cat'=> $course_parent_cat,'countries'=> $countries]);
    }

    public function update(Request $request, $id)
    {
        $request->validate([
            'title' => 'required',
            'university' => 'required',
            'category' => 'required',
            // 'sub_category' => 'required',
            'study_level' => 'required',
            'intake' => 'required'
        ]);
        $course = UniversityCourses::find($id);
        if (empty($course)) {
            return redirect()->route('admin.coursemanagement')->withFlashDanger('University Course id is invalid!');
        }
       
        $course->course_name = $request->get('title');
        $course->university = $request->get('university');
        if(!empty($request->get('sub_category'))){
            $cat = $request->get('sub_category');
        }else{
            $cat = $request->get('category');
        }

        $currency = "USD";
        if(!empty($request->get('currency'))){
            $currency = $request->get('currency');
        }

        $course->category = $cat;
        $course->study_level = $request->get('study_level');
        $course->campus = $request->get('campus');
        $course->duration = $request->get('duration');
        $course->intakes = implode(',', $request->get('intake'));

        $course->course_details = $request->get('course_details');
        $course->occupation_outcome = $request->get('occupation_outcome');
        $course->practical_work = $request->get('practical_work');
        $course->course_content = $request->get('course_content');
        $course->qualify_extended_visa = $request->get('qualify_extended_visa');
        $course->regional_location = $request->get('regional_location');
        $course->country_students_not_accepted = ($request->get('country_students_not_accepted'))? implode(",",$request->get('country_students_not_accepted')):'';
        
        // $course->year = implode(',', $request->get('year'));
        $course->website_url = $request->get('website_url');
        $course->entry_requirements = get_utf8_string($request->get('entry_requirements'));
        $course->ielts_score = $request->get('ielts_score');
        $course->ielts_no_band_less_than = $request->get('ielts_no_band_less_than');
        $course->toefl_score = $request->get('toefl_score');
        $course->toefl_no_band_less_than = $request->get('toefl_no_band_less_than');
        $course->pte_score = $request->get('pte_score');
        $course->pte_no_band_less_than = $request->get('pte_no_band_less_than');
        $course->gre_score = $request->get('gre_score');
        $course->gmat_score = $request->get('gmat_score');
        $course->application_deadline = $request->get('application_deadline');
        $course->application_fee = $request->get('application_fee');
        $course->yearly_tuition_fee = $request->get('yearly_tuition_fee');
        $course->total_tuition_fee = $request->get('total_tuition_fee');
        $course->non_tuition_fee = $request->get('non_tuition_fee');
        $course->currency = $currency;
        $course->commission_percentage = $request->get('commission_percentage');
        $course->sub_agent_commission_detail = $request->get('sub_agent_commission_detail');
        // $course->tuition_fee_onshore = $request->get('tuition_fee_onshore');
        // $course->tuition_fee_offshore = $request->get('tuition_fee_offshore');
        $course->scholarship_available = get_utf8_string($request->get('scholarship_available'));
        $course->scholarship_detail = $request->get('scholarship_detail');
        $course->backlog_range = $request->get('backlog_range');
        $course->remarks = get_utf8_string($request->get('remarks'));
        $course->esl_elp_detail = get_utf8_string($request->get('esl_elp_detail'));
        $course->applicationmode = $request->get('applicationmode');
        $course->detscore = $request->get('detscore');
        $course->status = $request->get('status');
        $course->concentration = $request->get('concentration');
        $course->save();
        return redirect()->route('admin.coursemanagement')->withFlashSuccess('University Course Updated!');
    }

    public function destroy($id) {
        $course = UniversityCourses::find($id);
        if (!empty($course)) {
            $check_pref = DB::table('course_preference')->where('course_id',$course->id)->count();
            if($check_pref > 0) {
                return redirect()->route('admin.coursemanagement')->withFlashDanger('This course is present in students course preferences, Could not delete Course');
            }
            $course->delete();
            return redirect()->route('admin.coursemanagement')->withFlashSuccess('University Course Deleted Successfully');
        }
        return redirect()->route('admin.coursemanagement')->withFlashDanger('Could not delete Course');
    }


    public function create_category(Request $request)
    {
        $categories = UniversityCourseCategories::where('parent_id', '=', 0)->get();
        return view("backend.course_management.create_category",compact('categories'));
    }

    public function store_category(Request $request)
    {
        $this->validate(
            $request,
            [
                'title' => 'required|max:100',
                'display_order' => 'required|numeric',
                'seo_url_slug' => 'required|regex:/^[a-z0-9-]+$/i|unique:university_course_categories,seo_url_slug',
            ]
        );

        $cat = new UniversityCourseCategories;
        $cat->title = $request->title;
        $cat->status = $request->status;
        $cat->description = $request->description;
        $cat->parent_id = $request->parentcat;
        $cat->display_order = $request->display_order;
        $cat->seo_url_slug = $request->seo_url_slug;
        $cat->save();

        return redirect()->route('admin.coursemanagement.categories')->withFlashSuccess('Category Created Successfully');
    }

    public function edit_category($id)
    {
        $cat_det = UniversityCourseCategories::where('id', '=', $id)->first();
        $categories = UniversityCourseCategories::where('parent_id', '=', 0)->where('id','!=',$id)->get();
        return view("backend.course_management.edit_category",compact('cat_det','categories'));
    }

    public function update_category(Request $request, $id)
    {
        $this->validate(
            $request,
            [
                'title' => 'required|max:100',
                'display_order' => 'required|numeric',
                'seo_url_slug' => 'required|regex:/^[a-z0-9-]+$/i|unique:university_course_categories,seo_url_slug,'.$id,
            ]
        );
        $course = UniversityCourseCategories::find($id);
        if (empty($course)) {
            return redirect()->route('admin.coursemanagement.categories')->withFlashDanger('Category id is invalid!');
        }
        UniversityCourseCategories::where('id', '=', $id)->update(['title' => $request->title, 'description' => $request->description, 'display_order' => $request->display_order, 'seo_url_slug' => $request->seo_url_slug, 'status' => $request->status]);
        return redirect()->route('admin.coursemanagement.categories')->withFlashSuccess('Category Updated Successfully');
    }

    public function destroy_category($id) {
        $category = UniversityCourseCategories::find($id);
        if(UniversityCourseCategories::where('parent_id',$id)->exists()){
            return redirect()->route('admin.coursemanagement.categories')->withFlashDanger('This category has sub category, Could not delete Category');
        }
        if (!empty($category)) {
            $courses = UniversityCourses::where('category', '=', $id)->get();
            if (count($courses) > 0) {
                return redirect()->route('admin.coursemanagement.categories')->withFlashDanger('This category has courses, Could not delete Category');
            }
            $category->delete();
            return redirect()->route('admin.coursemanagement.categories')->withFlashSuccess('Category Deleted Successfully');
        }
        return redirect()->route('admin.coursemanagement.categories')->withFlashDanger('Could not delete category');
    }


    public function create_university(Request $request)
    {
        if (\Auth::user()->isAgent()) {
        return redirect()->route('agent.dashboard.message')
            ->withFlashDanger('You do not have access permission to this feature.');
        }
        $countries = get_study_countries();
        return view("backend.course_management.create_university",compact('countries'));
    }


    public function store_university(Request $request)
    {
        if (\Auth::user()->isAgent()) {
            return redirect()->route('agent.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        $this->validate(
            $request,
                [
                    'university_name' => 'required|max:150',
                    'country' => 'required',
                ]
        );

        $image = "";
        if ($request->has('university_logo')) {
            $request->validate([
                'university_logo' => 'image|mimes:jpeg,jpg,png',
            ]);
            $image = $request->file('university_logo');
        } else {
            $image = false;
        }

        $university = new Universities;
        if ($image) {
            $extension = $image->extension();
            if (in_array($extension, array('jpg', 'jpeg', 'png'))) {
                if ($image->getSize() < 2048000) { //max size 2MB
                    $filename = pathinfo($image->getClientOriginalName(), PATHINFO_FILENAME);
                    if (strlen($filename) > 100) {
                        $filename = substr($filename, 0, 100);
                    }
                    $filename = $filename . Str::random(10) . "." . $image->getClientOriginalExtension();
                    $university->university_logo = $image->storeAs('/universityLogo', $filename, 'public');
                } else {
                    throw new GeneralException('University Logo size must not exceed 2MB');
                }
            } else {
                throw new GeneralException('You must upload a valid University Logo (jpg/png/jpeg).');
            }
        }

        $university->university_name = $request->university_name;
        $university->country = $request->country;
        $university->institution_type = $request->institution_type;
        $university->concurrent_course_allowed = $request->concurrent_course_allowed;
        $university->commission_count = $request->commission_count;
        $university->commission_percent = $request->commission_percent;
        $university->tax_type = $request->tax_type;
        $university->tax_percent = $request->tax_percent;
        $university->college_details = $request->college_details;
        $university->address = $request->address;
        $university->website = $request->website;
        $university->cricos_link = $request->cricos_link;
        $university->documents_link = $request->documents_link;
        $university->current_fee_list_link = $request->current_fee_list_link;
        $university->initial_deposit_options = $request->initial_deposit_options;
        $university->admission_process = $request->admission_process;
        $university->contact_person_name = $request->contact_person_name;
        $university->contact_person_phone = $request->contact_person_phone;
        $university->contact_person_email = $request->contact_person_email;
        //logic to hide university from agent
        $university->agent_uni = $request->has('agent_uni') ? 1 : 0;
        $university->save();

        return redirect()->route('admin.coursemanagement.universities')->withFlashSuccess('University Created Successfully');
    }

    public function edit_university($id){
        if (\Auth::user()->isAgent()) {
        return redirect()->route('agent.dashboard.message')
            ->withFlashDanger('You do not have access permission to this feature.');
        }
        $university = Universities::where('id', '=', $id)->first();
        $countries = get_study_countries();
        return view("backend.course_management.edit_university",compact('university','countries'));
    }

    public function update_university(Request $request, $id){
        if (\Auth::user()->isAgent()) {
        return redirect()->route('agent.dashboard.message')
            ->withFlashDanger('You do not have access permission to this feature.');
        }
        $this->validate($request, [
            'university_name' => 'required|max:150',
            'country' => 'required',
        ]);

        $university = Universities::find($id);
        $image = "";
        $university_logo = $university->university_logo;
        if ($request->has('university_logo')) {
            $request->validate([
                'university_logo' => 'image|mimes:jpeg,jpg,png',
            ]);
            $image = $request->file('university_logo');
        } else {
            $image = false;
        }

        if ($image) {
            $extension = $image->extension();
            if (in_array($extension, array('jpg', 'jpeg', 'png'))) {
                if ($image->getSize() < 2048000) { //max size 2MB
                    $filename = pathinfo($image->getClientOriginalName(), PATHINFO_FILENAME);
                    if (strlen($filename) > 100) {
                        $filename = substr($filename, 0, 100);
                    }
                    $filename = $filename . Str::random(10) . "." . $image->getClientOriginalExtension();

                    $image_path = public_path("storage/{$university_logo}");
                    if (File::exists($image_path)) {
                        File::delete($image_path);
                    }
                    $university_logo = $image->storeAs('/universityLogo', $filename, 'public');
                } else {
                    throw new GeneralException('University Logo size must not exceed 2MB');
                }
            } else {
                throw new GeneralException('You must upload a valid University Logo (jpg/png/jpeg).');
            }
        }

        if (empty($university)) {
            return redirect()->route('admin.coursemanagement.universities')->withFlashDanger('University id is invalid!');
        }

        $university->country = $request->country;
        $university->university_logo = $university_logo;
        $university->university_name = $request->university_name;
        $university->institution_type = $request->institution_type;
        $university->concurrent_course_allowed = $request->concurrent_course_allowed;
        $university->commission_count = $request->commission_count;
        $university->commission_percent = $request->commission_percent;
        $university->tax_type = $request->tax_type;
        $university->tax_percent = $request->tax_percent;
        $university->college_details = $request->college_details;
        $university->address = $request->address;
        $university->website = $request->website;
        $university->cricos_link = $request->cricos_link;
        $university->documents_link = $request->documents_link;
        $university->current_fee_list_link = $request->current_fee_list_link;
        $university->initial_deposit_options = $request->initial_deposit_options;
        $university->admission_process = $request->admission_process;
        $university->contact_person_name = $request->contact_person_name;
        $university->contact_person_phone = $request->contact_person_phone;
        $university->contact_person_email = $request->contact_person_email;
        $university->agent_uni = $request->has('agent_uni') ? 1 : 0;

        $university->save();

        return redirect()->route('admin.coursemanagement.universities')->withFlashSuccess('University Updated Successfully');
    }

    public function destroy_university($id) {
        if (\Auth::user()->isAgent()) {
        return redirect()->route('agent.dashboard.message')
            ->withFlashDanger('You do not have access permission to this feature.');
        }
        $university = Universities::find($id);
        if (!empty($university)) {
            $universityCourse = UniversityCourses::where('university', '=', $id)->get();
            if (count($universityCourse) > 0) {
                return redirect()->route('admin.coursemanagement.universities')->withFlashDanger('This university has courses, Could not delete University');
            }
            $university->delete();
            return redirect()->route('admin.coursemanagement.universities')->withFlashSuccess('University Deleted Successfully');
        }
        return redirect()->route('admin.coursemanagement.universities')->withFlashDanger('Could not delete University');
    }

    //NEW
    public function course_finder(Request $request){
        $countries = get_study_countries();
        $levels = get_study_levels();
        $categories = UniversityCourseCategories::where('parent_id', '=', 0)->where('status',1)->get();
        
        // if (Auth::user()->isAgent()) {
        //     $universities = Universities::where('agent_uni', 0)->get();
        //     $courses = UniversityCourses::whereHas('university', function ($query) {
        //         $query->where('agent_uni', 0);
        //     })
        //     ->where('status', 1)   
        //     ->get();
        // } else {
        //     $universities = Universities::all();
        //     $courses = UniversityCourses::all();
        // }

        return view("backend.course_management.course_finder_new", compact('countries', 'levels', 'categories'));
    }

    public function courseFinderData(Request $request){
        if (!$request->ajax()) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this page.');
        }
        echo $this->getDatatableCourseFinderlist($request);
    }

    public function getDatatableCourseFinderlist($request){
        // Enable query logging
        DB::enableQueryLog();
        
        // dd($request->all());
        $columns = array(
            0 => 'id',
            1 => 'course_name',
            2 => 'study_level',
            3 => 'university_name',
        );        
        //$courses = explode(',',$request->input('course_university_search'));
        // dd($courses);
        $course_university_search = $request->input('course');
        $country = $request->input('country');
        $universities = $request->input('universities');
        $study_level = $request->study_level;
        $yearly_tuition_fee = $request->input('yearly_tuition_fee');
        // $institution_turnaround_time = $request->input('institution_turnaround_time');


        $intake = $request->search_intake;
        $duration = $request->search_duration;
        $campus = $request->campus;
        $sort_by = $request->sort_by;

        //Score Filter
        $ielts = $request->ielts;
        $ielts_no_band = $request->ielts_no_band;
        $toefl = $request->toefl;
        $toefl_no_band = $request->toefl_no_band;
        $pte = $request->pte;
        $pte_no_band = $request->pte_no_band;
        $gre = $request->gre;
        $gmat = $request->gmat;
        $subcategory = $category = "";
        $subcategories = array();
        
        // if($request->category){
        //     if(!empty($request->sub_category)){
        //         $subcategory = $request->sub_category;
        //     }else{
        //         $subcategories = UniversityCourseCategories::where('parent_id',$request->category)->where('status',1)->pluck('id')->toArray();
        //         $subcategories = array_values($subcategories);
        //         $subcategories[] = $request->category;
            
        //     }
        // }
        if($request->filterstudyArea){
            if(!empty($request->filterdisciplineArea)){
                $subcategory = $request->filterdisciplineArea;
            }
            else{
                $subcategories = UniversityCourseCategories::where('parent_id',$request->filterstudyArea)->where('status',1)->pluck('id')->toArray();
                $subcategories = array_values($subcategories);
                $subcategories[] = $request->filterstudyArea;
            }
            
        }

        $limit = $request->input('length');
        $start = $request->input('start');
        $order = $columns[$request->input('order.0.column')];
        $dir = $request->input('order.0.dir');

        $get_search = $request->input('search.value');
        $totalData = DB::table('university_courses')
            ->join('universities', 'university_courses.university', '=', 'universities.id')
            ->join('countries', 'universities.country', '=', 'countries.id')
            ->when(Auth::check() && Auth::user()->isAgent(), function ($query) {
                return $query->where('universities.agent_uni', 0);
            })
            ->select('universities.country','universities.university_name', 'university_courses.*', 'countries.title_en as country_name', 'universities.university_logo')
            ->when(!empty($course_university_search), function ($query) use ($course_university_search) {
                return $query->Where(function ($query) use ($course_university_search) {
                    $query->where('university_courses.course_name', 'LIKE', "%$course_university_search%");
                    $query->orwhere('universities.university_name', 'LIKE', "%$course_university_search%");
                });
            })
           ->when(!empty($country), function ($query) use ($country) {
                return $query->whereIn('universities.country', function ($subquery) use ($country) {
                    $subquery->select('id')
                             ->from('countries')
                             ->whereIn('title_en', (array) $country);
                });
            })

            ->when(!empty($universities), function ($query) use ($universities) {
                return $query->where('universities.id',$universities);
            })
            // ->when(!empty($study_level), function ($query) use ($study_level) {
            //     return $query->where('university_courses.study_level',$study_level);
            // })
            ->when($study_level, function ($query) use ($study_level) {
                return $query->where('university_courses.study_level', $study_level);
            })
        
            ->when(!empty($duration), function ($query) use ($duration) {
                if($duration==12){
                    return $query->whereBetween('university_courses.duration', [1, 12]); 
                }else if($duration == 24){
                return $query->whereBetween('university_courses.duration', [12, 24]); 
                }else if($duration == 36){
                return $query->whereBetween('university_courses.duration', [24, 36]); 
                }else if($duration == 48){
                    return $query->whereBetween('university_courses.duration', [36, 48]); 
                } else{
                    return $query->where('university_courses.duration','>',48); 
                }
            })
            ->when(!empty($subcategory), function ($query) use ($subcategory) {
            return $query->where('university_courses.category', $subcategory);
            })
            ->when(!empty($subcategories) && count($subcategories), function ($query) use ($subcategories) {
            return $query->whereIn('university_courses.category', $subcategories);
            })
            ->when(!empty($category), function ($query) use ($category) {
                return $query->where('university_courses.category', $category);
            })
            ->when(!empty($campus), function ($query) use ($campus) {
                return $query->where('university_courses.campus', $campus);
            })
            ->when($intake && count($intake), function ($query) use ($intake) {
                return $query->Where(function ($query) use ($intake) {
                    foreach($intake as $keyword) {
                        $query->orWhere('university_courses.intakes', 'LIKE', "%$keyword%");
                    }
                });
            })
            ->when(!empty($ielts), function ($query) use ($ielts) {
                return $query->where('university_courses.ielts_score', "=", $ielts);
            })
            ->when(!empty($ielts_no_band), function ($query) use ($ielts_no_band) {
                return $query->where('university_courses.ielts_no_band_less_than', "=", $ielts_no_band);
            })
            ->when(!empty($toefl), function ($query) use ($toefl) {
                return $query->where('university_courses.toefl_score', ">=", $toefl);
            })
            ->when(!empty($toefl_no_band), function ($query) use ($toefl_no_band) {
                return $query->where('university_courses.toefl_no_band_less_than', ">=", $toefl_no_band);
            })
            ->when(!empty($pte), function ($query) use ($pte) {
                return $query->where('university_courses.pte_score', ">=", $pte);
            })
            ->when(!empty($pte_no_band), function ($query) use ($pte_no_band) {
                return $query->where('university_courses.pte_no_band_less_than', ">=", $pte_no_band);
            })
            ->when(!empty($gre), function ($query) use ($gre) {
                return $query->where('university_courses.gre_score', ">=", $gre);
            })
            ->when(!empty($gmat), function ($query) use ($gmat) {
                return $query->where('university_courses.gmat_score', ">=", $gmat);
            })
            ->when(!empty($sort_by), function ($query) use ($sort_by) {
                if($sort_by == "course_name"){
                    return $query->orderBy('course_name', 'asc');
                }
                else if($sort_by == "ielts_lth"){
                    return $query->orderBy('ielts_score', 'asc');
                }
                else if($sort_by == "ielts_htl"){
                    return $query->orderBy('ielts_score', 'desc');
                }
                else if($sort_by == "fee_lth"){
                    return $query->orderBy('yearly_tuition_fee', 'asc');
                }
                else if($sort_by == "fee_htl"){
                    return $query->orderBy('yearly_tuition_fee', 'desc');
                }
            })
            ->when(!empty($get_search), function ($query) use ($get_search) {
                return $query->Where(function ($query) use ($get_search) {
                    $query->Where('course_name', 'like', "%$get_search%")
                        ->orWhere('universities.university_name', 'like', "%$get_search%")
                        ->orWhere('universities.country', 'like', "%$get_search%")
                        ->orWhere('study_level', 'like', "%$get_search%")
                        ->orWhere('university_courses.id', '=', "$get_search")
                        ->orWhere('duration', 'like', "%$get_search%") 
                        ->orWhere('intakes', 'like', "%$get_search%") 
                        ->orWhere('year', 'like', "%$get_search%")
                        ->orWhere('remarks', 'like', "%$get_search%")
                        ->orWhere('university_courses.status', 'like', "%$get_search%");
                });
            })      
            ->where('university_courses.status', 1)           
            ->count();
            // $queries = DB::getQueryLog();
            // dd(end($queries));  

        $users = DB::table('university_courses')
            ->join('universities', 'university_courses.university', '=', 'universities.id')
            ->join('countries', 'universities.country', '=', 'countries.id')
            ->when(Auth::check() && Auth::user()->isAgent(), function ($query) {
                return $query->where('universities.agent_uni', 0);
            })
            ->select('universities.country','universities.university_name', 'university_courses.*', 'countries.title_en as country_name', 'universities.university_logo')
            ->when(!empty($course_university_search), function ($query) use ($course_university_search) {
                return $query->Where(function ($query) use ($course_university_search) {
                    $query->where('university_courses.course_name', 'LIKE', "%$course_university_search%");
                    $query->orwhere('universities.university_name', 'LIKE', "%$course_university_search%");
                });
            })
            // ->when(!empty($courses), function ($query) use ($courses) {
            //     return $query->Where(function ($query) use ($courses) {
            //         for ($i = 0; $i <b count($courses); $i++){
            //             $query->orwhere('universities.university_name', 'like',  '%' . $courses[$i] .'%');
            //             $query->orwhere('university_courses.course_name', 'like',  '%' . $courses[$i] .'%');
            //          } 
            //     });
            // })
            ->when(!empty($country), function ($query) use ($country) {
                return $query->whereIn('universities.country', function ($subquery) use ($country) {
                    $subquery->select('id')
                             ->from('countries')
                             ->whereIn('title_en', (array) $country);
                });
            })

            ->when(!empty($universities), function ($query) use ($universities) {
                return $query->where('universities.id',$universities);
            })
            // ->when(!empty($study_level), function ($query) use ($study_level) {
            //     return $query->where('university_courses.study_level',$study_level);
            // })
            ->when($study_level, function ($query) use ($study_level) {
                return $query->where('university_courses.study_level', $study_level);
            })

            ->when(!empty($duration), function ($query) use ($duration) {
                if($duration==12){
                    return $query->whereBetween('university_courses.duration', [1, 12]); 
                }else if($duration == 24){
                return $query->whereBetween('university_courses.duration', [12, 24]); 
                }else if($duration == 36){
                return $query->whereBetween('university_courses.duration', [24, 36]); 
                }else if($duration == 48){
                    return $query->whereBetween('university_courses.duration', [36, 48]); 
                } else{
                    return $query->where('university_courses.duration','>',48); 
                }
            })
            ->when(!empty($subcategory), function ($query) use ($subcategory) {
                return $query->where('university_courses.category', $subcategory);
            })
            ->when(!empty($subcategories) && count($subcategories), function ($query) use ($subcategories) {
                return $query->whereIn('university_courses.category', $subcategories);
            })
            ->when(!empty($category), function ($query) use ($category) {
                return $query->where('university_courses.category', $category);
            })
            ->when(!empty($campus), function ($query) use ($campus) {
                return $query->where('university_courses.campus', $campus);
            })
            ->when($intake && count($intake), function ($query) use ($intake) {
                return $query->Where(function ($query) use ($intake) {
                    foreach($intake as $keyword) {
                        $query->orWhere('university_courses.intakes', 'LIKE', "%$keyword%");
                    }
                });
            })
            ->when(!empty($ielts), function ($query) use ($ielts) {
                return $query->where('university_courses.ielts_score', "=", $ielts);
            })
            ->when(!empty($ielts_no_band), function ($query) use ($ielts_no_band) {
                return $query->where('university_courses.ielts_no_band_less_than', "=", $ielts_no_band);
            })
            ->when(!empty($toefl), function ($query) use ($toefl) {
                return $query->where('university_courses.toefl_score', ">=", $toefl);
            })
            ->when(!empty($toefl_no_band), function ($query) use ($toefl_no_band) {
                return $query->where('university_courses.toefl_no_band_less_than', ">=", $toefl_no_band);
            })
            ->when(!empty($pte), function ($query) use ($pte) {
                return $query->where('university_courses.pte_score', ">=", $pte);
            })
            ->when(!empty($pte_no_band), function ($query) use ($pte_no_band) {
                return $query->where('university_courses.pte_no_band_less_than', ">=", $pte_no_band);
            })
            ->when(!empty($gre), function ($query) use ($gre) {
                return $query->where('university_courses.gre_score', ">=", $gre);
            })
            ->when(!empty($gmat), function ($query) use ($gmat) {
                return $query->where('university_courses.gmat_score', ">=", $gmat);
            })
            ->when(!empty($sort_by), function ($query) use ($sort_by) {
                if($sort_by == "course_name"){
                    return $query->orderBy('course_name', 'asc');
                }
                else if($sort_by == "ielts_lth"){
                    return $query->orderBy('ielts_score', 'asc');
                }
                else if($sort_by == "ielts_htl"){
                    return $query->orderBy('ielts_score', 'desc');
                }
                else if($sort_by == "fee_lth"){
                    return $query->orderBy('yearly_tuition_fee', 'asc');
                }
                else if($sort_by == "fee_htl"){
                    return $query->orderBy('yearly_tuition_fee', 'desc');
                }
            })
            ->when(!empty($get_search), function ($query) use ($get_search) {
                return $query->Where(function ($query) use ($get_search) {
                    $query->Where('course_name', 'like', "%$get_search%")
                        ->orWhere('universities.university_name', 'like', "%$get_search%")
                        ->orWhere('universities.country', 'like', "%$get_search%")
                        ->orWhere('study_level', 'like', "%$get_search%")
                        ->orWhere('university_courses.id', '=', "$get_search")
                        ->orWhere('duration', 'like', "%$get_search%") 
                        ->orWhere('intakes', 'like', "%$get_search%") 
                        ->orWhere('year', 'like', "%$get_search%")
                        ->orWhere('remarks', 'like', "%$get_search%")
                        ->orWhere('university_courses.status', 'like', "%$get_search%");
                });
            })       
            ->where('university_courses.status', 1)       
            ->offset($start)
            ->limit($limit)
            ->orderBy($order, $dir)
            ->get();

        $totalFiltered = $totalData;

        $data = $courses = array();
        if (!empty($users)) {
            $count = 1;
            foreach ($users as $page) {
                $customResult['course_name'] = "<div class='row card py-3 mb-0 rounded ' style='border: 2px solid #e3e3e3; margin-top: 0px;'>";
                        $customResult['course_name'] .= "<div class='col-12 card-body pb-0' style='padding-top: 0px;'>";
                            $customResult['course_name'] .= "<div class='row'>";
                                $customResult['course_name'] .= "<div class='col-sm-12 col-lg-2 text-center p-0 agents-border' style='border-right:2px solid rgb(203, 211, 222);'>";
                                    $customResult['course_name'] .= $page->university_logo ? "<a href='".$page->website_url."' target='_blank'>
                                    <img src='".asset('storage/'.$page->university_logo)."' class='img-fluid p-2' style='height: 150px; width: 130px;'></a>":
                                    "<img src='".url('assets/images/university_15394744.png')."' class='img-fluid p-2' style='height: 150px; width: 130px;'>";
                                    $customResult['course_name'] .= $page->campus ? "<p class='text-center mt-3 mb-1'><b class='text-secondary'>" . $page->campus . "</b></p>" : "";
                                    $customResult['course_name'] .= $page->country ? "<p class='text-center mt-2'><b class='fw-bold text-dark'>" . $page->country_name . "</b></p>" : "";
                                $customResult['course_name'] .= "</div>";
                                $customResult['course_name'] .= "<div class='col-10'>";
                                    $customResult['course_name'] .= "<div class='row'>";
                                        $customResult['course_name'] .= "<div class='col-12'>";
                                            $customResult['course_name'] .=  $page->course_name ? '<a href="' . route('admin.coursemanagement.course-details', $page->id) . '" target="_blank" class="text-primary" > <b style="font-size: 24px !important;">' . $page->course_name . '</b></a>' : "--";
                                        $customResult['course_name'] .= "</div>";
                                        $customResult['course_name'] .= "<div class='col-4'>";
                                            $customResult['course_name'] .= '<br><b class="text-secondary">Id: <span class="fw-bold text-dark">' . $page->id . '</span></b>';
                                        $customResult['course_name'] .= "</div>";
                                        $customResult['course_name'] .= "<div class='col-4'>";
                                            $customResult['course_name'] .= $page->university_name ? '<br><b class="text-secondary"><i class="fa fa-university text-secondary"></i> </b> <a class="text-dark fw-bold" target="_blank" href="' . route('admin.coursemanagement.university-details', $page->university) . '">' . $page->university_name . '</a>' : "--";
                                        $customResult['course_name'] .= "</div>";
                                        $customResult['course_name'] .= "<div class='col-4'>";
                                            $customResult['course_name'] .= $page->country ? "<br><b class='text-secondary'><i class='fa fa-globe text-secondary'></i></b> <b class='fw-bold text-dark'>" . $page->country_name . "</b>" . "</b>" : "<br><b>Country:</b> --";
                                        $customResult['course_name'] .= "</div>";
                                    // $customResult['course_name'] .= "</div>";
                                // $customResult['course_name'] .= "</div>";
                            // $customResult['course_name'] .= "</div>";

                            // $customResult['course_name'] .= "<div class='row'>";
                                $customResult['course_name'] .= "<div class='col-4'>";
                                    $customResult['course_name'] .= "<br><b class='text-secondary'><i class='fa fa-graduation-cap text-secondary'></i> Study Level:</b><br> <b class='fw-bold text-dark'>".$page->study_level . "</b>";
                                $customResult['course_name'] .= "</div>";
                                $customResult['course_name'] .= "<div class='col-4'>";
                                    $customResult['course_name'] .= $page->duration ? "<br><b class='text-secondary'><i class='fa fa-clock text-secondary'></i> Duration:</b><br> <b class='fw-bold text-dark'>" . $page->duration . "</b>" : "<br><b class='text-secondary'><i class='fa fa-clock-o text-secondary'></i> Duration:</b><br> --";
                                $customResult['course_name'] .= "</div>";
                                $customResult['course_name'] .= "<div class='col-4'>";
                                    $intakes = array();
                                    if($page->intakes){
                                        $intakes = explode(",", $page->intakes);
                                    }
                                    $count = count($intakes);
                                    $allIntake = "";
                                    $i = 0;
                                    foreach($intakes as $intake){
                                        $i++;
                                        $allIntake .= substr($intake, 0, 3);
                                        if($i < $count){
                                            $allIntake .= ", ";
                                        }
                                    }
                                    $customResult['course_name'] .= $page->intakes ? "<br><b class='text-secondary'><i class='fa fa-book text-secondary'></i> Intakes:</b><br> <b class='fw-bold text-dark'>" . $allIntake . "</b>" : "<br><b class='text-secondary'><i class='fa fa-book text-secondary'></i> Intakes:</b><br> --";
                                $customResult['course_name'] .= "</div>";
                                $customResult['course_name'] .= "<div class='col-4'>";
                                    $customResult['course_name'] .= $page->yearly_tuition_fee ? "<br><b class='text-secondary'><i class='fas fa-money-bill-alt'></i> Tuition Fees:</b><br> <b class='fw-bold text-dark'>". $page->currency.' '.$page->yearly_tuition_fee . "</b>" : "<br><b class='text-secondary'><i class='fas fa-money-bill-alt'></i> Tuition Fees:</b><br> --";
                                $customResult['course_name'] .= "</div>";
                                $customResult['course_name'] .= "<div class='col-4'>";
                                    $customResult['course_name'] .= $page->application_fee ? "<br><b class='text-secondary'><i class='fas fa-money-bill-alt'></i> Application Fees:</b><br> <b class='fw-bold text-dark'>". $page->currency.' '.$page->application_fee . "</b>" : "<br><b class='text-secondary'><i class='fas fa-money-bill-alt'></i> Application Fees:</b><br> --";
                                $customResult['course_name'] .= "</div>";

                                $customResult['course_name'] .= "<div class='col-4'>";
                                    // if($page->commission_percentage && $page->discounted_fee){
                                    //     $commission = $page->currency.' '.$page->discounted_fee * $page->commission_percentage/100;
                                    //     if(\Auth::user()->isAgent()){
                                    //         $comm = $page->discounted_fee * $page->commission_percentage/100;
                                    //         $commission = $page->currency.' '.$comm * get_agent_commission_percentage(\Auth::user()->id)/100;
                                    //     }
                                    // } else 
                                    if($page->commission_percentage && $page->yearly_tuition_fee){
                                        $yearly_tuition_fee = (int) $page->yearly_tuition_fee;
                                        $commission_percentage = (int) $page->commission_percentage;
                                        $commission = $page->currency.' '.$yearly_tuition_fee * $commission_percentage/100;
                                        if(\Auth::user()->isAgent()){
                                            $comm = $yearly_tuition_fee * $commission_percentage/100;
                                            $commission = $page->currency.' '.$comm * get_agent_commission_percentage(\Auth::user()->id)/100;
                                        }
                                    } else{
                                        $commission = '--';
                                    }
                                    if(\Auth::user()->isAdmin() || \Auth::user()->isAgent() || \Auth::user()->can('Commission Rates')){
                                        if(\Auth::user()->isAgent())
                                            $sub_agent_commission_detail = "<br><br><b>Commission Detail:</b> $page->sub_agent_commission_detail";
                                        else $sub_agent_commission_detail = "";
                                        // $customResult['course_name'] .= "<br><b class='text-secondary'><i class='fas fa-money-bill-alt'></i> Commission:</b><br> <b class='fw-bold text-dark'>". $commission . "</b>";
                                        $customResult['course_name'] .= "<br><b class='text-secondary'><i class='fas fa-money-bill-alt'></i> Commission:</b><br> <b class='fw-bold text-dark'>
                                        <a href='#' class='text-primary commissionModal' data-commission='$commission' data-commission_detail='$sub_agent_commission_detail' data-bs-toggle='modal' data-bs-target='#commissionModal'>Show</a></b>";
                                    }
                                $customResult['course_name'] .= "</div>";
                            $customResult['course_name'] .= "</div>";
                        $customResult['course_name'] .= "</div>";
        
                        $customResult['course_name'] .= "<div class='row py-1 mt-3' style='background-color: rgb(244,244,244); margin: -20px; border-bottom: 2px solid #e3e3e3;'>";
                            $customResult['course_name'] .= "<div class='col-4 align-self-center'>";
                                $customResult['course_name'] .= '<div class="form-check">
                                        <label class="text-start">Select<input class="form-check-input data-checkboxs" onclick="javascript:checkboxCheckSelected();" name="multiselect" type="checkbox" value="'.$page->id.'" /></label></div></div>
                                        <div class="col-8">
                                            <p class="mb-1 mt-2 text-end">
                                                <a data-toggle="modal" id="courseButton" data-target="#courseModal" data-id="'.$page->id.'" class="btn btn-sm btn-warning py-2 px-3">
                                                    <i class="fa fa-arrow-circle-up"></i> Suggest to Student
                                                </a> 
                                                <a data-toggle="modal" id="coursePreferenceButton" data-target="#coursePreferenceModal" data-id="'.$page->id.'" title="Add to Student Course Preference" class="btn btn-sm btn-primary py-2 px-3">
                                                    <i class="fa fa-plus"></i> Add to Student Course Preference
                                                </a> 
                                            </p>';
                            $customResult['course_name'] .= "</div>";
                        $customResult['course_name'] .= "</div>";
                    

                    $customResult['course_name'] .= "</div>";
                    $customResult['course_name'] .= "</div>";   

                $customResult['course'] = $page->course_name ? $page->course_name : "--";
                $customResult['university'] = $page->university_name ? $page->university_name : "--";
                $customResult['country'] = $page->country ? $page->country_name : "--";
                $customResult['study_level'] = $page->study_level ? $page->study_level : "--";
                $customResult['duration'] = $page->duration ? $page->duration : "--";
                $customResult['intakes'] = $page->intakes ? $allIntake : "--";
                $customResult['application_fees'] = $page->application_fee ? $page->currency.' '.$page->application_fee : "--";
                $customResult['tution_fees'] = $page->yearly_tuition_fee ? $page->currency.' '.$page->yearly_tuition_fee : "--";
                
                $customResult['id'] = $page->id;
                $data[] = $customResult;
            }
        }

        // Get the logged queries
        $queries = DB::getQueryLog();

        // Count the queries
        $queryCount = count($queries);
        \Log::debug(end($queries));
        \Log::debug("COURSE FINDER DATABASE QUERY COUNT: " . $queryCount);

        $json_data = array(
            "draw"            => intval($request->input('draw')),
            "recordsTotal"    => intval($totalData),
            "recordsFiltered" => intval($totalFiltered),
            "data"            => $data,
            "search" => $get_search,
        );
        return json_encode($json_data);
    }
//NEW


//OLD
    public function course_finder_old(Request $request){
        $countries = get_study_countries();
        $levels = get_study_levels();
        $categories = UniversityCourseCategories::where('parent_id', '=', 0)->where('status',1)->get();
        // $years = get_course_suggest_years();
        $universities = Universities::query();

        if (Auth::check() && Auth::user()->isAgent()) {
            $universities = $universities->where('agent_uni', 0); // Hide from agents

            //Hide courses related to hidden universities
            $courses = UniversityCourses::whereHas('university', function ($query) {
                $query->where('agent_uni', 0);
            })->get();
        } else {
            $courses = UniversityCourses::all();
        }

        $universities = $universities->get();

        return view("backend.course_management.course_finder", compact('countries', 'levels', 'categories', 'universities', 'courses'));
    }
    public function course_finder_result(Request $request){
        DB::enableQueryLog();
        
        $course = $request->course;
        $intake = $request->intake;
        $country = $request->country;
        $study_level = $request->study_level;
        $duration = $request->duration;
        $campus = $request->campus;

        //Score Filter
        $ielts = $request->ielts;
        $ielts_no_band = $request->ielts_no_band;
        $toefl = $request->toefl;
        $toefl_no_band = $request->toefl_no_band;
        $pte = $request->pte;
        $pte_no_band = $request->pte_no_band;
        $gre = $request->gre;
        $gmat = $request->gmat;

        // $year = $request->year;
        $subcategory = $category = "";
        $subcategories = array();
        // if(!empty($request->sub_category)){
        //     $subcategory = $request->sub_category;
        // }else{
        //     $subcategories = UniversityCourseCategories::where('parent_id',$request->category)->where('status',1)->pluck('id')->toArray();
        //     $subcategories = array_values($subcategories);
        //     if(count($subcategories)==0){
        //         $category = $request->category;
        //     }
        // }
        
        if($request->category){
            if(!empty($request->sub_category)){
                $subcategory = $request->sub_category;
            }else{
                // $category = $request->category;
                $subcategories = UniversityCourseCategories::where('parent_id',$request->category)->where('status',1)->pluck('id')->toArray();
                $subcategories = array_values($subcategories);
                $subcategories[] = $request->category;
                // dd($subcategories);
            }
        }

        $result = DB::table('university_courses')
                ->join('universities', 'universities.id', '=', 'university_courses.university')
            ->when(Auth::check() && Auth::user()->isAgent(), function ($query) {
                return $query->where('universities.agent_uni', 0);
            })
                ->when(!empty($course), function ($query) use ($course) {
                    return $query->Where(function ($query) use ($course) {
                        $query->where('university_courses.course_name', 'LIKE', "%$course%");
                        $query->orwhere('universities.university_name', 'LIKE', "%$course%");
                    });
                })
                ->when(!empty($study_level), function ($query) use ($study_level) {
                   return $query->where('university_courses.study_level',$study_level);
                })
                // ->when(!empty($year), function ($query) use ($year) {
                //    return $query->where('university_courses.year', 'LIKE', "%$year%");
                // })
                ->when(!empty($duration), function ($query) use ($duration) {
                    if($duration==12){
                        return $query->whereBetween('university_courses.duration', [1, 12]); 
                    }else if($duration == 24){
                       return $query->whereBetween('university_courses.duration', [12, 24]); 
                    }else if($duration == 36){
                       return $query->whereBetween('university_courses.duration', [24, 36]); 
                    }else if($duration == 48){
                       return $query->whereBetween('university_courses.duration', [36, 48]); 
                    }
                    else{
                       return $query->where('university_courses.duration','>',48); 
                    }
                })
                ->when(!empty($subcategory), function ($query) use ($subcategory) {
                   return $query->where('university_courses.category', $subcategory);
                })
                ->when(!empty($subcategories) && count($subcategories), function ($query) use ($subcategories) {
                   return $query->whereIn('university_courses.category', $subcategories);
                })
                ->when(!empty($category), function ($query) use ($category) {
                    return $query->where('university_courses.category', $category);
                 })
                ->when(!empty($campus), function ($query) use ($campus) {
                    return $query->where('university_courses.campus', $campus);
                 })
                ->when($intake && count($intake), function ($query) use ($intake) {
                    return $query->Where(function ($query) use ($intake) {
                        foreach($intake as $keyword) {
                            $query->orWhere('university_courses.intakes', 'LIKE', "%$keyword%");
                        }
                    });
                })
                ->when(!empty($country), function ($query) use ($country) {
                    // foreach($countries as $country) {
                    //     $query->orWhere('universities.country', 'LIKE', "%$country%");
                    // }
                    return $query->where('universities.country', $country);
                    // return $query->Where(function ($query) use ($countries) {
                    //     foreach($countries as $country) {
                    //         $query->orWhere('universities.country',$country);
                    //     }
                    // });
                })
                ->when(!empty($ielts), function ($query) use ($ielts) {
                    return $query->where('university_courses.ielts_score', "=", $ielts);
                })
                ->when(!empty($ielts_no_band), function ($query) use ($ielts_no_band) {
                    return $query->where('university_courses.ielts_no_band_less_than', "=", $ielts_no_band);
                })
                ->when(!empty($toefl), function ($query) use ($toefl) {
                    return $query->where('university_courses.toefl_score', ">=", $toefl);
                })
                ->when(!empty($toefl_no_band), function ($query) use ($toefl_no_band) {
                    return $query->where('university_courses.toefl_no_band_less_than', ">=", $toefl_no_band);
                })
                ->when(!empty($pte), function ($query) use ($pte) {
                    return $query->where('university_courses.pte_score', ">=", $pte);
                })
                ->when(!empty($pte_no_band), function ($query) use ($pte_no_band) {
                    return $query->where('university_courses.pte_no_band_less_than', ">=", $pte_no_band);
                })
                ->when(!empty($gre), function ($query) use ($gre) {
                    return $query->where('university_courses.gre_score', ">=", $gre);
                })
                ->when(!empty($gmat), function ($query) use ($gmat) {
                    return $query->where('university_courses.gmat_score', ">=", $gmat);
                })
            ->where('university_courses.status', "1")
                ->select(DB::raw('university_courses.*, universities.country, universities.university_name , universities.id as uni_id'))
            ->orderBy('university_courses.created_at', 'desc')
                ->limit(200)
                ->get();

        $queries = DB::getQueryLog();
        // \Log::debug(end($queries));  
        foreach($result as $res){
            $res->country = get_country_name($res->country);
        }
        return response()->json(['status'=>'success', 'data'=>$result,'query'=>$queries]);
    }

    public function course_details($id){
        $course = UniversityCourses::find($id);
        if(!$course){
            return redirect()->route('admin.dashboard.message')->withFlashDanger('Course not found!');
        }
        $university = get_university_data($course->university);
        if (Auth::check() && Auth::user()->isAgent() && $university && $university->agent_uni == 1) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('Access denied to this course.');
        }
        return view("backend.course_management.course_details",compact('course'));
    }

    public function university_details($id)
    {
        $university = Universities::find($id);
        if(!$university){
            return redirect()->route('admin.dashboard.message')->withFlashDanger('University not found!');
        }
        if (Auth::check() && Auth::user()->isAgent() && $university->agent_uni == 1) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('Access denied to this university.');
        }
        return view("backend.course_management.university_details",compact('university'));
    }

    public function course_info($id)
    {
        $course = UniversityCourses::find($id);
    
        if (!$course) {
            return response()->json(['status' => 'error', 'message' => 'Course not found.'], 404);
        }

        // Restrict if agent and university is hidden
        $university = get_university_data($course->university);
        if (Auth::check() && Auth::user()->isAgent() && $university && $university->agent_uni == 1) {
            return response()->json(['status' => 'error', 'message' => 'Access denied for agents.'], 403);
        }
        $courseYears = explode(',',$course->year);

        $courseMonths = explode(',',$course->intakes);

        foreach($courseMonths as $months){
            $monthNum[] = array('moName'=>date("M", strtotime($months)),'moNum' => date("m", strtotime($months)));
        }

        $data = (['course_years'=> $courseYears,'course_months'=>$monthNum, 'course_name'=> $course->course_name, 'id'=> $course->id, 'university_id' => $course->university,'course_uni' => get_university_data($course->university)->university_name, 'university_country_id' => get_university_data($course->university)->country,'course_country' => get_country_name(get_university_data($course->university)->country)]);
        return response()->json(['status'=>'success','data'=>$data]);
    }

    public function suggestCourseToStudent(Request $request) {
        $course = $request->input('assign_course');
        $student = $request->input('stu_id');
        $is_multi = $request->input('is_multi');
        if($is_multi){
            $courses = explode(',',$course);
            // if(CourseShortlisted::where('student_id', $student)->whereIn('course_id', $courses)->exists()) {
            //     return response()->json(['error'=>'Preference already exists for selected course(s)!']);
            // }
            $count_sugg = 0;
            foreach($courses as $course){
                $chk_suggested = CourseShortlisted::where('student_id', $student)->where('course_id', $course)->first();
                if(!$chk_suggested){
                    CourseShortlisted::create([
                        'student_id' => $student,
                        'course_id' => $course,
                        'staff_suggested' => 1,
                        'staff_id' => \Auth::user()->id
                    ]);
                    $count_sugg++;

                    //email
                    $user = User::find($student);
                    $mail_tpl = MailTemplates::find("COURSE_SUGGESTION");
                    if(!$user->agent_id) {
                        if ($mail_tpl && $mail_tpl->status) {
                            $email_message = $mail_tpl->message;
                            \Log::debug("MAIL COURSE_SUGGESTION:");                       
                            $email_message = str_ireplace("#sitename#", app_name(), $email_message);
                            $email_message = str_ireplace("#siteurl#", url(''), $email_message);
                            $email_message = str_ireplace("#first_name#", $user->first_name, $email_message);
                            $email_message = str_ireplace("#full_name#", $user->full_name, $email_message);
                            try{
                                //$user->notify(new UserAlert($mail_tpl->subject, $email_message));
                                user_notify($user,$mail_tpl->subject, $email_message);
                            }catch (\Exception $e) {
                                $message = $e->getMessage();
                                \Log::debug("Mail Error: (Course Suggestion) ".$message);   
                            }
                        }
                        
                        //send whatsapp
                        if (get_site_config('whatsapp_enabled')) {
                            $numbers = [$user->whatsapp_dialcode.$user->whatsapp]; 

                            $hook = 'course_suggestion';

                            send_whatsapp_message($numbers, $hook, $user->id);
                        }
                    }  

                    // Send Push App Notification
                    if(\Auth::user()->isAgent()){
                        $message = "Course Suggested by Agent: " . get_user_full_name_by_id(\Auth::user()->id) . 
                        "<br>"._label('backend_student').": " . $user->full_name." (". $user->id. ")";
                        $email_message = "Hi, <br>Course Suggested by Agent: " . get_user_full_name_by_id(\Auth::user()->id) . 
                        "<br>"._label('backend_student').": " . $user->full_name." (". $user->id. ")";
                        $subject = "Course Suggested by Agent: " . get_user_full_name_by_id(\Auth::user()->id) . 
                        "<br>"._label('backend_student').": " . $user->full_name." (". $user->id. ")";
                    } else{
                        $message = "Course Suggested by Staff: " . get_user_full_name_by_id(\Auth::user()->id) . 
                        "<br>"._label('backend_student').": " . $user->full_name." (". $user->id. ")";
                        $email_message = "Hi, <br>Course Suggested by Staff: " . get_user_full_name_by_id(\Auth::user()->id) . 
                        "<br>"._label('backend_student').": " . $user->full_name." (". $user->id. ")";
                        $subject = "Course Suggested by Staff: " . get_user_full_name_by_id(\Auth::user()->id) . 
                        "<br>"._label('backend_student').": " . $user->full_name." (". $user->id. ")";
                    }

                    //Notify Staff
                    $email_message .= "<br><b>Course: ".$course;
                    send_mail_notification($student,"","COURSE_SUGGESTED_BY_STAFF",$subject,$email_message);

                    $subject = "";
                    send_in_app_notification(
                        $user->id, 
                        $user->id, 
                        "COURSE_SUGGESTED_BY_STAFF", 
                        $subject, 
                        $message,
                        "lead", 
                    );
                }
            }
            if($count_sugg == 0){
                return response()->json(['error'=>'Selected course(s) already Suggested or Bookmarked!']);
            }        

            return response()->json(['success'=> $count_sugg.' Course(s) suggested successfully!']);
        }
        else{
            if(CourseShortlisted::where('student_id', $student)->where('course_id', $course)->exists()) {
                return response()->json(['error'=>'Course Suggestion or Bookmark already exists!']);
            } else {
                CourseShortlisted::create([
                    'student_id' => $student,
                    'course_id' => $course,
                    'staff_suggested' => 1,
                    'staff_id' => \Auth::user()->id
                ]);

                //email
                $user = User::find($student);
                $mail_tpl = MailTemplates::find("COURSE_SUGGESTION");
                if(!$user->agent_id) {
                    if ($mail_tpl && $mail_tpl->status) {
                        $email_message = $mail_tpl->message;
                        \Log::debug("MAIL COURSE_SUGGESTION:");                       
                        $email_message = str_ireplace("#sitename#", app_name(), $email_message);
                        $email_message = str_ireplace("#siteurl#", url(''), $email_message);
                        $email_message = str_ireplace("#first_name#", $user->first_name, $email_message);
                        $email_message = str_ireplace("#full_name#", $user->full_name, $email_message);
                        try{
                            //$user->notify(new UserAlert($mail_tpl->subject, $email_message));
                            user_notify($user,$mail_tpl->subject, $email_message);
                        }catch (\Exception $e) {
                            $message = $e->getMessage();
                            \Log::debug("Mail Error: (Course Suggestion) ".$message);   
                        }
                    }  
                }           

                // Send Push App Notification
                if(\Auth::user()->isAgent()){
                    $message = "Course Suggested by Agent: " . get_user_full_name_by_id(\Auth::user()->id) . 
                    "<br>"._label('backend_student').": " . $user->full_name." (". $user->id. ")";
                    $email_message = "Hi, <br>Course Suggested by Agent: " . get_user_full_name_by_id(\Auth::user()->id) . 
                    "<br>"._label('backend_student').": " . $user->full_name." (". $user->id. ")";
                    $subject = "Course Suggested by Agent: " . get_user_full_name_by_id(\Auth::user()->id) . 
                    "<br>"._label('backend_student').": " . $user->full_name." (". $user->id. ")";
                } else{
                    $message = "Course Suggested by Staff: " . get_user_full_name_by_id(\Auth::user()->id) . 
                    "<br>"._label('backend_student').": " . $user->full_name." (". $user->id. ")";
                    $email_message = "Hi, <br>Course Suggested by Staff: " . get_user_full_name_by_id(\Auth::user()->id) . 
                    "<br>"._label('backend_student').": " . $user->full_name." (". $user->id. ")";
                    $subject = "Course Suggested by Staff: " . get_user_full_name_by_id(\Auth::user()->id) . 
                    "<br>"._label('backend_student').": " . $user->full_name." (". $user->id. ")";
                }

                //Notify Staff
                $email_message .= "<br><b>Course: ".$course;
                send_mail_notification($user->id,"","COURSE_SUGGESTED_BY_STAFF",$subject,$email_message);

                $subject = "";
                send_in_app_notification(
                    $user->id, 
                    $user->id, 
                    "COURSE_SUGGESTED_BY_STAFF", 
                    $subject, 
                    $message,
                    "lead", 
                );

                return response()->json(['success'=>'Course suggested successfully!']);
            }
        }
    }

    public function finder_suggestion_delete(Request $request){
        // if (!(\Auth::user()->isAdmin() || \Auth::user()->can('delete-lead-student-application-batch'))) {
        //     return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        // }

        $id = $request->suggestion_id;
        //dd($id);
        $suggestion = CourseShortlisted::where('id', $id)->where('student_approved', "0")->first();
        $user = User::where("id", $suggestion->student_id)->first();
        if($suggestion) {
            // Send Push App Notification
            if(\Auth::user()->isAgent()){
                $message = "Course Finder Suggestion Deleted by Agent: " . get_user_full_name_by_id(\Auth::user()->id) . 
                "<br>"._label('backend_student').": " . $user->full_name." (". $user->id. ")";
                $email_message = "Hi, <br>Course Finder Suggestion Deleted by Agent: " . get_user_full_name_by_id(\Auth::user()->id) . 
                "<br>"._label('backend_student').": " . $user->full_name." (". $user->id. ")";
                $subject = "Course Finder Suggestion Deleted by Agent: " . get_user_full_name_by_id(\Auth::user()->id) . 
                "<br>"._label('backend_student').": " . $user->full_name." (". $user->id. ")";
            } else{
                $message = "Course Finder Suggestion Deleted by Staff: " . get_user_full_name_by_id(\Auth::user()->id) . 
                "<br>"._label('backend_student').": " . $user->full_name." (". $user->id. ")";
                $email_message = "Hi, <br>Course Finder Suggestion Deleted by Staff: " . get_user_full_name_by_id(\Auth::user()->id) . 
                "<br>"._label('backend_student').": " . $user->full_name." (". $user->id. ")";
                $subject = "Course Finder Suggestion Deleted by Staff: " . get_user_full_name_by_id(\Auth::user()->id) . 
                "<br>"._label('backend_student').": " . $user->full_name." (". $user->id. ")";
            }

            //Notify Staff
            send_mail_notification($user->id,"","LEAD_STUDENT_APPLICATION_COURSEPREFERENCE_OTHERS_DELETE",$subject,$email_message);

            $subject = "";
            send_in_app_notification(
                $user->id, 
                $user->id, 
                "LEAD_STUDENT_APPLICATION_COURSEPREFERENCE_OTHERS_DELETE", 
                $subject, 
                $message,
                "student", 
            );

            $suggestion->delete();
            return redirect('admin/auth/user/'.$suggestion->student_id.'#coursefindersuggestion')->withFlashSuccess("Course Finder Suggestion Deleted Successfully.");
        } else {
            $stu_data = CourseShortlisted::where('id', $id)->first();
            return redirect('admin/auth/user/'.$stu_data->student_id.'#coursefindersuggestion')->withFlashDanger("Cannot delete. Suggestion already accepted by student.");
        }
    }

    public function bulk_course_delete(Request $request)
    {
        if (!(\Auth::user()->isAdmin() || \Auth::user()->can('overseas education configs'))) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        $coursedata_arr = array();
        $id_arr = explode(',', $request->input('id_arr'));
        if (is_array($id_arr)) {
            foreach ($id_arr as $course_id) {
                $course = UniversityCourses::find($course_id);
                if($course){
                    $check_pref = DB::table('course_preference')->where('course_id',$course->id)->count();
                    if($check_pref > 0) {
                        return redirect()->route('admin.coursemanagement')->withFlashDanger('Some of the courses are present in students course preferences, Could not delete Courses');
                    }else{
                     $course->delete();   
                    }
                }
            }
            return redirect()->route('admin.coursemanagement')->withFlashSuccess('Selected Course(s) Deleted Successfully');
        }
    }

    public function bulk_course_enable(Request $request)
    {
        if (!(\Auth::user()->isAdmin() || \Auth::user()->can('overseas education configs'))) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        $coursedata_arr = array();
        $id_arr = explode(',', $request->input('id_arr'));
        if (is_array($id_arr)) {
            foreach ($id_arr as $course_id) {
                $course = UniversityCourses::find($course_id);
                $course->status = 1;
                $course->save();
            }
            return redirect()->route('admin.coursemanagement')->withFlashSuccess('Selected Course(s) Enabled Successfully');
        }
    }

    public function bulk_course_disable(Request $request)
    {
        if (!(\Auth::user()->isAdmin() || \Auth::user()->can('overseas education configs'))) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
        }
        $coursedata_arr = array();
        $id_arr = explode(',', $request->input('id_arr'));
        if (is_array($id_arr)) {
            foreach ($id_arr as $course_id) {
                $course = UniversityCourses::find($course_id);
                $course->status = 0;
                $course->save();
            }
            return redirect()->route('admin.coursemanagement')->withFlashSuccess('Selected Course(s) Disabled Successfully');
        }
    }

    // public function courseImportOpen($currentPage){
    //     $userRepo = new UserRepository();
    //     $staff = $userRepo->getNonStudentsTrainers();
    //     $courseCategory = DB::table('university_course_categories')->get();
    //     return view('backend.import_export.index', compact('staff','courseCategory','currentPage'));
    // }

    public function createCoursePreference($id) {
        $course = CourseShortlisted::join('university_courses', 'university_courses.id', '=', 'course_shortlisted.course_id')
            ->join('universities', 'universities.id', '=', 'university_courses.university')
            ->select('course_shortlisted.*','university_courses.course_name','university_courses.university','university_courses.study_level','university_courses.campus','university_courses.duration','university_courses.intakes','university_courses.intakes','university_courses.year','universities.university_name','universities.country','universities.id')
            ->where('course_shortlisted.id', $id)
            ->first();

        if($course){
            $nmonth = date('m',strtotime($course->intakes));
            // $countryCoursePref = get_country_id_by_name($course->country);

            if (!DB::table('course_preference')
                ->where('userid', $course->student_id)
                ->where('course_id', $course->course_id)
                ->where('university_id', $course->university)
                ->where('country_id', $course->country)
                ->where('intake_year', $course->year)
                ->where('intake', $nmonth)
                ->exists()) {
                    DB::table('course_preference')
                    ->insert([
                        'userid' => $course->student_id, 
                        'course_id' => $course->course_id,  
                        'university_id' => $course->university, 
                        'country_id' => $course->country,
                        'intake_year' => $course->year, 
                        'intake' => $nmonth,
                        'created_by' => \Auth::user()->id,
                        'created_at' => now(),
                    ]);

                return redirect('admin/auth/user/'.$course->student_id.'#coursepreference')->withFlashSuccess("Course Preferences Added Successfully");
            }else{
                return redirect('admin/auth/user/'.$course->student_id.'#coursefindersuggestion')->withFlashDanger("Course Preferences Already Exists!");
            }
        }else{
            return redirect('admin/auth/user/'.$course->student_id.'#coursefindersuggestion')->withFlashDanger("Course Not Found!");
        }
    }

    public function downloadCourses(REQUEST $request)
    {
        $course_ids = $request->input('download_multi_course');
        $course_ids = explode(',',$course_ids);
        
        $courses = DB::table('university_courses')
            ->join('universities', 'university_courses.university', '=', 'universities.id')
            ->select('universities.country','universities.university_name', 'university_courses.*')
            ->whereIn('university_courses.id',$course_ids)
            ->get();
        $filename = "Courses-List";
        if($request->input('btn_pdf')){
        $pdf = PDF::loadView('backend.course_management.course_download',['courses' => $courses]);
        $pdf->set_option('isFontSubsettingEnabled', false);
        $pdf->setPaper('A4','landscape');
        return $pdf->download($filename.".pdf");
        }
        else if($request->input('btn_xls')){
            $fileName = "Courses-List.csv"; 
            $csvFile = fopen(public_path('storage/'.$fileName), 'w');
            $headers = array('Course','Institution Name','Country','Duration','Intakes','Application Fee','Tuition Fee','Study Level','Study Area');
            //$headers = array_keys((array) $data[0]);
            fputcsv($csvFile, $headers);

            foreach($courses as $course){
                $intakes = array();
                if($course->intakes){
                    $intakes = explode(",", $course->intakes);
                }
                $count = count($intakes);
                $allIntake = "";
                $i = 0;
                foreach($intakes as $intake){
                    $i++;
                    $allIntake .= substr($intake, 0, 3);
                    if($i < $count){
                        $allIntake .= ", ";
                    }
                }
                    
                $lineData = array($course->course_name ? $course->course_name : "--",
                        $course->university_name ? $course->university_name : "--",
                        $course->country ? get_country_name($course->country) : "--",
                        $course->duration ? $course->duration : "--",
                        $course->intakes ? $allIntake : "--",
                        $course->application_fee ? $course->currency." ".number_format($course->application_fee,2) : "--",
                        $course->yearly_tuition_fee ? $course->currency." ".number_format($course->yearly_tuition_fee,2) : "--",
                $course->commission_percentage ? $course->commission_percentage : "--",
                        $course->study_level ? $course->study_level : "--",
                        $course->category ? get_university_course_parent_category_name($course->category ) : "--");
                fputcsv($csvFile, (array) $lineData);
            }
            fclose($csvFile);
            return \Response::download(public_path('storage/'.$fileName))->deleteFileAfterSend(true);
                // // Excel file name for download 
                // $fileName = "Courses-List.xls"; 
                // // Headers for download 
                // header("Content-Type: application/vnd.ms-excel");
                // header("Content-Disposition: attachment; filename=\"$fileName\"");
                
                // // Column names 
                // $fields = array('Course','Institution Name','Country','Duration','Intakes','Application Fee','Tuition Fee','Study Level','Study Area'); 
                
                // // Display column names as first row 
                // $excelData = implode("\t", array_values($fields)) . "\n"; 
                
                // // Fetch records from database 
               
                // if(count($courses)){ 
                //     // Output each row of the data 
                //     foreach($courses as $course){
                //         $intakes = array();
                //         if($course->intakes){
                //             $intakes = explode(",", $course->intakes);
                //         }
                //         $count = count($intakes);
                //         $allIntake = "";
                //         $i = 0;
                //         foreach($intakes as $intake){
                //             $i++;
                //             $allIntake .= substr($intake, 0, 3);
                //             if($i < $count){
                //                 $allIntake .= ", ";
                //             }
                //         }
                            
                //         $lineData = array($course->course_name ? $course->course_name : "--",
                //                 $course->university_name ? $course->university_name : "--",
                //                 $course->country ? get_country_name($course->country) : "--",
                //                 $course->duration ? $course->duration : "--",
                //                 $course->intakes ? $allIntake : "--",
                //                 $course->application_fee ? $course->currency." ".number_format($course->application_fee,2) : "--",
                //                 $course->yearly_tuition_fee ? $course->currency." ".number_format($course->yearly_tuition_fee,2) : "--",
                //                 $course->study_level ? $course->study_level : "--",
                //                 $course->category ? get_university_course_parent_category_name($course->category ) : "--");
                       
                //         // array_walk($lineData, array($this, 'filterData')); 
                //         $excelData .= implode("\t", array_values($lineData)) . "\n"; 
                //     } 
                // }else{ 
                //     $excelData .= 'No records found...'. "\n"; 
                // } 
                
                // echo $excelData; 
                // exit;


            }
        else{
            
        }
    }

    public function search_student_data(Request $request){
        $users = User::where('id', $request->id)->first();
        $country_interested = "";
        $intake_month = "";
        if($users){
            if($users->country_interested){
                $country_interested = get_country_id_by_name($users->country_interested);
            }
    
            $user_details = UserDetails::where('userid', $request->id)->first(['intake_month']);
            
            if($user_details->intake_month){
                $intake_month = get_intake_month_name($user_details->intake_month);
            }
        }
        
        return ['results' => $users, 'intake_month' => $intake_month,'country_interested' => $country_interested];
    }

     //method to hide bulk universities
    public function bulkHideUniversities(Request $request)
    {
        if (\Auth::user()->isAgent()) {
            return redirect()->route('agent.dashboard.message')
                ->withFlashDanger('You do not have access permission to this feature.');
        }

        $ids = explode(',', $request->selected_ids);
        $action = $request->bulk_action;

        if (empty($ids) || !$action) {
            return redirect()->back()->withFlashDanger('Please select universities and an action.');
        }

        if ($action === 'hide') {
            Universities::whereIn('id', $ids)->update(['agent_uni' => 1]);
            return redirect()->back()->withFlashSuccess('Selected universities have been hidden from agents.');
        } elseif ($action === 'unhide') {
            Universities::whereIn('id', $ids)->update(['agent_uni' => 0]);
            return redirect()->back()->withFlashSuccess('Selected universities are now visible to agents.');
        } elseif ($action === 'delete') {
            Universities::whereIn('id', $ids)->delete();
            return redirect()->back()->withFlashSuccess('Selected universities have been deleted.');
        }

        return redirect()->back()->withFlashDanger('Invalid action selected.');
    }
}
