<?php

namespace App\Http\Controllers\Backend;

use App\Mail\MailSend;
use App\Models\Auth\Course;
use Illuminate\Support\Str;
use App\Models\Auth\Webinar;
use Illuminate\Http\Request;
use App\Models\MailTemplates;
use Illuminate\Support\Facades\DB;
use App\Models\Auth\StudentWebinar;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Mail;
use App\Models\Auth\UserNotification;
use App\Notifications\Backend\UserAlert;
use App\Mail\AgentWebinarRegistrationMail;
use App\Repositories\Backend\Auth\UserRepository;


class WebinarAgentController extends Controller
{
	public function __construct()
	{
		$this->middleware(function ($request, $next) {
			// Let all authenticated users through
			return $next($request);
		});
	}

	public function storeAgentRegistration(Request $request){
		try {
			$request->validate([
				'webinar_id' => 'required|exists:webinar,id',
			]);

			$user = auth()->user();
			if (!$user) {
				return response()->json(['success' => false, 'message' => 'User not authenticated'], 401);
			}

			$webinarId = $request->webinar_id;
			// Check if this agent already registered
			if (StudentWebinar::where('webinar_id', $request->webinar_id)->where('agent_id', $user->id)->exists()) {
				return response()->json(['success' => false, 'message' => 'You have already registered.']);
			}

			StudentWebinar::create([
				'webinar_id' => $request->webinar_id,
				'agent_id' => $user->id,
			]);

			// 🔔 Send confirmation mail
			$webinar = Webinar::find($webinarId);
			$subject = "Webinar Registration Confirmation";
			$message = "Dear {$user->full_name},\n\n
			You have successfully registered for the webinar titled '{$webinar->topic}' scheduled on {$webinar->date} at {$webinar->venue}.\n\n
			Thank you for registering. We look forward to your participation!\n\n";

			$message = "Agent $user->full_name has registered for the webinar: {$webinar->topic} scheduled on {$webinar->date} at {$webinar->venue}.";
			$noti = new UserNotification;
			$noti->user_id = 1;
			$noti->application_id = $webinar->id;
			$noti->application_type = "webinar";
			$noti->message = $message;
			$noti->from_user = $user->id;
			$noti->save();

			user_notify($user->email, $subject, $message);

			return response()->json(['success' => true, 'message' => 'Registered successfully.']);
		} catch (\Exception $e) {
			// Log the error for debugging
			\Log::error('Webinar registration error: ' . $e->getMessage());
			return response()->json(['success' => false, 'message' => "Something went wrong!"], 500);
		}
	}

	public function index(){
		if(get_site_config('agent_module_webinar')){
			$agentId = auth()->id();
			$webinars = DB::table('webinar')
				->select('id', 'topic', 'date', 'venue', 'link', 'image', 'target_user_type', 'description', 'public_slug')
				->whereIn('target_user_type', ['agent', 'student_agent'])
				->orderBy('webinar.date', 'ASC')
				->get();
	
			$registeredWebinarIds = StudentWebinar::where('agent_id', $agentId)->pluck('webinar_id')->toArray();
	
			return view('backend.webinar.webinaragent', [
				'pages' => $webinars,
				'registeredWebinarIds' => $registeredWebinarIds
			]);
		}

		return redirect()->route('admin.agent.dashboard')->withFlashDanger("Webinar and Events Feature Disabled!");
	}

	public function store(Request $request)
	{
		$path = null; 

		if ($request->hasFile('image')) {
			$file = $request->image->getClientOriginalName();
			$filename = pathinfo($file, PATHINFO_FILENAME);
			$filename = substr($filename, 0, 125);
			$filename = $filename . Str::random(20) . '.' . $request->image->getClientOriginalExtension();
			$path = $request->file('image')->storeAs('images', $filename, 'public');
		}

		$date = $request->input('date')
			? date("Y-m-d H:i", strtotime($request->input('date')))
			: null;

		$data = new Webinar([
			'topic' => $request->input('topic'),
			'image' => $path, // Safe even if null
			'date' => $date,
			'venue' => $request->input('venue'),
			'link' => $request->input('link'),
			'description' => $request->input('description'),
			'target_user_type' => $request->input('target_user_type'),
			'public_slug' => Str::uuid(),
		]);
		$data->save();

		return redirect()->route('admin.webinar.index')->withFlashSuccess('Webinar Added Successfully');
	}
}
