<?php

namespace App\Http\Controllers\Backend;

use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use App\Models\Auth\Course;
use Illuminate\Support\Str;
use App\Models\Auth\Webinar;
use App\Mail\MailSend;
use App\Notifications\Backend\UserAlert;
use App\Repositories\Backend\Auth\UserRepository;
use App\Models\MailTemplates;

class WebinarController extends Controller {

    public function __construct()
    {
        $this->middleware(function ($request, $next) {
            $user = \Auth::user();
            if (!(\Auth::user()->isAdmin() || \Auth::user()->can('cms-events'))) {
                return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
            }            
            return $next($request);
        });
    }

    public function index() {
        $pages = DB::table('webinar')
                // ->select('courses.title', 'webinar.topic', 'webinar.date', 'webinar.venue', 'webinar.link', 'webinar.image', 'webinar.id')
                ->select('webinar.topic', 'webinar.date', 'webinar.venue', 'webinar.link', 'webinar.image', 'webinar.id', 'webinar.target_user_type')
                // ->leftJoin('courses', 'courses.id', '=', 'webinar.course')
                ->orderBy('webinar.created_at', "DESC")
                ->get();
        
        return view('backend.webinar.webinardetails', ['pages' => $pages]);
    }

    public function create() {
        // $courses = Course::all();
        return view('backend.webinar.webinarcreate');
    }

    public function store(Request $request) {
        // dd($request->all());
        if ($request->file('image')) {
        $file = $request->image->getClientOriginalName();
        $filename = pathinfo($file, PATHINFO_FILENAME);
        if (strlen($filename) > 125)
            $filename = substr($filename, 0, 125);

        $filename = $filename . Str::random(20) . "." . $request->image->getClientOriginalExtension();
        $path = $request->file('image')->storeAs('/images', $filename, 'public');
        }

        if (!empty($request->input('date'))) {
            $date = date("Y-m-d H:i", strtotime($request->get('date')));
        } else {
            $date = NULL;
        }
        // $date = $request->get('date');

        $emailsRaw = $request->input('notify_email');
        $emails = collect(explode(',', $emailsRaw))
            ->map(fn($email) => trim($email))
            ->filter(fn($email) => filter_var($email, FILTER_VALIDATE_EMAIL))
            ->unique()
            ->values()
            ->all();

        $webinar_date = $request->get('date');
        $data = new Webinar([
            'topic' => $request->get('topic'),
            // 'course' => $request->get('course'),
            'image' => $path,
            'date' => $date,
            'venue' => $request->get('venue'),
            'link' => $request->get('link'),
            'notify_email' => $emailsRaw ? $emailsRaw : null,
            'target_user_type' => $request->input('target_user_type'),
            'description' => $request->get('description'),
            'public_slug' => Str::uuid(),
        ]);
        $data->save();

        // Send email to each
        $subject = "New Webinar: " . $request->get('topic');
        $message = "Hi,\n\nA new webinar has been scheduled:\n\n" .
            "Topic: " . $request->get('topic') . "\n" .
            "Date: " . $date . "\n" .
            "Venue: " . $request->get('venue') . "\n" .
            "Link: " . $request->get('link') . "\n\n" .
            "Description: " . $request->get('description') . "\n\n" .
            "Thank you for your interest in our webinars.\n\n";
        foreach ($emails as $email) {
            user_notify($email, $subject, $message);
        }

        return redirect()->route('admin.webinar.index')->withFlashSuccess('Webinar Added Successfully');
    }

    public function view($id) {
        $webinars = Webinar::find($id);
        $courses = Course::find($webinars->course);
        return view('backend.webinar.webinarview', ['webinars' => $webinars, 'courses' => $courses]);
    }

    public function edit($id) {
        $page = Webinar::find($id);
        $courses = Course::all();
        return view('backend.webinar.webinaredit', ['page' => $page, 'courses' => $courses]);
    }

    public function update(Request $request, $id) {
        $webinarupdate = Webinar::find($id);

        if ($request->file('image')) {
        $file = $request->image->getClientOriginalName();
        $filename = pathinfo($file, PATHINFO_FILENAME);
        if (strlen($filename) > 125)
            $filename = substr($filename, 0, 125);

        $filename = $filename . Str::random(20) . "." . $request->image->getClientOriginalExtension();
        $path = $request->file('image')->storeAs('/images', $filename, 'public');
        //Log::debug("::ticket attachment savedpath::$path");             
        $webinarupdate->image = $path;
        }

        if (!empty($request->input('date'))) {
            $date = date("Y-m-d H:i", strtotime($request->get('date')));
        } else {
            $date = NULL;
        }
        // $date = $request->get('date');

        $webinarupdate->topic = $request->get('topic');
        // $webinarupdate->course = $request->get('course');
        $webinarupdate->date = $date;
        $webinarupdate->venue = $request->get('venue');
        $webinarupdate->link = $request->get('link');
        $webinarupdate->description = $request->get('description');
        $webinarupdate->target_user_type = $request->get('target_user_type');
        $webinarupdate->save();
        return redirect()->route('admin.webinar.index')->withFlashSuccess('Webinar Information saved!');
    }

    public function destroy($id) {
        $webinardelete = Webinar::find($id);
        $webinardelete->delete();
        return redirect()->route('admin.webinar.index')->withFlashSuccess('Webinar Deleted Successfully');
    }

    public function details($id, $webinarType = "student") {
        $webinar = Webinar::find($id);
        if($webinarType == "agent"){
            $data = DB::table('student_webinar')
                    ->select('users.first_name', 'users.last_name', 'users.id', 'users.email', 'users.mobile', 'users.dialcode')
                    ->join('users', 'users.id', '=', 'student_webinar.agent_id')
                    ->join('webinar', 'webinar.id', '=', 'student_webinar.webinar_id')
                    ->where('student_webinar.webinar_id', $id)
                    ->orderBy('student_webinar.created_at', "DESC")
                    ->get();
        } elseif($webinarType == "student"){
            $data = DB::table('student_webinar')
                    ->select('users.first_name', 'users.last_name', 'users.id', 'users.email', 'users.mobile', 'users.dialcode')
                    ->join('users', 'users.id', '=', 'student_webinar.student_id')
                    ->join('webinar', 'webinar.id', '=', 'student_webinar.webinar_id')
                    ->where('student_webinar.webinar_id', $id)
                    ->orderBy('student_webinar.created_at', "DESC")
                    ->get();
        } else{
            $records = DB::table('student_webinar')
                ->join('webinar', 'webinar.id', '=', 'student_webinar.webinar_id')
                ->leftJoin('users as student', 'student.id', '=', 'student_webinar.student_id')
                ->leftJoin('users as agent', 'agent.id', '=', 'student_webinar.agent_id')
                ->where('student_webinar.webinar_id', $id)
                ->orderBy('student_webinar.created_at', 'DESC')
                ->get([
                    'student.id as student_id',
                    'student.first_name as student_first_name',
                    'student.last_name as student_last_name',
                    'student.email as student_email',
                    'student.mobile as student_mobile',
                    'student.dialcode as student_dialcode',
                    'agent.id as agent_id',
                    'agent.first_name as agent_first_name',
                    'agent.last_name as agent_last_name',
                    'agent.email as agent_email',
                    'agent.mobile as agent_mobile',
                    'agent.dialcode as agent_dialcode',
                ]);

            // Merge both student and agent into one collection
            $data = collect();

            foreach ($records as $row) {
                // Add student if exists
                if ($row->student_id) {
                    $data->push((object)[
                        'id' => $row->student_id,
                        'first_name' => $row->student_first_name,
                        'last_name' => $row->student_last_name,
                        'email' => $row->student_email,
                        'mobile' => $row->student_mobile,
                        'dialcode' => $row->student_dialcode,
                        'type' => 'Student',
                    ]);
                }

                // Add agent if exists
                if ($row->agent_id) {
                    $data->push((object)[
                        'id' => $row->agent_id,
                        'first_name' => $row->agent_first_name,
                        'last_name' => $row->agent_last_name,
                        'email' => $row->agent_email,
                        'mobile' => $row->agent_mobile,
                        'dialcode' => $row->agent_dialcode,
                        'type' => 'Agent',
                    ]);
                }
            }
        }

        return view('backend.webinar.webinarlist', ['data' => $data, 'webinar' => $webinar, 'webinarType' => $webinarType]);
    }
}
