<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\WhatsappTemplates;
use Illuminate\Http\Request;


class WhatsappTemplatesController extends Controller
{

    public function __construct()
    {
        $this->middleware(function ($request, $next) {
            $user = \Auth::user();
            if (!(\Auth::user()->isAdmin() || \Auth::user()->can('mail templates'))) {
                return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this feature.');
            }
            return $next($request);
        });
    }

    public function index()
    {
        $templates = WhatsappTemplates::orderByDesc('created_at')->get();
        return view('backend.whatsapp_templates.index', compact('templates'));
    }


    public function create()
    {
        $variables = "sitename,siteurl,first_name,full_name";
        return view('backend.whatsapp_templates.create', compact('variables'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  Page $template
     * @return view
     */
    public function edit($id)
    {
        $template = WhatsappTemplates::where('hook', '=', $id)->first();
        return view('backend.whatsapp_templates.edit', compact('template'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'hook'       => 'required|max:50|unique:whatsapp_templates',
            'message'    => 'required|max:1024',
            'media_type' => 'nullable|in:image,video,document',
            'media_file' => 'nullable|file|mimes:jpg,jpeg,png,mp4,mov,avi,pdf,doc,docx|max:10240',
        ]);

        preg_match_all('/#(\S+)#/', $request->message, $matches);
        foreach ($matches[0] as $vars) {
            if (!in_array($vars, ['#first_name#', '#full_name#', '#sitename#', '#siteurl#'])) {
                return redirect()->route('admin.whatsapptemplates.create')
                                ->withFlashDanger('Invalid variable '.$vars);
            }
        }

        $filePath = null;
        if ($request->hasFile('media_file')) {
            $file      = $request->file('media_file');
            $extension = $file->getClientOriginalExtension();
            $filename  = time() . '_' . uniqid() . '.' . $extension;
            $filePath  = $file->storeAs('uploads', $filename, 'public');
        }

        $hook = strtolower(str_replace(' ', '_', $request->get('hook')));

        WhatsappTemplates::create([
            'hook'       => $hook,
            'message'    => $request->get('message'),
            'status'     => 1,
            'variables'  => 'sitename,siteurl,first_name,full_name',
            'media_type' => $request->get('media_type'),
            'media_file' => $filePath, 
        ]);

        return redirect()
            ->route('admin.whatsapptemplates')
            ->withFlashSuccess('Whatsapp Template Created Successfully!');
    }


    public function update(Request $request, $id)
    {
        $request->validate([
            'message'    => 'required|max:1024',
            'media_type' => 'nullable|in:image,video,document',
            'media_file' => 'nullable|file|mimes:jpg,jpeg,png,mp4,mov,avi,pdf,doc,docx|max:10240',
        ]);

        preg_match_all('/#(\S+)#/', $request->message, $matches);
        // foreach($matches[0] as $vars){
        //     if(!in_array($vars, ['#first_name#', '#full_name#', '#sitename#', '#siteurl#'])){
        //         return redirect()->back()->withFlashDanger('Invalid variable '.$vars);
        //     }
        // }

        $whtsp_tpl = WhatsappTemplates::findOrFail($id);
        $whtsp_tpl->message = $request->get('message');
        $whtsp_tpl->media_type = $request->get('media_type');

        if ($request->hasFile('media_file')) {
            $file      = $request->file('media_file');
            $extension = $file->getClientOriginalExtension();
            $filename  = time() . '_' . uniqid() . '.' . $extension;
            $filePath  = $file->storeAs('uploads', $filename, 'public');
            $whtsp_tpl->media_file = $filePath;
        }

        $whtsp_tpl->save();

        return redirect()->route('admin.whatsapptemplates')->withFlashSuccess('Whatsapp Template Updated Successfully!');
    }

    public function changeStatus($hook, $status) {
        $whtsp_tpl = WhatsappTemplates::findOrFail($hook);
        $whtsp_tpl->status = $status;
        $whtsp_tpl->save();
        return redirect()->route('admin.whatsapptemplates')->withFlashSuccess('Whatsapp Template Updated!');
    }

    public function destroy($id)
    {
        $whtsp_tpl = WhatsappTemplates::findOrFail($id);
        if ($whtsp_tpl->system_tpl) {
            return redirect()->back()->withFlashDanger('This template cannot be deleted');
        }
        $whtsp_tpl->delete();
        return redirect()->back()->withFlashSuccess('Template deleted successfully');
    }

    public function ajaxgetwhtsptpl(Request $request) {
        if (!$request->ajax()) {
            return redirect()->route('admin.dashboard.message')->withFlashDanger('You do not have access permission to this page.');
        }
        $whtsp_tpl = WhatsappTemplates::findOrFail($request->hook);
        echo json_encode($whtsp_tpl);
    }
}
