<?php

namespace App\Http\Controllers\Frontend\Auth;

use App\Http\Controllers\Controller;
use App\Http\Requests\Frontend\Auth\RegisterRequest;
use App\Helpers\Frontend\Auth\Socialite;
use App\Events\Frontend\Auth\UserRegistered;
use Illuminate\Foundation\Auth\RegistersUsers;
use App\Repositories\Frontend\Auth\UserRepository;
use App\Recipients\AdminRecipient;
use App\Notifications\Frontend\AdminAlert;
use App\Models\Auth\User;
use App\Models\Auth\Lead;
use App\Models\Auth\LeadActivity;
use App\Models\Auth\LeadStatus;
use App\Notifications\Backend\UserAlert;
use App\Models\MailTemplates;
use App\Mail\MailSend;
use Illuminate\Support\Facades\Mail;
use App\Models\Auth\Country;
use App\Models\Auth\UserNotification;
use App\Models\UserActivity;
use Illuminate\Support\Facades\Cookie;
use DB;
use Log;
/**
 * Class RegisterController.
 */
class RegisterController extends Controller
{

    use RegistersUsers;

    /**
     * @var UserRepository
     */
    protected $userRepository;

    /**
     * RegisterController constructor.
     *
     * @param UserRepository $userRepository
     */
    public function __construct(UserRepository $userRepository)
    {
        $this->userRepository = $userRepository;
    }

    /**
     * Where to redirect users after login.
     *
     * @return string
     */
    public function redirectPath()
    {
        return route(home_route());
    }

    /**
     * Show the application registration form.
     *
     * @return \Illuminate\Http\Response
     */
    public function showRegistrationForm()
    {
        abort_unless(config('access.registration'), 404);
        $refered_by_cookie = $refered_by_cookie_name = "";
        $ref_code = Cookie::get('ref_code');

        $refered_by_name = "";
        if($ref_code) {
            Log::debug("STUDENT SIGNUP: Ref cookie found:".$ref_code);
            $ref_user = DB::table('users')->where('refcode', $ref_code)->where('active', 1)->first();
            if($ref_user) {
                $refered_by_cookie = $ref_user->id;
                $refered_by_name = $ref_user->first_name." ".$ref_user->last_name;
                $refered_by_cookie_name = $ref_user->refcode;
                Log::debug("STUDENT SIGNUP: Valid Ref code:".$refered_by_cookie . " (".$refered_by_cookie_name.")");
            }
        } else {
            Log::debug("STUDENT SIGNUP: No Ref cookie");
        }

        $defCountryCode = env('USER_DEFAULT_DIALCODE');
        $def_dialcode = $dialcodes = Country::where('tel', $defCountryCode)->first();

        return view('frontend.auth.register', compact('refered_by_cookie','refered_by_cookie_name','def_dialcode','refered_by_name'))
            ->withSocialiteLinks((new Socialite)->getSocialLinks());
    }

    public function showAgentRegistrationForm()
    {
        abort_unless(config('access.registration'), 404);
        abort_unless(env('AGENTS_ENABLE'), 404);

        $defCountryCode = env('USER_DEFAULT_DIALCODE');
        $def_dialcode = $dialcodes = Country::where('tel', $defCountryCode)->first();

        return view('frontend.auth.agentregister', ['def_dialcode' => $def_dialcode]);
    }


    /**
     * @param RegisterRequest $request
     *
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     * @throws \Throwable
     */
    public function register(RegisterRequest $request){
        abort_unless(config('access.registration'), 404);
        
        \Log::debug("SIGNUP: start");
        //Google Recaptcha Validation
        if(get_site_config('recaptcha_site_key')!="" && get_site_config('recaptcha_secret_key')!="" && get_site_config('student_signup_captcha')==1){
            $request->validate([
                'g-recaptcha-response' => 'required',
            ], [
                'g-recaptcha-response.required' => 'Please complete the reCAPTCHA verification.',
            ]);
        
            $response = \Http::asForm()->post('https://www.google.com/recaptcha/api/siteverify', [
                'secret' => get_site_config('recaptcha_secret_key'),
                'response' => $request->input('g-recaptcha-response'),
            ]);
        
            $responseBody = $response->json();
            if (!$responseBody['success']) {
                $errorCodes = $responseBody['error-codes'] ?? [];
                $errorMessage = 'reCAPTCHA validation failed.';
                if (in_array('timeout-or-duplicate', $errorCodes)) {
                    $errorMessage = 'The reCAPTCHA has timed out or is a duplicate. Please try again.';
                } elseif (in_array('invalid-input-response', $errorCodes)) {
                    $errorMessage = 'The reCAPTCHA response is invalid. Please try again.';
                }
                return back()->withErrors(['g-recaptcha-response' => $errorMessage])->withInput();
            }
        }
        //Google Recaptcha Validation

        $user = $this->userRepository->create($request->only(
            'first_name', 'last_name', 'email', 'full_number', 'dialcode', 'mobile', 'password', 'referral_code', 'whatsapp_number', 'whatsapp_full_number'
        ));
        event(new UserRegistered($user));

        $followup = new UserActivity();
        $followup->user_id = $user->id;
        $followup->type = 'others';
        $followup->staff = $user->id;
        $followup->detail = get_user_full_name_by_id($user->id) . " (LEAD) SIGNUP";
        $followup->save();

        if (config('app.product_code') == 'OVERSEAS') {
            $user->student_type = 'study_abroad';
        } elseif (config('app.product_code') == 'TRAINING') {
            $user->student_type = 'course';
        } else {
            $user->student_type = 'study_abroad_course';
        }

        $user->branch = $request->branch_id;
        $user->save();

        //remove cookie
        setcookie('ref_code', '', time() - 3600, '/');

        //Notify admin
        $msg_details = "A New User Registered.<br /><br />Name:" . $user->full_name . "<br />email:" . $user->email . "<br />Mobile:" . $user->mobile;
        // if ($msg_details) {
        //     $email_message = "Dear Admin,<br><br>" . $msg_details;
        //     $recipient = new AdminRecipient();
        //     try {
        //         $recipient->notify(new AdminAlert("New User Signup", $email_message));
        //     } catch (\Exception $e) {
        //         $message = $e->getMessage();
        //         \Log::debug("Mail Error: (ADMIN MAIL - New User Signup) " . $message);
        //     }
        // }
        $notify_emails = get_site_config('event_notification_email');
        if($notify_emails) {
            send_event_notification_email(explode(",", $notify_emails), "New User Signup", $msg_details);
        } 

        send_mail_notification($user->id,"","STUDENT_SIGNUP","New User: " . $user->full_name." (". $user->id. ")"." Signup",$msg_details,"",$user->id);

        // Send Push App Notification
        $message = "New "._label('backend_student')." Signup. Name:" . $user->full_name." (". $user->id. ")". ", Email: " . $user->email;
        $subject = "";
        send_in_app_notification(
            $user->id, 
            $user->id,
            "STUDENT_SIGNUP", 
            $subject, 
            $message,
            "lead",
            "",
            $user->id
        );


        if (get_site_config('whatsapp_enabled')) {
            $numbers = [$user->whatsapp_dialcode.$user->whatsapp];

            $hook = 'student_welcome';

            send_whatsapp_message($numbers, $hook, $user->id);
        }

        \Log::debug("SIGNUP: check if mail verify required");
        // If the user must confirm their email or their account requires approval,
        // create the account but don't log them in.
        if (config('access.users.confirm_email') || config('access.users.requires_approval')) {

            return redirect(route('frontend.auth.register.complete'))->withFlashSuccess(
                config('access.users.requires_approval') ?
                    __('exceptions.frontend.auth.confirmation.created_pending') :
                    __('exceptions.frontend.auth.confirmation.created_confirm')
            );
        } else {

            //email        
            \Log::debug("SIGNUP: sending welcome mail");
            $mail_tpl = MailTemplates::find("STUDENT_WELCOME");
            if ($mail_tpl && $mail_tpl->status) {
                $email_message = $mail_tpl->message;
                //\Log::debug("MAIL BEFORE:".$email_message);                       
                $email_message = str_ireplace("#sitename#", app_name(), $email_message);
                $email_message = str_ireplace("#siteurl#", url(''), $email_message);
                $email_message = str_ireplace("#first_name#", $user->first_name, $email_message);
                $email_message = str_ireplace("#full_name#", $user->full_name, $email_message);
                $email_message = str_ireplace("#email#", $user->email, $email_message);
                $email_message = str_ireplace("#mobile#", $user->mobile, $email_message);
                //\Log::debug("MAIL TPL FINAL:".$email_message);
                try {

                    //$user->notify(new UserAlert($mail_tpl->subject, $email_message));
                    user_notify($user,$mail_tpl->subject, $email_message);
                } catch (\Exception $e) {
                    $message = $e->getMessage();
                    \Log::debug("Mail Error: (USER SIGNUP)" . $message);
                }
            }
            //sms
            \Log::debug("SIGNUP: sending sms");
            $notify_sms = get_sms_template('STUDENT_WELCOME');
            if ($notify_sms && $notify_sms->status) {
                $notify_sms->message = str_replace("#sitename#", app_name(), $notify_sms->message);
                $notify_sms->message = str_replace("#siteurl#", url('/'), $notify_sms->message);
                $notify_sms->message = str_replace("#first_name#", $user->first_name, $notify_sms->message);
                $notify_sms->message = str_replace("#full_name#", $user->full_name, $notify_sms->message);
                $user->sms($notify_sms->message, $notify_sms->template_id);
            }

            //Whatsapp
            \Log::debug("SIGNUP: sending sms[Whatsapp]");
            /*
            $notify_wtsp = get_whatsapp_template('STUDENT_WELCOME');
            if ($notify_wtsp && $notify_wtsp->status) {
                $notify_wtsp->message = str_replace("#sitename#", app_name(), $notify_wtsp->message);
                $notify_wtsp->message = str_replace("#siteurl#", url('/'), $notify_wtsp->message);
                $notify_wtsp->message = str_replace("#first_name#", $user->first_name, $notify_wtsp->message);
                $notify_wtsp->message = str_replace("#full_name#", $user->full_name, $notify_wtsp->message);
                $user->whatsapp($notify_wtsp->message, $notify_wtsp->hook);
            }*/
            if (get_site_config('whatsapp_enabled')) {
                $numbers = [$user->whatsapp_dialcode.$user->whatsapp];

                $hook = 'student_welcome';

                send_whatsapp_message($numbers, $hook, $user->id);
            }

            \Log::debug("SIGNUP: Auto Login");
            auth()->login($user);

            //event(new UserRegistered($user));
            \Log::debug("SIGNUP: completed. Redirecting");

            //return redirect($this->redirectPath());
            return redirect()->route('frontend.home')->withFlashSuccess('Account created! Please complete your profile.');
        }
    }

    public function registerAgent(RegisterRequest $request)
    {
        abort_unless(config('access.registration'), 404);
        abort_unless(env('AGENTS_ENABLE'), 404);
        
        // dd($request->all());
        \Log::debug("AGENT SIGNUP: start");
        //Google Recaptcha Validation
        if(get_site_config('recaptcha_site_key')!="" && get_site_config('recaptcha_secret_key')!="" && get_site_config('student_signup_captcha')==1){
            $request->validate([
                'g-recaptcha-response' => 'required',
            ], [
                'g-recaptcha-response.required' => 'Please complete the reCAPTCHA verification.',
            ]);
        
            $response = \Http::asForm()->post('https://www.google.com/recaptcha/api/siteverify', [
                'secret' => get_site_config('recaptcha_secret_key'),
                'response' => $request->input('g-recaptcha-response'),
            ]);
        
            $responseBody = $response->json();
            if (!$responseBody['success']) {
                $errorCodes = $responseBody['error-codes'] ?? [];
                $errorMessage = 'reCAPTCHA validation failed.';
                if (in_array('timeout-or-duplicate', $errorCodes)) {
                    $errorMessage = 'The reCAPTCHA has timed out or is a duplicate. Please try again.';
                } elseif (in_array('invalid-input-response', $errorCodes)) {
                    $errorMessage = 'The reCAPTCHA response is invalid. Please try again.';
                }
                return back()->withErrors(['g-recaptcha-response' => $errorMessage])->withInput();
            }
        }
        //Google Recaptcha Validation
        $user = $this->userRepository->createAgent($request->only('first_name', 'last_name', 'email', 'mobile', 'full_number', 'password'));

        //Notify admin
        $msg_details = "A New Agent Registered.<br /><br />Agent Name:" . get_user_full_name_by_id($user->id) . "<br />email:" . $user->email . "<br />Mobile:" . $user->mobile;
        // if ($msg_details) {
        //     $email_message = "Dear Admin,<br><br>" . $msg_details;
        //     $recipient = new AdminRecipient();
        //     try {
        //         $recipient->notify(new AdminAlert("New Agent Signup", $email_message));
        //     } catch (\Exception $e) {
        //         $message = $e->getMessage();
        //         \Log::debug("Mail Error: (ADMIN MAIL - New Agent Signup) " . $message);
        //     }
        // }

        $notify_emails = get_site_config('event_notification_email');
        if($notify_emails) {
            send_event_notification_email(explode(",", $notify_emails), "New Agent Signup", $msg_details);
        } 

        send_mail_notification($user->id,"","AGENT_SIGNUP","New Agent Signup",$msg_details,"",$user->id);

        // Send Push App Notification
        $message = "Agent Signup: " . get_user_full_name_by_id($user->id) . "<br> Email: " . $user->email;
        $subject = "";
        send_in_app_notification(
            $user->id, 
            $user->id,
            "AGENT_SIGNUP", 
            $subject, 
            $message,
            "agent",
            "",
            $user->id
        );

        \Log::debug("AGENT SIGNUP: check if mail verify required");
        // If the user must confirm their email or their account requires approval,
        // create the account but don't log them in.
        if (config('access.users.confirm_email') || config('access.users.requires_approval')) {
            event(new UserRegistered($user));

            return redirect(route('frontend.auth.register.complete'))->withFlashSuccess(
                config('access.users.requires_approval') ?
                    __('exceptions.frontend.auth.confirmation.created_pending') :
                    __('exceptions.frontend.auth.confirmation.created_confirm')
            );
        } else {

            //email        
            // \Log::debug("SIGNUP: sending welcome mail");
            // $mail_tpl = MailTemplates::find("welcome");
            // $email_message = $mail_tpl->message;
            // //\Log::debug("MAIL BEFORE:".$email_message);                       
            // $email_message = str_ireplace("#sitename#", app_name(), $email_message);
            // $email_message = str_ireplace("#siteurl#", url(''), $email_message);
            // $email_message = str_ireplace("#first_name#", $user->first_name, $email_message);
            // $email_message = str_ireplace("#full_name#", $user->full_name, $email_message);
            // $email_message = str_ireplace("#email#", $user->email, $email_message);
            // $email_message = str_ireplace("#mobile#", $user->mobile, $email_message);
            // //\Log::debug("MAIL TPL FINAL:".$email_message);
            // $user->notify(new UserAlert($mail_tpl->subject, $email_message));

            // //sms
            // \Log::debug("SIGNUP: sending sms");
            // $notify_sms = get_sms_template('Welcome');
            // if ($notify_sms) {
            //     $notify_sms->message = str_replace("{#var#}", "%s", $notify_sms->message);
            //     $notify_sms->message = sprintf($notify_sms->message, $user->first_name);
            //     $user->sms($notify_sms->message, $notify_sms->template_id);
            // }

            \Log::debug("AGENT SIGNUP: Auto Login");
            auth()->login($user);

            //event(new UserRegistered($user));
            \Log::debug("AGENT SIGNUP: completed. Redirecting");

            //return redirect($this->redirectPath());
            return redirect()->route('admin.agent.dashboard');
        }
    }


    public function registerComplete()
    {
        abort_unless(config('access.registration'), 404);

        return view('frontend.auth.register-complete');
    }
}
