<?php
/*
 * Project: eductrlcrm-consultancy
 * -----
 * Last Modified: Tuesday, 4th July 2023 10:14:22 am
 * -----
 * Copyright (c) 2020 - 2023, Sash Info Services Private Limited. All rights reserved. This software is the property of Sash Info Services Private Limited and is protected by copyright laws. Unauthorized reproduction, distribution, or modification of this software, in whole or in part, is strictly prohibited.
 */


namespace App\Http\Controllers;

use App\Http\Controllers\Controller;

use Illuminate\Support\Facades\DB;
use App\Models\Auth\Posts;
use App\Models\Auth\Course;
use App\Models\Auth\OnlineCourse;
use App\Models\Auth\CountryInfo;
use App\Models\Auth\User;
use App\Models\Auth\Batch;
use App\Models\Banners;
use Illuminate\Http\Request;
use App\Helpers\General\Time;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Validator;

/**
 * Class HomeController.
 */
class HomeController extends Controller
{
    /**
     * @return \Illuminate\View\View
     */

    public function __construct()
    {
        $this->menu_details = array();
        //$main_menus = Posts::Select('id','title','menu_title','slug')->where('status', '=', 1)->where('show_in_main_menu', '=', 1)->where('module_type', '!=', 'home')->orderBy('order')->get();

        if(config('app.product_code') == 'TRAINING') {
            $main_menus = Posts::Select('id', 'title', 'menu_title', 'slug')
            ->where('status', '=', 1)
            ->where('show_in_main_menu', '=', 1)
            ->where(function ($query) {
                $query->Where('module_type', '<>', "home")
                    ->Where('module_type', '<>', "country")
                    ->orWhere('module_type', NULL);
            })
            ->orderBy('order')
            ->get();
        } else if(config('app.product_code') == 'OVERSEAS') {
            $main_menus = Posts::Select('id', 'title', 'menu_title', 'slug')
            ->where('status', '=', 1)
            ->where('show_in_main_menu', '=', 1)
            ->where(function ($query) {
                $query->Where('module_type', '<>', "home")
                    ->Where('module_type', '<>', "coaching")
                    ->Where('module_type', '<>', "courses")
                    ->orWhere('module_type', NULL);
            })
            ->orderBy('order')
            ->get();

        } else {
            $main_menus = Posts::Select('id', 'title', 'menu_title', 'slug')
            ->where('status', '=', 1)
            ->where('show_in_main_menu', '=', 1)
            ->where(function ($query) {
                $query->Where('module_type', '<>', "home")
                    ->orWhere('module_type', NULL);
            })
            ->orderBy('order')
            ->get();
        }

        foreach ($main_menus as $main_menu) {
            $menu_title = $main_menu['menu_title'] != '' ? $main_menu['menu_title'] : $main_menu['title'];
            $this->menu_details[$menu_title] =  route('frontend.view-article', [$main_menu['slug']]);
        }
    }

    public function web_lead_error()
    {
        return view("frontend.web-lead-error-page");
    }


    public function index()
    {
        \Log::debug("index");
        $article_det = Posts::Where('module_type', '=', 'home')->first();
        $coaching_list = Course::OrderBy('display_order')->limit(4)->get();
        $course_list = OnlineCourse::OrderBy('display_order')->limit(4)->get();
        $country_list = CountryInfo::Where('parent', '=', 0)->orderBy('name')->limit(4)->get();

        $coaching_intro = $coaching_slug = $course_intro = $course_slug = $country_intro = $country_slug = "";
        $res = Posts::Where('module_type', '=', 'coaching')->first();
        if($res) {
        $coaching_intro = $res->intro;
        $coaching_slug = $res->slug;
        }

        $res = Posts::Where('module_type', '=', 'courses')->first();
        if($res) {        
        $course_intro = $res->intro;
        $course_slug = $res->slug;
        }

        $res = Posts::Where('module_type', '=', 'country')->first();
        if($res) {                
        $country_intro = $res->intro;
        $country_slug = $res->slug;
        }
        
        $banners = Banners::orderBy('display_order')->get();
        //$content_block_1 = get_content_block('Home page block 1');

        return view('frontend.index', [
            'cms_page' => 1, 
            'main_menu_arr' => $this->menu_details, 
            'article_det' => $article_det, 
            'coaching_classes' => $coaching_list, 
            'courses' => $course_list, 
            'countries' => $country_list, 
            'coaching_intro' => $coaching_intro,
            'course_intro' => $course_intro, 
            'country_intro' => $country_intro, 
            'banners' => $banners,
            'coaching_slug' => $coaching_slug,
            'course_slug' => $course_slug,
            'country_slug' => $country_slug,
            //'content_block_1' => $content_block_1,
        ]);
    }


    public function viewArticle($slug)
    {
        $list_det = array();
        $cms_page = "1";
        $main_menu_arr = $this->menu_details;
        $article_det = Posts::Where('slug', '=', $slug)->where('status', '=', 1)->first();

        if (!isset($article_det)) {
            return redirect()->route('index');
        }

        if ($article_det->module_type == 'coaching') {
            $list_det = Course::OrderBy('display_order')->get();
        } elseif ($article_det->module_type == 'courses') {
            $list_det = OnlineCourse::OrderBy('display_order')->get();
        } else if ($article_det->module_type == 'country') {
            $list_det = CountryInfo::Select(['id', 'name as title','image'])->where('parent', '=', 0)->orderBy('name')->get();
        }

        return view("frontend.article", compact("article_det", "cms_page", "main_menu_arr", "list_det"));
    }

    public function viewCourseDetail($course)
    {
        $course = Course::findOrFail($course);
        return view("frontend.course-detail", ['cms_page' => 1, 'main_menu_arr' => $this->menu_details, 'course' => $course]);
    }

    public function viewOnlineCourseDetail($course)
    {
        //$course = OnlineCourse::findOrFail($course);
        $onlineCourseObj = OnlineCourse::getCourseWithChaptersObjById($course);
        $chaptersCount = (!empty($onlineCourseObj->chapters)) ? $onlineCourseObj->chapters->count() : 0;
        $totalDurationInHour = "00:00:00";
        $chapters = array();
        if ($chaptersCount > 0) {
            $totalMinutes = 0;
            foreach ($onlineCourseObj->chapters as $key=>$chapter) {
                $totalMinutes += floor($chapter->duration);
                $chapters[$key]['id'] = $chapter->id;
                $chapters[$key]['title'] = $chapter->title;
                $chapters[$key]['duration'] = ($chapter->duration > 0) ? Time::secToHR(floor($chapter->duration) * 60) : "00:00:00";
            }
            $totalDurationInHour = ($totalMinutes > 0) ? Time::secToHR($totalMinutes * 60) : "00:00:00";
        }
        return view("frontend.online-course-detail", ['cms_page' => 1, 'main_menu_arr' => $this->menu_details, 'onlineCourseObj' => $onlineCourseObj, 'chaptersCount' => $chaptersCount,'totalDurationInHour'=>$totalDurationInHour,'chapters'=>$chapters,'courseId'=>$course]);
    }

    public function viewCountryDetail($folderid = 0)
    {

        $access_denied = false;
        $folders = array();
        $rows = DB::table('country_info_folder')->where('parent', $folderid)->get();
        if ($rows) {
            foreach ($rows as $row) {
                $folders[] = array("id" => $row->id, "name" => $row->name, "sharing" => $row->sharing, "image" => $row->image);
            }
        }

        $cur_folder = $folderid;
        $folder_breadcrumb = array();
        $i = 0;
        while (true) {
            if ($cur_folder == 0) {
                $folder_breadcrumb[$i++] = array('id' => $cur_folder, 'name' => 'Documents');
                break;
            } else {
                $row = DB::table('country_info_folder')->where('id', $cur_folder)->first();
                $folder_breadcrumb[$i++] = array('id' => $row->id, 'name' => $row->name);
                $cur_folder = $row->parent;
            }
        }

        //docs
        $docs = array();
        if ($folderid > 0) {

            //check permissions
            $allow_access = false;
            $folderinfo = DB::table('country_info_folder')->where('id', $folderid)->first();
            if ($folderinfo->sharing == 'public') {
                $allow_access = true;
            } else {
                $user = auth()->user();
                if(!$user) {
                    return redirect()->route('frontend.auth.login');
                }
                if ($folderinfo->sharing == 'member') {
                    if (DB::table('user_academics')->where('userid', $user->id)->exists()) {
                        $allow_access = true;
                    }
                } elseif ($folderinfo->sharing == 'private') {
                    if (DB::table('user_folder_permission')->where(['userid' => $user->id, 'folderid' => $folderid])->exists()) {
                        $allow_access = true;
                    }
                }
            }

            if ($allow_access) {
                $rows = DB::table('country_info_doc')->where('folder', $folderid)->get();
                if ($rows) {
                    foreach ($rows as $row) {
                        $description = $row->description;
                        $filepath = url('storage/' . $row->file);
                        $docs[] = array("id" => $row->id, "title" => $row->title, "file" => $filepath, "description" => $description);
                    }
                }
            } else {
                $access_denied = true;
            }
        }
        return view('frontend.country-detail', ['cms_page' => 1, 'main_menu_arr' => $this->menu_details,'folders' => $folders, 'docs' => $docs, 'folder_breadcrumb' => $folder_breadcrumb, 'current_folder' => $folderid, 'access_denied' => $access_denied]);
    }


    public function searchUser(Request $request) {
        //DB::enableQueryLog();
        $users = User::where('first_name', 'LIKE', $request->input('q', '').'%')
        ->orWhere('email', '=', $request->input('q', ''))
        ->when(\Auth::user()->isAgent(), function ($q) {
            $q->where('agent_id', \Auth::user()->id);
        })
        ->select('id', DB::raw("concat(COALESCE(`first_name`,''), ' ', COALESCE(`last_name`,''), ' | ', email) as text"))
        ->get();
        //$queries = DB::getQueryLog();
        //\Log::debug(end($queries));        
        return ['results' => $users];        
    }    

    public function searchStudent(Request $request) {
        $users = User::with('roles')
        ->active()
        ->whereHas('roles', function ($q) {
            $q->whereIn('name', ['user']);
        })
        ->when(\Auth::user()->isAgent(), function ($q) {
            $q->where('agent_id', \Auth::user()->id);
        })
        ->where('first_name', 'LIKE', $request->input('q', '').'%')
        ->orWhere('email', '=', $request->input('q', ''))
        ->select('id', DB::raw("concat(COALESCE(`first_name`,''), ' ', COALESCE(`last_name`,''), ' | ', email) as text"))
        ->get();
        return ['results' => $users];        
    } 

    public function searchCourse(Request $request){       
        $q =  $request->input('q', '');
        $s_level =  $request->input('s_level');
        //$degree = get_study_level_degree_level($s_level);

        //we need these, as in course data we have different naming
        // if($degree == "Pre-bachelors"){
        //     $s_level = "Pre-bachelors";
        // }
        // elseif($degree == "Bachelors"){
        //     $s_level = "Undergraduate";
        // }
        // elseif($degree == "Masters"){
        //     $s_level = "Postgraduate";
        // }
        // elseif($degree == "Ph.D"){
        //     $s_level = "PhD";
        // }
        // else{
        //     $s_level = "Others";
        // }
        // Query debug
        // DB::enableQueryLog();

        $course = DB::table('university_courses')
            ->join('universities', 'university_courses.university', '=', 'universities.id')
            ->where(function ($query) use ($q,$s_level) {
                $query->where('course_name', 'LIKE','%'. $q . '%')
                    ->where('study_level', $s_level);
            })
            // ->select('id', DB::raw("course_name as text"))
            ->select('university_courses.id', DB::raw("CONCAT(university_courses.course_name, ' | ', universities.university_name) as text"))
            // ->groupBy('text')
            ->get();

        // $queries = DB::getQueryLog();
        // \Log::debug(end($queries));
        // \Log::debug("Data:- ".$course);
    
        return ['results' => $course,'s_level'=>$s_level];
    }

    public function searchUniversity(Request $request)
    {
        
        $q = $request->input('q', '');
        $university = DB::table('universities')->where(function ($query) use ($q) {
                $query->where('university_name', 'LIKE', $q . '%');
                //->orwhere('last_name', 'LIKE', $q . '%')
            })
            ->select('id', DB::raw("university_name as text"))
            ->get();
        return ['results' => $university];
    }


    //For admin\orders\create order

    public function search_student(Request $request)
    {
        $my_branch = get_staff_branch_ids(\Auth::user()->id);
        $q =  $request->input('q', '');
        $users = User::where(function ($query) use ($q) {
                $query->where('first_name', 'LIKE', $q . '%')
                ->orwhere('last_name', 'LIKE', $q . '%')
                ->orWhere('email', '=', $q)
                ->orWhere('mobile', '=', $q);
            })
            ->whereIn('branch', $my_branch)
            // ->select('id', DB::raw("concat(first_name, ' ', last_name, ' | ', email) as text"))
            ->select('id', DB::raw("concat(IF($q IS NOT NULL, CONCAT(first_name), ''), ' ', IF(last_name IS NOT NULL, CONCAT(last_name), ''), ' | ', email) as text"))
            ->get();
        return ['results' => $users,'branch' => $my_branch];
    }
    public function searchStudentData(Request $request)
    {
        $users = User::where('id', $request->id)
            ->first();
        $batch = Batch::join('batch_students', 'batch.id', '=', 'batch_students.batch')
            ->where('student', $request->id)
            ->get(['batch.id', 'batch.code']);
        return ['results' => $users, 'batch' => $batch];
    }

    public function paymentCancel(Request $request)
    {
         return redirect()->route("frontend.home")->withFlashDanger("Oops! Payment Cancelled.");    
    }    

    public function paymentSuccess(Request $request)
    {
         return redirect()->route("frontend.home")->withFlashSuccess("Thank You! Order created. Please check 'My Orders' section for the details.");    
    }
    public function invoicePaymentSuccess(Request $request)
    {
         return redirect()->route("frontend.home")->withFlashSuccess("Thank You! Payment Received.");    
    }
    public function invoicepaymentCancel(Request $request)
    {
        if(\Auth::user()->isAgent()){
            return redirect()->route("admin.agent.dashboard")->withFlashDanger("Oops! Payment Cancelled.");
        }else{
         return redirect()->route("frontend.home")->withFlashDanger("Oops! Payment Cancelled.");
        }
    }
    
    public function getVersionInfo() {
        echo "EduCtrl v".config('version.major').".".config('version.minor').".".config('version.date');
    }


    public function validate_email_mobile(Request $request){
        if(get_site_config('lead_create_admin_mandatory') == 'email'){
            $validator = Validator::make($request->all(), [
                'email' => 'required|email:filter,filter_unicode|unique:users',
                'mobile' => ['nullable', 'numeric', 'digits_between:'.config('app.USER_MOBILE_MIN_LENGTH').','.config('app.USER_MOBILE_MAX_LENGTH')],
            ]);
        } elseif(get_site_config('lead_create_admin_mandatory') == 'email_and_mobile'){
            $validator = Validator::make($request->all(), [
                'email' => 'required|email:filter,filter_unicode|unique:users',
                'mobile' => ['required', 'numeric', 'digits_between:'.config('app.USER_MOBILE_MIN_LENGTH').','.config('app.USER_MOBILE_MAX_LENGTH'), Rule::unique('users')],
            ]);
        } elseif(get_site_config('lead_create_admin_mandatory') == 'email_or_mobile'){
            $validator = Validator::make($request->all(), [
                'email' => 'nullable|email:filter,filter_unicode|unique:users|required_without:mobile',
                'mobile' => ['nullable', 'numeric', 'required_without:email', 'digits_between:'.config('app.USER_MOBILE_MIN_LENGTH').','.config('app.USER_MOBILE_MAX_LENGTH'), Rule::unique('users')],
            ]);
        } elseif(get_site_config('lead_create_admin_mandatory') == 'mobile'){
            $validator = Validator::make($request->all(), [
                'email' => 'nullable|email:filter,filter_unicode|unique:users',
                'mobile' => ['required', 'numeric', 'digits_between:'.config('app.USER_MOBILE_MIN_LENGTH').','.config('app.USER_MOBILE_MAX_LENGTH'), Rule::unique('users')],
            ]);
        }

        if ($validator->fails()) {
            $data = [];
            $messages = $validator->errors();
            if ($messages->first('email')) {
                $data['email_err'] = $messages->first('email');
            }
            if ($messages->first('mobile')) {
                $data['mobile_err'] = $messages->first('mobile');
            }
            return json_encode($data);
        }
        return json_encode("Valid Email & Mobile");
    }

    public function generate_student_ref_codes() {
        $results = User::whereNull('refcode')->get(['id', 'first_name','last_name', 'refcode']);
        foreach ($results as $row) {
            echo "<pre>";
            $refcode = generate_refcode($row->first_name);
            echo "New Refcode: " . $refcode . "<br>";
            DB::table('users')->where('id', $row->id)->update(['refcode' => $refcode]);
        }
    }

    function get_ref_users_list(Request $request) {
        $z =  $request->input('q', '');
        $exclude_user = $request->input('exclude_user', '');
        $users = User::with('roles')
        ->active()
        ->where('is_lead',0)
        ->whereHas('roles', function ($q) {
            $q->whereIn('name', ['user']);
        })
        ->where(function ($query) use ($z) {
            $query->where('first_name', 'LIKE', $z . '%')
            ->orwhere('last_name', 'LIKE', $z . '%')
            ->orWhere('id', '=', $z)
            ->orWhere('mobile', '=', $z);
        })
        ->when(!empty($exclude_user), function ($query) use ($exclude_user) {
            return $query->where('id',"!=",$exclude_user);
        })
        ->select('id', DB::raw("concat(COALESCE(`first_name`,''), ' ', COALESCE(`last_name`,''), ' (', id, ') ') as text"))
        ->get();
        return ['results' => $users];     
    }

    public function showPublicWebinar($webinar_id){
        $webinar = DB::table('webinar')
            ->select('id', 'topic', 'date', 'venue', 'link', 'image', 'target_user_type', 'description', 'public_slug')
            ->where('public_slug', $webinar_id)
            ->first();

        if (!$webinar) {
            return redirect()->route('index')->withFlashDanger("Webinar not found.");
        }

        return view('frontend.webinar.public_webinar', [
            'cms_page' => 1,
            'webinar' => $webinar,
            'main_menu_arr' => $this->menu_details,
        ]);
    }

}
