<?php

namespace App\Models\Auth;

use App\Http\Controllers\Backend\OnlineCoursesController;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;

/**
 * Class Course.
 */
class OnlineCourse extends Model
{

    /**
     * The database table used by the model.
     *
     * @var string
     */
    protected $table = 'online_course';
    protected $primaryKey = "id";
    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = ['title', 'short_desc', 'type', 'display_order'];

    public function scopeActive($query)
    {
        return $query->where('status', '=', 1)->orderBy('id','asc');
    }

    public static function getCourseObjById($id){
        if($id > 0){
            return OnlineCourse::where('id','=',$id)->first();
        }
        return '';
    }
    
    public static function addOrUpdateOnlineCourse($data, $courseId = 0)
    {
        if (!empty($data)) {
            if ($courseId > 0) {
                $newObj = OnlineCourse::where('id', '=', $courseId)->first();
            } else {
                $newObj = new OnlineCourse();
            }
            $newObj->title = $data['title'];
            $newObj->short_desc = $data['short_description'];
            $newObj->description = $data['description'];
            if (isset($data['created_by'])) {
                $newObj->created_by = $data['created_by'];
            }
            $newObj->status = 1;
            if (isset($data['status'])) {
                $newObj->status = $data['status'];
            }
            if (isset($data['image']) && (!empty($data['image']))) {
                $newObj->image = $data['image'];
            }
            $newObj->display_order = $data['display_order'];
            $newObj->price = $data['price'];
            $newObj->offer_price = $data['offer_price'];
            $newObj->save();
            return $newObj->id;
        }
        return 0;
    }

    public function chapters(){

        return $this->hasMany('\App\Models\Auth\Chapters', 'online_course_id', 'id')->displayOrder();
    }

    public static function getCourseWithChaptersObjById($id){
        if($id > 0){
            return OnlineCourse::with(['chapters'=>function($query){
                $query->displayOrder();
            }])->where('id','=',$id)->first();
        }
        return '';
    }

    public static function getCourseWithChaptersObjByIds($courseIds){
        if(!empty($courseIds)){
            return OnlineCourse::with(['chapters'=>function($query){
                $query->active();
            }])->whereIn('id',$courseIds)->get();
        }
        return '';
    }

    public static function getCourseObjByIds($courseIds){
        if(!empty($courseIds)){
            return OnlineCourse::whereIn('id',$courseIds)->get();
        }
        return '';
    }

    public static function uploadCourseImage($fileObj){
        $file = $fileObj->getClientOriginalName();
        $filename = pathinfo($file, PATHINFO_FILENAME);
        if (strlen($filename) > 125){
            $filename = substr($filename, 0, 125);
        }
        $filename = $filename . Str::random(20) . "." . $fileObj->getClientOriginalExtension();
        return $fileObj->storeAs('/images', $filename, 'public');
    }

    public static function getAllOnlineCourses()
    {
        return OnlineCourse::active()->orderBy('display_order', 'asc')->get();
    }
    
}
