<?php

namespace App\Models\Auth\Traits\Attribute;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;

/**
 * Trait UserAttribute.
 */
trait StaffAttribute {

    /**
     * @param $password
     */
    public function setPasswordAttribute($password): void {
        // If password was accidentally passed in already hashed, try not to double hash it
        if (
                (\strlen($password) === 60 && preg_match('/^\$2y\$/', $password)) ||
                (\strlen($password) === 95 && preg_match('/^\$argon2i\$/', $password))
        ) {
            $hash = $password;
        } else {
            $hash = Hash::make($password);
        }

        // Note: Password Histories are logged from the \App\Observer\User\UserObserver class
        $this->attributes['password'] = $hash;
    }

    /**
     * @return string
     */
    public function getStatusLabelAttribute() {
        if ($this->isActive()) {
            return "<span class='badge badge-success'>" . __('labels.general.active') . '</span>';
        }

        return "<span class='badge badge-danger'>" . __('labels.general.inactive') . '</span>';
    }

    /**
     * @return string
     */
    public function getConfirmedLabelAttribute() {
        if ($this->isConfirmed()) {
            if ($this->id != 1 && $this->id != auth()->id()) {
                return '<a href="' . route(
                                'admin.auth.user.unconfirm', $this
                        ) . '" data-toggle="tooltip" data-placement="top" title="' . __('buttons.backend.access.users.unconfirm') . '" name="confirm_item"><span class="badge badge-success" style="cursor:pointer">' . __('labels.general.yes') . '</span></a>';
            } else {
                return '<span class="badge badge-success">' . __('labels.general.yes') . '</span>';
            }
        }

        return '<a href="' . route('admin.auth.user.confirm', $this) . '" data-toggle="tooltip" data-placement="top" title="' . __('buttons.backend.access.users.confirm') . '" name="confirm_item"><span class="badge badge-danger" style="cursor:pointer">' . __('labels.general.no') . '</span></a>';
    }

    /**
     * @return string
     */
    public function getRolesLabelAttribute() {
        $roles = $this->getRoleNames()->toArray();

        if (\count($roles)) {
            return implode(', ', array_map(function ($item) {
                        return ucwords($item);
                    }, $roles));
        }

        return 'N/A';
    }

    /**
     * @return string
     */
    public function getPermissionsLabelAttribute() {
        $permissions = $this->getDirectPermissions()->toArray();

        if (\count($permissions)) {
            return implode(', ', array_map(function ($item) {
                        return ucwords($item['name']);
                    }, $permissions));
        }

        return 'N/A';
    }

    /**
     * @return string
     */
    public function getFullNameAttribute() {
        return $this->last_name ? $this->first_name . ' ' . $this->last_name : $this->first_name;
    }

    /**
     * @return string
     */
    public function getNameAttribute() {
        return $this->full_name;
    }

    /**
     * @return mixed
     */
    public function getPictureAttribute() {
        return $this->getPicture();
    }

    /**
     * @return string
     */
    public function getSocialButtonsAttribute() {
        $accounts = [];

        foreach ($this->providers as $social) {
            $accounts[] = '<a href="' . route(
                            'admin.auth.user.social.unlink', [$this, $social]
                    ) . '" data-toggle="tooltip" data-placement="top" title="' . __('buttons.backend.access.users.unlink') . '" data-method="delete"><i class="fab fa-' . $social->provider . '"></i></a>';
        }

        return \count($accounts) ? implode(' ', $accounts) : __('labels.general.none');
    }

    /**
     * @return string
     */
    public function getLoginAsButtonAttribute() {
        /*
         * If the admin is currently NOT spoofing a user
         */
        if (!session()->has('admin_user_id') || !session()->has('temp_user_id')) {
            //Won't break, but don't let them "Login As" themselves
            if ($this->id != auth()->id()) {
                return '<a href="' . route(
                                'admin.auth.user.login-as', $this
                        ) . '" class="dropdown-item">' . __('buttons.backend.access.users.login_as', ['user' => e($this->full_name)]) . '</a> ';
            }
        }

        return '';
    }

    /**
     * @return string
     */
    public function getClearSessionButtonAttribute() {
        if ($this->id != auth()->id() && config('session.driver') == 'database') {
            return '<a href="' . route('admin.auth.user.clear-session', $this) . '"
			 	 data-trans-button-cancel="' . __('buttons.general.cancel') . '"
                 data-trans-button-confirm="' . __('buttons.general.continue') . '"
                 data-trans-title="' . __('strings.backend.general.are_you_sure') . '"
                 class="dropdown-item" name="confirm_item">' . __('buttons.backend.access.users.clear_session') . '</a> ';
        }

        return '';
    }

    /**
     * @return string
     */
    public function getShowButtonAttribute() {
        return '<a href="' . route('admin.auth.user.show', $this) . '" data-toggle="tooltip" data-placement="top" title="' . __('buttons.general.crud.view') . '" class="btn btn-info"><i class="ri-eye-fill"></i></a>';
    }

    /**
     * @return string
     */
    public function getEditButtonAttribute() {
        return '<a href="' . route('admin.auth.user.edit', $this) . '" class="dropdown-item">Edit</a>';
    }

    /**
     * @return string
     */
    public function getChangePasswordButtonAttribute() {
        return '<a href="' . route('admin.auth.user.change-password', $this) . '" class="dropdown-item">' . __('buttons.backend.access.users.change_password') . '</a> ';
    }

    /**
     * @return string
     */
    public function getStatusButtonAttribute() {
        if ($this->id != auth()->id()) {
            switch ($this->active) {
                case 0:
                    return '<a href="' . route('admin.auth.user.mark', [
                                $this,
                                1,
                            ]) . '" class="dropdown-item">' . __('buttons.backend.access.users.activate') . '</a> ';

                case 1:
                    return '<a href="' . route('admin.auth.user.mark', [
                                $this,
                                0,
                            ]) . '" class="dropdown-item">' . __('buttons.backend.access.users.deactivate') . '</a> ';

                default:
                    return '';
            }
        }

        return '';
    }

    /**
     * @return string
     */
    public function getConfirmedButtonAttribute() {
        if (!$this->isConfirmed() && !config('access.users.requires_approval')) {
            return '<a href="' . route('admin.auth.user.account.confirm.resend', $this) . '" class="dropdown-item">' . __('buttons.backend.access.users.resend_email') . '</a> ';
        }

        return '';
    }

    /**
     * @return string
     */
    public function getDeleteButtonAttribute() {
        if ($this->id != auth()->id() && $this->id != 1) {
            return '<a href="' . route('admin.auth.user.destroy', $this) . '"
                 data-method="delete"
                 data-trans-button-cancel="' . __('buttons.general.cancel') . '"
                 data-trans-button-confirm="' . __('buttons.general.crud.delete') . '"
                 data-trans-title="' . __('strings.backend.general.are_you_sure') . '"
                 class="dropdown-item">' . __('buttons.general.crud.delete') . '</a> ';
        }

        return '';
    }

    /**
     * @return string
     */
    public function getDeletePermanentlyButtonAttribute() {
        return '<a href="' . route('admin.auth.user.delete-permanently', $this) . '" name="confirm_item" class="btn btn-danger"><i class="bx bxs-trash" data-toggle="tooltip" data-placement="top" title="' . __('buttons.backend.access.users.delete_permanently') . '"></i></a> ';
    }

    /**
     * @return string
     */
    public function getRestoreButtonAttribute() {
        return '<a href="' . route('admin.auth.user.restore', $this) . '" name="confirm_item" class="btn btn-info"><i class="bx bx-sync" data-toggle="tooltip" data-placement="top" title="' . __('buttons.backend.access.users.restore_user') . '"></i></a> ';
    }

    /**
     * @return string
     */
    public function getActionButtonsAttribute() {
        if ($this->trashed()) {
            return '
				<div class="btn-group" role="group" aria-label="' . __('labels.backend.access.users.user_actions') . '">
				  ' . $this->restore_button . '
				  ' . $this->delete_permanently_button . '
				</div>';
        }

        return '
    	<div class="btn-group" role="group" aria-label="' . __('labels.backend.access.users.user_actions') . '">
		  ' . $this->show_button . '
		  ' . $this->email_button . '
		  ' . $this->sms_button . '        
		  ' . $this->whatsapp_button . '                              
		  <div class="btn-group btn-group-sm" role="group">
			<button id="userActions" type="button" class="btn btn-secondary dropdown-toggle" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false">
			  ' . __('labels.general.more') . '
			</button>
			<div class="dropdown-menu" aria-labelledby="userActions">
			  ' . $this->clear_session_button . '
                          ' . $this->edit_button . '                      
                          ' . $this->demo_class_permission_button . '        
                          ' . $this->countryinfo_permission_button . '
			  ' . $this->login_as_button . '
			  ' . $this->change_password_button . '
			  ' . $this->status_button . '
			  ' . $this->confirmed_button . '
			  ' . $this->delete_button . '
			</div>
		  </div>
		</div>';
    }

    public function getTestLabelAttribute() {
        return "<span class='badge badge-success'>TEST</span>";
    }

    public function getTagsLabelAttribute() {
        $ret = "";
        $rows = DB::table('user_tags')->where('userid', $this->id)->get();
        if ($rows) {
            foreach ($rows as $row) {
                $ret .= "<span class='badge badge-success'>" . $row->tag . "</span> ";
            }
        }
        return $ret;
    }

    public function getAcademicsAttribute() {
        $rows = DB::table('user_academics')->where('userid', $this->id)->get();
        return $rows;
    }

    public function getEnglishcertAttribute() {
        $rows = DB::table('user_certification_eng')->where('userid', $this->id)->get();
        return $rows;
    }

    public function getLanguagecertAttribute() {
        $rows = DB::table('user_certification_lang')->where('userid', $this->id)->get();
        return $rows;
    }

    public function getGreGmatAttribute() {
        $rows = DB::table('user_gre_gmat')->where('userid', $this->id)->get();
        return $rows;
    }

    public function getAdditionalcoursesAttribute() {
        $rows = DB::table('user_courses')->where('userid', $this->id)->get();
        return $rows;
    }

    public function getExperiencesAttribute() {
        $rows = DB::table('user_experience')->where('userid', $this->id)->get();
        return $rows;
    }

    public function getCoursepreferencesAttribute() {
        $ret = array();
        $rows = DB::table('course_preference')->where('userid', $this->id)->get();
        if ($rows) {
            foreach ($rows as $row) {
                $intake_str = "";
                $intake = $row->intake;
                if ($intake) {
                    $month_nums = explode(",", $intake);
                    if (count($month_nums)) {
                        foreach ($month_nums as $month) {
                            if (is_numeric($month)) {
                                $str = date("Y") . "-" . $month . "-01";
                                $intake_str .= date("M", strtotime($str)) . ",";
                            }
                        }
                    }
                }
                array_push($ret, array('country' => $row->country, 'course' => $row->course, 'university' => $row->university, 'intake' => $intake_str));
            }
        }
        return $ret;
    }

    public function getCourseSuggestionHistoryAttribute() {
        $ret = array();
        $rows = DB::table('course_recommendation')->where('userid', $this->id)->orderBy('upload_date', 'desc')->get();
        if ($rows) {
            foreach ($rows as $row) {
                $title = $row->title;
                $filename = $row->file;
                $ext = pathinfo($filename, PATHINFO_EXTENSION);
                $filepath = url('storage/' . $filename);
                $file_link = "<a href='" . $filepath . "'>$title (.$ext)</a>";
                $ret[] = array('date' => $row->upload_date, 'file' => $file_link, 'satisfied' => ($row->satisfied == 'none') ? '-' : ucfirst($row->satisfied));
            }
        }
        return $ret;
    }

    public function getApplicationsAttribute() {
        $ret = array();
        $rows = DB::table('university_application')->where('userid', $this->id)->orderBy('id', 'asc')->get();
        if ($rows) {
            foreach ($rows as $row) {
                $intake_str = "";
                $intake = $row->intake;
                if ($intake) {
                    $str = date("Y") . "-" . $intake . "-01";
                    $intake_str = date("M", strtotime($str));
                }
                $status = "";
                if ($row->offer_letter_ts) {
                    $status = "Offer Letter Given on " . $row->offer_letter_ts;
                } elseif ($row->result != 'none') {
                    $status = "Application Result: " . ucfirst($row->result);
                } elseif ($row->application_submitted == 'yes') {
                    $status = "Application Submitted on: " . $row->application_submitted_ts;
                } elseif ($row->application_fee_paid != 'none') {
                    $status = "Application Fees Paid: " . ufirst($row->application_fee_paid);
                    if ($row->application_fee_paid == 'yes')
                        $status .= " On " . $row->application_fee_paid_ts;
                } elseif ($row->application_mode != 'none') {
                    $status = "Application Mode: " . ucfirst($row->application_mode);
                } elseif ($row->admission_criteria_met != 'none') {
                    $status = "Admission Criteria: " . ucfirst($row->admission_criteria_met);
                } elseif ($row->doc_ready != 'none') {
                    $status = "Doc ready: " . ucfirst($row->doc_ready);
                } else {
                    $status = "None";
                }

                array_push($ret, array('id' => $row->id, 'country' => $row->country, 'course' => $row->course, 'university' => $row->university, 'year' => $row->year, 'intake' => $intake_str, 'open_date' => $row->application_open_date, 'close_date' => $row->application_close_date, 'status' => $status));
            }
        }
        return $ret;
    }

    public function getCountryinfoPermissionButtonAttribute() {
        return '<a href="' . route('admin.auth.user.countrypermission', $this->id) . '" class="dropdown-item" target="_blank">Country Permission</a>';
    }

    public function getDemoClassPermissionButtonAttribute() {
        return '<a href="' . route('admin.auth.user.democlasspermission', $this->id) . '" class="dropdown-item" target="_blank">Demo Class Permission</a>';
    }
    
    public function getEducationDocsAttribute() {
        $ret = array();
        $row = DB::table('user_documents')->where('userid', $this->id)->first();
        if ($row) {
            if ($row->class10) {
                $ret["10th"] = "<a target=\"_blank\" href='" . url('storage/' . $row->class10) . "'>" . substr($row->class10, 5) . "</a>";
            }
            if ($row->class12) {
                $ret["12th"] = "<a target=\"_blank\" href='" . url('storage/' . $row->class12) . "'>" . substr($row->class12, 5) . "</a>";
            }
            if ($row->diploma) {
                $ret["Diploma"] = "<a target=\"_blank\" href='" . url('storage/' . $row->diploma) . "'>" . substr($row->diploma, 5) . "</a>";
            }
            if ($row->grad_all_sem) {
                $ret["Bachelors All Sem"] = "<a target=\"_blank\" href='" . url('storage/' . $row->grad_all_sem) . "'>" . substr($row->grad_all_sem, 5) . "</a>";
            }
            if ($row->grad_degree) {
                $ret["Bachelors Degree"] = "<a target=\"_blank\" href='" . url('storage/' . $row->grad_degree) . "'>" . substr($row->grad_degree, 5) . "</a>";
            }
            if ($row->grad_transcripts) {
                $ret["Graducation Transcripts"] = "<a target=\"_blank\" href='" . url('storage/' . $row->grad_transcripts) . "'>" . substr($row->grad_transcripts, 5) . "</a>";
            }
            if ($row->pg_all_sem) {
                $ret["Masters All Sem"] = "<a target=\"_blank\" href='" . url('storage/' . $row->pg_all_sem) . "'>" . substr($row->pg_all_sem, 5) . "</a>";
            }
            if ($row->pg_degree) {
                $ret["Masters Degree"] = "<a target=\"_blank\" href='" . url('storage/' . $row->pg_degree) . "'>" . substr($row->pg_degree, 5) . "</a>";
            }
            if ($row->pg_transcripts) {
                $ret["Masters Transcripts"] = "<a target=\"_blank\" href='" . url('storage/' . $row->pg_transcripts) . "'>" . substr($row->pg_transcripts, 5) . "</a>";
            }
        }
        return $ret;
    }

    public function getLangDocsAttribute() {
        $ret = array();
        $row = DB::table('user_documents')->where('userid', $this->id)->first();
        if ($row) {
            if ($row->lang_ielts) {
                $ret["IELTS"] = "<a target=\"_blank\" href='" . url('storage/' . $row->lang_ielts) . "'>" . substr($row->lang_ielts, 5) . "</a>";
            }
            if ($row->lang_pte) {
                $ret["PTE"] = "<a target=\"_blank\" href='" . url('storage/' . $row->lang_pte) . "'>" . substr($row->lang_pte, 5) . "</a>";
            }
            if ($row->lor) {
                $ret["LOR"] = "<a target=\"_blank\" href='" . url('storage/' . $row->lor) . "'>" . substr($row->lor, 5) . "</a>";
            }
            if ($row->sop) {
                $ret["SOP"] = "<a target=\"_blank\" href='" . url('storage/' . $row->sop) . "'>" . substr($row->sop, 5) . "</a>";
            }
            if ($row->letter_recom) {
                $ret["Letter of Recommendation"] = "<a target=\"_blank\" href='" . url('storage/' . $row->letter_recom) . "'>" . substr($row->letter_recom, 5) . "</a>";
            }
            if ($row->gre) {
                $ret["GRE"] = "<a target=\"_blank\" href='" . url('storage/' . $row->gre) . "'>" . substr($row->gre, 5) . "</a>";
            }
            if ($row->gmat) {
                $ret["GMAT"] = "<a target=\"_blank\" href='" . url('storage/' . $row->gmat) . "'>" . substr($row->gmat, 5) . "</a>";
            }
        }
        return $ret;
    }

    public function getProfDocsAttribute() {
        $ret = array();
        $rows = DB::table('user_docs_other')->where('userid', $this->id)->get();
        if ($rows) {
            foreach ($rows as $row) {
                if ($row->doc_type == 'internship') {
                    $ret["Internship"] = "<a target=\"_blank\" href='" . url('storage/' . $row->file) . "'>" . substr($row->file, 5) . "</a>";
                } elseif ($row->doc_type == 'offer') {
                    $ret["Offer Letter"] = "<a target=\"_blank\" href='" . url('storage/' . $row->file) . "'>" . substr($row->file, 5) . "</a>";
                } elseif ($row->doc_type == 'experience') {
                    $ret["Experience Letter"] = "<a target=\"_blank\" href='" . url('storage/' . $row->file) . "'>" . substr($row->file, 5) . "</a>";
                } elseif ($row->doc_type == 'payslip') {
                    $ret["Payslip"] = "<a target=\"_blank\" href='" . url('storage/' . $row->file) . "'>" . substr($row->file, 5) . "</a>";
                } elseif ($row->doc_type == 'other') {
                    $ret["Others"] = "<a target=\"_blank\" href='" . url('storage/' . $row->file) . "'>" . substr($row->file, 5) . "</a>";
                }
            }
        }
        return $ret;
    }

    public function getEmailButtonAttribute() {
        return '<a href="' . route('admin.communicate.mail', $this->id) . '" data-toggle="tooltip" data-placement="top" title="Email" class="btn btn-primary"><i class="bi bi-envelope-fill"></i></a>';
    }

    public function getSmsButtonAttribute() {
        return '<a href="' . route('admin.communicate.sms', $this->id) . '" data-toggle="tooltip" data-placement="top" title="SMS" class="btn btn-info"><i class="bx bx-message"></i></a>';
    }

    public function getWhatsappButtonAttribute() {
        return '<a href="https://wa.me/' . $this->mobile . '" target="_blank" data-toggle="tooltip" data-placement="top" title="Whatsapp" class="btn btn-primary"><i class="bx bxs-trash"></i></a>';
    }

}
