<?php

namespace App\Models\Auth\Traits\Attribute;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;

/**
 * Trait UserAttribute.
 */
trait UserAttribute {

    /**
     * @param $password
     */
    public function setPasswordAttribute($password): void {
        // If password was accidentally passed in already hashed, try not to double hash it
        if (
                (\strlen($password) === 60 && preg_match('/^\$2y\$/', $password)) ||
                (\strlen($password) === 95 && preg_match('/^\$argon2i\$/', $password)) 
        ) {
            $hash = $password;
        } else {
            $hash = Hash::make($password);
        }

        // Note: Password Histories are logged from the \App\Observer\User\UserObserver class
        $this->attributes['password'] = $hash;
    }

    /**
     * @return string
     */
    public function getStatusLabelAttribute() {
        if ($this->isActive()) {
            return "<span class='badge badge-success'>" . __('labels.general.active') . '</span>';
        }

        return "<span class='badge badge-danger'>" . __('labels.general.inactive') . '</span>';
    }

    /**
     * @return string
     */
    public function getConfirmedLabelAttribute() {
        if ($this->isConfirmed()) {
            if ($this->id != 1 && $this->id != auth()->id()) {
                return '<a href="' . route(
                                'admin.auth.user.unconfirm', $this
                        ) . '" data-toggle="tooltip" data-placement="top" title="' . __('buttons.backend.access.users.unconfirm') . '" name="confirm_item"><span class="badge badge-success" style="cursor:pointer">' . __('labels.general.yes') . '</span></a>';
            } else {
                return '<span class="badge badge-success">' . __('labels.general.yes') . '</span>';
            }
        }

        // return '<a href="' . route('admin.auth.user.confirm', $this) . '" data-toggle="tooltip" data-placement="top" title="' . __('buttons.backend.access.users.confirm') . '" name="confirm_item"><span class="badge badge-danger" style="cursor:pointer">' . __('labels.general.no') . '</span></a>';
        return '<a href="' . route('admin.auth.user.confirm', $this) . '" data-toggle="tooltip" data-placement="top" title="' . __('buttons.backend.access.users.confirm') . '" name="confirm_item"><span class="" style="cursor:pointer">Mark Email Confirmed </span></a>';
    }

    /**
     * @return string
     */
    public function getRolesLabelAttribute() {
        $roles = $this->getRoleNames()->toArray();

        if (\count($roles)) {
            return implode(', ', array_map(function ($item) {
                        return ucwords($item);
                    }, $roles));
        }

        return 'N/A';
    }

    /**
     * @return string
     */
    public function getPermissionsLabelAttribute() {
        $permissions = $this->getDirectPermissions()->toArray();

        if (\count($permissions)) {
            return implode(', ', array_map(function ($item) {
                        return ucwords($item['name']);
                    }, $permissions));
        }

        return 'N/A';
    }

    public function getStudentTypeLabelAttribute() {
        if ($this->student_type == 'course') {
            return "Trainings";
        } elseif ($this->student_type == 'study_abroad') {
            return "Study Abroad";
        } elseif ($this->student_type == 'study_abroad_course') {
            return "Trainings & Study Abroad";
        } else {
            return "-";
        }
    }

    /**
     * @return string
     */
    public function getFullNameAttribute() {
        return $this->last_name ? $this->first_name . ' ' . $this->last_name : $this->first_name;
    }

    /**
     * @return string
     */
    public function getNameAttribute() {
        return $this->full_name;
    }

    /**
     * @return mixed
     */
    public function getPictureAttribute() {
        return $this->getPicture();
    }

    /**
     * @return string
     */
    public function getSocialButtonsAttribute() {
        $accounts = [];

        foreach ($this->providers as $social) {
            $accounts[] = '<a href="' . route(
                            'admin.auth.user.social.unlink', [$this, $social]
                    ) . '" data-toggle="tooltip" data-placement="top" title="' . __('buttons.backend.access.users.unlink') . '" data-method="delete"><i class="fab fa-' . $social->provider . '"></i></a>';
        }

        return \count($accounts) ? implode(' ', $accounts) : __('labels.general.none');
    }

    /**
     * @return string
     */
    public function getLoginAsButtonAttribute() {
        /*
         * If the admin is currently NOT spoofing a user
         */
        // if (!session()->has('admin_user_id') || !session()->has('temp_user_id')) {
            //Won't break, but don't let them "Login As" themselves
            if ($this->id != auth()->id()) {
                return '<a href="' . route(
                                'admin.auth.user.login-as', $this
                        ) . '" class="py-2 dropdown-item"><i class="fas fa-user text-sub"></i> &nbsp;' . __('buttons.backend.access.users.login_as', ['user' => e($this->full_name)]) . '</a> ';
            }
        // }

        return '';
    }

    /**
     * @return string
     */
    public function getClearSessionButtonAttribute() {
        if ($this->id != auth()->id() && config('session.driver') == 'database') {
            return '<a href="' . route('admin.auth.user.clear-session', $this) . '"
			 	 data-trans-button-cancel="' . __('buttons.general.cancel') . '"
                 data-trans-button-confirm="' . __('buttons.general.continue') . '"
                 data-trans-title="' . __('strings.backend.general.are_you_sure') . '"
                 class="dropdown-item" name="confirm_item">' . __('buttons.backend.access.users.clear_session') . '</a> ';
        }

        return '';
    }

    /**
     * @return string
     */
    public function getShowButtonAttribute() {
        return '<a style="padding: 4px 8px;" href="' . route('admin.auth.user.show', $this) . '" data-toggle="tooltip" data-placement="top" title="' . __('buttons.general.crud.view') . '" class="btn btn-outline-info"><i class="fas fa-eye"></i></a>';
    }

    /**
     * @return string
     */
    public function getEditButtonAttribute() {
        return '<a href="' . route('admin.auth.user.edit', $this) . '" class="py-2 dropdown-item"><i class="fas fa-edit text-sub"></i> &nbsp;Edit</a>';
    }
    public function getEditProfileButtonAttribute() {
        return '<a href="' . route('admin.student.edit', $this) . '" class="py-2 dropdown-item"><i class="fas fa-edit text-sub"></i> &nbsp;Edit Profile</a>';
    }

    /**
     * @return string
     */
    public function getChangePasswordButtonAttribute() {
        return '<a href="' . route('admin.auth.user.change-password', $this) . '" class="py-2 dropdown-item"><i class="fas fa-exchange-alt text-sub"></i> &nbsp;' . __('buttons.backend.access.users.change_password') . '</a> ';
    }

    public function getArchiveButtonAttribute() {
        if(!\Auth::user()->isAgent()){
            return '<a href="' . route('admin.auth.user.archive-user', $this) . '" class="py-2 dropdown-item" name="confirm_item" data-trans-title="Are you sure want to Archive this student ?"><i class="fas fa-archive text-sub"></i> &nbsp;Archive</a> ';
        }
    }

    /**
     * @return string
     */
    public function getStatusButtonAttribute() {
        if ($this->id != auth()->id()) {
            switch ($this->active) {
                case 0:
                    return '<a href="' . route('admin.auth.user.mark', [
                                $this,
                                1,
                            ]) . '" class="dropdown-item">' . __('buttons.backend.access.users.activate') . '</a> ';

                case 1:
                    return '<a href="' . route('admin.auth.user.mark', [
                                $this,
                                0,
                            ]) . '" class="dropdown-item">' . __('buttons.backend.access.users.deactivate') . '</a> ';

                default:
                    return '';
            }
        }

        return '';
    }

    /**
     * @return string
     */
    public function getConfirmedButtonAttribute() {
        if (!$this->isConfirmed() && !config('access.users.requires_approval') && !\Auth::user()->isAgent()) {
            return '<a href="' . route('admin.auth.user.account.confirm.resend', $this) . '" class="dropdown-item"><i class="bi bi-arrow-clockwise text-sub"></i> ' . __('buttons.backend.access.users.resend_email') . '</a> ';
        }

        return '';
    }

    /**
     * @return string
     */
    public function getDeleteButtonAttribute() {
        if ($this->id != auth()->id() && $this->id != 1 && !\Auth::user()->isAgent()) {
            return '<a href="' . route('admin.auth.user.destroy', $this) . '"
                 data-method="delete"
                 data-message="You can restore this student later from Deleted Students page if required"
                 data-trans-button-cancel="' . __('buttons.general.cancel') . '"
                 data-trans-button-confirm="' . __('buttons.general.crud.delete') . '"
                 data-trans-title="' . __('strings.backend.general.are_you_sure') . '"
                 class="py-2 dropdown-item"><i class="fas fa-trash-alt text-sub"></i> &nbsp;' . __('buttons.general.crud.delete') . '</a> '; 
        }

        return '';
    }

    /**
     * @return string
     */
    public function getDeletePermanentlyButtonAttribute() {
        if(!\Auth::user()->isAgent()){
            return '<a href="' . route('admin.auth.user.delete-permanently', $this) . '" name="confirm_item" class="btn btn-danger" data-trans-title="' . __('strings.backend.general.delete_user_permanently') . '"><i class="bi bi-trash" data-toggle="tooltip" data-placement="top" title="' . __('buttons.backend.access.users.delete_permanently') . '"></i></a> ';
        }
    }

    /**
     * @return string
     */
    public function getRestoreButtonAttribute() {
        if(!\Auth::user()->isAgent()){
            return '<a href="' . route('admin.auth.user.restore', $this) . '" name="confirm_item" class="btn btn-info"><i class="fas fa-sync" data-toggle="tooltip" data-placement="top" title="' . __('buttons.backend.access.users.restore_user') . '"></i></a> ';
        }
    }

    /**
     * @return string
     */
    public function getActionButtonsAttribute() {
        if ($this->trashed()) {
            return '
				<div class="btn-group" role="group" aria-label="' . __('labels.backend.access.users.user_actions') . '">
				  ' . $this->restore_button . '
				  ' . $this->delete_permanently_button . '
				</div>';
        }

        $ret = '
    	<div class="" aria-label="' . __('labels.backend.access.users.user_actions') . '">
		  ' . $this->show_button . '
          
		  <div class="btn-group" role="group" >
			 <button style="padding: 4px 8px;" type="button" class="btn btn-outline-danger dropdown-toggle action_dropdown_btn"  data-bs-toggle="dropdown" aria-expanded="true">
             <i class="bi bi-three-dots-vertical"></i>
			</button>
			<div class="dropdown-menu dropdown-menu-end dropdown-menu-arrow" data-popper-placement="bottom-end">              
            ' . $this->confirmed_button ;
        
            $ret .= $this->login_as_button . '
            ' . $this->edit_button . '
            <a href="' . route('admin.student.edit', $this) . '" class="py-2 dropdown-item"><i class="fas fa-user-edit text-sub"></i> &nbsp;Edit Profile</a>
            ' . $this->change_password_button . '
            ' . $this->clear_session_button . '               
            ' . $this->archive_button . '               
            ' . $this->delete_button . '      
                            
			</div>
		  </div>
		</div>';
        // ' . $this->status_button . '
        // <a style="padding: 4px 8px;" href="' . route('admin.communicate.mail', $this->id) . '" data-toggle="tooltip" data-placement="top" title="Email" class="btn btn-outline-success"><i class="fa fa-envelope"></i></a>
        //    <a style="padding: 4px 8px;" href="' . route('admin.communicate.sms', $this->id)  . '" data-toggle="tooltip" data-placement="top" title="SMS" class="btn btn-outline-primary"><i class="fa fa-comment"></i></a>
        // ' 
        //     ' . $this->edit_button
        return $ret;
    }

    public function getTestLabelAttribute() {
        return "<span class='badge badge-success'>TEST</span>";
    }

    public function getTagsLabelAttribute() {
        $ret = "";
        $rows = DB::table('user_tags')->where('userid', $this->id)->get();
        if ($rows) {
            foreach ($rows as $row) {
                $ret .= "<span class='badge badge-success'>" . $row->tag . "</span> ";
            }
        }
        return $ret;
    }

    public function getAcademicsAttribute() {
        $rows = DB::table('user_academics')->where('userid', $this->id)->get();
        return $rows;
    }

    public function getEnglishcertAttribute() {
        $rows = DB::table('user_certification_eng')->where('userid', $this->id)->get();
        return $rows;
    }

    public function getLanguagecertAttribute() {
        $rows = DB::table('user_certification_lang')->where('userid', $this->id)->get();
        return $rows;
    }

    public function getGreGmatAttribute() {
        $rows = DB::table('user_gre_gmat')->where('userid', $this->id)->get();
        return $rows;
    }

    public function getAdditionalcoursesAttribute() {
        $rows = DB::table('user_courses')->where('userid', $this->id)->get();
        return $rows;
    }

    public function getExperiencesAttribute() {
        $rows = DB::table('user_experience')->where('userid', $this->id)->get();
        return $rows;
    }
    public function getEmploymentAttribute() {
        $rows = DB::table('employment_detail')->where('userid', $this->id)->get();
        return $rows;
    }
    public function getVisaAttribute() {
        $rows = DB::table('visa_application_history')->where('user_id', $this->id)->get();
        return $rows;
    }
    public function getTravelAttribute() {
        $rows = DB::table('travel_history')->where('userid', $this->id)->get();
        return $rows;
    }
    public function getFamilyAttribute() {
        $rows = DB::table('user_family')->where('userid', $this->id)->get();
        return $rows;
    }

    public function getCoursepreferencesAttribute() {
        $ret = array();
        $rows = DB::table('course_preference')->where('userid', $this->id)->orderBy('id', 'DESC')->get();
        if ($rows) {
            foreach ($rows as $row) {
                $intake_str = "";
                $intake = $row->intake;
                if ($intake) {
                    $month_nums = explode(",", $intake);
                    if (count($month_nums)) {
                        foreach ($month_nums as $month) {
                            if (is_numeric($month)) {
                                $str = date("Y") . "-" . $month . "-01";
                                $intake_str .= date("M", strtotime($str)) . ",";
                            }
                        }
                        // dd(strrpos($intake_str,","));
                        if(strrpos($intake_str,",")){
                            $intake_str = substr($intake_str, 0, strrpos($intake_str,","));
                        }
                    }
                }
                array_push($ret, array('country_id' => $row->country_id, 'course_id' => $row->course_id,'course_other' => $row->course_other,'university_other' => $row->university_other, 'university_id' => $row->university_id,'created_by' => $row->created_by,'created_at' => $row->created_at,'country' => $row->country, 'course' => $row->course, 'university' => $row->university, 'intake' => $intake_str,'id' => $row->id, 'intake_year' => $row->intake_year));
            }
        }
        return $ret;
    }

    public function getVisapreferencesAttribute() {
        $ret = array();
        $rows = DB::table('visa_preference')
        ->join('visa_service_types', 'visa_service_types.id', '=', 'visa_preference.service_type')
        ->join('visa_letter_types', 'visa_letter_types.id', '=', 'visa_preference.letter_type')
        ->where('userid', $this->id)
        ->select('visa_preference.id','visa_preference.country','visa_service_types.title as service_type','visa_letter_types.title as letter_type')
        ->get();
        if ($rows) {
            foreach ($rows as $row) {
                $country = $row->country;
                if ($country) {
                    array_push($ret, array('country' => $row->country, 'service_type' => $row->service_type, 'letter_type' => $row->letter_type, 'id' => $row->id));
                }
            }
        }
        return $ret;
    }


    public function getCourseSuggestionHistoryAttribute() {
        $ret = array();
        $rows = DB::table('course_recommendation')->where('userid', $this->id)->orderBy('upload_date', 'desc')->get();
        if ($rows) {
            foreach ($rows as $row) {
                $title = $row->title;
                $filename = $row->file;
                $ext = pathinfo($filename, PATHINFO_EXTENSION);
                $filepath = url('storage/' . $filename);
                $file_link = "<a class='text-primary' href='" . $filepath . "'>$title (.$ext)</a>";
                $ret[] = array('date' => $row->upload_date, 'file' => $file_link, 'satisfied' => ($row->satisfied == 'none') ? '-' : ucfirst($row->satisfied));
            }
        }
        return $ret;
    }

    public function getCourseFinderSuggestionHistoryAttribute() {
        $ret = array();
        $rows = DB::table('course_shortlisted')->where('student_id', $this->id)->orderBy('created_at', 'desc')->get();
        if ($rows) {
            foreach ($rows as $row) {
                $course_info = "";
                $course_row = DB::table('university_courses')->where('id', $row->course_id)->first();
                if($course_row) {
                    $univ_detail = DB::table('universities')->where('id', $course_row->university)->first();
                    $course_info .= "<a target='_blank' class='text-primary' href='".route('admin.coursemanagement.course-details', $course_row->id)."'><i class='bi bi-book'></i> <strong>".$course_row->course_name."</strong></a><br />";
                    if($univ_detail) {
                        $course_info .= '<a target="_blank" class="text-primary" href="'.route('admin.coursemanagement.university-details', $univ_detail->id).'"><svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-buildings" viewBox="0 0 16 16">
                        <path d="M14.763.075A.5.5 0 0 1 15 .5v15a.5.5 0 0 1-.5.5h-3a.5.5 0 0 1-.5-.5V14h-1v1.5a.5.5 0 0 1-.5.5h-9a.5.5 0 0 1-.5-.5V10a.5.5 0 0 1 .342-.474L6 7.64V4.5a.5.5 0 0 1 .276-.447l8-4a.5.5 0 0 1 .487.022ZM6 8.694 1 10.36V15h5V8.694ZM7 15h2v-1.5a.5.5 0 0 1 .5-.5h2a.5.5 0 0 1 .5.5V15h2V1.309l-7 3.5V15Z"/>
                        <path d="M2 11h1v1H2v-1Zm2 0h1v1H4v-1Zm-2 2h1v1H2v-1Zm2 0h1v1H4v-1Zm4-4h1v1H8V9Zm2 0h1v1h-1V9Zm-2 2h1v1H8v-1Zm2 0h1v1h-1v-1Zm2-2h1v1h-1V9Zm0 2h1v1h-1v-1ZM8 7h1v1H8V7Zm2 0h1v1h-1V7Zm2 0h1v1h-1V7ZM8 5h1v1H8V5Zm2 0h1v1h-1V5Zm2 0h1v1h-1V5Zm0-2h1v1h-1V3Z"/>
                      </svg> '.$univ_detail->university_name;
                        if($course_row->campus) {
                            $course_info .= " (".$course_row->campus.")";
                        }
                        $course_info .= "</a>";
                        $course_info .= "<br>";
                        $course_info .= "<i class='bi bi-globe'></i> ".get_country_name($univ_detail->country);
                        $course_info .= "<br>";
                    }
                    $course_info .= '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-mortarboard" viewBox="0 0 16 16">
                    <path d="M8.211 2.047a.5.5 0 0 0-.422 0l-7.5 3.5a.5.5 0 0 0 .025.917l7.5 3a.5.5 0 0 0 .372 0L14 7.14V13a1 1 0 0 0-1 1v2h3v-2a1 1 0 0 0-1-1V6.739l.686-.275a.5.5 0 0 0 .025-.917l-7.5-3.5ZM8 8.46 1.758 5.965 8 3.052l6.242 2.913L8 8.46Z"/>
                    <path d="M4.176 9.032a.5.5 0 0 0-.656.327l-.5 1.7a.5.5 0 0 0 .294.605l4.5 1.8a.5.5 0 0 0 .372 0l4.5-1.8a.5.5 0 0 0 .294-.605l-.5-1.7a.5.5 0 0 0-.656-.327L8 10.466 4.176 9.032Zm-.068 1.873.22-.748 3.496 1.311a.5.5 0 0 0 .352 0l3.496-1.311.22.748L8 12.46l-3.892-1.556Z"/>
                    </svg> '.$course_row->study_level;
                    // $course_info .= "<br><a target='_blank' class='text-primary' href='".route('admin.coursemanagement.course-details', $course_row->id)."'>Details &gt;&gt;</a>";
                    $course_info .= "<br><button class='btn btn-sm btn-outline-primary mt-2 course_preference' data-sugg_id='$row->id' title='Create'>Add To Course Preference</button>";
                    if(!$row->staff_suggested){
                        $course_info .= "<br><span style='font-style: italic;'>Bookmarked By Student</span>";
                    }else{
                        $course_info .= "<br><span style='font-style: italic;'>Suggested By ".get_user_full_name_by_id($row->staff_id)."</span>";
                    }
                }
                $ret[] = array('id'=> $row->id, 'date' => $row->created_at, 'course' => $course_info, 'approved' => $row->student_approved);
            }
        }
        return $ret;
    }

    public function getApplicationsAttribute() {
        $ret = array();
        $rows = DB::table('university_application')->where('userid', $this->id)->orderBy('id', 'asc')->get();
        if ($rows) {
            foreach ($rows as $row) {
                $intake_str = "";
                $intake = $row->intake;
                if ($intake) {
                    $str = date("Y") . "-" . $intake . "-01";
                    $intake_str = date("M", strtotime($str));
                }

                if($row->type == 'visa') {
                    $visa_info = DB::table('visa_preference')->where('id', $row->visa_pref_id)->first();
                    $country = $row->country . "<br /><strong>Service Type:</strong> ".get_visa_service_type_name($visa_info->service_type)."<br /><strong>Letter Type:</strong> ".get_visa_letter_type_name($visa_info->letter_type);
                    $status = $row->visa_status;
                } else {
                    $country = $row->country;
                    $status = $row->status;
                }
               
                // if ($row->offer_letter_ts) {
                //     $status = "Offer Letter Given on " . $row->offer_letter_ts;
                // } elseif ($row->result != 'none') {
                //     $status = "Application Result: " . ucfirst($row->result);
                // } elseif ($row->application_submitted == 'yes') {
                //     $status = "Application Submitted on: " . $row->application_submitted_ts;
                // } elseif ($row->application_fee_paid != 'none') {
                //     $status = "Application Fees Paid: " . ucfirst($row->application_fee_paid);
                // if ($row->application_fee_paid == 'yes')
                //     $status .= " On " . $row->application_fee_paid_ts;
                // } elseif ($row->application_mode != 'none') {
                //     $status = "Application Mode: " . ucfirst($row->application_mode);
                // } elseif ($row->admission_criteria_met != 'none') {
                //     $status = "Admission Criteria: " . ucfirst($row->admission_criteria_met);
                // } elseif ($row->doc_ready != 'none') {
                //     $status = "Doc ready: " . ucfirst($row->doc_ready);
                // } else {
                //     $status = "None";
                // }

                array_push($ret, array('id' => $row->id,'appl_id' => $row->appl_id, 'country' => $country, 'course' => $row->course, 'university' => $row->university, 'year' => $row->year, 'intake' => $intake_str, 'open_date' => $row->application_open_date, 'close_date' => $row->application_close_date, 'status' => $status));
            }
        }
        return $ret;
    }

    public function getCountryinfoPermissionButtonAttribute() {
        return '<a href="' . route('admin.auth.user.countrypermission', $this->id) . '" class="dropdown-item">Country Permission</a>';
    }

    public function getDemoClassPermissionButtonAttribute() {
        return '<a href="' . route('admin.auth.user.democlasspermission', $this->id) . '" class="dropdown-item">Demo Class Permission</a>';
    }
    
    public function getEducationDocsAttribute() {
        $ret = array();
        $row = DB::table('user_documents')->where('userid', $this->id)->first();
        if ($row) {
            if ($row->class10) {
                $ret["10th Certificate"] = "<a target=\"_blank\" href='" . url('storage/' . $row->class10) . "'>" . substr($row->class10, 5) . "</a>";
            }
            if ($row->class10_marks_sheet) {
                $ret["10th Marks Sheet"] = "<a target=\"_blank\" href='" . url('storage/' . $row->class10_marks_sheet) . "'>" . substr($row->class10_marks_sheet, 5) . "</a>";
            }            
            if ($row->class12) {
                $ret["12th Certificate"] = "<a target=\"_blank\" href='" . url('storage/' . $row->class12) . "'>" . substr($row->class12, 5) . "</a>";
            }
            if ($row->class12_marks_sheet) {
                $ret["12th Marks Sheet"] = "<a target=\"_blank\" href='" . url('storage/' . $row->class12_marks_sheet) . "'>" . substr($row->class12_marks_sheet, 5) . "</a>";
            }
            if ($row->diploma) {
                $ret["Diploma Certificate"] = "<a target=\"_blank\" href='" . url('storage/' . $row->diploma) . "'>" . substr($row->diploma, 5) . "</a>";
            }
            if ($row->diploma_marks_sheet) {
                $ret["Diploma Marks Sheet"] = "<a target=\"_blank\" href='" . url('storage/' . $row->diploma_marks_sheet) . "'>" . substr($row->diploma_marks_sheet, 5) . "</a>";
            }
            if ($row->grad_degree) {
                $ret["Bachelors Certificate"] = "<a target=\"_blank\" href='" . url('storage/' . $row->grad_degree) . "'>" . substr($row->grad_degree, 5) . "</a>";
            }
            if ($row->grad_transcripts) {
                $ret["Bachelors Transcripts"] = "<a target=\"_blank\" href='" . url('storage/' . $row->grad_transcripts) . "'>" . substr($row->grad_transcripts, 5) . "</a>";
            }
            if ($row->grad_all_sem) {
                $ret["Bachelors All Sem Marks Sheet"] = "<a target=\"_blank\" href='" . url('storage/' . $row->grad_all_sem) . "'>" . substr($row->grad_all_sem, 5) . "</a>";
            }

            if ($row->pg_degree) {
                $ret["Masters Certificate"] = "<a target=\"_blank\" href='" . url('storage/' . $row->pg_degree) . "'>" . substr($row->pg_degree, 5) . "</a>";
            }
            if ($row->pg_transcripts) {
                $ret["Masters Transcripts"] = "<a target=\"_blank\" href='" . url('storage/' . $row->pg_transcripts) . "'>" . substr($row->pg_transcripts, 5) . "</a>";
            }
            if ($row->pg_all_sem) {
                $ret["Masters All Sem Marks Sheet"] = "<a target=\"_blank\" href='" . url('storage/' . $row->pg_all_sem) . "'>" . substr($row->pg_all_sem, 5) . "</a>";
            }

        }
        return $ret;
    }

    public function getLangDocsAttribute() {
        $ret = array();
        $row = DB::table('user_documents')->where('userid', $this->id)->first();
        if ($row) {
            if ($row->lang_ielts) {
                $ret["IELTS"] = "<a target=\"_blank\" href='" . url('storage/' . $row->lang_ielts) . "'>" . substr($row->lang_ielts, 5) . "</a>";
            }
            if ($row->lang_pte) {
                $ret["PTE"] = "<a target=\"_blank\" href='" . url('storage/' . $row->lang_pte) . "'>" . substr($row->lang_pte, 5) . "</a>";
            }
            if ($row->gre) {
                $ret["GRE"] = "<a target=\"_blank\" href='" . url('storage/' . $row->gre) . "'>" . substr($row->gre, 5) . "</a>";
            }
            if ($row->gmat) {
                $ret["GMAT"] = "<a target=\"_blank\" href='" . url('storage/' . $row->gmat) . "'>" . substr($row->gmat, 5) . "</a>";
            }
        }
        return $ret;
    }

    public function getProfDocsAttribute() {
        $ret = array();
        $Internship = $Offer_Letter = $Experience_Letter = $Payslip = $Others = $SOP = $LOR = $CV = $Passport = "";
        $rows = DB::table('user_docs_other')->where('userid', $this->id)->get();
        if ($rows) {
            foreach ($rows as $row) {
                if ($row->doc_type == 'internship') {
                    $Internship .= "<a target=\"_blank\" href='" . url('storage/' . $row->file) . "'>" . substr($row->file, 5) . "</a> <a title=\"delete document\" data-id=\"$this->id\" data-file=\"$row->file\" data-type=\"Internship\" href=\"javascript:void(0)\" class=\"cross del-pro-btn ml-1\"></a><br>";
                } elseif ($row->doc_type == 'offer') {
                    $Offer_Letter .= "<a target=\"_blank\" href='" . url('storage/' . $row->file) . "'>" . substr($row->file, 5) . "</a> <a title=\"delete document\" data-id=\"$this->id\" data-file=\"$row->file\" data-type=\"Offer Letter\" href=\"javascript:void(0)\" class=\"cross del-pro-btn ml-1\"></a><br>";
                } elseif ($row->doc_type == 'experience') {
                    $Experience_Letter .= "<a target=\"_blank\" href='" . url('storage/' . $row->file) . "'>" . substr($row->file, 5) . "</a> <a title=\"delete document\" data-id=\"$this->id\" data-file=\"$row->file\" data-type=\"Experience Letter\" href=\"javascript:void(0)\" class=\"cross del-pro-btn ml-1\"></a><br>";
                } elseif ($row->doc_type == 'payslip') {
                    $Payslip .= "<a target=\"_blank\" href='" . url('storage/' . $row->file) . "'>" . substr($row->file, 5) . "</a> <a title=\"delete document\" data-id=\"$this->id\" data-file=\"$row->file\" data-type=\"Payslip\" href=\"javascript:void(0)\" class=\"cross del-pro-btn ml-1\"></a><br>";
                } elseif ($row->doc_type == 'other') {
                    $Others .= "<a target=\"_blank\" href='" . url('storage/' . $row->file) . "'>" . substr($row->file, 5) . "</a> <a title=\"delete document\" data-id=\"$this->id\" data-file=\"$row->file\" data-type=\"Others\" href=\"javascript:void(0)\" class=\"cross del-pro-btn ml-1\"></a><br>";
                } elseif ($row->doc_type == 'sop') {
                    $SOP .= "<a target=\"_blank\" href='" . url('storage/' . $row->file) . "'>" . substr($row->file, 5) . "</a> <a title=\"delete document\" data-id=\"$this->id\" data-file=\"$row->file\" data-type=\"SOP\" href=\"javascript:void(0)\" class=\"cross del-pro-btn ml-1\"></a><br>";
                } elseif ($row->doc_type == 'letter_of_recommendation') {
                    $LOR .= "<a target=\"_blank\" href='" . url('storage/' . $row->file) . "'>" . substr($row->file, 5) . "</a> <a title=\"delete document\" data-id=\"$this->id\" data-file=\"$row->file\" data-type=\"LOR\" href=\"javascript:void(0)\" class=\"cross del-pro-btn ml-1\"></a><br>";
                } elseif ($row->doc_type == 'cv') {
                    $CV = "<a target=\"_blank\" href='" . url('storage/' . $row->file) . "'>" . substr($row->file, 5) . "</a> <a title=\"delete document\" data-id=\"$this->id\" data-file=\"$row->file\" data-type=\"CV\" href=\"javascript:void(0)\" class=\"cross del-pro-btn ml-1\"></a><br>";
                } elseif ($row->doc_type == 'passport') {
                    $Passport = "<a target=\"_blank\" href='" . url('storage/' . $row->file) . "'>" . substr($row->file, 5) . "</a> <a title=\"delete document\" data-id=\"$this->id\" data-file=\"$row->file\" data-type=\"Passport\" href=\"javascript:void(0)\" class=\"cross del-pro-btn ml-1\"></a><br>";
                }
            }
        }

        if($Internship) {
            $ret['Internship'] = $Internship;
        }
        if($Offer_Letter) {
            $ret['Offer Letter'] = $Offer_Letter;
        }
        if($Experience_Letter) {
            $ret['Experience Letter'] = $Experience_Letter;
        }
        if($Payslip) {
            $ret['Payslip'] = $Payslip;
        }
        if($Others) {
            $ret['Others'] = $Others;
        }
        if($SOP) {
            $ret['SOP'] = $SOP;
        }
        if($LOR) {
            $ret['LOR'] = $LOR;
        }                           
        if($CV) {
            $ret['CV'] = $CV;
        }                           
        if($Passport) {
            $ret['Passport'] = $Passport;
        }

        return $ret;
    }

    public function getEmailButtonAttribute() {
        return '<a style="    padding:0.455rem 0.75rem;color:#fff;" href="' . route('admin.communicate.mail', $this->id) . '" data-toggle="tooltip" data-placement="top" title="Email" class="btn btn-primary"><i class="bi bi-envelope"></i></a>';
    }

    public function getSmsButtonAttribute() {
        return '<a style="    padding:0.455rem 0.75rem;color:#fff;" href="' . route('admin.communicate.sms', $this->id) . '" data-toggle="tooltip" data-placement="top" title="SMS" class="btn btn-info"><i class="bx bx-message"></i></a>';
    }

    public function getWhatsappButtonAttribute() {
        return '<a style="padding:0.455rem 0.75rem;color:#fff;" href="https://wa.me/' . $this->dialcode.$this->mobile . '" target="_blank" data-toggle="tooltip" data-placement="top" title="Whatsapp" class="btn btn-primary"><i class="bi bi-whatsapp"></i></a>';
    } 

    
    
    public function getEmailButtonLinkAttribute() {
        return '<a href="' . route('admin.communicate.mail', $this->id) . '" target="_blank" title="Email" class="dropdown-item">Email</a>';
    }

    public function getSmsButtonLinkAttribute() {
        return '<a href="' . route('admin.communicate.sms', $this->id) . '" target="_blank" title="SMS" class="dropdown-item">SMS</a>';
    }

    public function getWhatsappButtonLinkAttribute() {
        // if(env('WATI_WHATSAPP_TOKEN')){
            if(get_site_config('whatsapp_enabled')){
            return '<a href="' . route('admin.communicate.whatsapp', $this->id) . '" title="Whatsapp" class="dropdown-item">Whatsapp</a>';
        }
        return '<a href="https://wa.me/' . $this->dialcode.$this->mobile . '" target="_blank" title="Whatsapp" class="dropdown-item">Whatsapp</a>';
    }

    public function getLeadFollowupsAttribute() {
        $ret = '';
        $followups = DB::table('user_activity')->where('user_id', $this->id)->where('type', 'followup')->orderBy('created_at','ASC')->get();
        for($i=0; $i < get_max_followups_count(); $i++ ){
            if(isset($followups[$i])){
                $ret .= '<td style="display: none;">'.$followups[$i]->created_at.'</td>';
                $ret .= '<td style="display: none;">'.$followups[$i]->detail.'</td>';
            }else{
                $ret .= '<td style="display: none;">--</td>';
                $ret .= '<td style="display: none;">--</td>';
            }
        }
        return $ret;
    }

    public function getUserActivityAttribute()
    {
        $ret = '';
        $events = DB::table('user_activity')->where('user_id', $this->id)->orderByDesc('created_at')->get();
        if (count($events)) {
            // $ret .= '<div class="" style=" border-radius: 5px; padding: 10px !important; background-color: rgb(240,240,240);">
            // <div style="max-height: 300px; overflow-y: scroll;">';
            $ret .= '<div class="" id="activity_box" style="border:1px solid rgb(167 167 167); border-radius: 5px; padding: 10px !important;">';
            foreach ($events as $event) {
                if($event->type == "email_sent" || $event->type == "outbound_call" || $event->type == "whatsapp_sent" || $event->type == "sms_sent" || $event->type == "message_sent_to_student"){
                    $event_type = "outbound";
                    $icon_type = 'bi bi-box-arrow-left';
                }
                else if($event->type == "email_received" || $event->type == "inbound_call" || $event->type == "whatsapp_received" || $event->type == "message_received_from_student"){
                    $event_type = "inbound";
                    $icon_type = 'bi bi-box-arrow-right';
                }
                else if($event->type == "followup"){
                    $event_type = "followup";
                    $icon_type = 'bi bi-arrow-down-up';
                }
                else{
                    $event_type = $event->type;
                    $icon_type = 'bi bi-info-circle';
                }
                $ret .= '<div class="mt-3 filter '.$event_type.'">';
                $ret .= '<div class="d-flex align-items-start"><div><i class="'.$icon_type.' btn btn-sm btn-outline-info border-0" style="font-size: 20px !important;"></i></div><div class="ms-2 w-100">';
                    if($event->title) {
                        $ret .= '<p style="color: #353535 !important;"><strong style="font-size: 18px !important;">' . $event->title . '</strong></p>';
                    } else {
                        if($event->type == "student_assignment" || $event->type == "lead_assignment"){
                            $ret .= '<p style="color: #353535 !important;"><strong style="font-size: 18px !important;">Staff Assignment</strong></p>';
                        }
                        if($event->type == "followup"){
                            $ret .= '<p style="color: #353535 !important;"><strong style="font-size: 18px !important;">Followup</strong></p>';
                        }
                        else if($event->type == "others"){
                            $ret .= '<p style="color: #353535 !important;"><strong style="font-size: 18px !important;">Other activity</strong></p>';
                        }
                        $ret .= '<p></p>';
                    }
                    if($event->detail) {
                        $ret .= '<p class="text-dark">' . nl2br($event->detail) . '</p>';
                    }
                    if($event->attachment){
                        if($event->type == "message_sent_to_student" || $event->type == "message_received_from_student"){
                            $ret .= '<b>Attachments:</b> ';
                            $attachments = explode(',',$event->attachment);   
                            foreach($attachments as $file) { 
                                $ret .=' <a class="btn btn-sm btn-outline-primary" href="'.url("storage/".$file).'" target="_blank">'.\Str::limit($file, 20).'</a>';
                            }
                        } else{
                            $ret .= '<b>Attachments:</b> ';
                            $attachments = explode(',',$event->attachment);   
                            foreach($attachments as $file) { 
                                $ret .=' <a class="btn btn-sm btn-outline-primary" href="'.url("storage/email_attachments/".$file).'" target="_blank">'.\Str::limit($file, 20).'</a>';
                            } 
                        } 
                    }
                    $ret .= '
                    <div class="text-xs text-secondary mb-0" style="font-style: italic;">
                    ' . date('d M Y g:i A', strtotime($event->created_at)) . ' ' . get_user_full_name_by_id($event->staff) . '
                    </div><hr style="height: 1px; color: rgb(164 176 247);"></div></div></div>';
                // $ret .= '<div class="card">
                //     <div class="card-body" style="padding: 0px 20px 0px 20px;">';
                //     if($event->title) {
                //         $ret .= '<p class="card-text" style="color: #353535 !important;padding-top:15px;padding-bottom: 7px;">
                //         <strong>' . $event->title . '</strong>
                //         </p>';
                //     } else {
                //         $ret .= '<p></p>';
                //     }
                //     if($event->detail) {
                //         $ret .= '<p>' . $event->detail . '</p>';
                //     }
                //     $ret .= '</div>
                //     <div class="card-footer" style="padding: 7px 20px;color: #6e6e6e;font-style: italic;">
                //     ' . date('d M Y g:i A', strtotime($event->created_at)) . ' ' . get_user_name_by_id($event->staff) . '
                //     </div>
                // </div>';
            }
            $ret .= '<div class="mt-2 filter not_found" style="display: none;"><div class="d-flex align-items-start"><p class="text-dark mb-1">No activities found</p></div></div>';
            $ret .= '</div>';
            return $ret;
        } else {
            return 'No records found!';
        }
    }
    
}
