<?php

namespace App\Repositories\Backend\Auth;

use App\Models\Auth\User;
use Illuminate\Support\Facades\DB;
use App\Exceptions\GeneralException;
use App\Repositories\BaseRepository;
use App\Events\Backend\Auth\User\UserCreated;
use App\Events\Backend\Auth\User\UserUpdated;
use App\Events\Backend\Auth\User\UserRestored;
use App\Events\Backend\Auth\User\UserConfirmed;
use Illuminate\Pagination\LengthAwarePaginator;
use App\Events\Backend\Auth\User\UserDeactivated;
use App\Events\Backend\Auth\User\UserReactivated;
use App\Events\Backend\Auth\User\UserUnconfirmed;
use App\Events\Backend\Auth\User\UserPasswordChanged;
use App\Notifications\Backend\Auth\UserAccountActive;
use App\Events\Backend\Auth\User\UserPermanentlyDeleted;
use App\Notifications\Frontend\Auth\UserNeedsConfirmation;
use App\Models\Auth\Branch;
use App\Models\StaffBranch;
use App\Models\StaffDetails;
use Log;

/**
 * Class StaffRepository.
 */
class StaffRepository extends BaseRepository
{

    /**
     * @return string
     */
    public function model()
    {
        return User::class;
    }




    public function getStaffPaginated($paged = 25, $orderBy = 'created_at', $sort = 'desc'): LengthAwarePaginator
    {
        return $this->model
            ->with('roles', 'permissions', 'providers')
            ->active()
            ->whereHas('roles', function ($q) {
                $q->whereNotIn('name', ['user', 'trainer', 'agent']);
            })
            ->orderBy($orderBy, $sort)
            ->paginate($paged);
    }


    public function getStaffFilteredPaginated($find, $paged = 25, $orderBy = 'created_at', $sort = 'desc'): LengthAwarePaginator
    {
        return $this->model
            ->with('roles', 'permissions', 'providers')
            ->active()
            ->where(function ($query) use ($find) {
                $query->where('first_name', 'like', "%$find%")
                    ->orWhere('last_name', 'like', "%$find%")
                    ->orWhere('email', '=', $find)
                    ->orWhere('mobile', '=', $find);
            })
            ->whereHas('roles', function ($q) {
                $q->whereNotIn('name', ['user', 'trainer', 'agent']);
            })
            ->orderBy($orderBy, $sort)
            ->paginate($paged);
    }



    /**
     * @param int    $paged
     * @param string $orderBy
     * @param string $sort
     *
     * @return LengthAwarePaginator
     */
    public function getInactivePaginated($paged = 25, $orderBy = 'created_at', $sort = 'desc'): LengthAwarePaginator
    {
        return $this->model
            ->with('roles', 'permissions', 'providers')
            ->active(false)
            ->whereHas('roles', function ($q) {
                $q->whereNotIn('name', ['user', 'trainer', 'agent']);
            })
            ->orderBy($orderBy, $sort)
            ->paginate($paged);
    }



    /**
     * @param int    $paged
     * @param string $orderBy
     * @param string $sort
     *
     * @return LengthAwarePaginator
     */
    public function getDeletedPaginated($paged = 25, $orderBy = 'created_at', $sort = 'desc'): LengthAwarePaginator
    {
        return $this->model
            ->with('roles', 'permissions', 'providers')
            ->onlyTrashed()
            ->whereHas('roles', function ($q) {
                $q->whereNotIn('name', ['user', 'trainer', 'agent']);
            })
            ->orderBy($orderBy, $sort)
            ->paginate($paged);
    }



    //san
    // public function getTrainers()
    // {
    //     return $this->model
    //         ->with('roles', 'permissions', 'providers')
    //         ->active()
    //         ->whereHas('roles', function ($q) {
    //             $q->whereIn('name', ['trainer']);
    //         })
    //         ->orderBy("first_name", "ASC")
    //         ->get();
    // }

    // public function getStudents()
    // {
    //     return $this->model
    //         ->with('roles', 'permissions', 'providers')
    //         ->active()
    //         ->whereHas('roles', function ($q) {
    //             $q->whereIn('name', ['user']);
    //         })
    //         ->orderBy("id", "DESC")
    //         ->get();
    // }

    public function getNonStudentsTrainers()
    {
        return $this->model
            ->with('roles', 'permissions', 'providers')
            ->active()
            ->whereHas('roles', function ($q) {
                $q->whereNotIn('name', ['user', 'trainer', 'agent']);
            })
            ->orderBy("name", "ASC")
            ->get();
    }


    public function getStaffUsers($branchs)
    {
        //\Log::debug("StaffRepository: getStaffUsers, branch:".var_export($branchs, true));
        return $this->model
            ->with('roles')
            ->join('staff_branch', 'staff_branch.staff_id', '=', 'users.id')
            ->whereIn('staff_branch.branch_id', $branchs)            
            ->active()
            ->whereHas('roles', function ($q) {
                $q->whereNotIn('name', ['user', 'trainer', 'agent']);
            })
            ->orderBy("users.first_name", "ASC")
            ->groupBy('users.id')
            ->get();
    }

    public function getActiveStaffCount(): int
    {
        return  $this->model
            ->with('roles', 'permissions', 'providers')
            ->active()
            ->whereHas('roles', function ($q) {
                $q->whereNotIn('name', ['user', 'trainer', 'agent']);
            })
            ->count();
    }


    public function getDatatableUsers($request)
    {

        $columns = array(
            0 => 'first_name',
            // 1 => 'last_name',
            1 => 'email',
            2 => 'mobile',
            // 3 => 'role',
            // 4 => 'branch',
            4 => 'created_at',
            // 6 => 'options',
        );

        $totalData = $this->getActiveStaffCount();
        $totalFiltered = $totalData;

        $limit = $request->input('length');
        $start = $request->input('start');
        $order = $columns[$request->input('order.0.column')];
        $dir = $request->input('order.0.dir');

        if (empty($request->input('search.value'))) {
            $users = $this->model
                ->with('roles', 'permissions', 'providers')
                ->active()
                ->whereHas('roles', function ($q) {
                    $q->whereNotIn('name', ['user', 'trainer', 'agent']);
                })
                ->offset($start)
                ->limit($limit)
                ->orderBy($order, $dir)
                ->get();
        } else {
            $get_search = $request->input('search.value');
            $users = $this->model
                ->with('roles', 'permissions', 'providers')
                ->active()
                ->where(function ($query) use ($get_search) {
                    $query->where('first_name', 'like', "%$get_search%")
                        ->orWhere('last_name', 'like', "%$get_search%")
                        ->orWhereRaw("CONCAT(`first_name`, ' ', `last_name`) LIKE ?", ['%'.$get_search.'%'])
                        ->orWhere('email', 'like', "%$get_search%")
                        ->orWhere('mobile', 'like', "%$get_search%");
                })
                ->whereHas('roles', function ($q) {
                    $q->whereNotIn('name', ['user', 'trainer', 'agent']);
                })
                ->offset($start)
                ->limit($limit)
                ->orderBy($order, $dir)
                ->get();

            $totalFiltered = $this->model
                ->with('roles', 'permissions', 'providers')
                ->active()
                ->where(function ($query) use ($get_search) {
                    $query->where('first_name', 'like', "%$get_search%")
                        ->orWhere('last_name', 'like', "%$get_search%")
                        ->orWhereRaw("CONCAT(`first_name`, ' ', `last_name`) LIKE ?", ['%'.$get_search.'%'])
                        ->orWhere('email', 'like', "%$get_search%")
                        ->orWhere('mobile', 'like', "%$get_search%");
                })
                ->whereHas('roles', function ($q) {
                    $q->whereNotIn('name', ['user', 'trainer', 'agent']);
                })
                ->count();
        }

        $data = array();
        if (!empty($users)) {
            foreach ($users as $page) {
                // $customResult['first_name'] = $page->first_name;
                // $customResult['last_name'] = $page->last_name;
                $customResult['name'] = "<a href='" . route('admin.auth.staff.show', $page->id) . "'><strong><span style='white-space:nowrap;'><i class='fas fa-user-tie text-sub'></i> " . $page->first_name . " " . $page->last_name . "</span></strong></a>";
                $customResult['email'] = $page->email;
                $customResult['mobile'] = ($page->dialcode ? "+".$page->dialcode."-" : '') . $page->mobile;

                foreach ($page->roles as $role) {
                    $customResult['role']  = ucfirst($role->name) . ', ';
                }
                if($page->id == '1'){
                    $customResult['role']  .= '<br>(Super&nbsp;Admin)';
                }
                
                $customResult['details'] = "<p class='mb-1'><b>Role(s): </b>".$customResult['role']."</p>";
                $customResult['details'] .= "<p class='mb-1'><b>Branch: </b>".get_staff_branch_names($page->id)."</p>";
                if($page->id != '1'){
                    $customResult['details'] .= "<p class='mb-1'><b>Reporting To: </b>".get_user_full_name_by_id($page->reporting_to)."</p>";
                }
                // $customResult['created_at'] = $page->created_at->diffForHumans();
                $customResult['created_at'] = $page->created_at ? date('d M Y', strtotime($page->created_at)) : "--";
                //$customResult['options'] = $page->action_buttons;

                $options =       '<div class="" aria-label="'.__('labels.backend.access.users.user_actions').'">
                                <a href="' . route('admin.auth.staff.show', $page->id) . '" data-toggle="tooltip" data-placement="top" title="'.__('buttons.general.crud.view').'" class="btn btn-outline-info"><i class="ri-eye-fill"></i> </a>
                                    
                                <div class="btn-group btn-group-sm" role="group">
                                    <button style="padding: 0.430rem 0.75rem;margin-bottom:0px;" id="userActions" type="button" class="btn btn-outline-danger dropdown-toggle" data-bs-toggle="dropdown" aria-expanded="true"><i class="bi bi-three-dots-vertical"></i>
                                    </button>
                                    <div class="dropdown-menu dropdown-menu-end">';
                                    $user_loggedin = \Auth::user();
                                    
                                    if($user_loggedin->user_unlock){
                $options .='
                                    
                                        <a href="' . route('admin.auth.staff.edit', $page->id) . '" class="dropdown-item border-bottom"><i class="fas fa-edit text-sub"></i> &nbsp;&nbsp;&nbsp;Edit</a>';
                                    }
                
                if(\Auth::user()->isManager() && $role->name == 'administrator'){
                // $options .='<a href="' . route('admin.auth.staff.login-as', $page->id) . '" class="dropdown-item"> Login As ' . $page->first_name . '</a>'; 
                }
                elseif(\Auth::user()->can('view staff') && $role->name == 'manager'){

                }
                else{
                    $options .='<a href="' . route('admin.auth.staff.login-as', $page->id) . '" class="dropdown-item border-bottom"><i class="fas fa-user text-sub"></i> &nbsp;&nbsp;&nbsp;&nbsp;Login As ' . $page->first_name . '</a>'; 
                }
                $options .='<a href="' . route('admin.auth.staff.change-password', $page->id) . '" class="dropdown-item border-bottom"><i class="fas fa-exchange-alt text-sub"></i> &nbsp;&nbsp;&nbsp;Change Password</a>';


                // if(env('WATI_WHATSAPP_TOKEN')){
                    if(get_site_config('whatsapp_enabled')){
                        $whatsapp_link = '<a target="_blank" href="'.route('admin.communicate.whatsapp', $page->id) .'" class="dropdown-item border-bottom"><i class="bi bi-whatsapp text-sub"></i> Whatsapp</a>';
                    }else{
                        $whatsapp_link = '<a target="_blank" href="https://wa.me/' . $page->mobile . '" class="dropdown-item border-bottom"><i class="bi bi-whatsapp text-sub"></i> Whatsapp</a>';
                    }

                    $options .=  '<a href="' . route('admin.communicate.sms', $page->id) . '" class="dropdown-item border-bottom"><i class="bx bx-message text-sub"></i> SMS</a>
                                        '. $whatsapp_link.'        
                                        <a href="' . route('admin.communicate.mail', $page->id) . '" class="dropdown-item border-bottom"><i class="bi bi-envelope text-sub"></i> Email</a>';
                    
                    if ($page->id != 1 && $page->id != auth()->id()) {
                        if ($page->active) {
                            $options .= '<a href="' . route('admin.auth.staff.mark', [$page->id, 0]) . '" class="dropdown-item border-bottom"><i class="fas fa-eye-slash text-sub"></i> &nbsp;&nbsp;&nbsp;Deactivate</a>';
                        }

                        
                        if ($page->user_unlock) {
                           
                            $options .= '<a href="' . route('admin.auth.staff.marklock', ['user'=>$page->id, 'status'=> 0]) . '" class="dropdown-item border-bottom"><i class="fas fa-solid fa-lock"></i> &nbsp;&nbsp;&nbsp;Lock</a>';
                        }
                        if ($page->user_unlock == 0) {
                            $options .= '<a href="' . route('admin.auth.staff.marklock', ['user'=>$page->id,'status'=>1]) . '" class="dropdown-item border-bottom"><i class="fas fa-solid fa-unlock"></i> &nbsp;&nbsp;&nbsp;Unlock</a>';
                        }
    
                        if (!$page->confirmed && !config('access.users.requires_approval')) {
                            $options .= '<a href="' . route('admin.auth.staff.account.confirm.resend', $page->id) . '" class="dropdown-item border-bottom"><i class="fas fa-share-square text-sub"></i> Resend Email</a>';
                        }
    
    
                        if ($page->deleted_at == NULL) {
                            $options .= '<a href="' . route('admin.auth.staff.destroy', $page->id) . '"
                                                    data-method="delete"
                                                    data-trans-button-cancel="Cancel"
                                                    data-trans-button-confirm="Delete"
                                                    data-trans-title="Are you sure to Delete?"
                                                    class="dropdown-item border-bottom"><i class="bi bi-trash text-sub"></i> Delete</a>';
                        } else {
                            $options .= '<a href="' . route('admin.auth.staff.restore', $page->id) . '" name="confirm_item" class="dropdown-item border-bottom">Restore user</a>';
                        }
                    }

                    $options .=  '</div>
                                </div>';

                $customResult['options'] = $options;
                $data[] = $customResult;
            }
        }

        $json_data = array(
            "draw"            => intval($request->input('draw')),
            "recordsTotal"    => intval($totalData),
            "recordsFiltered" => intval($totalFiltered),
            "data"            => $data
        );

        return json_encode($json_data);
    }







    /**
     * @param array $data
     *
     * @return User
     * @throws \Exception
     * @throws \Throwable
     */

    /**
     * @param User  $user
     * @param array $data
     *
     * @return User
     * @throws GeneralException
     * @throws \Exception
     * @throws \Throwable
     */
    public function update(User $user, array $data): User
    {
        $this->checkUserByEmail($user, $data['email']);

        // See if adding any additional permissions
        if (!isset($data['permissions']) || !count($data['permissions'])) {
            $data['permissions'] = [];
        }

        if (!isset($data['reporting_to'])) {
            $data['reporting_to'] = 1;
        }

        // if (isset($data["birthday"]) && $data["birthday"]) {
        //     $dob = date('Y-m-d', strtotime($data["birthday"]));
        // } else {
        //     $dob = NULL;
        // }
        // $data['dob'] = $dob;

        return DB::transaction(function () use ($user, $data) {
            // dd($data);
            if ($user->update([
                'first_name' => $data['first_name'],
                'last_name' => $data['last_name'],
                'is_staff' => 1,
                'email' => $data['email'],
                'dialcode' => $data['dialcode'],
                'reporting_to' => $data['reporting_to'],
                'whatsapp_dialcode' => $data['whatsapp_dialcode'],
                'mobile' => $data['mobile'],
                'whatsapp' => $data['whatsapp_number'],
                'alternate_contact' => $data['alternate_contact'],
                'updated_by'=>\Auth::user()->id
            ])) {
                // Add selected roles/permissions
                $user->syncRoles($data['roles']);
                $user->syncPermissions($data['permissions']);

                //checking and adding smtp credentials
                //if(isset($data['email_integration'])){
                    if(isset($data['smtp_password'])){
                        $new_pass = $data['smtp_password'];
                    }else{
                        $new_pass = (isset($data['old_smtp_password'])) ? $data['old_smtp_password'] : NULL;
                    }
                    StaffDetails::updateOrCreate(
                        [
                            'staff_id'   => $user->id,
                        ],
                        [
                        'email_integration' => isset($data['email_integration']) && $data['email_integration'] == '1' ? 1 : 0,
                        'smtp_port' => (isset($data['smtp_port'])) ? $data['smtp_port'] : NULL,
                        'smtp_host' => (isset($data['smtp_host'])) ? $data['smtp_host'] : NULL,
                        // 'imap_host' => $data['imap_host'],
                        // 'imap_port' => $data['imap_port'],
                        'smtp_encryption' => (isset($data['smtp_encryption'])) ? $data['smtp_encryption'] : NULL,
                        'smtp_username' => $user->email,
                        'smtp_password' => $new_pass,
                        'modified_by'=>\Auth::user()->id

                    ]); 

                //}
                //update branch
                StaffBranch::where('staff_id', $user->id)->delete();
                if(in_array('all', $data['branch'])) {
                    $all_branch = Branch::all();
                    foreach($all_branch as $branch) {
                        StaffBranch::create(["staff_id" => $user->id, "branch_id" => $branch->id]);
                    }
                } else {
                    foreach($data['branch'] as $branch_id) {
                        StaffBranch::create(["staff_id" => $user->id, "branch_id" => $branch_id]);
                    }                    
                }

                event(new UserUpdated($user));

                return $user;
            }

            throw new GeneralException(__('exceptions.backend.access.users.update_error'));
        });
    }

    /**
     * @param User $user
     * @param      $input
     *
     * @return User
     * @throws GeneralException
     */
    public function updatePassword(User $user, $input): User
    {
        if ($user->update(['password' => $input['password']])) {
            event(new UserPasswordChanged($user));

            return $user;
        }

        throw new GeneralException(__('exceptions.backend.access.users.update_password_error'));
    }

    /**
     * @param User $user
     * @param      $status
     *
     * @return User
     * @throws GeneralException
     */
    public function mark(User $user, $status): User
    {
        if(env('DEMO_MODE') && in_array($user->id, config('app.demo_users')))
            return redirect()->route('admin.demo-user-restriction'); 
                    
        if (auth()->id() == $user->id && $status == 0) {
            throw new GeneralException(__('exceptions.backend.access.users.cant_deactivate_self'));
        }

        if ($user->id == 1) {
            throw new GeneralException('This operation is not allowed');
        }        

        $user->active = $status;

        switch ($status) {
            case 0:
                event(new UserDeactivated($user));
                break;

            case 1:
                event(new UserReactivated($user));
                break;
        }

        if ($user->save()) {
            return $user;
        }

        throw new GeneralException(__('exceptions.backend.access.users.mark_error'));
    }

    /**
     * @param User $user
     *
     * @return User
     * @throws GeneralException
     */
    public function confirm(User $user): User
    {
        if ($user->confirmed) {
            throw new GeneralException(__('exceptions.backend.access.users.already_confirmed'));
        }

        $user->confirmed = 1;
        $confirmed = $user->save();

        if ($confirmed) {
            event(new UserConfirmed($user));

            // Let user know their account was approved
            if (config('access.users.requires_approval')) {
                try{
                    //$user->notify(new UserAccountActive);

                    $email_message = __('strings.emails.auth.account_confirmed');
                    $email_message .= "<a href='".route('frontend.auth.login')."'>".__('labels.frontend.auth.login_button')."</a>";
                    $email_message .= __('strings.emails.auth.thank_you_for_using_app');
                    user_notify($user,app_name(), $email_message);
                }catch (\Exception $e) {
                    $message = $e->getMessage();
                    \Log::debug("Mail Error: (User Account Active) ".$message);   
                }
            }

            return $user;
        }

        throw new GeneralException(__('exceptions.backend.access.users.cant_confirm'));
    }

    /**
     * @param User $user
     *
     * @return User
     * @throws GeneralException
     */
    public function unconfirm(User $user): User
    {
        if (!$user->confirmed) {
            throw new GeneralException(__('exceptions.backend.access.users.not_confirmed'));
        }

        if ($user->id == 1) {
            // Cant un-confirm admin
            throw new GeneralException(__('exceptions.backend.access.users.cant_unconfirm_admin'));
        }

        if ($user->id == auth()->id()) {
            // Cant un-confirm self
            throw new GeneralException(__('exceptions.backend.access.users.cant_unconfirm_self'));
        }

        $user->confirmed = 0;
        $unconfirmed = $user->save();

        if ($unconfirmed) {
            event(new UserUnconfirmed($user));

            return $user;
        }

        throw new GeneralException(__('exceptions.backend.access.users.cant_unconfirm'));
    }

    /**
     * @param User $user
     *
     * @return User
     * @throws GeneralException
     * @throws \Exception
     * @throws \Throwable
     */
    public function forceDelete(User $user): User
    {
        if (is_null($user->deleted_at)) {
            throw new GeneralException(__('exceptions.backend.access.users.delete_first'));
        }

        return DB::transaction(function () use ($user) {
            // Delete associated relationships
            $user->passwordHistories()->delete();
            $user->providers()->delete();
            $user->sessions()->delete();

            if ($user->forceDelete()) {
                event(new UserPermanentlyDeleted($user));

                return $user;
            }

            throw new GeneralException(__('exceptions.backend.access.users.delete_error'));
        });
    }

    /**
     * @param User $user
     *
     * @return User
     * @throws GeneralException
     */
    public function restore(User $user): User
    {
        if (is_null($user->deleted_at) && $user->removed == 0) {
            throw new GeneralException(__('exceptions.backend.access.users.cant_restore'));
        }

        if ($user->restore()) {
            event(new UserRestored($user));

            return $user;
        }

        throw new GeneralException(__('exceptions.backend.access.users.restore_error'));
    }

    /**
     * @param User $user
     * @param      $email
     *
     * @throws GeneralException
     */
    protected function checkUserByEmail(User $user, $email)
    {
        //Figure out if email is not the same
        if ($user->email != $email) {
            //Check to see if email exists
            if ($this->model->where('email', '=', $email)->first()) {
                throw new GeneralException(trans('exceptions.backend.access.users.email_error'));
            }
        }
    }
}
