<?php

namespace App\Repositories\Backend\Auth;

use App\Models\Auth\User;
use Illuminate\Support\Facades\DB;
use App\Exceptions\GeneralException;
use App\Repositories\BaseRepository;
use App\Events\Backend\Auth\User\UserCreated;
use App\Events\Backend\Auth\User\UserUpdated;
use App\Events\Backend\Auth\User\UserRestored;
use App\Events\Backend\Auth\User\UserConfirmed;
use App\Events\Backend\Auth\User\UserLocked;
use App\Events\Backend\Auth\User\UserUnlocked;
use Illuminate\Pagination\LengthAwarePaginator;
use App\Events\Backend\Auth\User\UserDeactivated;
use App\Events\Backend\Auth\User\UserReactivated;
use App\Notifications\Frontend\AdminAlert;
use App\Events\Backend\Auth\User\UserUnconfirmed;
use App\Models\Auth\UserDetails;
use App\Models\UserActivity;
use App\Events\Backend\Auth\User\UserPasswordChanged;
use App\Notifications\Backend\Auth\UserAccountActive;
use App\Events\Backend\Auth\User\UserPermanentlyDeleted;
use App\Notifications\Frontend\Auth\UserNeedsConfirmation;
use App\Notifications\Backend\UserAlert;
use App\Models\MailTemplates;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use App\Recipients\AdminRecipient;
use Illuminate\Support\Facades\File;
use App\Models\Auth\UserNotification;
use Illuminate\Support\Facades\Mail;
use App\Mail\MailSend;
use Illuminate\Support\HtmlString;
use App\Models\StaffDetails;

/**
 * Class UserRepository.
 */
class UserRepository extends BaseRepository
{

    /**
     * @return string
     */
    public function model()
    {
        return User::class;
    }

    /**
     * @return mixed
     */
    public function getUnconfirmedCount(): int
    {
        return $this->model
            ->where('confirmed', 0)
            ->count();
    }


    public function getActiveCount(): int
    {
        return  $this->model
            ->with('roles', 'permissions', 'providers')
            ->active()
            ->whereHas('roles', function ($q) {
                $q->whereIn('name', ['user']);
            })
            ->count();
    }


    public function getDatatableUsers($request)
    {

        // \Log::debug("getDatatableUsers:".var_export($request->all(), true));
        // DB::enableQueryLog();
        $columns = array(
            1 => 'id',
            8 => 'first_name',
            9 => 'next_followup',
            11 => 'assigned_to',
            12 => 'student_created_at',
        );

        $my_branch = "";
        $overdue = ""; 
        $followup_date_from = "";
        $followup_date_to = "";  
        $today = ""; 
        $tomorrow = ""; 
        $in_next_7_days = "";
        $created_date_from = "";
        $created_date_to = ""; 

        if(\Auth::user()->can('view assigned data only')) {
            $my_user_id = \Auth::user()->id;
        } else {
            $my_user_id = $request->input('mystu') == 1 ? \Auth::user()->id : '';        
            //$my_branch = get_staff_branch_ids(\Auth::user()->id);
        }
        if(\Auth::user()->isAgent() && !\Auth::user()->isAdmin()){
            // No need to check branches for agents
            $my_branch = get_all_branch_ids();
        } else if($request->input('branch') == "all"){
            $my_branch = get_staff_branch_ids(\Auth::user()->id);
        } else{
            $my_branch = array($request->input('branch'));
        }
        // dd($my_branch);

        $exclude_agent_student = $request->input('excludeAgentStudent');
        $agentListOnly = $request->input('agentListOnly');
        $before_student_age = "";

        if(!empty($request->input('student_age'))) {
            $student_age = $request->input('student_age');
            $before_student_age = date("Y-m-d 00:00:00", strtotime("-".$student_age." days"));
        }

        if(!empty($request->input('followup_daterange'))) {
            $followup_dts_str = $request->input('followup_daterange');
            $followup_dts = explode(" to ", $followup_dts_str);
            $followup_date_from = date("Y-m-d 00:00:00", strtotime($followup_dts[0]));
            $followup_date_to = date("Y-m-d 23:59:59", strtotime($followup_dts[1]));
        }
        //\Log::debug('Lead follow dt: '.$followup_date_from. ' to '.$followup_date_to);

        $next_followup_date = $request->input('next_followup');
    
        if($next_followup_date == "overdue"){
            $overdue = date("Y-m-d 00:00:00");
        } elseif($next_followup_date == "today"){
            $today = date("Y-m-d");
        } elseif($next_followup_date == "tomorrow"){
            $tomorrow = date("Y-m-d", strtotime("+1 days"));
        } elseif($next_followup_date == "in_next_7_days"){
            $in_next_7_days = date("Y-m-d 23:59:59", strtotime("+8 days")); 
            //$current_days = date("Y-m-d 00:00:00");
        } elseif($next_followup_date == "custom"){     
            if(!empty($request->input('followup_daterange'))) {
                $followup_dts_str = $request->input('followup_daterange');
                $followup_dts = explode(" to ", $followup_dts_str);
                $followup_date_from = date("Y-m-d 00:00:00", strtotime($followup_dts[0]));
                $followup_date_to = date("Y-m-d 23:59:59", strtotime($followup_dts[1]));
            }
            //\Log::debug('Lead follow dt: '.$followup_date_from. ' to '.$followup_date_to);
        }

        $created_date = $request->input('created_date');
        if(!empty($created_date)) {
            $created_dts_str = $created_date;
            $created_dts = explode(" to ", $created_dts_str);
            $created_date_from = date("Y-m-d 00:00:00", strtotime($created_dts[0]));
            $created_date_to = date("Y-m-d 23:59:59", strtotime($created_dts[1]));
            \Log::debug('Lead created dt: '.$created_date_from. ' to '.$created_date_to);
        }

        $limit = $request->input('length');
        $start = $request->input('start');
        $order = $columns[$request->input('order.0.column')];
        $tags_filter = $request->input('tags_filter');
        $dir = $request->input('order.0.dir');
        $student_type = $request->input('student_type');
        $overseas_status = $request->input('overseas_status');
        $coaching_status = $request->input('coaching_status');
        $assigned_to = $request->input('assigned_to');
        //$my_user_id = $request->input('mystu') == 1 ? \Auth::user()->id : '';
        $user_country = $request->input('user_country');

        $country_pref = "";
        if($request->input('country_pref') != ""){
            $country_pref = implode(',',$request->input('country_pref'));
        }

        $service_pref = "";
        if($request->input('service_pref') != ""){
            $service_pref = implode(',',$request->input('service_pref'));
        }
        $study_level_filter = $request->input('study_level_filter');
        $course_interested_filter = $request->input('course_interested_filter');
        $intatake_arr = "";
        if (!empty($request->input('intake_filter'))) {
            $intake = $request->input('intake_filter');
            $intatake_arr =  explode(" ", $intake);
        }

        $totalData = $this->model
            ->with('roles', 'permissions', 'providers')
            ->leftjoin('countries', 'users.country', '=', 'countries.id')
            ->leftjoin('states', 'users.state', '=', 'states.id')
            ->leftjoin('branch', 'users.branch', '=', 'branch.id')
            ->leftjoin('user_details', 'users.id', '=', 'user_details.userid')
                ->select("users.*", "user_details.intake_year", "user_details.intake_month", "user_details.student_disqualified_reason", "countries.title_en as country_name", "states.name as state_name", "branch.branch_name as branch_name")
                ->active()
                ->whereHas('roles', function ($q) {
                    $q->whereIn('name', ['user']);
                })
                ->when(!empty($my_branch), function ($query) use ($my_branch) {
                    return $query->whereIn('users.branch', $my_branch);
                })
                ->when(\Auth::user()->isAgent(), function ($q) {
                    $q->where('agent_id', \Auth::user()->id);
                })
                ->when(!empty($tags_filter), function ($query) use ($tags_filter) {
                    return $query->Where(function ($query) use ($tags_filter) {
                        foreach ($tags_filter as $tag) {
                            $query->orWhere("tags", "like", "%" . $tag . "%");
                        }
                    });
                })
                ->when(!empty($user_country), function ($query) use ($user_country) {
                    return  $query->Where('users.country', '=', "$user_country");
                })
                ->when(!empty($assigned_to), function ($query) use ($assigned_to) {
                    return  $query->Where('assigned_to', '=', "$assigned_to");
                })
                ->when(!empty($my_user_id), function ($query) use ($my_user_id) {
                    return  $query->Where('assigned_to', '=', "$my_user_id");
                })
                ->when($exclude_agent_student, function ($query) {
                    return $query->where('agent_id', 0);
                })
                ->when($agentListOnly, function ($query) {
                    return $query->where('agent_id', '!=', 0);
                })
                ->when(!empty($before_student_age), function ($query) use ($before_student_age) {
                    return  $query->Where('student_created_at', '<=', "$before_student_age");
                    // return $query->whereRaw("IF(lead_converted_date IS NULL, created_at <= '$before_student_age', lead_converted_date <= '$before_student_age')");
                })
                ->when(!empty($followup_date_from), function ($query) use ($followup_date_from) {
                    return  $query->Where('next_followup', '>=', "$followup_date_from");
                })
                ->when(!empty($followup_date_to), function ($query) use ($followup_date_to) {
                    return  $query->Where('next_followup', '<=', "$followup_date_to");
                })
                ->when(!empty($overdue), function ($query) use ($overdue) {
                    return  $query->Where('next_followup', '<', "$overdue");
                })
                ->when(!empty($today), function ($query) use ($today) {
                    return  $query->Where('next_followup', 'like', "$today%");
                })
                ->when(!empty($tomorrow), function ($query) use ($tomorrow) {
                    return  $query->Where('next_followup', 'like', "$tomorrow%");
                })
                ->when(!empty($in_next_7_days), function ($query) use ($in_next_7_days) {
                    return $query->Where(function ($query) use ($in_next_7_days) {
                        $query->where('next_followup', '>=', date("Y-m-d"))
                            ->Where('next_followup', '<=', "$in_next_7_days");
                    });
                })
                ->when(!empty($created_date_from), function ($query) use ($created_date_from) {
                    return  $query->Where('student_created_at', '>=', "$created_date_from");
                })
                ->when(!empty($created_date_to), function ($query) use ($created_date_to) {
                    return  $query->Where('student_created_at', '<=', "$created_date_to");
                })
                ->when(!empty($country_pref), function ($query) use ($country_pref) {
                    return  $query->where('country_interested', 'like', "%$country_pref%");
                })
                ->when(!empty($study_level_filter), function ($query) use ($study_level_filter) {
                    return  $query->Where('overseas_study_level_selection', '=', "$study_level_filter");
                })
                ->when(!empty($course_interested_filter), function ($query) use ($course_interested_filter) {
                    return  $query->Where('overseas_course_intersted', '=', "$course_interested_filter");
                })
                ->when(!empty($service_pref), function ($query) use ($service_pref) {
                    return  $query->where('services', 'like', "%$service_pref%");
                })
                ->when(!empty($intatake_arr), function ($query) use ($intatake_arr) {
                    return $query->Where(function ($query) use ($intatake_arr) {
                        $query->where('user_details.intake_year', '=', $intatake_arr[1])
                            ->Where('user_details.intake_month', '=', $intatake_arr[0]);
                    });
                })
                ->where('is_lead', 0)
                ->where('archived', 0)
                ->count();

            $totalFiltered = $totalData;

        if (empty($request->input('search.value')) && empty($student_type) && empty($overseas_status) && empty($coaching_status)) {
            $users = $this->model
                ->with('roles', 'permissions', 'providers')
                ->active()
                ->leftjoin('countries', 'users.country', '=', 'countries.id')
                ->leftjoin('states', 'users.state', '=', 'states.id')
                ->leftjoin('branch', 'users.branch', '=', 'branch.id')
                ->leftjoin('user_details', 'users.id', '=', 'user_details.userid')
                ->select("users.*", "user_details.intake_year", "user_details.intake_month", "user_details.student_disqualified_reason", "countries.title_en as country_name", "states.name as state_name", "branch.branch_name as branch_name")
                ->whereHas('roles', function ($q) {
                    $q->whereIn('name', ['user']);
                })
                ->when(!empty($my_branch), function ($query) use ($my_branch) {
                    return $query->whereIn('users.branch', $my_branch);
                })
                ->when(\Auth::user()->isAgent(), function ($q) {
                    $q->where('agent_id', \Auth::user()->id);
                })
                ->when(!empty($tags_filter), function ($query) use ($tags_filter) {
                    return $query->Where(function ($query) use ($tags_filter) {
                        foreach ($tags_filter as $tag) {
                            $query->orWhere("tags", "like", "%" . $tag . "%");
                        }
                    });
                })
                ->when(!empty($user_country), function ($query) use ($user_country) {
                    return  $query->Where('users.country', '=', "$user_country");
                })
                ->when(!empty($assigned_to), function ($query) use ($assigned_to) {
                    return  $query->Where('assigned_to', '=', "$assigned_to");
                })
                ->when(!empty($my_user_id), function ($query) use ($my_user_id) {
                    return  $query->Where('assigned_to', '=', "$my_user_id");
                })
                ->when($exclude_agent_student, function ($query) {
                    return $query->where('agent_id', 0);
                })
                ->when($agentListOnly, function ($query) {
                    return $query->where('agent_id', '!=', 0);
                })
                ->when(!empty($before_student_age), function ($query) use ($before_student_age) {
                    return  $query->Where('student_created_at', '<=', "$before_student_age");
                    // return $query->whereRaw("IF(lead_converted_date IS NULL, created_at <= '$before_student_age', lead_converted_date <= '$before_student_age')");
                })
                ->when(!empty($followup_date_from), function ($query) use ($followup_date_from) {
                    return  $query->Where('next_followup', '>=', "$followup_date_from");
                })
                ->when(!empty($followup_date_to), function ($query) use ($followup_date_to) {
                    return  $query->Where('next_followup', '<=', "$followup_date_to");
                })
                ->when(!empty($overdue), function ($query) use ($overdue) {
                    return  $query->Where('next_followup', '<', "$overdue");
                })
                ->when(!empty($today), function ($query) use ($today) {
                    return  $query->Where('next_followup', 'like', "$today%");
                })
                ->when(!empty($tomorrow), function ($query) use ($tomorrow) {
                    return  $query->Where('next_followup', 'like', "$tomorrow%");
                })
                ->when(!empty($in_next_7_days), function ($query) use ($in_next_7_days) {
                    return $query->Where(function ($query) use ($in_next_7_days) {
                        $query->where('next_followup', '>=', date("Y-m-d"))
                            ->Where('next_followup', '<=', "$in_next_7_days");
                    });
                })
                ->when(!empty($created_date_from), function ($query) use ($created_date_from) {
                    return  $query->Where('student_created_at', '>=', "$created_date_from");
                })
                ->when(!empty($created_date_to), function ($query) use ($created_date_to) {
                    return  $query->Where('student_created_at', '<=', "$created_date_to");
                })
                ->when(!empty($country_pref), function ($query) use ($country_pref) {
                    return  $query->where('country_interested', 'like', "%$country_pref%");
                })
                ->when(!empty($study_level_filter), function ($query) use ($study_level_filter) {
                    return  $query->Where('overseas_study_level_selection', '=', "$study_level_filter");
                })
                ->when(!empty($course_interested_filter), function ($query) use ($course_interested_filter) {
                    return  $query->Where('overseas_course_intersted', '=', "$course_interested_filter");
                })
                ->when(!empty($service_pref), function ($query) use ($service_pref) {
                    return  $query->where('services', 'like', "%$service_pref%");
                })
                ->when(!empty($intatake_arr), function ($query) use ($intatake_arr) {
                    return $query->Where(function ($query) use ($intatake_arr) {
                        $query->where('user_details.intake_year', '=', $intatake_arr[1])
                            ->Where('user_details.intake_month', '=', $intatake_arr[0]);
                    });
                })
                ->where('is_lead', 0)
                ->where('archived', 0)
                ->offset($start)
                ->limit($limit)
                ->orderBy($order, $dir)
                ->get();

            // $queries = DB::getQueryLog();
            // \Log::debug($queries);

            $totalFiltered = $this->model
                ->with('roles', 'permissions', 'providers')
                ->active()
                ->leftjoin('countries', 'users.country', '=', 'countries.id')
                ->leftjoin('states', 'users.state', '=', 'states.id')
                ->leftjoin('branch', 'users.branch', '=', 'branch.id')
                ->leftjoin('user_details', 'users.id', '=', 'user_details.userid')
                ->select("users.*", "user_details.intake_year", "user_details.intake_month", "user_details.student_disqualified_reason", "countries.title_en as country_name", "states.name as state_name", "branch.branch_name as branch_name")
                ->whereHas('roles', function ($q) {
                    $q->whereIn('name', ['user']);
                })
                ->when(!empty($my_branch), function ($query) use ($my_branch) {
                    return $query->whereIn('users.branch', $my_branch);
                })
                ->when(\Auth::user()->isAgent(), function ($q) {
                    $q->where('agent_id', \Auth::user()->id);
                })
                ->when(!empty($tags_filter), function ($query) use ($tags_filter) {
                    return $query->Where(function ($query) use ($tags_filter) {
                        foreach ($tags_filter as $tag) {
                            $query->orWhere("tags", "like", "%" . $tag . "%");
                        }
                    });
                })
                ->when(!empty($user_country), function ($query) use ($user_country) {
                    return  $query->Where('users.country', '=', "$user_country");
                })
                ->when(!empty($assigned_to), function ($query) use ($assigned_to) {
                    return  $query->Where('assigned_to', '=', "$assigned_to");
                })
                ->when(!empty($my_user_id), function ($query) use ($my_user_id) {
                    return  $query->Where('assigned_to', '=', "$my_user_id");
                })
                ->when($exclude_agent_student, function ($query) {
                    return $query->where('agent_id', 0);
                })
                ->when($agentListOnly, function ($query) {
                    return $query->where('agent_id', '!=', 0);
                })
                ->when(!empty($before_student_age), function ($query) use ($before_student_age) {
                    return  $query->Where('student_created_at', '<=', "$before_student_age");
                    // return $query->whereRaw("IF(lead_converted_date IS NULL, created_at <= '$before_student_age', lead_converted_date <= '$before_student_age')");
                })
                ->when(!empty($followup_date_from), function ($query) use ($followup_date_from) {
                    return  $query->Where('next_followup', '>=', "$followup_date_from");
                })
                ->when(!empty($followup_date_to), function ($query) use ($followup_date_to) {
                    return  $query->Where('next_followup', '<=', "$followup_date_to");
                })
                ->when(!empty($overdue), function ($query) use ($overdue) {
                    return  $query->Where('next_followup', '<', "$overdue");
                })
                ->when(!empty($today), function ($query) use ($today) {
                    return  $query->Where('next_followup', 'like', "$today%");
                })
                ->when(!empty($tomorrow), function ($query) use ($tomorrow) {
                    return  $query->Where('next_followup', 'like', "$tomorrow%");
                })
                ->when(!empty($in_next_7_days), function ($query) use ($in_next_7_days) {
                    return $query->Where(function ($query) use ($in_next_7_days) {
                        $query->where('next_followup', '>=', date("Y-m-d"))
                            ->Where('next_followup', '<=', "$in_next_7_days");
                    });
                })
                ->when(!empty($created_date_from), function ($query) use ($created_date_from) {
                    return  $query->Where('student_created_at', '>=', "$created_date_from");
                })
                ->when(!empty($created_date_to), function ($query) use ($created_date_to) {
                    return  $query->Where('student_created_at', '<=', "$created_date_to");
                })
                ->when(!empty($country_pref), function ($query) use ($country_pref) {
                    return  $query->where('country_interested', 'like', "%$country_pref%");
                })
                ->when(!empty($study_level_filter), function ($query) use ($study_level_filter) {
                    return  $query->Where('overseas_study_level_selection', '=', "$study_level_filter");
                })
                ->when(!empty($course_interested_filter), function ($query) use ($course_interested_filter) {
                    return  $query->Where('overseas_course_intersted', '=', "$course_interested_filter");
                })
                ->when(!empty($service_pref), function ($query) use ($service_pref) {
                    return  $query->where('services', 'like', "%$service_pref%");
                })
                ->when(!empty($intatake_arr), function ($query) use ($intatake_arr) {
                    return $query->Where(function ($query) use ($intatake_arr) {
                        $query->where('user_details.intake_year', '=', $intatake_arr[1])
                            ->Where('user_details.intake_month', '=', $intatake_arr[0]);
                    });
                })
                ->where('is_lead', 0)
                ->where('archived', 0)
                ->count();

        } else {
            $get_search = $request->input('search.value');
            $users = $this->model
                ->with('roles', 'permissions', 'providers')
                ->active()
                ->leftjoin('countries', 'users.country', '=', 'countries.id')
                ->leftjoin('states', 'users.state', '=', 'states.id')
                ->leftjoin('branch', 'users.branch', '=', 'branch.id')
                ->leftjoin('user_details', 'users.id', '=', 'user_details.userid')
                ->select("users.*", "user_details.intake_year", "user_details.intake_month", "user_details.student_disqualified_reason", "countries.title_en as country_name", "states.name as state_name", "branch.branch_name as branch_name")
                ->where(function ($query) use ($get_search) {
                    $query->where('users.first_name', 'like', "%$get_search%")
                        ->orWhere('users.last_name', 'like', "%$get_search%")
                        ->orWhereRaw("CONCAT(`users`.`first_name`, ' ', `users`.`last_name`) LIKE ?", ['%'.$get_search.'%'])
                        ->orWhere('users.email', 'like', "%$get_search%")
                        ->orWhere('users.id', 'like', "%$get_search%")
                        ->orWhere('users.mobile', 'like', "%$get_search%")
                        ->orWhere('users.student_type', 'like', "%$get_search%");
                })
                ->whereHas('roles', function ($q) {
                    $q->whereIn('name', ['user']);
                })
                ->when(!empty($my_branch), function ($query) use ($my_branch) {
                    return $query->whereIn('users.branch', $my_branch);
                })
                ->when(\Auth::user()->isAgent(), function ($q) {
                    $q->where('agent_id', \Auth::user()->id);
                })
                ->when(!empty($tags_filter), function ($query) use ($tags_filter) {
                    return $query->Where(function ($query) use ($tags_filter) {
                        foreach ($tags_filter as $tag) {
                            $query->orWhere("tags", "like", "%" . $tag . "%");
                        }
                    });
                })
                ->when(!empty($user_country), function ($query) use ($user_country) {
                    return  $query->Where('users.country', '=', "$user_country");
                })
                ->when(!empty($student_type), function ($query) use ($student_type) {
                    return  $query->Where('student_type', '=', "$student_type");
                })            
                ->when(!empty($overseas_status), function ($query) use ($overseas_status) {
                    // if($overseas_status == 'Profile') {
                    //     return  $query->WhereIn('overseas_status', ['Profile','Course Preferences','Documents']);
                    // } elseif($overseas_status == 'Course Preferences') {
                    //     return  $query->WhereIn('overseas_status', ['Course Preferences','Documents']);
                    // } else {
                        return  $query->WhereIn('users.overseas_status', $overseas_status);
                    // }
                })
                ->when(!empty($coaching_status), function ($query) use ($coaching_status) {
                    return  $query->Where('coaching_status', '=', "$coaching_status");
                })
                ->when(!empty($assigned_to), function ($query) use ($assigned_to) {
                    return  $query->Where('assigned_to', '=', "$assigned_to");
                })
                ->when(!empty($my_user_id), function ($query) use ($my_user_id) {
                    return  $query->Where('assigned_to', '=', "$my_user_id");
                })
                ->when($exclude_agent_student, function ($query) {
                    return $query->where('agent_id', 0);
                })
                ->when($agentListOnly, function ($query) {
                    return $query->where('agent_id', '!=', 0);
                })
                ->when(!empty($before_student_age), function ($query) use ($before_student_age) {
                    return  $query->Where('student_created_at', '<=', "$before_student_age");
                    // return $query->whereRaw("IF(lead_converted_date IS NULL, created_at <= '$before_student_age', lead_converted_date <= '$before_student_age')");
                })
                ->when(!empty($followup_date_from), function ($query) use ($followup_date_from) {
                    return  $query->Where('next_followup', '>=', "$followup_date_from");
                })
                ->when(!empty($followup_date_to), function ($query) use ($followup_date_to) {
                    return  $query->Where('next_followup', '<=', "$followup_date_to");
                })
                ->when(!empty($overdue), function ($query) use ($overdue) {
                    return  $query->Where('next_followup', '<', "$overdue");
                })
                ->when(!empty($today), function ($query) use ($today) {
                    return  $query->Where('next_followup', 'like', "$today%");
                })
                ->when(!empty($tomorrow), function ($query) use ($tomorrow) {
                    return  $query->Where('next_followup', 'like', "$tomorrow%");
                })
                ->when(!empty($in_next_7_days), function ($query) use ($in_next_7_days) {
                    return $query->Where(function ($query) use ($in_next_7_days) {
                        $query->where('next_followup', '>=', date("Y-m-d"))
                            ->Where('next_followup', '<=', "$in_next_7_days");
                    });
                })
                ->when(!empty($created_date_from), function ($query) use ($created_date_from) {
                    return  $query->Where('student_created_at', '>=', "$created_date_from");
                })
                ->when(!empty($created_date_to), function ($query) use ($created_date_to) {
                    return  $query->Where('student_created_at', '<=', "$created_date_to");
                })
                ->when(!empty($country_pref), function ($query) use ($country_pref) {
                    return  $query->where('country_interested', 'like', "%$country_pref%");
                })
                ->when(!empty($study_level_filter), function ($query) use ($study_level_filter) {
                    return  $query->Where('overseas_study_level_selection', '=', "$study_level_filter");
                })
                ->when(!empty($course_interested_filter), function ($query) use ($course_interested_filter) {
                    return  $query->Where('overseas_course_intersted', '=', "$course_interested_filter");
                })
                ->when(!empty($service_pref), function ($query) use ($service_pref) {
                    return  $query->where('services', 'like', "%$service_pref%");
                })
                ->when(!empty($intatake_arr), function ($query) use ($intatake_arr) {
                    return $query->Where(function ($query) use ($intatake_arr) {
                        $query->where('user_details.intake_year', '=', $intatake_arr[1])
                            ->Where('user_details.intake_month', '=', $intatake_arr[0]);
                    });
                })
                ->where('is_lead', 0)
                ->where('archived', 0)
                ->offset($start)
                ->limit($limit)
                ->orderBy($order, $dir)
                ->get();

            // $queries = DB::getQueryLog();
            // \Log::debug($queries);

            $totalFiltered = $this->model
                ->with('roles', 'permissions', 'providers')
                ->active()
                ->leftjoin('countries', 'users.country', '=', 'countries.id')
                ->leftjoin('states', 'users.state', '=', 'states.id')
                ->leftjoin('branch', 'users.branch', '=', 'branch.id')
                ->leftjoin('user_details', 'users.id', '=', 'user_details.userid')
                ->select("users.*", "user_details.intake_year", "user_details.intake_month", "user_details.student_disqualified_reason", "countries.title_en as country_name", "states.name as state_name", "branch.branch_name as branch_name")
                ->where(function ($query) use ($get_search) {
                    $query->where('users.first_name', 'like', "%$get_search%")
                        ->orWhere('users.last_name', 'like', "%$get_search%")
                        ->orWhereRaw("CONCAT(`users`.`first_name`, ' ', `users`.`last_name`) LIKE ?", ['%'.$get_search.'%'])
                        ->orWhere('users.email', 'like', "%$get_search%")
                        ->orWhere('users.id', 'like', "%$get_search%")
                        ->orWhere('users.mobile', 'like', "%$get_search%")
                        ->orWhere('users.student_type', 'like', "%$get_search%");
                })
                ->whereHas('roles', function ($q) {
                    $q->whereIn('name', ['user']);
                })
                ->when(!empty($my_branch), function ($query) use ($my_branch) {
                    return $query->whereIn('users.branch', $my_branch);
                })
                ->when(\Auth::user()->isAgent(), function ($q) {
                    $q->where('agent_id', \Auth::user()->id);
                })
                ->when(!empty($tags_filter), function ($query) use ($tags_filter) {
                    return $query->Where(function ($query) use ($tags_filter) {
                        foreach ($tags_filter as $tag) {
                            $query->orWhere("tags", "like", "%" . $tag . "%");
                        }
                    });
                })
                ->when(!empty($user_country), function ($query) use ($user_country) {
                    return  $query->Where('users.country', '=', "$user_country");
                })
                ->when(!empty($student_type), function ($query) use ($student_type) {
                    return  $query->Where('student_type', '=', "$student_type");
                })                
                ->when(!empty($overseas_status), function ($query) use ($overseas_status) {
                    // if($overseas_status == 'Profile') {
                    //     return  $query->WhereIn('overseas_status', ['Profile','Course Preferences','Documents']);
                    // } elseif($overseas_status == 'Course Preferences') {
                    //     return  $query->WhereIn('overseas_status', ['Course Preferences','Documents']);
                    // } else {
                        return  $query->WhereIn('users.overseas_status', $overseas_status);
                    // }
                    
                })
                ->when(!empty($coaching_status), function ($query) use ($coaching_status) {
                    return  $query->Where('coaching_status', '=', "$coaching_status");
                })
                ->when(!empty($assigned_to), function ($query) use ($assigned_to) {
                    return  $query->Where('assigned_to', '=', "$assigned_to");
                })
                ->when(!empty($my_user_id), function ($query) use ($my_user_id) {
                    return  $query->Where('assigned_to', '=', "$my_user_id");
                })
                ->when($exclude_agent_student, function ($query) {
                    return $query->where('agent_id', 0);
                })
                ->when($agentListOnly, function ($query) {
                    return $query->where('agent_id', '!=', 0);
                })
                ->when(!empty($before_student_age), function ($query) use ($before_student_age) {
                    return  $query->Where('student_created_at', '<=', "$before_student_age");
                    // return $query->whereRaw("IF(lead_converted_date IS NULL, created_at <= '$before_student_age', lead_converted_date <= '$before_student_age')");
                })
                ->when(!empty($followup_date_from), function ($query) use ($followup_date_from) {
                    return  $query->Where('next_followup', '>=', "$followup_date_from");
                })
                ->when(!empty($followup_date_to), function ($query) use ($followup_date_to) {
                    return  $query->Where('next_followup', '<=', "$followup_date_to");
                })
                ->when(!empty($overdue), function ($query) use ($overdue) {
                    return  $query->Where('next_followup', '<', "$overdue");
                })
                ->when(!empty($today), function ($query) use ($today) {
                    return  $query->Where('next_followup', 'like', "$today%");
                })
                ->when(!empty($tomorrow), function ($query) use ($tomorrow) {
                    return  $query->Where('next_followup', 'like', "$tomorrow%");
                })
                ->when(!empty($in_next_7_days), function ($query) use ($in_next_7_days) {
                    return $query->Where(function ($query) use ($in_next_7_days) {
                        $query->where('next_followup', '>=', date("Y-m-d"))
                            ->Where('next_followup', '<=', "$in_next_7_days");
                    });
                })
                ->when(!empty($created_date_from), function ($query) use ($created_date_from) {
                    return  $query->Where('student_created_at', '>=', "$created_date_from");
                })
                ->when(!empty($created_date_to), function ($query) use ($created_date_to) {
                    return  $query->Where('student_created_at', '<=', "$created_date_to");
                })
                ->when(!empty($country_pref), function ($query) use ($country_pref) {
                    return  $query->where('country_interested', 'like', "%$country_pref%");
                })
                ->when(!empty($study_level_filter), function ($query) use ($study_level_filter) {
                    return  $query->Where('overseas_study_level_selection', '=', "$study_level_filter");
                })
                ->when(!empty($course_interested_filter), function ($query) use ($course_interested_filter) {
                    return  $query->Where('overseas_course_intersted', '=', "$course_interested_filter");
                })
                ->when(!empty($service_pref), function ($query) use ($service_pref) {
                    return  $query->where('services', 'like', "%$service_pref%");
                })
                ->when(!empty($intatake_arr), function ($query) use ($intatake_arr) {
                    return $query->Where(function ($query) use ($intatake_arr) {
                        $query->where('user_details.intake_year', '=', $intatake_arr[1])
                            ->Where('user_details.intake_month', '=', $intatake_arr[0]);
                    });
                })
                ->where('is_lead', 0)            
                ->where('archived', 0)
                ->count();
        }

        $data = array();
        $disqualified_student_status = get_disqualified_student_status();
        if (!empty($users)) {
            foreach ($users as $page) {
                $customResult['id'] = $page->id;
                $folow_text = $this->get_user_last_followup($page->id);
                // $customResult['name'] = "<a href='" . route('admin.auth.user.show', $page->id) . "'><strong><span style='white-space:nowrap; font-size: 16px !important;'>" . $page->first_name . " " . $page->last_name . "</span></strong></a>";
                
                $customResult['details'] = "<a href='" . route('admin.auth.user.show', $page->id) . "'><strong><span style='white-space:nowrap; font-size: 16px !important;'>" . $page->first_name . " " . $page->last_name . "</span></strong></a><br />";
                //Tags
                $customResult['details'] .= "<p class='text-sub m-0 mt-1'><i class='bi bi-tags-fill'></i> ";
                if($page->tags){
                    $tags_array = explode(',', $page->tags);
                    foreach($tags_array as $tags)
                    {
                        $customResult['details'] .="<span class='badge text-dark me-1 mb-1' title='".$tags."' style='cursor: pointer;background: #E9F8F4;'>".$tags."<span class='close text-secondary remove-tag ps-2' aria-hidden='true' id='".$page->id."' data-tag_name='".$tags."'> &times;</span></span>";
                    }
                }
                $customResult['details'] .=  "<a class='btn actionBtn add-tag-btn mb-0' style='font-size:12px;' data-id='". $page->id."' data-tooltip='Add' title='Add Tag' href='javascript:void(0)'><i class='fa fa-plus'></i></a>
                </p>";
                //Tags
                if($page->confirmed) {
                    $confirmed = " <i title='Confirmed' class=\"fas fa-check text-success\"></i>";
                } else {
                    $confirmed = " <i title='Not Confirmed' class=\"fas fa-ban text-muted\"></i>";
                }

                $customResult['details'] .= "<span style='white-space:nowrap;'><i class='fas fa-envelope text-sub'></i> <span class=''>" . $page->email . "</span>".$confirmed."</span><br />";
                if($page->mobile){
                    $customResult['details'] .= "<span style='white-space:nowrap;'><i class='fas fa-phone-alt text-sub'></i> <span class=''>" . ($page->dialcode ? "+".$page->dialcode."-" : '') . $page->mobile . "</span></span><br />";

                    //For Excel export
                    $customResult['mobile'] = ($page->dialcode ? "+".$page->dialcode."-" : '') . $page->mobile;
                    //For Excel export
                }else{
                    //For Excel export
                    $customResult['mobile'] = "";
                    //For Excel export 
                }

                // if($page->user_country) {
                //     $customResult['details'] .= "<i class='fas fa-globe text-sub'></i> <span class=''>".strtoupper($page->user_country->title_en)."</span>";
                // }

                if ($page->whatsapp) {
                    if ($page->whatsapp_dialcode != '') {
                        $whtsp = '+' . $page->whatsapp_dialcode . '-' . $page->whatsapp;
                    } else {
                        $whtsp = $page->whatsapp;
                    }
                    $customResult['details'] .= "<span style='white-space:nowrap;'><i class='fab fa-whatsapp'></i> <span class=''>" . $whtsp . "</span></span><br />";

                    //For Excel export
                    $customResult['whatsapp'] = $whtsp;
                    //For Excel export
                }else{
                    //For Excel export
                    $customResult['whatsapp'] = "";
                    //For Excel export
                }

                //For Excel export
                $customResult['export_name'] =  $page->first_name . " " . $page->last_name;
                $customResult['email'] = $page->email;
                $customResult['branch'] =  $page->branch_name;
                //For Excel export

                //For Excel export
                $customResult['location'] = "";
                //For Excel export
                if ($page->country || $page->city || $page->state) {
                    $customResult['details'] .= "<span style='white-space:nowrap;'><i class='fas fa-globe text-sub'></i> <span class=''>";
                    if ($page->city) {
                        $city = get_city_name($page->city);
                        $customResult['details'] .= $city . ", ";

                        //For Excel export
                        $customResult['location'] .= $city . ", ";
                        //For Excel export
                    }
                    if ($page->state) {
                        $customResult['details'] .= $page->state_name . ", ";

                        //For Excel export
                        $customResult['location'] .= $page->state_name.", ";
                        //For Excel export
                    }
                    if($page->country){
                        $customResult['details'] .= $page->country_name." ";

                        //For Excel export
                        $customResult['location'] .= $page->country_name." ";
                        //For Excel export
                    }
                }
                $customResult['details'] .= "</span></span>";
                $customResult['next_followup'] = $page->next_followup ? date('d M Y, g:ia', strtotime($page->next_followup)) : "--";

                $customResult['details'] .= '<div>';
                if(!\Auth::user()->isAgent()){
                    //$customResult['details'] .= '<br>';
                    if(get_site_config('outbound_call_enabled')){
                        if($page->mobile){
                            $customResult['details'] .= '<a target="_blank" class="btn actionBtn" title="Call" href="' . route('admin.communicate.phoneCall', $page->id) . '"><i class="bi bi-telephone-outbound-fill contactIcon"></i></a>';
                        }
                    }else{
                        if($page->mobile){
                            $customResult['details'] .= '<span class="btn actionBtnInactive" title="Outbound Call Not Enabled"><i class="bi bi-telephone-outbound-fill contactIcon"></i></span>';
                        }
                    }
                    if($page->mobile){
                        if(get_site_config('sms_enabled')){
                            $customResult['details'] .= '<a target="_blank" class="btn actionBtn" title="Send SMS" href="' . route('admin.communicate.sms', $page->id) . '"><i class="fa fa-comment-dots contactIcon"></i></a>';
                        } else {
                            $customResult['details'] .= '<span class="btn actionBtnInactive" title="SMS Not Enabled"><i class="fa fa-comment-dots contactIcon"></i></span>';
                        }
                    }

                    if($page->email){
                        $customResult['details'] .= '<a target="_blank" class="btn actionBtn" title="Send Email" href="' . route('admin.communicate.mail', $page->id) . '"><i class="fas fa-envelope contactIcon"></i></a>';
                    }
                    if(get_site_config('whatsapp_enabled')){
                        $whatsapp_link = '<a target="_blank" class="btn actionBtn" title="Send Whatsapp" href="' . route('admin.communicate.whatsapp', $page->id) . '"><i class="fab fa-whatsapp contactIcon"></i></a>';
                    } else {
                        $whatsapp_link = '<a target="_blank" class="btn actionBtn" title="Open Whatsapp Web" href="https://wa.me/' . $page->dialcode . $page->mobile . '"><i class="fab fa-whatsapp contactIcon"></i></a>';
                    }
                    if($page->whatsapp){
                        $customResult['details'] .= $whatsapp_link;
                    }
                    if($folow_text) {
                        $customResult['details'] .= "<div class='edu-tooltip px-2 py-2' style='border-radius: 8px; width: 300px !important; text-wrap: wrap; background-color: #EBEBEB; margin: 0px 0px 5px 0px !important;'><span class='pt-2' style='font-size: 13px !important;font-style: italic'>".substr($folow_text, 0, 110) ."<span style='font-size: 20px;'>.....</span>
                        <span class='edu-tooltiptext'>".$folow_text."</span>
                        </span></div>";
                    } else {
                        $customResult['details'] .= "<div></div>";
                    }
                }
                $customResult['details'] .= '</div>';

                

                // Calculating data for tags
                $tagCount = 1;
                $today_date = time();
                $created_at_date = strtotime($page->student_created_at);

                $total_days_sec = (abs($today_date - $created_at_date));
                $lead_age = round($total_days_sec / (60 * 60 * 24));
                $customResult['details'] .= "<div style='width: 300px !important; display: flex; flex-wrap: wrap;'><span class='tagSuccess mb-1'>Profile Age: " . $lead_age . " Days</span>";

                //appl count
                $appl_count = DB::table('university_application')->where('userid', $page->id)->count();
                if($appl_count){
                    $customResult['details'] .= "<span><a href='".url('admin/auth/user/'.$page->id.'#application') ."' class='tagPrimary py-1'>Applications: ".$appl_count." </a></span>";
                }

                $services_count = DB::table('service_applications')->where('user_id', $page->id)->count();
                if($services_count){
                    $customResult['details'] .= "<span><a href='".url('admin/auth/user/'.$page->id.'#services') ."' class='tagPurple py-1'>Visa & Services: ".$services_count." </a></span>";
                }

                // $customResult['details'] .= "<p style='margin: 0px 0px 3px 0px !important;padding: 0px !important;'>";
                if($page->country_interested){
                    $country_interested = explode(",",$page->country_interested);
                    foreach($country_interested as $country_interest){
                        $customResult['details'] .= "<span class='tagOrange mb-1'>" . $country_interest . "</span>";
                    }
                }

                if($page->services){
                    $services_interested = explode(",",$page->services);
                    $customResult['details'] .= get_all_services($services_interested);
                }
                if($page->overseas_study_level_selection){
                    $customResult['details'] .= "<span class='tagPurple mb-1'>" . $page->overseas_study_level_selection . "</span>";
                }                
		        if($page->overseas_course_intersted) {
                    $customResult['details'] .= "<span class='tagPrimary mb-1'>" . ucwords($page->overseas_course_intersted) . "</span>";
                }
		        if($page->intake_year && $page->intake_month) {
                    $customResult['details'] .= "<span class='tagOrange mb-1'>" . $page->intake_year . " " . date("M", mktime(0, 0, 0, $page->intake_month, 10)) . "</span>";
                }
		        if($page->branch && !\Auth::user()->isAgent()) {
                    $customResult['details'] .= "<span class='tagPink mb-1'>" . $page->branch_name . "</span>";
                }

                $customResult['details'] .= "</div>";

                if(!\Auth::user()->isAgent()){
                    if($page->agent_id) {
                        $customResult['details'] .= "<p style='padding: 0px !important; margin: 0px !important;'><i title='Partner/Agent' class='fas fa-user-tie text-sub'></i> <span style='white-space:nowrap;' title='Partner/Agent'>" . get_user_full_name_by_id($page->agent_id) . "</span></p>";
                    }
                }

                $customResult['status'] = "";

                if ($page->student_type == 'study_abroad') {
                    $customResult['student_type'] = '<span class="badge badge-primary">Study Abroad</span>';
                } elseif ($page->student_type == 'course') {
                    $customResult['student_type'] = '<span class="badge badge-default">Coaching</span>';
                } elseif ($page->student_type == 'study_abroad_course') {
                    $customResult['student_type'] = '<span class="badge badge-primary">Study Abroad</span><br /> <span class="badge badge-default">Coaching</span>';
                } elseif ($page->student_type == 'visa') {
                    $customResult['student_type'] = '<span class="badge badge-warning">Visa</span>';
                } else {
                    $customResult['student_type'] = '';
                }

                $overseas_status = $course_status = "";

                $customResult['status'] = '<span class="badge p-1" style="font-size: 14px!important;font-weight: 700; color:'.get_overseas_status_color($page->overseas_status).';">'.$page->overseas_status.'</span>'; 
                
                if($page->student_disqualified_reason && $page->overseas_status == $disqualified_student_status){
                    $customResult['status'] .= '<p class="mt-1" style="font-size: 11px">' . $page->student_disqualified_reason . '</p>';
                }

                $customResult['assigned_to_name'] = "<div style='display: flex; justify-content: space-between;'>";
                $customResult['assigned_to_name'] .= $page->assigned_to ? get_user_full_name_by_id($page->assigned_to) : "<span style='color: red;'>Unassigned</span>";
                if (\Auth::user()->isAdmin() || \Auth::user()->can('student-assignment')) {
                    // $customResult['assigned_to_name'] .= '<a target="_blank" style="padding:0;font-size:12px;float:right;" href="' . route('admin.auth.user.assignStudent', $page->id) . '" data-toggle="tooltip" data-placement="right" title="Assign Student" class="btn"><i class="fas fa-user-edit"></i></a>';
                    $customResult['assigned_to_name'] .= '<button style="font-size:12px;" data-id="' . $page->id . '" class="assigned_to_staff btn actionBtn" data-toggle="tooltip" data-placement="right" title="Change Staff"><i class="fas fa-user-edit"></i></button>';
                }
                $customResult['assigned_to_name'] .= "</div>";

                $customResult['created_at'] = date('d M Y', strtotime($page->student_created_at));

                $customResult['options'] = $page->action_buttons;
                $data[] = $customResult;
            }
        }

        $json_data = array(
            "draw"            => intval($request->input('draw')),
            "recordsTotal"    => intval($totalData),
            "recordsFiltered" => intval($totalFiltered),
            "data"            => $data,
            "branch" =>     $my_branch
        );

        return json_encode($json_data);
    }

    public function get_user_last_followup($user_id){
        $last_followup = DB::table('user_activity')
            ->select('user_activity.*','users.id','users.student_created_at')
            ->join('users','users.id','=','user_activity.user_id')
            ->where("user_id", $user_id)
            ->where("type", "followup")
            ->latest()
            ->first();

        if($last_followup){
            $followup = DB::table('user_activity')
                ->select('user_activity.*','users.id','users.student_created_at')
                ->join('users','users.id','=','user_activity.user_id')
                // ->where("user_activity.created_at", ">", "users.student_created_at")
                ->where("user_activity.created_at", ">", $last_followup->student_created_at)
                ->where("user_activity.user_id", $user_id)
                ->where("user_activity.type", "followup")
                ->latest()
                ->first();

            if ($followup) {
                return date('d M, g:i A', strtotime($followup->created_at)) . ": " . $followup->detail;
            } else {
                return 0;
            }
        }

        return 0;

    }



    /**
     * @param int    $paged
     * @param string $orderBy
     * @param string $sort
     *
     * @return mixed
     */
    // public function getActivePaginated($paged = 25, $orderBy = 'created_at', $sort = 'desc'): LengthAwarePaginator
    // {
    //     return $this->model
    //         ->with('roles', 'permissions', 'providers')
    //         ->active()
    //         ->whereHas('roles', function ($q) {
    //             $q->whereIn('name', ['user']);
    //         })
    //         ->orderBy($orderBy, $sort)
    //         ->paginate($paged);
    // }

    /**
     * @param int    $paged
     * @param string $orderBy
     * @param string $sort
     *
     * @return LengthAwarePaginator
     */
    public function getInactivePaginated($paged = 25, $orderBy = 'created_at', $sort = 'desc'): LengthAwarePaginator
    {
        return $this->model
            ->with('roles', 'permissions', 'providers')
            ->active(false)
            ->whereHas('roles', function ($q) {
                $q->whereIn('name', ['user']);
            })
            ->when(\Auth::user()->isAgent(), function ($q) {
                $q->where('agent_id', \Auth::user()->id);
            })
            ->orderBy($orderBy, $sort)
            ->paginate($paged);
    }

    /**
     * @param int    $paged
     * @param string $orderBy
     * @param string $sort
     *
     * @return LengthAwarePaginator
     */
    public function getDeletedPaginated($paged = 25, $orderBy = 'deleted_at', $sort = 'desc'): LengthAwarePaginator
    {
        return $this->model
            ->with('roles', 'permissions', 'providers')
            ->onlyTrashed()
            ->where('removed', "0")
            ->where('archived', "0")
            ->whereHas('roles', function ($q) {
                $q->whereIn('name', ['user']);
            })
            ->when(\Auth::user()->isAgent(), function ($q) {
                $q->where('agent_id', \Auth::user()->id);
            })
            ->orderBy($orderBy, $sort)
            ->paginate($paged);
    }

    // public function getFilteredPaginated($find, $paged = 25, $orderBy = 'created_at', $sort = 'desc'): LengthAwarePaginator
    // {
    //     return $this->model
    //         ->with('roles', 'permissions', 'providers')
    //         ->active()
    //         ->where(function ($query) use ($find) {
    //             $query->where('first_name', 'like', "%$find%")
    //                 ->orWhere('last_name', 'like', "%$find%")
    //                 ->orWhere('email', '=', $find)
    //                 ->orWhere('mobile', '=', $find);
    //         })
    //         ->whereHas('roles', function ($q) {
    //             $q->whereIn('name', ['user']);
    //         })
    //         ->orderBy($orderBy, $sort)
    //         ->paginate($paged);
    // }

    ////san
    //public function getStaffPaginated($paged = 25, $orderBy = 'created_at', $sort = 'desc'): LengthAwarePaginator {
    //    return $this->model
    //                    ->with('roles', 'permissions', 'providers')
    //                    ->active()
    //                    ->whereHas('roles', function($q) {
    //                        $q->whereIn('name', ['administrator','manager','team member','trainer']);
    //                    })
    //                    ->orderBy($orderBy, $sort)
    //                    ->paginate($paged);
    //}
    //
    //
    ////san
    //public function getStaffFilteredPaginated($find, $paged = 25, $orderBy = 'created_at', $sort = 'desc'): LengthAwarePaginator {
    //    return $this->model
    //                    ->with('roles', 'permissions', 'providers')
    //                    ->active()
    //                    ->where('first_name', 'like', "%$find%")
    //                    ->orWhere('last_name', 'like', "%$find%")
    //                    ->orWhere('email', '=', $find)
    //                    ->orWhere('mobile', '=', $find)
    //                    ->whereHas('roles', function($q) {
    //                        $q->whereIn('name', ['administrator','manager','team member','trainer']);
    //                    })                
    //                    ->orderBy($orderBy, $sort)
    //                    ->paginate($paged);
    //}


    public function getTrainers()
    {
        return $this->model
            ->with('roles', 'permissions', 'providers')
            ->active()
            ->whereHas('roles', function ($q) {
                $q->whereIn('name', ['trainer']);
            })
            ->orderBy("first_name", "ASC")
            ->get();
    }

    public function getTrainersWhatsapp()
    {
        return $this->model
            ->with('roles', 'permissions', 'providers')
            ->active()
            ->whereHas('roles', function ($q) {
                $q->whereIn('name', ['trainer']);
            })
            ->where('whatsapp',"!=",'')
            ->orderBy("first_name", "ASC")
            ->get();
    }

    public function getAgents()
    {
        return $this->model
            ->with('roles', 'permissions', 'providers')
            ->active()
            ->whereHas('roles', function ($q) {
                $q->whereIn('name', ['agent']);
            })
            ->orderBy("first_name", "ASC")
            ->get();
    }

    public function getAgentsWhatsapp()
    {
        return $this->model
            ->with('roles', 'permissions', 'providers')
            ->active()
            ->whereHas('roles', function ($q) {
                $q->whereNotIn('name', ['agent']);
            })
            ->where('whatsapp',"!=",'')
            ->orderBy("first_name", "ASC")
            ->get();
    }

    public function getStaff()
    {
        return $this->model
            ->with('roles', 'permissions', 'providers')
            ->active()
            ->whereHas('roles', function ($q) {
                $q->whereNotIn('name', ['user', 'trainer', 'agent']);
            })
            ->orderBy("first_name", "ASC")
            ->get();
    }

    public function getStaffWhatsapp()
    {
        return $this->model
            ->with('roles', 'permissions', 'providers')
            ->active()
            ->whereHas('roles', function ($q) {
                $q->whereNotIn('name', ['user', 'trainer', 'agent']);
            })
            ->where('whatsapp',"!=",'')
            ->orderBy("first_name", "ASC")
            ->get();
    }


    public function getStudents()
    {
        return $this->model
            ->with('roles', 'permissions', 'providers')
            ->active()
            ->whereHas('roles', function ($q) {
                $q->whereIn('name', ['user']);
            })
            ->orderBy("id", "DESC")
            ->get();
    }

    public function getStudentsWhatsapp()
    {
        return $this->model
            ->with('roles', 'permissions', 'providers')
            ->active()
            ->whereHas('roles', function ($q) {
                $q->whereIn('name', ['user']);
            })
            ->where('whatsapp',"!=",'')
            ->orderBy("id", "DESC")
            ->get();
    }

    public function getNonStudentsTrainers()
    {
        return $this->model
            ->with('roles', 'permissions', 'providers')
            ->active()
            ->whereHas('roles', function ($q) {
                $q->whereNotIn('name', ['user', 'trainer', 'agent']);
            })
            ->orderBy("first_name", "ASC")
            ->get();
    }

    /**
     * @param array $data
     *
     * @return User
     * @throws \Exception
     * @throws \Throwable
     */
    public function create(array $data): User{
        
        if (!isset($data['title']))
            $data['title'] = NULL;

        if (!isset($data['middle_name']))
            $data['middle_name'] = '';

        if (!isset($data['student_type']))
            $data['student_type'] = NULL;

        if (!isset($data['gender']))
            $data['gender'] = 'male';

        // $country_interested = '';
        // if (isset($data['country_interested']) && count($data['country_interested'])) {
        //     foreach ($data['country_interested'] as $country) {
        //         $country_interested .= $country . ',';
        //     }
        //     $country_interested = rtrim($country_interested, ',');
        // }

        $country_interested = '';
        if (isset($data['country_interested']) && count($data['country_interested'])) {
            $country_interested = implode(",", $data['country_interested']);
        }
        $data['country_interested'] = $country_interested;

        $service_interested = NULL;
        $service_pref = "";
        if (isset($data['services']))
            $service_pref = $data['services'];
        if (!empty($service_pref) && count($service_pref)) {
            $service_interested = implode(",", $data['services']);
        }
        $data['service_interested'] = $service_interested;

        if (!isset($data['address_1']))
            $data['address_1'] = null;

        if (!isset($data['mobile']))
            $data['mobile'] = null;

        if (!isset($data['whatsapp_number']))
            $data['whatsapp_number'] = null;
            
        if (!isset($data['whatsapp_number']))
            $data['whatsapp_dialcode'] = null;

        if (!isset($data['alternate_contact']))
            $data['alternate_contact'] = null;

        if (!isset($data['city']))
            $data['city'] = null;

        if (!isset($data['overseas_study_level']))
            $data['overseas_study_level'] = null;

        if (!isset($data['overseas_study_level_selection']))
            $data['overseas_study_level_selection'] = null;

        if (!isset($data['course_interested']))
            $data['course_interested'] = null;

        if (!isset($data['intake_year']))
            $data['intake_year'] = null;

        if (!isset($data['intake_month']))
            $data['intake_month'] = null;

        if (!isset($data['qualification']))
            $data['qualification'] = null;

        if (!isset($data['passout_year']))
            $data['passout_year'] = null;

        if (!isset($data['qualification_score']))
            $data['qualification_score'] = null;

        if (!isset($data['current_course']))
            $data['current_course'] = null;

        if (!isset($data['work_experience']))
            $data['work_experience'] = null;

        if (!isset($data['lead_source']))
            $data['lead_source'] = null;

        if (!isset($data['lead_source_detail']))
            $data['lead_source_detail'] = null;

        if (!isset($data['campaign_name']))
            $data['campaign_name'] = null;

        if (!isset($data['branch'])){
            if(\Auth::user()->isAgent()){
                $data['branch'] = \Auth::user()->branch;  
            } else{
                $data['branch'] = 1;
            }
        }

        if (!isset($data['assigned_to']))
            $data['assigned_to'] = 0;

        if (!isset($data['roles']))
            $data['roles'] = ["user"];

        if (!isset($data['birthday']) || $data['birthday'] == ""){
            $data['birthday'] = NULL;
        } else {
            $data['birthday'] = date('Y-m-d',strtotime($data['birthday']));
        }

        if (!isset($data['country']))
            $data['country'] = 0;

        if (\Auth::user()->isAgent()) {
            $data['branch'] = \Auth::user()->branch;
            $data['agent_id'] = \Auth::user()->id;
            $data['password'] = Str::random(10);//Agent added students will not login to site, set some random pswd
        }
        elseif(isset($data['agent_id']) && $data['agent_id'] != "") {
            $data['agent_id'] = $data['agent_id'];
        } else{
            $data['agent_id'] = 0;
        }
        
        //checking created mode
        if (\Auth::user()->isAgent()) {
            $data['created_mode'] = "agent";
        }
        else if(isset($data['created_mode'])){
            $data['created_mode'] = $data['created_mode'];
        }else{
            $data['created_mode'] = "staff";
        }
        //checking created mode
        // if (!isset($data['lead_status'])){
        //     $data['lead_status'] = '';
        // }
        // else{
        //     if($data['lead_status'] =='lead') {
        //         $data['lead_status'] = "Lead";
        //     }else{
        //         $data['lead_status'] = "Pending";
        //     }
        // }  
        if (isset($data['study_level']) && $data['study_level'] !="")
            $data['overseas_study_level'] = get_study_level_degree_level($data['study_level']);
        else
            $data['overseas_study_level'] = NULL;

        if (isset($data['intake']) && $data['intake']!="") {
            $intake = $data['intake'];
            $intatake_arr =  explode(" ", $intake);
        } else {
            $intatake_arr = array(null, null);
        }
        $data['intake'] = $intatake_arr;
        
        $data['lead_status'] = get_default_student_status();          
        $data['refcode'] = generate_refcode($data['first_name']);

        if(isset($data['referred_by']) && $data['referred_by']!="") {
            $data['referred_by'] = $data['referred_by'];
        } else {
            $data['referred_by'] = 0;
        }
        return DB::transaction(function () use ($data) {
            $user = parent::create([
                'first_name' => $data['first_name'],
                'middle_name' => $data['middle_name'],
                'last_name' => $data['last_name'],
                'gender' => $data['gender'],
                'email' => $data['email'],
                'birthday' => $data['birthday'],
                'title' => $data['title'],
                'dialcode' => $data['dialcode'],
                'mobile' => $data['mobile'],
                'whatsapp_dialcode' => $data['whatsapp_dialcode'],
                'whatsapp' => $data['whatsapp_number'],
                'alternate_contact' => $data['alternate_contact'],
                'country' => $data['country'],
                'state' => $data['state'],
                'city' => $data['city'],
                'country_interested' => $data['country_interested'],
                'overseas_study_level' => $data['overseas_study_level'],
                'overseas_study_level_selection' => $data['study_level'],
                'overseas_course_intersted' => $data['course_interested'],
                'password' => $data['password'],
                'lead_source' => $data['lead_source'],
                'branch' => $data['branch'],
                'assigned_to' => $data['assigned_to'],
                'created_mode' => $data['created_mode'],
                'overseas_status' => $data['lead_status'],
                'student_type' => $data['student_type'],
                'address_1' => $data['address_1'],
                'services' => $data['service_interested'],
                'ref_by' => $data['referred_by'],
                'refcode' => $data['refcode'],
                'agent_id' => $data['agent_id'],
                'active' => isset($data['active']) && $data['active'] == '1' ? 1 : 0,
                'confirmation_code' => md5(uniqid(mt_rand(), true)),
                'confirmed' => isset($data['confirmed']) && $data['confirmed'] == '1' ? 1 : 0,

                'student_created_at' => now(),

            ]);

            $userPassword = $data['password'];

            // See if adding any additional permissions
            // if (!isset($data['permissions']) || !count($data['permissions'])) {
            //     $data['permissions'] = [];
            // }

            if ($user) {
                // User must have at least one role
                if (!count($data['roles'])) {
                    throw new GeneralException(__('exceptions.backend.access.users.role_needed_create'));
                }
                
                $custom_attributes = DB::table('custom_attributes')->where('for_student', 1)->orderBy('display_order','asc')->get(["custom_attributes.*"]);
                $attributes_json = "";
                if(count($custom_attributes)){
                    $all_atts = array();
                    foreach($custom_attributes as $attrs){
                        $name = $attrs->attr_name;
                        $all_atts[$attrs->attr_name] = (isset($data[$name]))? $data[$name] : null;
                    }
                $attributes_json = json_encode($all_atts);
                }
                
                // echo "<pre>";
                // print_r($data);
                // exit();

                $userdetails = new UserDetails([
                    'userid'    => $user->id,
                    'intake_year' => $data['intake'][1],
                    'intake_month' => $data['intake'][0],
                    'highest_level_education' => $data['qualification'],
                    'passout_year' => $data['passout_year'],
                    'user_data_js' => $attributes_json,
                    'qualification_score' => $data['qualification_score'],
                    'current_course' => $data['currently_studying_course'],
                    'work_experience' => $data['work_experience'],
                    'lead_source_detail' => $data['lead_source_detail'],
                    'lead_campaign_name' => $data['campaign_name'],
                ]);
                $userdetails->save();

                //Create lead
                // if($data['lead_status']=='Lead') {
                //     $user->is_lead = 1;
                //     $user->lead_status = get_default_lead_status();
                //     $user->save();
                // }

                $followup = new UserActivity;
                $followup->user_id = $user->id;
                $followup->type = 'others';
                $followup->staff = \Auth::user()->id;
                $followup->detail = "Staff Created Student";
                $followup->save();

                // Add selected roles/permissions
                $user->syncRoles($data['roles']);
                //$user->syncPermissions($data['permissions']);

                //Send confirmation email if requested and account approval is off
                if (isset($data['confirmation_email']) && $user->confirmed == 0 && !config('access.users.requires_approval')) {
                    try{
                        //$user->notify(new UserNeedsConfirmation($user->confirmation_code));

                        $email_message = __('strings.emails.auth.click_to_confirm');
                        $email_message .= new HtmlString("<a style='font-family:Avenir,Helvetica,sans-serif;box-sizing:border-box;border-radius:3px;color:#fff;display:inline-block;text-decoration:none;background-color:#3097d1;border-top:10px solid #3097d1;border-right:18px solid #3097d1;border-bottom:10px solid #3097d1;border-left:18px solid #3097d1' href='".route('frontend.auth.account.confirm', $user->confirmation_code)."'>Confirm Account</a>");
                        user_notify($user, app_name().': '.__('exceptions.frontend.auth.confirmation.confirm'), $email_message);
                    }catch (\Exception $e) {
                        $message = $e->getMessage();
                        \Log::debug("Mail Error: (Confirmation Email) ".$message);   
                    }
                }

                event(new UserCreated($user));
                //Notification
                if(!\Auth::user()->isAgent()){
                    //Student added by staff
                    $mail_tpl = MailTemplates::find("STUDENT_CREATE_WELCOME");
                    if($mail_tpl && $mail_tpl->status){
                        $email_message = $mail_tpl->message;                        
                        $email_message = str_ireplace("#sitename#", app_name(), $email_message);
                        $email_message = str_ireplace("#siteurl#", url(''), $email_message);
                        $email_message = str_ireplace("#first_name#", $user->first_name, $email_message);
                        $email_message = str_ireplace("#full_name#", $user->full_name, $email_message);
                        $email_message = str_ireplace("#email#", $user->email, $email_message);
                        $email_message = str_ireplace("#password#", $userPassword, $email_message);
                        try{
                            //$user->notify(new UserAlert($mail_tpl->subject, $email_message));
                            user_notify($user,$mail_tpl->subject, $email_message);
                        }catch (\Exception $e) {
                            $message = $e->getMessage();
                            \Log::debug("Mail Error: (New User - Notify User) ".$message);   
                        }
                    }
                    $notify_sms = get_sms_template('STUDENT_CREATE');
                    if ($notify_sms && $notify_sms->status) {
                        $notify_sms->message = str_replace("#sitename#", app_name(), $notify_sms->message);
                        $notify_sms->message = str_replace("#siteurl#", url('/'), $notify_sms->message);
                        $notify_sms->message = str_replace("#first_name#", $user->first_name, $notify_sms->message);
                        $notify_sms->message = str_replace("#full_name#", $user->full_name, $notify_sms->message);
                        $notify_sms->message = str_replace("#email#", $user->email, $notify_sms->message);
                        $notify_sms->message = str_replace("#password#", $userPassword, $notify_sms->message);

                        $user->sms($notify_sms->message, $notify_sms->template_id);
                    }

                    //Whatsapp
                    if (get_site_config('whatsapp_enabled')) {
                        $numbers = [$user->whatsapp_dialcode.$user->whatsapp];

                        $hook = 'student_create';

                        $customMessage = [
                            'email'  => $user->email,
                            'password'  => $userPassword,
                        ];

                        send_whatsapp_message($numbers, $hook, $user->id, $customMessage);
                    }
                    /*
                    $notify_wtsp = get_whatsapp_template('STUDENT_CREATE');
                    if ($notify_wtsp && $notify_wtsp->status) {
                        $notify_wtsp->message = str_replace("#sitename#", app_name(), $notify_wtsp->message);
                        $notify_wtsp->message = str_replace("#siteurl#", url('/'), $notify_wtsp->message);
                        $notify_wtsp->message = str_replace("#first_name#", $user->first_name, $notify_wtsp->message);
                        $notify_wtsp->message = str_replace("#full_name#", $user->full_name, $notify_wtsp->message);
                        $notify_wtsp->message = str_replace("#email#", $user->email, $notify_wtsp->message);
                        $notify_wtsp->message = str_replace("#password#", $userPassword, $notify_wtsp->message);
                        $user->whatsapp($notify_wtsp->message,$notify_wtsp->hook);
                    }
                    */

                }

                return $user;
            }

            throw new GeneralException(__('exceptions.backend.access.users.create_error'));
        });
    }

    /**
     * @param User  $user
     * @param array $data
     *
     * @return User
     * @throws GeneralException
     * @throws \Exception
     * @throws \Throwable
     */
    public function update(User $user, array $data): User
    {
        $this->checkUserByEmail($user, $data['email']);

        // See if adding any additional permissions
        if (!isset($data['permissions']) || !count($data['permissions'])) {
            $data['permissions'] = [];
        }

        return DB::transaction(function () use ($user, $data) {

            if (!isset($data['title']))
                $data['title'] = 'NULL';

            if (!isset($data['middle_name']))
                $data['middle_name'] = '';
    
            // if (!isset($data['student_type']))
            //     $data['student_type'] = NULL;
    
            if (!isset($data['gender']))
                $data['gender'] = 'male';

            $country_interested = '';
            if (isset($data['country_interested']) && count($data['country_interested'])) {
                $country_interested = implode(",", $data['country_interested']);
            }
            $data['country_interested'] = $country_interested;

            $service_interested = NULL;
            $service_pref = "";
            if (isset($data['services']))
                $service_pref = $data['services'];
            if (!empty($service_pref) && count($service_pref)) 
                $service_interested = implode(",", $data['services']);
            $data['service_interested'] = $service_interested;

            if (!isset($data['mobile']))
            $data['mobile'] = null;

            if (!isset($data['whatsapp_number']))
                $data['whatsapp_number'] = null;

            if (!isset($data['whatsapp_number']))
                $data['whatsapp_dialcode'] = null;

            if (!isset($data['alternate_contact']))
                $data['alternate_contact'] = null;

            if (!isset($data['city']))
                $data['city'] = null;

            if (isset($data['study_level'])){
                $data['overseas_study_level'] = get_study_level_degree_level($data['study_level']);
            }else{
                $data['overseas_study_level'] = NULL;
            }

            if (!isset($data['course_interested']))
                $data['course_interested'] = null;

            if (!isset($data['intake_year']))
                $data['intake_year'] = null;

            if (!isset($data['intake_month']))
                $data['intake_month'] = null;

            if (!isset($data['qualification']))
                $data['qualification'] = null;

            if (!isset($data['passout_year']))
                $data['passout_year'] = null;

            if (!isset($data['qualification_score']))
                $data['qualification_score'] = null;

            if (!isset($data['current_course']))
                $data['current_course'] = null;

            if (!isset($data['work_experience']))
                $data['work_experience'] = null;

            if (!isset($data['lead_source']))
                $data['lead_source'] = null;

            if (!isset($data['lead_source_detail']))
                $data['lead_source_detail'] = null;

            if (!isset($data['campaign_name']))
                $data['campaign_name'] = null;

            if (!isset($data['branch']))
                $data['branch'] = 1; 
            
            if (!isset($data['birthday']) || $data['birthday'] == ""){
                $data['birthday'] = NULL;
            } else {
                $data['birthday'] = date('Y-m-d',strtotime($data['birthday']));
            }
            
            if (!isset($data['address_1']))
                $data['address_1'] = null;
            
            if (isset($data['intake'])) {
                $intake = $data['intake'];
                $intatake_arr =  explode(" ", $intake);
            } else {
                $intatake_arr = array(null, null);
            }
            $data['intake'] = $intatake_arr;

            if(isset($data['referred_by']) && $data['referred_by']!="") {
                $data['referred_by'] = $data['referred_by'];
            } else {
                $data['referred_by'] = 0;
            }

            $update_data = [
                'first_name' => $data['first_name'],
                'middle_name' => $data['middle_name'],
                'last_name' => $data['last_name'],
                'gender' => $data['gender'],
                'email' => $data['email'],
                'birthday' => $data['birthday'],
                'dialcode' => $data['dialcode'],
                'mobile' => $data['mobile'],
                'whatsapp' => $data['whatsapp_number'],
                'whatsapp_dialcode' => $data['whatsapp_dialcode'],
                'alternate_contact' => $data['alternate_contact'],
                'country' => $data['country'],
                'state' => $data['state'],
                'city' => $data['city'],
                'country_interested' => $data['country_interested'],
                'overseas_study_level' => $data['overseas_study_level'],
                'overseas_study_level_selection' => $data['study_level'],
                'overseas_course_intersted' => $data['course_interested'],
                'lead_source' => $data['lead_source'],
                'branch' => $data['branch'],
                'ref_by' => $data['referred_by'],
                'services' => $service_interested,

                // 'overseas_status' => $data['lead_status'],
                // 'student_type' => $data['student_type'],
                'address_1' => $data['address_1'],
            ];

            if ($user->update($update_data)) {
                $custom_attributes = DB::table('custom_attributes')->where('for_student', 1)->orderBy('display_order','asc')->get(["custom_attributes.*"]);
                $attributes_json = "";
                if(count($custom_attributes)){
                    $all_atts = array();
                    foreach($custom_attributes as $attrs){
                        $name = $attrs->attr_name;
                        $all_atts[$attrs->attr_name] = (isset($data[$name]))? $data[$name] : null;
                    }
                $attributes_json = json_encode($all_atts);
                }
                $user_details = UserDetails::where('userid', $user->id)->first();
                $user_details->intake_year = $data['intake'][1];
                $user_details->intake_month = $data['intake'][0];
                $user_details->highest_level_education = $data['qualification'];
                $user_details->passout_year = $data['passout_year'];
                $user_details->user_data_js = $attributes_json;
                $user_details->qualification_score = $data['qualification_score'];
                $user_details->current_course = $data['currently_studying_course'];
                $user_details->work_experience = $data['work_experience'];
                $user_details->lead_source_detail = $data['lead_source_detail'];
                $user_details->lead_campaign_name = $data['campaign_name'];
                $user_details->save();

                // Add selected roles/permissions
                if (isset($data['roles']) && count($data['roles'])) {
                    $user->syncRoles($data['roles']);
                }
                $user->syncPermissions($data['permissions']);

                event(new UserUpdated($user));

                // Send Push App Notification
                if(\Auth::user()->isAgent()){
                    $message = _label('backend_student')." <strong>".$user->full_name."</strong> (".$user->id.") updated by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
                    $email_message = "Hi, <br><br>"._label('backend_student')." Updated by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
                    $subject = _label('backend_student').": " . $user->full_name." (". $user->id. ") Updated by Agent: " . get_user_full_name_by_id(\Auth::user()->id);
                } else{
                    $message = _label('backend_student')." <strong>".$user->full_name."</strong> (".$user->id.") updated by staff";
                    $email_message = "Hi, <br><br>"._label('backend_student')." Updated by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
                    $subject = _label('backend_student').": " . $user->full_name." (". $user->id. ") Updated by Staff: " . get_user_full_name_by_id(\Auth::user()->id);
                }

                //Sending email Notification
                $email_message_part = "<br />Mobile:$user->mobile<br /><br />Email: $user->email <br/><br/><a href=\"" . url("admin/auth/user/" . $user->id) . "\">View "._label('backend_student')." Details</a>";
                $email_message .= "<br><br>"._label('backend_student').": " . $user->full_name . " (".$user->id.") Updated by Staff: " . get_user_full_name_by_id(\Auth::user()->id)."<br />" . $email_message_part;
                send_mail_notification($user->id,"","STUDENT_PROFILE_UPDATED",$subject,$email_message);

                send_in_app_notification(
                    $user->id, 
                    $user->id,
                    "STUDENT_PROFILE_UPDATED", 
                    $subject, 
                    $message,
                    "student"
                );

                return $user;
            }

            throw new GeneralException(__('exceptions.backend.access.users.update_error'));
        });
    }

    /**
     * @param User $user
     * @param      $input
     *
     * @return User
     * @throws GeneralException
     */
    public function updatePassword(User $user, $input): User
    {
        if(env('DEMO_MODE') && in_array($user->id, config('app.demo_users')))
            return redirect()->route('admin.demo-user-restriction');    

        if ($user->id == 1 && auth()->id() != 1) {
            throw new GeneralException('This operation is not allowed');
        }

        if ($user->update(['password' => $input['password']])) {
            event(new UserPasswordChanged($user));
            //send new password to user
            $student = new UserActivity;
            $student->user_id = $user->id;
            $student->title = "User Password Changed.";
            $student->type = 'admin_event';
            $student->staff = \Auth::user()->id;
            $student->detail = "User Password Changed" . get_user_full_name_by_id(\Auth::user()->id);
            $student->save();


            // Send Whatsapp
            /* 
            if (get_site_config('whatsapp_enabled')) {
                $numbers = [$user->whatsapp_dialcode.$user->whatsapp];

                $hook = 'student_password';

                $customMessage = [
                    'email'     => $user->email,
                    'password'  => $input['password']
                ];

                send_whatsapp_message($numbers, $hook, $user->id, $customMessage);
            }
            */

            if ($user->email) {
                if ($user->agent_id == 0) {
                    $subject = "Password Reset";
                    $email_message = "Hi,<br>Your password has been reset by the support team.<br>Here is your new password: ".$input['password'];
                    try {
                        //Mail::to($user->email)->send(new MailSend($subject, $email_message));
                        user_notify($user->email,$subject, $email_message);
                        \Log::debug("Password reset mail Successfully Sent to: ".$user->email." Password: ".$input['password']);
                    } catch (\Exception $e) {
                        $err_message = $e->getMessage();
                        \Log::debug("Mail Error: (Password reset mail) ".$err_message);   
                    }
                }else{
                    \Log::debug("Password reset mail cannot be sent for Agent's user: ".$user->id);
                }
            }else{
                \Log::debug("Mail Error: (Password reset mail): Email not found for user: ".$user->id);   
            }
            return $user;
        }

        throw new GeneralException(__('exceptions.backend.access.users.update_password_error'));
    }




    /**
     * @param      $input
     * @param bool $expired
     *
     * @return bool
     * @throws GeneralException
     */
    public function updateSelfPassword($input, $expired = false)
    {
        $user = $this->getById(auth()->id());

        if (Hash::check($input['old_password'], $user->password)) {
            if ($expired) {
                $user->password_changed_at = Carbon::now()->toDateTimeString();
            }

            return $user->update(['password' => $input['password']]);
        }

        throw new GeneralException(__('exceptions.frontend.auth.password.change_mismatch'));
    }






    /**
     * @param User $user
     * @param      $status
     *
     * @return User
     * @throws GeneralException
     */
    public function mark(User $user, $status): User
    {

        if(env('DEMO_MODE') && in_array($user->id, config('app.demo_users')))
            return redirect()->route('admin.demo-user-restriction');    


        if (auth()->id() == $user->id && $status == 0) {
            throw new GeneralException(__('exceptions.backend.access.users.cant_deactivate_self'));
        }

        if ($user->id == 1) {
            throw new GeneralException('This operation is not allowed');
        }

        $user->active = $status;

        switch ($status) {
            case 0:
                event(new UserDeactivated($user));
                break;

            case 1:
                event(new UserReactivated($user));
                break;
        }

        if ($user->save()) {
            return $user;
        }

        throw new GeneralException(__('exceptions.backend.access.users.mark_error'));
    }

     public function marklock(User $user, $status): User
    {
    //    dd("table value:",$user->user_unlock , $status);
        if(env('DEMO_MODE') && in_array($user->id, config('app.demo_users')))
            return redirect()->route('admin.demo-user-restriction');    


        if (auth()->id() == $user->id && $status == 0) {
            throw new GeneralException(__('exceptions.backend.access.users.cant_deactivate_self'));
        }

        if ($user->id == 1) {
            throw new GeneralException('This operation is not allowed');
        }
        $user->user_unlock = $status;
    
        switch ($status) {
            case 0:
                event(new UserUnlocked($user));
                
                break;

            case 1:
                event(new UserLocked($user));
                
                break;
        }
        // dd("table value:",$user->user_unlock , $status);
        if ($user->save()) {
            return $user;
        }

        throw new GeneralException(__('exceptions.backend.access.users.mark_error'));
    }

    /**
     * @param User $user
     *
     * @return User
     * @throws GeneralException
     */
    public function confirm(User $user): User
    {
        if ($user->confirmed) {
            throw new GeneralException(__('exceptions.backend.access.users.already_confirmed'));
        }

        $user->confirmed = 1;
        $confirmed = $user->save();

        if ($confirmed) {
            event(new UserConfirmed($user));

            // Let user know their account was approved
            if (config('access.users.requires_approval')) {
                try{
                    //$user->notify(new UserAccountActive);
                    $email_message = __('strings.emails.auth.account_confirmed');
                    $email_message .= "<a href='".route('frontend.auth.login')."'>".__('labels.frontend.auth.login_button')."</a>";
                    $email_message .= __('strings.emails.auth.thank_you_for_using_app');
                    user_notify($user,app_name(), $email_message);
                }catch (\Exception $e) {
                    $message = $e->getMessage();
                    \Log::debug("Mail Error: (User Confirmation) ".$message);   
                }
            }

            return $user;
        }

        throw new GeneralException(__('exceptions.backend.access.users.cant_confirm'));
    }

    /**
     * @param User $user
     *
     * @return User
     * @throws GeneralException
     */
    public function unconfirm(User $user): User
    {
        if (!$user->confirmed) {
            throw new GeneralException(__('exceptions.backend.access.users.not_confirmed'));
        }

        if ($user->id == 1) {
            // Cant un-confirm admin
            throw new GeneralException(__('exceptions.backend.access.users.cant_unconfirm_admin'));
        }

        if ($user->id == auth()->id()) {
            // Cant un-confirm self
            throw new GeneralException(__('exceptions.backend.access.users.cant_unconfirm_self'));
        }

        $user->confirmed = 0;
        $unconfirmed = $user->save();

        if ($unconfirmed) {
            event(new UserUnconfirmed($user));

            return $user;
        }

        throw new GeneralException(__('exceptions.backend.access.users.cant_unconfirm'));
    }

    /**
     * @param User $user
     *
     * @return User
     * @throws GeneralException
     * @throws \Exception
     * @throws \Throwable
     */
    public function forceDelete(User $user): User
    {
        if (is_null($user->deleted_at)) {
            throw new GeneralException(__('exceptions.backend.access.users.delete_first'));
        }

        return DB::transaction(function () use ($user) {
            // Delete associated relationships
            $user->passwordHistories()->delete();
            $user->providers()->delete();
            $user->sessions()->delete();

            if ($user->forceDelete()) {
                event(new UserPermanentlyDeleted($user));

                return $user;
            }

            throw new GeneralException(__('exceptions.backend.access.users.delete_error'));
        });
    }

    /**
     * @param User $user
     *
     * @return User
     * @throws GeneralException
     */
    public function restore(User $user): User
    {
        if (is_null($user->deleted_at) && $user->removed != 0) {
            throw new GeneralException(__('exceptions.backend.access.users.cant_restore'));
        }

        if ($user->restore()) {
            event(new UserRestored($user));

            return $user;
        }

        throw new GeneralException(__('exceptions.backend.access.users.restore_error'));
    }

    

    /**
     * @param User $user
     * @param      $email
     *
     * @throws GeneralException
     */
    protected function checkUserByEmail(User $user, $email)
    {
        //Figure out if email is not the same
        if ($user->email != $email) {
            //Check to see if email exists
            if ($this->model->where('email', '=', $email)->first()) {
                throw new GeneralException(trans('exceptions.backend.access.users.email_error'));
            }
        }
    }

    // public function updateBasicInfo($id, array $input)
    // {
    //     $user = $this->getById($id);
    //     $user->first_name = $input['first_name'];
    //     $user->last_name = $input['last_name'];
    //     // $user->title = $input['title'];
    //     $user->email = $input['email'];
    //     $user->mobile = $input['mobile'];
    //     $user->address_1 = $input['address_1'];

    //     return $user->save();
    // }


     public function updateBasicInfo($request,$id, array $input)
    {
         // Retrieve user by ID
        $user = $this->getById($id);
        //dd($request->all());
    
        $request->validate([
        'birthday' => 'required|date_format:d-m-Y',
        'gender'=>'required',
        'emergency_contact_ph_no'=>'nullable',
        'work_phone_no'=>'nullable',    
        'passport_issue_date' => 'nullable|date_format:d-m-Y',
        'passport_expiry_date' => 'nullable|date_format:d-m-Y',
        'visa_expiry_date' => 'nullable|date_format:d-m-Y', 
        'cos_issue_date' => 'nullable|date_format:d-m-Y',
        'cos_expiry_date' => 'nullable|date_format:d-m-Y',
        'rwc_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:8192',
        'emp_id' => 'nullable',
        'designation' => 'nullable',

        ]);
        
    
        

        // Update user fields
        $user->first_name = $input['first_name'] ?? $user->first_name;
        $user->last_name = $input['last_name'] ?? $user->last_name;
      //  $user->preferred_name = $input['preferred_name'] ?? $user->preferred_name;
        $user->email = $input['email'] ?? $user->email;
        $user->whatsapp = $input['whatsapp'] ?? $user->whatsapp;
        $user->mobile = $input['mobile'] ?? $user->mobile;
        $user->dialcode = $input['country_code'] ?? $user->dialcode;
         $user->whatsapp_dialcode = $input['country_code'] ?? $user->whatsapp_dialcode;

        $user->gender = $input['gender'] ?? $user->gender;
        $user->birthday = \Carbon\Carbon::createFromFormat('d-m-Y', $request->birthday)->format('Y-m-d');
        $user->address_2 = $input['address_2'] ?? $user->address_2;
        $user->address_1 = $input['address_1'] ?? $user->address_1;
        $user->country= $input['country'] ?? $user->country;
        $user->state = $input['state'] ?? $user->state;
        $user->city = $input['city'] ?? $user->city;
        $user->post_code = $input['postal_code'] ?? $user->post_code;
    
        

        // Retrieve or create staff details
        $staff_details = StaffDetails::where('staff_id', $id)->first();
        \Log::info([$staff_details]);
        //dd($staff_details->passport_issue_date , $staff_details->passport_expiry_date ,$staff_details->visa_expiry_date);
        // Check if staff details exist, if not create a new one
        if (!$staff_details) {
            $staff_details = new StaffDetails();
            $staff_details->staff_id = $id;
        }

        // Update staff details
        $staff_details->gender = $input['gender'] ?? $staff_details->gender;
        $staff_details->bank_account_no = $input['account_number'] ?? $staff_details->bank_account_no;
        $staff_details->bank_payee = $input['account_holder_name'] ?? $staff_details->bank_payee;
        $staff_details->bank_bank_name = $input['bank_name'] ?? $staff_details->bank_bank_name;
        $staff_details->bank_ifsc =$input['ifsc_code'] ?? $staff_details->bank_ifsc;
        $staff_details->bank_branch =$input['branch_name'] ?? $staff_details->bank_branch;


        $staff_details->personal_email = $input['personal_email'] ?? $staff_details->personal_email;
        $staff_details->personal_phone = $input['personal_phone'] ?? $staff_details->personal_phone;
        $staff_details->extension = $input['extension'] ?? $staff_details->extension;
        $staff_details->work_phone_no =$input['work_phone_no'] ?? $staff_details->work_phone_no;
        $staff_details->country = $input['country'] ?? $staff_details->country;
        $staff_details->aadhar_no= $input['aadhar_no'] ?? $staff_details->aadhar_no;
        $staff_details->state = $input['state'] ?? $staff_details->state;
        $staff_details->city = $input['city'] ?? $staff_details->city;
        $staff_details->postal_code = $input['postal_code'] ?? $staff_details->postal_code;
        $staff_details->entry_clearance = $input['entry_clearance'] ?? $staff_details->entry_clearance;
        $staff_details->dob = \Carbon\Carbon::createFromFormat('d-m-Y', $request->birthday)->format('Y-m-d');
        $staff_details->passport_issue_date = !empty($request->passport_issue_date) ? \Carbon\Carbon::createFromFormat('d-m-Y', $request->passport_issue_date)->format('Y-m-d') : null;
        $staff_details->passport_expiry_date = $request->passport_expiry_date ? \Carbon\Carbon::createFromFormat('d-m-Y', $request->passport_expiry_date)->format('Y-m-d') : null;
        $staff_details->modified_by= $id ?? \Auth::user()->id;
        
        $staff_details->visa_expiry_date = $request->visa_expiry_date ? \Carbon\Carbon::createFromFormat('d-m-Y', $request->visa_expiry_date)->format('Y-m-d') : null;
        
        $staff_details->cos_issue_date = $request->cos_issue_date ? \Carbon\Carbon::createFromFormat('d-m-Y', $request->cos_issue_date)->format('Y-m-d') : null;
        
        $staff_details->cos_expiry_date = $request->cos_expiry_date ? \Carbon\Carbon::createFromFormat('d-m-Y', $request->cos_expiry_date)->format('Y-m-d') : null;
        if (!empty($request->input('share_code_entry_date'))) {
            $share_code_entry_date = substr($request->input('share_code_entry_date'), 6, 4) . "-" . substr($request->input('share_code_entry_date'), 3, 2) . "-" . substr($request->input('share_code_entry_date'), 0, 2);
        } else {
            $share_code_entry_date= NULL;
        }
        if (!empty($request->input('share_code_expiry_date'))) {
            $share_code_expiry_date = substr($request->input('share_code_expiry_date'), 6, 4) . "-" . substr($request->input('share_code_expiry_date'), 3, 2) . "-" . substr($request->input('share_code_expiry_date'), 0, 2);
        } else {
            $share_code_expiry_date = NULL;
        }

    
        
        // dd( $staff_details->emp_start_date);

        if (!empty($request->emp_start_date)) {
            try {
                $staff_details->emp_start_date = \Carbon\Carbon::createFromFormat('d-m-Y', trim($request->emp_start_date))->format('Y-m-d');
            } catch (\Exception $e) {
                return back()->withErrors(['emp_start_date' => 'Invalid date format. Please enter in dd-mm-yyyy format.']);
            }
        } else {
            $staff_details->emp_start_date = null;
        }

        if (!empty($request->emp_end_date)) {
            try {
                $staff_details->emp_end_date = \Carbon\Carbon::createFromFormat('d-m-Y', trim($request->emp_end_date))->format('Y-m-d');
            } catch (\Exception $e) {
                return back()->withErrors(['emp_end_date' => 'Invalid date format. Please enter in dd-mm-yyyy format.']);
            }
        } else {
            $staff_details->emp_end_date = null;
        }
        if (!empty($request->rwc_check_date)) {
            try {
                $staff_details->rwc_check_date = \Carbon\Carbon::createFromFormat('d-m-Y', trim($request->rwc_check_date))->format('Y-m-d');
            } catch (\Exception $e) {
                return back()->withErrors(['rwc_check_date' => 'Invalid date format. Please enter in dd-mm-yyyy format.']);
            }
        } else {
            $staff_details->rwc_check_date = null;
        }
    
        
        //dd($staff_details->passport_issue_date,$staff_details->visa_expiry_date,$staff_details->passport_expiry_date ,  $staff_details->visa_expiry_date , $staff_details->cos_issue_date ,   $staff_details->cos_expiry_date);
        $staff_details->emp_id =$input['employee_ID'] ?? $staff_details->emp_id;
        $staff_details->reporting_to = $input['reporting_to'] ?? $staff_details->reporting_to;
        $staff_details->pan_no = $input['pan_number'] ?? $staff_details->pan_no;
        
        $staff_details->designation = $input['employee_designation'] ?? $staff_details->designation;
        $staff_details->department = $input['department'] ?? $staff_details->department;
        $staff_details->emp_type =$input['employee_type'] ?? $staff_details->emp_type;
        $staff_details->onboarding_status = $input['onboarding_status'] ?? $staff_details->onboarding_status;
        $staff_details->emp_status = $input['emp_status'] ?? $staff_details->emp_status;
        $staff_details->hiring_source = $input['hiring_source'] ?? $staff_details->hiring_source;
        $staff_details->aadhar_no = $input['aadhar_number'] ?? $staff_details->aadhar_no;
        $staff_details->ctc= $input['current_ctc'] ?? $staff_details->ctc;
    
        $staff_details->permanent_postal_code = $input['permanent_postal_code'] ?? $staff_details->permanent_postal_code;
        $staff_details->permanent_country =$input['country-current'] ?? $staff_details->permanent_country;
        $staff_details->permanent_state = $input['state-current'] ?? $staff_details->permanent_state;
        $staff_details->permanent_city = $input['permanent_city'] ?? $staff_details->permanent_city;
        $staff_details->permanent_postal_code = $input['permanent_postal_code'] ?? $staff_details->permanent_postal_code;
        $staff_details->marital_status = $input['marital_status']?? $staff_details->marital_status;
        $staff_details->father_name = $input['father_name']?? $staff_details->father_name;
        $staff_details->mother_name = $input['mother_name']?? $staff_details->mother_name;
        $staff_details->blood_group = $input['blood_group']?? $staff_details->blood_group;
        
        $staff_details->past_experience_year = $input['past_experience_year']?? $staff_details->past_experience_year;
        $staff_details->past_experience_month = $input['past_experience_month']?? $staff_details->past_experience_month;
        $staff_details->about= $input['about']?? $staff_details->about;
        $staff_details->passport_no = $input['passport_no']?? $staff_details->passport_no;
        //dd($staff_details->cos_no);
        $staff_details->passport_issue_country = $input['passport_issue_country']?? $staff_details->passport_issue_country;
        $staff_details->visa_no = $input['visa_no']?? $staff_details->visa_no;
        $staff_details->brp= $input['brp']?? $staff_details->brp; 
        $staff_details->cos_no = $input['cos_no']?? $staff_details->cos_no;
        $staff_details->cos_status = $input['cos_status']?? $staff_details->cos_status;
        $staff_details->share_code_right_work = $input['share_code_right_work']??$staff_details->share_code_right_work;
        $staff_details->share_code_entry_date = $share_code_entry_date;
        $staff_details->share_code_expiry_date=$share_code_expiry_date;
        // $staff_details->rwc_check_date =$rwc_check_date;

    
        // dd($staff_details->passport_issue_date , $staff_details->passport_expiry_date ,$staff_details->visa_expiry_date);
    
    
        $staff_details->emergency_contact_name = $input['emergency_contact_name']?? $staff_details->emergency_contact_name;
        $staff_details->emergency_contact_email = $input['emergency_contact_email']?? $staff_details->emergency_contact_email;
        $staff_details->emergency_contact_ph_no = $input['emergency_contact_ph_no']?? $staff_details->emergency_contact_ph_no;
        $staff_details->emergency_contact_address = $input['emergency_contact_address']?? $staff_details->emergency_contact_address;
        $staff_details->seating_location = $input['seating_location']?? $staff_details->seating_location;
        
    
        $request->validate([
            'photo'  =>'mimes:jpeg,jpg,png|max:2048',
            

        ]);

        // if ($request->hasFile('rwc_file')) {
        //     $file = $request->file('rwc_file');
        //     $filename = 'rwc_file' . time() . '.' . $file->getClientOriginalExtension();
        //     $path = $file->storeAs('uploads/staff_profiles', $filename, 'public');
        
        //     $staff_Details->rwc_file = $path;
            
        // }
        if ($request->hasFile('rwc_file')) {
            $file = $request->file('rwc_file');
            $filename = 'rwc_file' . time() . '.' . $file->getClientOriginalExtension();
            $path = $file->storeAs('/staff_doc', $filename, 'public');
        
            $staff_details->rwc_file = $path; // Ensure $staff_details is properly initialized
        
        }

        // Handle file uploads
        if ($request->hasFile('photo')) {
            // Delete old photo 
            // if it exists
            if (!empty($staff_details->photo) && \Storage::disk('public')->exists($staff_details->photo)) {
                \Storage::disk('public')->delete($staff_details->photo);
            }
        
            // Store new photo
            $file = $request->file('photo');
            $originalFileName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
            $extension = strtolower($file->getClientOriginalExtension()); 


            $filename = Str::slug($originalFileName) . '.' . $extension; 
            $path = $file->storeAs('uploads/staff_profiles', $filename, 'public');
        
            // Save new photo path
            $staff_details->photo = 'uploads/staff_profiles/' . $filename;
        }
        

        $staff_details->save();
        $user->save();
    

        return true; // Indicates success
    }

    public function eductrlDelete(User $user): User{
        if (is_null($user->deleted_at) && $user->removed != 0) {
            throw new GeneralException(__('exceptions.backend.access.users.delete_first'));
        }

        $id = $user->id;
        if($id){
            // Deleting all record from all table wherever this user id is used

            $course_preference = DB::table('course_preference')->where('userid', $id)->delete();
            $course_recommendation = DB::table('course_recommendation')->where('userid', $id)->delete();
            $course_shortlisted = DB::table('course_shortlisted')->where('student_id', $id)->delete();

            $invoice = DB::table('invoice')->where('student_id', $id)->delete();

            $orders = DB::table('orders')->where('user_id', $id)->delete();

            $password_histories = DB::table('password_histories')->where('user_id', $id)->delete();

            $payment_gateway_trans = DB::table('payment_gateway_trans')->where('user_id', $id)->delete();

            $service_applications = DB::table('service_applications')->where('user_id', $id)->delete();
            $service_application_messages = DB::table('service_application_messages')->where('user_id', $id)->get();
            if(count($service_application_messages)){
                foreach($service_application_messages as $application_message){
                    \File::delete(public_path("storage/".$application_message->attachment_1));
                    \File::delete(public_path("storage/".$application_message->attachment_2));
                    \File::delete(public_path("storage/".$application_message->attachment_3));
                    \File::delete(public_path("storage/".$application_message->attachment_4));
                }
                $service_application_messages = DB::table('service_application_messages')->where('user_id', $id)->delete();
            }

            $social_accounts = DB::table('social_accounts')->where('user_id', $id)->delete();

            $student_course_certificate = DB::table('student_course_certificate')->where('student_id', $id)->delete();
            $student_exams = DB::table('student_exams')->where('student_id', $id)->delete();
            $student_exam_result = DB::table('student_exam_result')->where('student_id', $id)->delete();
            $student_marks = DB::table('student_marks')->where('student_id', $id)->delete();
            $student_online_course = DB::table('student_online_course')->where('student_id', $id)->delete();
            $student_quiz_answer = DB::table('student_quiz_answer')->where('student_id', $id)->delete();
            $student_quiz_result = DB::table('student_quiz_result')->where('student_id', $id)->delete();
            $student_webinar = DB::table('student_webinar')->where('student_id', $id)->delete();

            $tickets = DB::table('tickets')->where('userid', $id)->delete();

            $application_messages = DB::table('application_messages')->where('user_id', $id)->get();
            if(count($application_messages)){
                foreach($application_messages as $application_message){
                    \File::delete(public_path("storage/".$application_message->attachment_1));
                    \File::delete(public_path("storage/".$application_message->attachment_2));
                    \File::delete(public_path("storage/".$application_message->attachment_3));
                    \File::delete(public_path("storage/".$application_message->attachment_4));
                }
                $application_messages = DB::table('application_messages')->where('user_id', $id)->delete();;
            }

            $university_application = DB::table('university_application')->where('userid', $id)->delete();
            $user_academics = DB::table('user_academics')->where('userid', $id)->delete();
            $user_activity = DB::table('user_activity')->where('user_id', $id)->delete();
            $user_certification_eng = DB::table('user_certification_eng')->where('userid', $id)->delete();
            $user_certification_lang = DB::table('user_certification_lang')->where('userid', $id)->delete();
            $user_courses = DB::table('user_courses')->where('userid', $id)->delete();
            // $user_course_preference = DB::table('user_course_preference')->where('userid', $id)->delete();
            $user_democlass_permission = DB::table('user_democlass_permission')->where('userid', $id)->delete();
            $user_details = DB::table('user_details')->where('userid', $id)->delete();

            $user_docs_other = DB::table('user_docs_other')->where('userid', $id)->get();;
            if(count($user_docs_other)){
                foreach($user_docs_other as $user_docs){
                    \File::delete(public_path("storage/".$user_docs->file));
                }
                $user_docs_other = DB::table('user_docs_other')->where('userid', $id)->delete();;
            }

            $user_documents = DB::table('user_documents')->where('userid', $id)->get();
            if(count($user_documents)){
                foreach($user_documents as $user_document){
                    \File::delete(public_path("storage/".$user_document->class10));
                    \File::delete(public_path("storage/".$user_document->class10_marks_sheet));
                    \File::delete(public_path("storage/".$user_document->class12));
                    \File::delete(public_path("storage/".$user_document->class12_marks_sheet));
                    \File::delete(public_path("storage/".$user_document->diploma));
                    \File::delete(public_path("storage/".$user_document->diploma_marks_sheet));
                    \File::delete(public_path("storage/".$user_document->grad_all_sem));
                    \File::delete(public_path("storage/".$user_document->grad_degree));
                    \File::delete(public_path("storage/".$user_document->cmm));
                    \File::delete(public_path("storage/".$user_document->grad_provisional));
                    \File::delete(public_path("storage/".$user_document->grad_transcripts));
                    \File::delete(public_path("storage/".$user_document->pg_all_sem));
                    \File::delete(public_path("storage/".$user_document->pg_degree));
                    \File::delete(public_path("storage/".$user_document->pg_provisional));
                    \File::delete(public_path("storage/".$user_document->pg_transcripts));
                    \File::delete(public_path("storage/".$user_document->lang_ielts));
                    \File::delete(public_path("storage/".$user_document->lang_pte));
                    \File::delete(public_path("storage/".$user_document->gre));
                    \File::delete(public_path("storage/".$user_document->gmat));
                    \File::delete(public_path("storage/".$user_document->toefl));
                    \File::delete(public_path("storage/".$user_document->duolingo));
                }
                $user_documents = DB::table('user_documents')->where('userid', $id)->delete();
            }

            $user_experience = DB::table('user_experience')->where('userid', $id)->delete();
            $user_folder_permission = DB::table('user_folder_permission')->where('userid', $id)->delete();
            $user_gre_gmat = DB::table('user_gre_gmat')->where('userid', $id)->delete();
            $user_internship = DB::table('user_internship')->where('userid', $id)->delete();
            $user_notification = DB::table('user_notification')->where('user_id', $id)->delete();
            $user_shipping_addresses = DB::table('user_shipping_addresses')->where('user_id', $id)->delete();
            // $user_tags = DB::table('user_tags')->where('userid', $id)->delete();
            // $visa_preference = DB::table('visa_preference')->where('userid', $id)->delete();
        }

        return DB::transaction(function () use ($user) {
            // Delete associated relationships
            $user->passwordHistories()->delete();
            $user->providers()->delete();
            $user->sessions()->delete();

            $user->removed = 1;
            $user->email = 'deleted_' . time() . '_' . $user->email;
            $user->first_name = 'DELETED';
            $user->middle_name = '';
            $user->last_name = '';
            $user->mobile = '';
            \File::delete(public_path("storage/".$user->avatar_location));
            $user->avatar_location = NULL;

            if ($user->save()) {
                event(new UserPermanentlyDeleted($user));
                return $user;
            }

            throw new GeneralException(__('exceptions.backend.access.users.delete_error'));
        });
    }

    public function eductrlDeleteLead(User $user): User{
        if (is_null($user->deleted_at) && $user->removed != 0) {
            throw new GeneralException(__('exceptions.backend.access.users.delete_first_lead'));
        }
        return DB::transaction(function () use ($user) {
            // Delete associated relationships
            $user->passwordHistories()->delete();
            $user->providers()->delete();
            $user->sessions()->delete();

            $user->removed = 1;
            $user->email = 'deleted_' . time() . '_' . $user->email;
            $user->first_name = 'DELETED';
            $user->middle_name = '';
            $user->last_name = '';
            $user->mobile = '';

            if ($user->save()) {
                event(new UserPermanentlyDeleted($user));
                return $user;
            }

            throw new GeneralException(__('exceptions.backend.access.users.delete_error_lead'));
        });
    }
}
