<?php

namespace App\Repositories\Frontend\Auth;

use Carbon\Carbon;
use App\Models\Auth\User;
use Illuminate\Http\UploadedFile;
use App\Models\Auth\SocialAccount;
use Illuminate\Support\Facades\DB;
use App\Exceptions\GeneralException;
use App\Repositories\BaseRepository;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use App\Events\Frontend\Auth\UserConfirmed;
use App\Events\Frontend\Auth\UserProviderRegistered;
use App\Notifications\Frontend\Auth\UserNeedsConfirmation;
use App\Models\Auth\UserDetails;
use App\Models\MailTemplates;
use App\Notifications\Backend\UserAlert;
use Illuminate\Support\Str;
use App\Models\StaffBranch;
use Illuminate\Support\HtmlString;
/**
 * Class UserRepository.
 */
class UserRepository extends BaseRepository
{
    /**
     * @return string
     */
    public function model()
    {
        return User::class;
    }

    /**
     * @param $token
     *
     * @return bool|\Illuminate\Database\Eloquent\Model
     */
    public function findByPasswordResetToken($token)
    {
        foreach (DB::table(config('auth.passwords.users.table'))->get() as $row) {
            if (password_verify($token, $row->token)) {
                return $this->getByColumn($row->email, 'email');
            }
        }

        return false;
    }

    /**
     * @param $uuid
     *
     * @return mixed
     * @throws GeneralException
     */
    public function findByUuid($uuid)
    {
        $user = $this->model
            ->uuid($uuid)
            ->first();

        if ($user instanceof $this->model) {
            return $user;
        }

        throw new GeneralException(__('exceptions.backend.access.users.not_found'));
    }

    /**
     * @param $code
     *
     * @return mixed
     * @throws GeneralException
     */
    public function findByConfirmationCode($code)
    {
        $user = $this->model
            ->where('confirmation_code', $code)
            ->first();

        if ($user instanceof $this->model) {
            return $user;
        }

        throw new GeneralException(__('exceptions.backend.access.users.not_found'));
    }

    /**
     * @param array $data
     *
     * @return \Illuminate\Database\Eloquent\Model|mixed
     * @throws \Exception
     * @throws \Throwable
     */
    public function create(array $data)
    {
        if (isset($data['full_number']) || $data['full_number'] == "") {
            $phoneNumberSize = strlen($data['mobile']);
            $dialcode = substr($data['full_number'], 0, -$phoneNumberSize);
            $dialcodeRemovePlus = substr($dialcode, 1);
            $data['dialcode'] = $dialcodeRemovePlus;
        }
        if (!isset($data['dialcode']) || $data['dialcode'] == "") {
            $data['dialcode'] = env('USER_DEFAULT_DIALCODE', '91');
        }
        if (!isset($data['country']) || $data['country'] == "") {
            $data['country'] = env('USER_DEFAULT_COUNTRY');
        }

        // Whatsapp no and dial code
        if (!empty($data['whatsapp_full_number'])) {
            $wpNumberSize = strlen($data['whatsapp_number']);
            $wpDialcode = substr($data['whatsapp_full_number'], 0, -$wpNumberSize);
            $data['whatsapp_dialcode'] = ltrim($wpDialcode, '+');
        }
        $ref_by = 0;
        if (isset($data['referral_code'])) {
            \Log::debug("Referral code used: " . $data['referral_code']);
            $refUser = User::where('refcode', $data['referral_code'])->where('active', '1')->first(['id', 'email']);
            if (!$refUser) {
                \Log::debug("Referral code used: Invalid Referral Code!");
                throw new GeneralException('Invalid Referral Code!');
            }
            $ref_by = $refUser->id;
            \Log::debug("Referral code used: Referred by- ".$ref_by);
        }
        $data['ref_by'] = $ref_by;
        $data['refcode'] = generate_refcode($data['first_name']);

        $data['lead_status'] = get_default_lead_status();
        return DB::transaction(function () use ($data) {
            $user = parent::create([
                'first_name'        => $data['first_name'],
                'last_name'        => $data['last_name'],
                'email'             => $data['email'],
                'dialcode'          => $data['dialcode'],
                'country'          => $data['country'],
                'mobile'            => $data['mobile'],
                'whatsapp_dialcode' => $data['whatsapp_dialcode'] ?? $data['dialcode'],
                'whatsapp' => $data['whatsapp_number'] ?? $data['mobile'],
                'is_lead'           => 1,
                'created_mode' =>   "front_end",
                'lead_status'       => $data['lead_status'],
                'lead_source'       => 'Website',
                'confirmation_code' => md5(uniqid(mt_rand(), true)),
                'active'            => 1,
                'overseas_status' => get_default_student_status(),
                'password'          => $data['password'],
                'refcode'           => $data['refcode'],
                //'agent_id'            => $data['ref_by'],
                'ref_by'            => $data['ref_by'],
                // If users require approval or needs to confirm email
                'confirmed'         => config('access.users.requires_approval') || config('access.users.confirm_email') ? 0 : 1,
            ]);

            if ($user) {
                /*
                 * Add the default site role to the new user
                 */
                $user->assignRole(config('access.users.default_role'));

                $userdetails = new UserDetails([
                    'userid'    => $user->id,
                ]);
                $userdetails->save();                

            }

            /*
             * If users have to confirm their email and this is not a social account,
             * and the account does not require admin approval
             * send the confirmation email
             *
             * If this is a social account they are confirmed through the social provider by default
             */
            if (config('access.users.confirm_email')) {
                // Pretty much only if account approval is off, confirm email is on, and this isn't a social account.
                try {
                    //$user->notify(new UserNeedsConfirmation($user->confirmation_code));
                    $email_message = __('strings.emails.auth.click_to_confirm');
                    $email_message .= new HtmlString("<a style='font-family:Avenir,Helvetica,sans-serif;box-sizing:border-box;border-radius:3px;color:#fff;display:inline-block;text-decoration:none;background-color:#3097d1;border-top:10px solid #3097d1;border-right:18px solid #3097d1;border-bottom:10px solid #3097d1;border-left:18px solid #3097d1' href='".route('frontend.auth.account.confirm', $user->confirmation_code)."'>Confirm Account</a>");
                    user_notify($user, app_name().': '.__('exceptions.frontend.auth.confirmation.confirm'), $email_message);
                } catch (\Exception $e) {
                    $message = $e->getMessage();
                    \Log::debug("Mail Error: (Confirm Email) " . $message);
                }
            }

            /*
             * Return the user object
             */
            return $user;
        });
    }

    public function createAgent(array $data)
    {

        if (isset($data['full_number']) || $data['full_number'] == "") {
            $phoneNumberSize = strlen($data['mobile']);
            $dialcode = substr($data['full_number'], 0, -$phoneNumberSize);
            $dialcodeRemovePlus = substr($dialcode, 1);
            $data['dialcode'] = $dialcodeRemovePlus;
        }
        if (!isset($data['dialcode']) || $data['dialcode'] == "") {
            $data['dialcode'] = env('USER_DEFAULT_DIALCODE');
        }

        if (!isset($data['country']) || $data['country'] == "") {
            $data['country'] = env('USER_DEFAULT_COUNTRY');
        }

        return DB::transaction(function () use ($data) {
            $user = parent::create([
                'first_name'        => $data['first_name'],
                'last_name'        => $data['last_name'],
                'is_agent' => 1,
                'email'             => $data['email'],
                'dialcode'          => $data['dialcode'],
                'country'          => $data['country'],
                'mobile'             => $data['mobile'],
                'branch'             => 1,
                'confirmation_code' => md5(uniqid(mt_rand(), true)),
                'active'            => 0,
                'password'          => $data['password'],
                // If users require approval or needs to confirm email
                'confirmed'         => config('access.users.requires_approval') || config('access.users.confirm_email') ? 0 : 1,
            ]);

            if ($user) {
                /*
                 * Add the default site role to the new user
                 */
                $user->assignRole('agent');
                StaffBranch::create(["staff_id" => $user->id, "branch_id" => 1]);
            }

            /*
             * If users have to confirm their email and this is not a social account,
             * and the account does not require admin approval
             * send the confirmation email
             *
             * If this is a social account they are confirmed through the social provider by default
             */
            if (config('access.users.confirm_email')) {
                // Pretty much only if account approval is off, confirm email is on, and this isn't a social account.
                try {
                    //$user->notify(new UserNeedsConfirmation($user->confirmation_code));
                    $email_message = __('strings.emails.auth.click_to_confirm');
                        $email_message .= new HtmlString("<a style='font-family:Avenir,Helvetica,sans-serif;box-sizing:border-box;border-radius:3px;color:#fff;display:inline-block;text-decoration:none;background-color:#3097d1;border-top:10px solid #3097d1;border-right:18px solid #3097d1;border-bottom:10px solid #3097d1;border-left:18px solid #3097d1' href='".route('frontend.auth.account.confirm', $user->confirmation_code)."'>Confirm Account</a>");
                        user_notify($user, app_name().': '.__('exceptions.frontend.auth.confirmation.confirm'), $email_message);
                } catch (\Exception $e) {
                    $message = $e->getMessage();
                    \Log::debug("Mail Error: (Create Agent - Confirm Email) " . $message);
                }
            }

            /*
             * Return the user object
             */
            return $user;
        });
    }


    /**
     * @param       $id
     * @param array $input
     * @param bool|UploadedFile  $image
     *
     * @return array|bool
     * @throws GeneralException
     */
    public function update($id, array $input, $image = false)
    {
        $user = $this->getById($id);
        $user->first_name = $input['first_name'];
        $user->last_name = $input['last_name'];
        $user->avatar_type = $input['avatar_type'];

        // Upload profile image if necessary
        if ($image) {
            $user->avatar_location = $image->store('/avatars', 'public');
        } else {
            // No image being passed
            if ($input['avatar_type'] == 'storage') {
                // If there is no existing image
                if (!strlen(auth()->user()->avatar_location)) {
                    throw new GeneralException('You must supply a profile image.');
                }
            } else {
                // If there is a current image, and they are not using it anymore, get rid of it
                if (strlen(auth()->user()->avatar_location)) {
                    Storage::disk('public')->delete(auth()->user()->avatar_location);
                }

                $user->avatar_location = null;
            }
        }

        if ($user->canChangeEmail()) {
            //Address is not current address so they need to reconfirm
            if ($user->email != $input['email']) {
                //Emails have to be unique
                if ($this->getByColumn($input['email'], 'email')) {
                    throw new GeneralException(__('exceptions.frontend.auth.email_taken'));
                }

                // Force the user to re-verify his email address if config is set
                if (config('access.users.confirm_email')) {
                    $user->confirmation_code = md5(uniqid(mt_rand(), true));
                    $user->confirmed = 0;
                    try {
                        //$user->notify(new UserNeedsConfirmation($user->confirmation_code));
                        $email_message = __('strings.emails.auth.click_to_confirm');
                        $email_message .= new HtmlString("<a style='font-family:Avenir,Helvetica,sans-serif;box-sizing:border-box;border-radius:3px;color:#fff;display:inline-block;text-decoration:none;background-color:#3097d1;border-top:10px solid #3097d1;border-right:18px solid #3097d1;border-bottom:10px solid #3097d1;border-left:18px solid #3097d1' href='".route('frontend.auth.account.confirm', $user->confirmation_code)."'>Confirm Account</a>");
                        user_notify($user, app_name().': '.__('exceptions.frontend.auth.confirmation.confirm'), $email_message);
                    } catch (\Exception $e) {
                        $message = $e->getMessage();
                        \Log::debug("Mail Error: (User Update - confirm email) " . $message);
                    }
                }
                $user->email = $input['email'];
                $updated = $user->save();

                // Send the new confirmation e-mail

                return [
                    'success' => $updated,
                    'email_changed' => true,
                ];
            }
        }

        return $user->save();
    }

    /**
     * @param      $input
     * @param bool $expired
     *
     * @return bool
     * @throws GeneralException
     */
    public function updatePassword($input, $expired = false)
    {
        $user = $this->getById(auth()->id());

        if (Hash::check($input['old_password'], $user->password)) {
            if ($expired) {
                $user->password_changed_at = Carbon::now()->toDateTimeString();
            }

            return $user->update(['password' => $input['password']]);
        }

        throw new GeneralException(__('exceptions.frontend.auth.password.change_mismatch'));
    }

    /**
     * @param $code
     *
     * @return bool
     * @throws GeneralException
     */
    public function confirm($code)
    {
        $user = $this->findByConfirmationCode($code);

        if ($user->confirmed == 1) {
            throw new GeneralException(__('exceptions.frontend.auth.confirmation.already_confirmed'));
        }

        if ($user->confirmation_code == $code) {

            //email        
            \Log::debug("Email Confirmed: sending welcome mail");
            $mail_tpl = MailTemplates::find("STUDENT_WELCOME");
            $email_message = $mail_tpl->message;
            //\Log::debug("MAIL BEFORE:".$email_message);                       
            $email_message = str_ireplace("#sitename#", app_name(), $email_message);
            $email_message = str_ireplace("#siteurl#", url(''), $email_message);
            $email_message = str_ireplace("#first_name#", $user->first_name, $email_message);
            $email_message = str_ireplace("#full_name#", $user->full_name, $email_message);
            $email_message = str_ireplace("#email#", $user->email, $email_message);
            $email_message = str_ireplace("#mobile#", $user->mobile, $email_message);
            //\Log::debug("MAIL TPL FINAL:".$email_message);
            try {
                //$user->notify(new UserAlert($mail_tpl->subject, $email_message));
                user_notify($user,$mail_tpl->subject, $email_message);
            } catch (\Exception $e) {
                $message = $e->getMessage();
                \Log::debug("Mail Error: (Welcome Email) " . $message);
            }
            //sms
            \Log::debug("Email Confirmed: sending sms");
            $notify_sms = get_sms_template('STUDENT_WELCOME');
            if ($notify_sms) {
                $notify_sms->message = str_replace("{#var#}", "%s", $notify_sms->message);
                $notify_sms->message = sprintf($notify_sms->message, $user->first_name);
                $user->sms($notify_sms->message, $notify_sms->template_id);
            }

            $user->confirmed = 1;

            event(new UserConfirmed($user));

            return $user->save();
        }

        throw new GeneralException(__('exceptions.frontend.auth.confirmation.mismatch'));
    }

    /**
     * @param $data
     * @param $provider
     *
     * @return mixed
     * @throws GeneralException
     */
    public function findOrCreateProvider($data, $provider)
    {
        // User email may not provided.
        $user_email = $data->email ?: "{$data->id}@{$provider}.com";

        // Check to see if there is a user with this email first.
        $user = $this->getByColumn($user_email, 'email');

        /*
         * If the user does not exist create them
         * The true flag indicate that it is a social account
         * Which triggers the script to use some default values in the create method
         */
        if (!$user) {
            // Registration is not enabled
            if (!config('access.registration')) {
                throw new GeneralException(__('exceptions.frontend.auth.registration_disabled'));
            }

            // Get users first name and last name from their full name
            $nameParts = $this->getNameParts($data->getName());

            $user = parent::create([
                'first_name'  => $nameParts['first_name'],
                'last_name'  => $nameParts['last_name'],
                'email' => $user_email,
                'active' => 1,
                'confirmed' => 1,
                'password' => null,
                'avatar_type' => $provider,
            ]);

            event(new UserProviderRegistered($user));
        }

        // See if the user has logged in with this social account before
        if (!$user->hasProvider($provider)) {
            // Gather the provider data for saving and associate it with the user
            $user->providers()->save(new SocialAccount([
                'provider'    => $provider,
                'provider_id' => $data->id,
                'token'       => $data->token,
                'avatar'      => $data->avatar,
            ]));
        } else {
            // Update the users information, token and avatar can be updated.
            $user->providers()->update([
                'token'       => $data->token,
                'avatar'      => $data->avatar,
            ]);

            $user->avatar_type = $provider;
            $user->update();
        }

        // Return the user object
        return $user;
    }

    /**
     * @param $fullName
     *
     * @return array
     */
    protected function getNameParts($fullName)
    {
        $parts = array_values(array_filter(explode(' ', $fullName)));
        $size = count($parts);
        $result = [];

        if (empty($parts)) {
            $result['first_name'] = null;
            $result['last_name'] = null;
        }

        if (!empty($parts) && $size == 1) {
            $result['first_name'] = $parts[0];
            $result['last_name'] = null;
        }

        if (!empty($parts) && $size >= 2) {
            $result['first_name'] = $parts[0];
            $result['last_name'] = $parts[1];
        }

        return $result;
    }

    //san

    public function updateBasicInfo($id, array $input, $image = false)
    {
        if (!isset($input['dialcode']) || $input['dialcode'] == "") {
            $input['dialcode'] = env('USER_DEFAULT_DIALCODE', '91');
        }
        
        $user = $this->getById($id);
        $user->first_name = $input['first_name'];
        $user->last_name = $input['last_name'];
        $user->middle_name = $input['middle_name'];

        $user->gender = $input['gender'];
        $user->email = $input['email'];

        $user->dialcode = $input['mobile_hidden'];
        $user->mobile = $input['mobile'];
        $user->whatsapp_dialcode = $input['whatsapp_number_hidden'];
        $user->whatsapp = $input['whatsapp_number'];
        // $user->phone_dialcode = $input['phone_number_hidden'];
        // $user->phone_number = $input['phone_number'];

        $user->country = $input['country'];
        $user->state = $input['state'];
        $user->city = $input['city'];

        if (isset($input['student_type'])) {
            $user->student_type = $input['student_type'];
        }
        $user->avatar_type = 'storage';
        // Upload profile image if necessary
        if ($image) {
            $extension = $image->extension();
            if (in_array($extension, array('jpg', 'jpeg', 'png'))) {
                if ($image->getSize() < 2048000) { //max size 2MB
                    $filename = pathinfo($image->getClientOriginalName(), PATHINFO_FILENAME);
                    if (strlen($filename) > 100) {
                        $filename = substr($filename, 0, 100);
                    }
                    $filename = $filename . Str::random(10) . "." . $image->getClientOriginalExtension();
                    $user->avatar_location = $image->storeAs('/avatars', $filename, 'public');
                } else {
                    throw new GeneralException('Profile pic size must not exceed 2MB');
                }
            } else {
                throw new GeneralException('You must upload a valid profile pic (jpg/png).');
            }
        } else {
            // If there is no existing image
            //if (! strlen(auth()->user()->avatar_location)) {
            //    throw new GeneralException('You must supply a profile image.');
            //}
        }
        $custom_attributes = DB::table('custom_attributes')->where('for_student', 1)->orderBy('display_order','asc')->get(["custom_attributes.*"]);
                $attributes_json = "";
                if(count($custom_attributes)){
                    $all_atts = array();
                    foreach($custom_attributes as $attrs){
                        $name = $attrs->attr_name;
                        $all_atts[$attrs->attr_name] = (isset($input[$name]))? $input[$name] : null;
                    }
                $attributes_json = json_encode($all_atts);
                }
        $user_details = UserDetails::where('userid', $id)->first();
        $user_details->user_data_js = $attributes_json;
        $user_details->save();
        return $user->save();
    }


    public function updatePaymentPreference($id, array $input)
    {
        $user = $this->getById($id);
        $user->first_name = $input['first_name'];
        $user->last_name = $input['last_name'];
        $user->avatar_type = $input['avatar_type'];

        // Upload profile image if necessary
        if ($image) {
            $user->avatar_location = $image->store('/avatars', 'public');
        } else {
            // No image being passed
            if ($input['avatar_type'] == 'storage') {
                // If there is no existing image
                if (!strlen(auth()->user()->avatar_location)) {
                    throw new GeneralException('You must supply a profile image.');
                }
            } else {
                // If there is a current image, and they are not using it anymore, get rid of it
                if (strlen(auth()->user()->avatar_location)) {
                    Storage::disk('public')->delete(auth()->user()->avatar_location);
                }

                $user->avatar_location = null;
            }
        }



        return $user->save();
    }

}
